<?php
/*
Plugin Name: Mailsoftly Form Embed
Description: This plugin allows users to embed forms from Mailsoftly into their WordPress site.
Version: 1.3
Author: Mailsoftly Dev.
License: GPLv2 or later
License URI: https://www.gnu.org/licenses/gpl-2.0.html
*/

function mailsoftly_plugin_activation() {
    global $wpdb;
    $table_name = $wpdb->prefix . 'mailsoftly_api_keys';
    $charset_collate = $wpdb->get_charset_collate();

    $sql = "CREATE TABLE $table_name (
        id INT(11) NOT NULL AUTO_INCREMENT,
        api_key VARCHAR(255) NOT NULL,
        PRIMARY KEY (id)
    ) $charset_collate;";

    require_once ABSPATH . 'wp-admin/includes/upgrade.php';
    dbDelta($sql);
}
register_activation_hook(__FILE__, 'mailsoftly_plugin_activation');

function mailsoftly_plugin_deactivation() {
    global $wpdb;
    $table_name = esc_sql($wpdb->prefix . 'mailsoftly_api_keys');
    $wpdb->query("DROP TABLE IF EXISTS `$table_name`");
}
register_deactivation_hook(__FILE__, 'mailsoftly_plugin_deactivation');

function mailsoftly_save_api_key_to_db($api_key) {
    global $wpdb;
    $table_name = $wpdb->prefix . 'mailsoftly_api_keys';
    $sanitized_api_key = sanitize_text_field($api_key);

    $wpdb->replace(
        $table_name,
        array(
            'id' => 1,
            'api_key' => $sanitized_api_key
        ),
        array('%d', '%s')
    );
}


function mailsoftly_delete_api_key_from_db() {
    global $wpdb;
    $table_name = $wpdb->prefix . 'mailsoftly_api_keys';
    $wpdb->delete($table_name, array('id' => 1), array('%d'));
}


function mailsoftly_get_api_key_from_db() {
    global $wpdb;
    $table_name = $wpdb->prefix . 'mailsoftly_api_keys';
    return $wpdb->get_var($wpdb->prepare("SELECT api_key FROM $table_name WHERE id = %d", 1));
}


function mailsoftly_verify_api_key($api_key) {
    $sanitized_api_key = sanitize_text_field($api_key);
    $response = wp_remote_get("https://app.mailsoftly.com/forms_meta_data?api_key=" . urlencode($sanitized_api_key));
    if (is_wp_error($response)) return false;
    $body = wp_remote_retrieve_body($response);
    $data = json_decode($body, true);
    return !isset($data['error']);
}


function mailsoftly_enqueue_thickbox() { add_thickbox(); }
add_action('admin_enqueue_scripts', 'mailsoftly_enqueue_thickbox');


function mailsoftly_admin_page() {
    if (
        isset($_SERVER['REQUEST_METHOD']) &&
        $_SERVER['REQUEST_METHOD'] === 'POST' &&
        isset($_POST['api_key']) &&
        isset($_POST['mailsoftly_nonce']) &&
        wp_verify_nonce(sanitize_text_field(wp_unslash($_POST['mailsoftly_nonce'])), 'mailsoftly_save_api_key')
    ) {
        $api_key = sanitize_text_field(wp_unslash($_POST['api_key']));
        if (mailsoftly_verify_api_key($api_key)) {
            mailsoftly_save_api_key_to_db($api_key);
            echo '<div class="updated"><p>API Key has been verified and saved!</p></div>';
            echo '<script>window.location.href = window.location.href;</script>';
        } else {
            mailsoftly_delete_api_key_from_db();
            echo '<div class="error"><p>API Key could not be verified, please try again.</p></div>';
        }
    }

    $saved_api_key = mailsoftly_get_api_key_from_db();
?>
<div class="wrap">
    <div class="ms-plugin">
        <a href="https://mailsoftly.com" target="_blank">
            <img src="https://mailsoftly.com/wp-content/uploads/2024/02/mailsoftly_logo_3.png" style="width:200px;" alt="">
        </a>
        <span>Form Embed Plugin</span>
    </div>

    <style>
        .ms-plugin {
            width: 100%;
            border-bottom: 1px solid gray;
            display: flex;
            flex-direction: column;
            padding-bottom: .5rem;
            margin-bottom: 1rem;
        }
    </style>

    <?php if (!$saved_api_key) : ?>
        <p>To use your Mailsoftly Forms on your website, you need to obtain an API key from your Mailsoftly account. Don't have an account? <a href="https://app.mailsoftly.com/tr/users/sign_up" target="_blank">Get a free account.</a></p>
        <form method="POST">
            <?php wp_nonce_field('mailsoftly_save_api_key', 'mailsoftly_nonce'); ?>
            <label for="api_key">Your API Key:</label>
            <input type="text" id="api_key" name="api_key" placeholder="API Key" required />
            <button type="submit" class="button button-primary">Submit API Key</button>
        </form>
    <?php else : ?>
        <h2>Your Forms</h2>
        <table id="forms-table" style="width:100%; border-collapse:collapse;">
            <thead style="border-bottom:1px solid black;">
                <tr>
                    <th style="text-align:left;">Form Name</th>
                    <th style="text-align:left;">Form Code (Shortcode)</th>
                    <th style="text-align:left;">Action</th>
                </tr>
            </thead>
            <tbody id="forms-list">
                <tr><td colspan="3">Loading Forms...</td></tr>
            </tbody>
        </table>
    <?php endif; ?>
</div>


<div id="form-modal" style="display:none; position:fixed; top:50%; left:50%; transform:translate(-50%, -50%); z-index:1000; background:white; padding:20px; width:400px; border-radius:8px; box-shadow:0 4px 8px rgba(0,0,0,0.1);">
    <div id="form-modal-content"></div>
</div>
<div id="form-modal-overlay" style="display:none; position:fixed; top:0; left:0; width:100%; height:100%; background:rgba(0,0,0,0.5); z-index:999;"></div>

<script>
document.addEventListener('DOMContentLoaded', function() {
    const formsList = document.getElementById('forms-list');
    fetch(ajaxurl + '?action=mailsoftly_get_forms')
        .then(response => response.json())
        .then(result => {
            if (!result.success) throw new Error(result.data);
            const data = result.data;
            formsList.innerHTML = '';
            if (data.length === 0) {
            formsList.innerHTML = `
                <tr>
                    <td colspan="3">
                        No forms found. 
                        <a href="https://app.mailsoftly.com/tr/soft_forms" target="_blank" style="margin-left: 6px; text-decoration: underline;">
                            Create one
                        </a>
                    </td>
                </tr>`;
                    return;
            }

            data.forEach(form => {
                const formName = form.name && form.name.trim() !== '' ? form.name : 'Unnamed Form';
                const shortcode = `[mailsoftly_form id="${form.id}"]`;
                formsList.innerHTML += `
                    <tr style="border-bottom:1px solid gray;">
                        <td>${formName}</td>
                        <td><code>${shortcode}</code></td>
                        <td>
                            <button class="button button-primary use-this-form" data-form-code='${shortcode}'>Use</button>
                            <button class="button button-secondary preview-form" data-form-id='${form.id}'>Preview</button>
                        </td>
                    </tr>`;
            });

            document.querySelectorAll('.use-this-form').forEach(btn => {
                btn.addEventListener('click', function() {
                    const code = this.getAttribute('data-form-code');
                    document.getElementById('form-modal-content').innerHTML = `
                        <h2>Form Code</h2>
                        <p>Copy this shortcode and paste it into your page:</p>
                        <textarea style="width:100%;height:100px;" readonly>${code}</textarea>
                        <button id="close-modal" class="button button-secondary" style="margin-top:10px;">Close</button>`;
                    showModal();
                });
            });

            document.querySelectorAll('.preview-form').forEach(btn => {
                btn.addEventListener('click', function() {
                    const formId = this.getAttribute('data-form-id');
                    fetch(ajaxurl + '?action=mailsoftly_get_form_data&form_id=' + formId)
                        .then(response => response.json())
                        .then(result => {
                            if (!result.success) throw new Error(result.data);
                            const data = result.data;
                            document.getElementById('form-modal-content').innerHTML = `
                                <h2>Preview of "${data.name}"</h2>
                                <iframe width='400' height='400' style='border:none;' src='https://app.mailsoftly.com/soft_forms/${data.id}'></iframe>
                                <button id="close-modal" class="button button-secondary" style="margin-top:10px;">Close</button>`;
                            showModal();
                        })
                        .catch(err => {
                            alert('Error fetching form: ' + err.message);
                        });
                });
            });

            function showModal() {
                document.getElementById('form-modal').style.display = 'block';
                document.getElementById('form-modal-overlay').style.display = 'block';
                document.getElementById('close-modal').addEventListener('click', hideModal);
                document.getElementById('form-modal-overlay').addEventListener('click', hideModal);
            }

            function hideModal() {
                document.getElementById('form-modal').style.display = 'none';
                document.getElementById('form-modal-overlay').style.display = 'none';
            }
        })
        .catch(error => {
            console.error('Error fetching forms:', error);
            formsList.innerHTML = '<tr><td colspan="3">Error loading forms.</td></tr>';
        });
});
</script>
<?php
}

function mailsoftly_form_embed_menu() {
    add_menu_page(
        'Mailsoftly Form Embed',
        'Mailsoftly Embed',
        'manage_options',
        'mailsoftly-form-embed',
        'mailsoftly_admin_page',
        'dashicons-email-alt',
        80
    );
}
add_action('admin_menu', 'mailsoftly_form_embed_menu');

function mailsoftly_form_shortcode_handler($atts) {
    $atts = shortcode_atts(array('id' => ''), $atts);
    $api_key = mailsoftly_get_api_key_from_db();
    if (!$api_key) return '<p>API Key not found</p>';

    $response = wp_remote_get("https://app.mailsoftly.com/form_data?api_key=" . urlencode($api_key) . "&form_id=" . urlencode($atts['id']));
    if (is_wp_error($response)) return '<p>Error: ' . $response->get_error_message() . '</p>';
    $body = wp_remote_retrieve_body($response);
    $form_data = json_decode($body, true);
    if (isset($form_data['form_data'])) return $form_data['form_data'];
    return '<p>Form not found</p>';
}
add_shortcode('mailsoftly_form', 'mailsoftly_form_shortcode_handler');

add_action('wp_ajax_mailsoftly_get_forms', 'mailsoftly_get_forms_via_server');
function mailsoftly_get_forms_via_server() {
    $api_key = mailsoftly_get_api_key_from_db();
    if (!$api_key) wp_send_json_error('API key not found');
    $response = wp_remote_get("https://app.mailsoftly.com/forms_meta_data?api_key=" . urlencode($api_key));
    if (is_wp_error($response)) wp_send_json_error($response->get_error_message());
    $body = wp_remote_retrieve_body($response);
    wp_send_json_success(json_decode($body, true));
}

add_action('wp_ajax_mailsoftly_get_form_data', 'mailsoftly_get_form_data_via_server');
function mailsoftly_get_form_data_via_server() {
    $api_key = mailsoftly_get_api_key_from_db();
    $form_id = isset($_GET['form_id']) ? sanitize_text_field($_GET['form_id']) : null;
    if (!$api_key || !$form_id) wp_send_json_error('Missing parameters');
    $response = wp_remote_get("https://app.mailsoftly.com/form_data?api_key=" . urlencode($api_key) . "&form_id=" . urlencode($form_id));
    if (is_wp_error($response)) wp_send_json_error($response->get_error_message());
    $body = wp_remote_retrieve_body($response);
    wp_send_json_success(json_decode($body, true));
}
