<?php
/**
 * Plugin Name: MailMoo for Mailgun
 * Plugin URI: https://wordpress.org/plugins/mailmoo-for-mailgun/
 * Description: A simple, no bloat, no paid features email sending plugin using Mailgun API for reliable email delivery.
 * Version: 1.0.1
 * Author: creatorcow
 * Author URI: https://creatorcow.com
 * License: MIT
 * Text Domain: mailmoo-for-mailgun
 * Requires at least: 6.5
 * Tested up to: 6.9.1
 * Requires PHP: 7.4
 */

// Prevent direct access
if (!defined('ABSPATH')) {
    exit;
}

// Plugin constants
define('MAILMOO_MAILGUN_VERSION', '1.0.0');
define('MAILMOO_MAILGUN_PLUGIN_URL', plugin_dir_url(__FILE__));
define('MAILMOO_MAILGUN_PLUGIN_PATH', plugin_dir_path(__FILE__));

// Include Composer autoloader if available
if (file_exists(MAILMOO_MAILGUN_PLUGIN_PATH . 'vendor/autoload.php')) {
    require_once MAILMOO_MAILGUN_PLUGIN_PATH . 'vendor/autoload.php';
}

/**
 * Main MailMoo Mailgun class
 */
class MailMooMailgun {
    
    /**
     * Constructor
     */
    public function __construct() {
        add_action('init', array($this, 'init'));
        add_action('admin_menu', array($this, 'admin_menu'));
        add_action('admin_init', array($this, 'admin_init'));
        add_action('admin_enqueue_scripts', array($this, 'admin_enqueue_scripts'));
        add_action('wp_mail_failed', array($this, 'log_failed_email'));
        add_filter('pre_wp_mail', array($this, 'send_via_mailgun'), 10, 2);
        add_action('wp_ajax_mailmoo_for_mailgun_get_log_details', array($this, 'get_log_details'));
        add_action('admin_notices', array($this, 'admin_notices'));
        
        // Activation and deactivation hooks
        register_activation_hook(__FILE__, array($this, 'activate'));
        register_deactivation_hook(__FILE__, array($this, 'deactivate'));
    }
    
    /**
     * Initialize plugin
     */
    public function init() {
        // Translation loading is handled automatically by WordPress for plugins hosted on WordPress.org
    }
    
    /**
     * Plugin activation
     */
    public function activate() {
        $this->create_email_log_table();
        
        // Set default options
        $defaults = array(
            'mailmoo_for_mailgun_api_key' => '',
            'mailmoo_for_mailgun_domain' => '',
            'mailmoo_for_mailgun_from_email' => get_option('admin_email'),
            'mailmoo_for_mailgun_from_name' => get_option('blogname'),
            'mailmoo_for_mailgun_reply_to_email' => '',
            'mailmoo_for_mailgun_reply_to_name' => '',
            'mailmoo_for_mailgun_enable_logging' => 'true'
        );
        
        foreach ($defaults as $key => $value) {
            if (get_option($key) === false) {
                add_option($key, $value);
            }
        }
    }
    
    /**
     * Plugin deactivation
     */
    public function deactivate() {
        // Cleanup if needed
    }
    
    /**
     * Create email log table
     */
    public function create_email_log_table() {
        global $wpdb;
        
        $table_name = $wpdb->prefix . 'mailmoo_for_mailgun_email_logs';
        
        $charset_collate = $wpdb->get_charset_collate();
        
        $sql = "CREATE TABLE $table_name (
            id mediumint(9) NOT NULL AUTO_INCREMENT,
            timestamp datetime DEFAULT CURRENT_TIMESTAMP NOT NULL,
            to_email varchar(255) NOT NULL,
            subject varchar(255) NOT NULL,
            message longtext NOT NULL,
            headers text,
            attachments text,
            status varchar(20) DEFAULT 'sent' NOT NULL,
            error_message text,
            mailgun_message_id varchar(255),
            PRIMARY KEY (id)
        ) $charset_collate;";
        
        require_once(ABSPATH . 'wp-admin/includes/upgrade.php');
        dbDelta($sql);
    }
    
    /**
     * Add admin menu
     */
    public function admin_menu() {
        add_options_page(
            __('MailMoo for Mailgun Settings', 'mailmoo-for-mailgun'),
            __('MailMoo for Mailgun', 'mailmoo-for-mailgun'),
            'manage_options',
            'mailmoo-for-mailgun-settings',
            array($this, 'admin_page')
        );
        
        add_submenu_page(
            'options-general.php',
            __('Mailgun Email Logs', 'mailmoo-for-mailgun'),
            __('Mailgun Email Logs', 'mailmoo-for-mailgun'),
            'manage_options',
            'mailmoo-for-mailgun-logs',
            array($this, 'logs_page')
        );
    }
    
    /**
     * Initialize admin settings
     */
    public function admin_init() {
        register_setting('mailmoo_for_mailgun_settings', 'mailmoo_for_mailgun_api_key', array(
            'sanitize_callback' => 'sanitize_text_field'
        ));
        register_setting('mailmoo_for_mailgun_settings', 'mailmoo_for_mailgun_domain', array(
            'sanitize_callback' => 'sanitize_text_field'
        ));
        register_setting('mailmoo_for_mailgun_settings', 'mailmoo_for_mailgun_from_email', array(
            'sanitize_callback' => 'sanitize_email'
        ));
        register_setting('mailmoo_for_mailgun_settings', 'mailmoo_for_mailgun_from_name', array(
            'sanitize_callback' => 'sanitize_text_field'
        ));
        register_setting('mailmoo_for_mailgun_settings', 'mailmoo_for_mailgun_reply_to_email', array(
            'sanitize_callback' => 'sanitize_email'
        ));
        register_setting('mailmoo_for_mailgun_settings', 'mailmoo_for_mailgun_reply_to_name', array(
            'sanitize_callback' => 'sanitize_text_field'
        ));
        register_setting('mailmoo_for_mailgun_settings', 'mailmoo_for_mailgun_enable_logging', array(
            'sanitize_callback' => array($this, 'sanitize_enable_logging')
        ));
    }
    
    /**
     * Sanitize enable logging option
     */
    public function sanitize_enable_logging($value) {
        return in_array($value, array('true', 'false'), true) ? $value : 'true';
    }
    
    /**
     * Configure PHPMailer to prevent conflicts
     */
    public function configure_phpmailer($phpmailer) {
        // Don't interfere with PHPMailer if we're handling via Mailgun
        return;
    }
    
    /**
     * Send email via Mailgun API using pre_wp_mail filter
     */
    public function send_via_mailgun($null, $atts) {
        // Convert wp_mail arguments to our expected format
        $args = array(
            'to' => $atts['to'],
            'subject' => $atts['subject'],
            'message' => $atts['message'],
            'headers' => isset($atts['headers']) ? $atts['headers'] : '',
            'attachments' => isset($atts['attachments']) ? $atts['attachments'] : array()
        );
        if (!class_exists('Mailgun\Mailgun')) {
            $this->log_email($args, 'failed', 'Mailgun SDK is not available');
            return null; // Let WordPress handle it
        }
        
        $api_key = get_option('mailmoo_for_mailgun_api_key');
        $domain = get_option('mailmoo_for_mailgun_domain');
        if (empty($api_key) || empty($domain)) {
            $this->log_email($args, 'failed', 'Mailgun API key or domain not configured');
            return null; // Let WordPress handle it
        }
        
        try {
            $mailgun = \Mailgun\Mailgun::create($api_key);
            
            $from_email = get_option('mailmoo_for_mailgun_from_email', get_option('admin_email'));
            $from_name = get_option('mailmoo_for_mailgun_from_name', get_option('blogname'));
            
            // Handle multiple recipients
            $to_emails = is_array($args['to']) ? $args['to'] : array($args['to']);
            
            // Build message parameters
            $message_params = array(
                'from' => $from_name . ' <' . $from_email . '>',
                'to' => implode(', ', $to_emails),
                'subject' => $args['subject']
            );
            
            // Determine content type
            $content_type = 'text/html';
            if (!empty($args['headers'])) {
                $headers = is_array($args['headers']) ? $args['headers'] : explode("\n", $args['headers']);
                foreach ($headers as $header) {
                    if (strpos($header, ':') !== false) {
                        list($key, $value) = explode(':', $header, 2);
                        $key = trim($key);
                        $value = trim($value);
                        
                        if (strtolower($key) === 'content-type' && strpos($value, 'text/plain') !== false) {
                            $content_type = 'text/plain';
                            break;
                        }
                    }
                }
            }
            
            // Set message content
            if ($content_type === 'text/plain') {
                $message_params['text'] = $args['message'];
            } else {
                $message_params['html'] = $args['message'];
            }
            
            // Add reply-to if configured
            $reply_to_email = get_option('mailmoo_for_mailgun_reply_to_email');
            if (!empty($reply_to_email)) {
                $reply_to_name = get_option('mailmoo_for_mailgun_reply_to_name');
                $message_params['h:Reply-To'] = !empty($reply_to_name) ? 
                    $reply_to_name . ' <' . $reply_to_email . '>' : $reply_to_email;
            }
            
            // Handle attachments
            $attachments = array();
            if (!empty($args['attachments'])) {
                $attachment_files = is_array($args['attachments']) ? $args['attachments'] : array($args['attachments']);
                foreach ($attachment_files as $attachment) {
                    if (file_exists($attachment)) {
                        $attachments[] = array(
                            'filePath' => $attachment,
                            'filename' => basename($attachment)
                        );
                    }
                }
            }
            
            $response = $mailgun->messages()->send($domain, $message_params, $attachments);
            
            if ($response->getId()) {
                $this->log_email($args, 'sent', '', $response->getId());
                // Return true to indicate we handled the email successfully
                return true;
            } else {
                $this->log_email($args, 'failed', 'Mailgun API Error: Unknown error');
                return null; // Let WordPress handle it
            }
            
        } catch (Exception $e) {
            $error_message = 'Mailgun Exception: ' . $e->getMessage();
            if ($e instanceof \Mailgun\Exception\HttpClientException) {
                $error_message .= ' (HTTP Client Exception)';
            } elseif ($e instanceof \Mailgun\Exception\HttpServerException) {
                $error_message .= ' (HTTP Server Exception)';
            }
            $this->log_email($args, 'failed', $error_message);
            return null; // Let WordPress handle it
        }
    }
    
    /**
     * Log email sending
     */
    public function log_email($args, $status = 'sent', $error_message = '', $message_id = '') {
        if (get_option('mailmoo_for_mailgun_enable_logging') !== 'true') {
            return;
        }
        
        global $wpdb;
        $table_name = $wpdb->prefix . 'mailmoo_for_mailgun_email_logs';
        
        $to_email = is_array($args['to']) ? implode(', ', $args['to']) : $args['to'];
        $headers = is_array($args['headers']) ? implode("\n", $args['headers']) : $args['headers'];
        $attachments = is_array($args['attachments']) ? serialize($args['attachments']) : $args['attachments'];
        
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
        $wpdb->insert(
            $table_name,
            array(
                'to_email' => $to_email,
                'subject' => $args['subject'],
                'message' => $args['message'],
                'headers' => $headers,
                'attachments' => $attachments,
                'status' => $status,
                'error_message' => $error_message,
                'mailgun_message_id' => $message_id
            ),
            array('%s', '%s', '%s', '%s', '%s', '%s', '%s', '%s')
        );
    }
    
    /**
     * Log failed email
     */
    public function log_failed_email($wp_error) {
        if (get_option('mailmoo_for_mailgun_enable_logging') !== 'true') {
            return;
        }
        
        global $wpdb;
        $table_name = $wpdb->prefix . 'mailmoo_for_mailgun_email_logs';
        
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
        $wpdb->insert(
            $table_name,
            array(
                'to_email' => 'Unknown',
                'subject' => 'Failed Email',
                'message' => '',
                'status' => 'failed',
                'error_message' => $wp_error->get_error_message()
            ),
            array('%s', '%s', '%s', '%s', '%s')
        );
    }
    
    /**
     * Admin notices
     */
    public function admin_notices() {
        $api_key = get_option('mailmoo_for_mailgun_api_key');
        $domain = get_option('mailmoo_for_mailgun_domain');
        if (empty($api_key) || empty($domain)) {
            echo '<div class="notice notice-warning"><p>';
            echo esc_html__('MailMoo Mailgun: Please configure your Mailgun API key and domain in ', 'mailmoo-for-mailgun');
            echo '<a href="' . esc_url(admin_url('options-general.php?page=mailmoo-for-mailgun-settings')) . '">';
            echo esc_html__('settings', 'mailmoo-for-mailgun');
            echo '</a></p></div>';
        }
        
        if (!class_exists('Mailgun\Mailgun')) {
            echo '<div class="notice notice-error"><p>';
            echo esc_html__('MailMoo Mailgun: Mailgun PHP SDK is not available.', 'mailmoo-for-mailgun');
            echo '</p></div>';
        }
    }
    
    /**
     * Enqueue admin scripts and styles
     */
    public function admin_enqueue_scripts($hook) {
        // Only enqueue on our plugin pages
        if ($hook === 'settings_page_mailmoo-for-mailgun-settings' || $hook === 'settings_page_mailmoo-for-mailgun-logs') {
            // Enqueue admin styles
            wp_enqueue_style(
                'mailmoo-for-mailgun-admin',
                MAILMOO_MAILGUN_PLUGIN_URL . 'assets/css/admin.css',
                array(),
                MAILMOO_MAILGUN_VERSION
            );
            
            // Only enqueue scripts on logs page
            if ($hook === 'settings_page_mailmoo-for-mailgun-logs') {
                // Enqueue jQuery (included in WordPress)
                wp_enqueue_script('jquery');
                
                // Enqueue admin scripts
                wp_enqueue_script(
                    'mailmoo-for-mailgun-admin',
                    MAILMOO_MAILGUN_PLUGIN_URL . 'assets/js/admin.js',
                    array('jquery'),
                    MAILMOO_MAILGUN_VERSION,
                    true
                );
                
                // Localize script with AJAX URL and nonce
                wp_localize_script('mailmoo-for-mailgun-admin', 'mailmoo_ajax', array(
                    'ajax_url' => admin_url('admin-ajax.php'),
                    'nonce' => wp_create_nonce('mailmoo_for_mailgun_get_log_details')
                ));
            }
        }
    }
    
    /**
     * Admin settings page
     */
    public function admin_page() {
        include MAILMOO_MAILGUN_PLUGIN_PATH . 'includes/admin-page.php';
    }
    
    /**
     * Email logs page
     */
    public function logs_page() {
        include MAILMOO_MAILGUN_PLUGIN_PATH . 'includes/logs-page.php';
    }
    
    /**
     * AJAX handler for getting log details
     */
    public function get_log_details() {
        // Check if nonce exists and verify it
        if (!isset($_POST['nonce']) || !wp_verify_nonce(sanitize_text_field(wp_unslash($_POST['nonce'])), 'mailmoo_for_mailgun_get_log_details')) {
            wp_die('Security check failed');
        }
        
        if (!current_user_can('manage_options')) {
            wp_die('Insufficient permissions');
        }
        
        // Check if log_id exists and sanitize it
        if (!isset($_POST['log_id'])) {
            wp_send_json_error('Missing log ID');
        }
        
        $log_id = intval(sanitize_text_field(wp_unslash($_POST['log_id'])));
        
        global $wpdb;
        
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
        $log = $wpdb->get_row($wpdb->prepare(
            "SELECT * FROM `{$wpdb->prefix}mailmoo_for_mailgun_email_logs` WHERE id = %d",
            $log_id
        ));
        
        if (!$log) {
            wp_send_json_error('Log not found');
        }
        
        $html = '<div class="log-detail">';
        $html .= '<label>' . __('Timestamp:', 'mailmoo-for-mailgun') . '</label>';
        $html .= '<div class="log-detail-content">' . esc_html($log->timestamp) . '</div>';
        $html .= '</div>';
        
        $html .= '<div class="log-detail">';
        $html .= '<label>' . __('To:', 'mailmoo-for-mailgun') . '</label>';
        $html .= '<div class="log-detail-content">' . esc_html($log->to_email) . '</div>';
        $html .= '</div>';
        
        $html .= '<div class="log-detail">';
        $html .= '<label>' . __('Subject:', 'mailmoo-for-mailgun') . '</label>';
        $html .= '<div class="log-detail-content">' . esc_html($log->subject) . '</div>';
        $html .= '</div>';
        
        $html .= '<div class="log-detail">';
        $html .= '<label>' . __('Message:', 'mailmoo-for-mailgun') . '</label>';
        $html .= '<div class="log-detail-content">' . esc_html($log->message) . '</div>';
        $html .= '</div>';
        
        if (!empty($log->headers)) {
            $html .= '<div class="log-detail">';
            $html .= '<label>' . __('Headers:', 'mailmoo-for-mailgun') . '</label>';
            $html .= '<div class="log-detail-content">' . esc_html($log->headers) . '</div>';
            $html .= '</div>';
        }
        
        if (!empty($log->attachments)) {
            $html .= '<div class="log-detail">';
            $html .= '<label>' . __('Attachments:', 'mailmoo-for-mailgun') . '</label>';
            $html .= '<div class="log-detail-content">' . esc_html($log->attachments) . '</div>';
            $html .= '</div>';
        }
        
        $html .= '<div class="log-detail">';
        $html .= '<label>' . __('Status:', 'mailmoo-for-mailgun') . '</label>';
        $html .= '<div class="log-detail-content">' . esc_html($log->status) . '</div>';
        $html .= '</div>';
        
        if (!empty($log->mailgun_message_id)) {
            $html .= '<div class="log-detail">';
            $html .= '<label>' . __('Mailgun Message ID:', 'mailmoo-for-mailgun') . '</label>';
            $html .= '<div class="log-detail-content">' . esc_html($log->mailgun_message_id) . '</div>';
            $html .= '</div>';
        }
        
        if (!empty($log->error_message)) {
            $html .= '<div class="log-detail">';
            $html .= '<label>' . __('Error Message:', 'mailmoo-for-mailgun') . '</label>';
            $html .= '<div class="log-detail-content">' . esc_html($log->error_message) . '</div>';
            $html .= '</div>';
        }
        
        wp_send_json_success($html);
    }
}

// Initialize the plugin
new MailMooMailgun();
