<?php
/**
 * Plugin Name: MailcatcherClient
 * Plugin URI: https://mailcatcher.dev
 * Description: Redirects all emails to Mailcatcher for testing.
 * Version: 1.0.0
 * Requires at least: 6.0
 * Requires PHP: 8.0
 * Author: jop-software Inh. Johannes Przymusinski
 * Author URI: https://jop-software.de
 * License: GPL v3 or later
 */

// Exit if accessed directly
if (!defined('ABSPATH')) {
    exit;
}

class MailcatcherClient {

    // Default settings
    private $options = [
        'tenant_slug' => '',
        'project_name' => '',
        'project_password' => ''
    ];

    // Fixed SMTP settings
    private $smtp_host = 'mailcatcher.cloud';
    private $smtp_port = '1026';
    private $smtp_secure = ''; // No encryption

    public function __construct() {
        // Load options from database
        $saved_options = get_option('mailcatcher_options');
        if (is_array($saved_options)) {
            $this->options = array_merge($this->options, $saved_options);
        }

        // Hook into WordPress's mail function
        add_action('phpmailer_init', array($this, 'configure_smtp'));

        // Add admin menu
        add_action('admin_menu', array($this, 'add_admin_menu'));

        // Register settings
        add_action('admin_init', array($this, 'register_settings'));

        // Register and enqueue script on the correct hook
        add_action('admin_enqueue_scripts', array($this, 'enqueue_admin_scripts'));
    }

    // Configure PHPMailer to use SMTP
    public function configure_smtp($phpmailer) {
        // Only proceed if all required settings are present
        if (empty($this->options['tenant_slug']) ||
            empty($this->options['project_name']) ||
            empty($this->options['project_password'])) {
            return;
        }

        $phpmailer->isSMTP();
        $phpmailer->Host = $this->smtp_host;
        $phpmailer->Port = $this->smtp_port;

        // Build username from project name and tenant slug
        $username = $this->options['project_name'] . '.' . $this->options['tenant_slug'];

        $phpmailer->SMTPAuth = true;
        $phpmailer->Username = $username;
        $phpmailer->Password = $this->options['project_password'];

        // No encryption
        $phpmailer->SMTPSecure = $this->smtp_secure;
        $phpmailer->SMTPAutoTLS = false;
    }

    // Add plugin settings page to admin menu
    public function add_admin_menu() {
        add_options_page(
            'Mailcatcher Settings',
            'Mailcatcher',
            'manage_options',
            'mailcatcher',
            array($this, 'render_settings_page')
        );
    }

    // Register plugin settings
    public function register_settings() {
        register_setting(
            'mailcatcher_group',
            'mailcatcher_options',
            array($this, 'sanitize_options')
        );

        add_settings_section(
            'mailcatcher_section',
            'SMTP Configuration',
            array($this, 'section_info'),
            'mailcatcher'
        );

        add_settings_field(
            'tenant_slug',
            'Tenant Slug',
            array($this, 'render_text_field'),
            'mailcatcher',
            'mailcatcher_section',
            ['field' => 'tenant_slug', 'placeholder' => 'Your tenant slug']
        );

        add_settings_field(
            'project_name',
            'Project Name',
            array($this, 'render_text_field'),
            'mailcatcher',
            'mailcatcher_section',
            ['field' => 'project_name', 'placeholder' => 'Your project name']
        );

        add_settings_field(
            'project_password',
            'Project Password',
            array($this, 'render_password_field'),
            'mailcatcher',
            'mailcatcher_section',
            ['field' => 'project_password', 'placeholder' => 'Your project password']
        );
    }

    // Sanitize user inputs
    public function sanitize_options($input) {
        $sanitized = [];

        $sanitized['tenant_slug'] = sanitize_text_field($input['tenant_slug']);
        $sanitized['project_name'] = sanitize_text_field($input['project_name']);

        // Only update password if one was entered
        if (!empty($input['project_password'])) {
            $sanitized['project_password'] = $input['project_password'];
        } else {
            // Keep the old password if none was entered
            $old_options = get_option('mailcatcher_options');
            $sanitized['project_password'] = $old_options['project_password'];
        }

        return $sanitized;
    }

    // Render the settings page
    public function render_settings_page() {
        ?>
        <div class="wrap">
            <h1>Mailcatcher Configuration</h1>
            <form method="post" action="options.php">
                <?php
                settings_fields('mailcatcher_group');
                do_settings_sections('mailcatcher');
                submit_button();
                ?>
                <hr>
                <h2>Test Email</h2>
                <p>Send a test email to verify your settings:</p>
                <p class="submit">
                    <button type="button" id="send_test_email" data-nonce="<?php echo esc_js(wp_create_nonce('smtp_test_email_nonce')); ?>" class="button button-secondary">Send Test Email</button>
                    <span id="test_result" style="margin-left: 10px;"></span>
                </p>
            </form>
        </div>
        <?php
    }

    // Display section information
    public function section_info() {
        echo 'Enter your mailcatcher.cloud credentials below:';
    }

    // Render a text field
    public function render_text_field($args) {
        $field = $args['field'];
        $placeholder = isset($args['placeholder']) ? $args['placeholder'] : '';
        ?>
        <input type="text" name="mailcatcher_options[<?php echo esc_attr($field); ?>]"
               value="<?php echo esc_attr($this->options[$field]); ?>"
               class="regular-text"
               placeholder="<?php echo esc_attr($placeholder); ?>">
        <?php
    }

    // Render a password field
    public function render_password_field($args) {
        $field = $args['field'];
        $placeholder = isset($args['placeholder']) ? $args['placeholder'] : '';
        ?>
        <input type="password" name="mailcatcher_options[<?php echo esc_attr($field); ?>]"
               value="<?php echo $this->options[$field] ? '••••••••' : ''; ?>"
               class="regular-text"
               placeholder="<?php echo esc_attr($placeholder); ?>">
        <?php
    }

    // Enqueue admin scripts
    public function enqueue_admin_scripts($hook) {
        // Only enqueue on the plugin settings page
        if ($hook !== 'settings_page_mailcatcher') {
            return;
        }
        wp_register_script(
            'mailcatcher_send-test-email',
            plugins_url('js/mailcatcher_send-test-email.js', __FILE__),
            array('jquery'),
            '1.0.0',
            true
        );
        wp_enqueue_script('mailcatcher_send-test-email');
    }
}

// Initialize plugin
$mailcatcher = new MailcatcherClient();

// AJAX handler for test email
add_action('wp_ajax_mailcatcher_smtp_test_email', 'mailcatcher_smtp_test_email_callback');
function mailcatcher_smtp_test_email_callback() {
    // Check nonce
    if (!isset($_POST['nonce']) || !wp_verify_nonce(sanitize_text_field(wp_unslash($_POST['nonce'])), 'smtp_test_email_nonce')) {
        wp_send_json_error('Security check failed.');
    }

    $to = "no-reply@mailcatcher.dev";
    $subject = 'Mailcatcher works!';
    $message = 'This is a test email sent from the Mailcatcher Plugin in your WordPress installation! Your instance is ready to work with Mailcatcher.';

    // Configure PHPMailer to log errors
    add_action('wp_mail_failed', function($error) {
        wp_send_json_error($error->get_error_message());
    });

    // Send the email
    $result = wp_mail($to, $subject, $message);

    if ($result) {
        wp_send_json_success();
    } else {
        wp_send_json_error('Failed to send email. Check your SMTP settings.');
    }
}
