<?php
declare(strict_types=1);

/**
 * Helper Functions
 *
 * @package Resend\WordPress
 */

if (!defined('ABSPATH')) {
    exit;
}

/**
 * Send email via Resend (helper function)
 *
 * @param string|array<string> $to Email address(es)
 * @param string $subject Email subject
 * @param string $message Email message
 * @param string|array<string> $headers Optional. Additional headers
 * @param string|array<string> $attachments Optional. File paths
 * @return bool|WP_Error True on success, WP_Error on failure
 */
function resend_wp($to, string $subject, string $message, $headers = '', $attachments = []): bool|\WP_Error {
    return wp_mail($to, $subject, $message, $headers, $attachments);
}

/**
 * Get Resend API key (decrypted)
 *
 * Automatically migrates unencrypted keys to encrypted format on first access.
 *
 * @return string Empty string if no key is set or decryption fails
 */
function resend_get_api_key(): string {
    $encrypted_key = get_option('resend_api_key', '');

    if (empty($encrypted_key)) {
        if (resend_is_debug_mode()) {
            error_log('[Resend] API key is empty in database');
        }
        return '';
    }

    // Check if the key is already encrypted
    if (\Resend\WordPress\Encryption::is_encrypted($encrypted_key)) {
        $decrypted = \Resend\WordPress\Encryption::decrypt($encrypted_key);
        
        if (resend_is_debug_mode()) {
            if ($decrypted === false) {
                error_log('[Resend] Decryption failed for encrypted API key');
            } else {
                $decrypted_length = strlen($decrypted);
                $decrypted_prefix = $decrypted_length > 7 ? substr($decrypted, 0, 7) . '...' : '***';
                error_log(sprintf(
                    '[Resend] API key decrypted successfully - Length: %d, Prefix: %s',
                    $decrypted_length,
                    $decrypted_prefix
                ));
            }
        }
        
        return $decrypted !== false ? $decrypted : '';
    }

    // Key is not encrypted - migrate it now
    if (resend_is_debug_mode()) {
        error_log('[Resend] API key is not encrypted, migrating to encrypted format');
    }
    
    $encrypted = \Resend\WordPress\Encryption::encrypt($encrypted_key);
    if ($encrypted !== false) {
        update_option('resend_api_key', $encrypted);
        // Mark that we've migrated
        update_option('resend_api_key_encrypted', true);
        return $encrypted_key; // Return original key since we just encrypted it
    }

    // Encryption failed, return original (shouldn't happen, but be safe)
    if (resend_is_debug_mode()) {
        error_log('[Resend] Encryption failed during migration, returning original key');
    }
    return $encrypted_key;
}

/**
 * Set Resend API key (encrypted)
 *
 * @param string $api_key API key to encrypt and store
 * @return bool True on success, false on failure
 */
function resend_set_api_key(string $api_key): bool {
    if (empty($api_key)) {
        // If empty, just delete the option
        delete_option('resend_api_key');
        delete_option('resend_api_key_encrypted');
        return true;
    }

    $encrypted = \Resend\WordPress\Encryption::encrypt(trim($api_key));
    if ($encrypted === false) {
        return false;
    }

    $result = update_option('resend_api_key', $encrypted);
    if ($result) {
        update_option('resend_api_key_encrypted', true);
    }

    return $result;
}

/**
 * Check if Resend debug mode is enabled
 *
 * @return bool True if debug mode is enabled, false otherwise
 */
function resend_is_debug_mode(): bool {
    return (bool) get_option('resend_debug_mode', false);
}

