<?php

declare(strict_types=1);

/**
 * Main Plugin Class
 *
 * @package Resend\WordPress
 */

namespace Resend\WordPress;

if (!defined('ABSPATH')) {
    exit;
}

/**
 * Main plugin class
 */
class Plugin
{
    /**
     * Plugin instance
     *
     * @var self|null
     */
    private static ?self $instance = null;

    /**
     * Service container
     *
     * @var ServiceContainer
     */
    private ServiceContainer $container;

    /**
     * Get plugin instance
     *
     * @return self
     */
    public static function get_instance(): self
    {
        if (null === self::$instance) {
            self::$instance = new self();
        }

        return self::$instance;
    }

    /**
     * Constructor
     */
    private function __construct()
    {
        $this->container = new ServiceContainer();
    }

    /**
     * Initialize plugin
     *
     * @return void
     */
    public function init(): void
    {
        // Load helper functions
        require_once RESEND_WP_PLUGIN_DIR . 'includes/functions.php';

        // Ensure database tables exist (in case activation hook didn't run or failed)
        $this->ensure_database_tables();

        // Initialize hooks
        $hooks = new Hooks($this->container);
        $hooks->init();

        if (is_multisite()) {
            add_action('wpmu_new_blog', [self::class, 'activate_new_site'], 10, 1);
        }

        // Initialize admin if in admin area
        if (is_admin()) {
            $this->init_admin();
        }
    }

    /**
     * Ensure database tables exist, create if missing
     *
     * @return void
     */
    private function ensure_database_tables(): void
    {
        global $wpdb;
        $table_name = $wpdb->prefix . 'resend_email_log';
        
        // Check if table exists
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery,WordPress.DB.DirectDatabaseQuery.NoCaching -- Table check requires direct query
        $table_exists = $wpdb->get_var($wpdb->prepare(
            "SHOW TABLES LIKE %s",
            $table_name
        )) === $table_name;
        
        if (!$table_exists) {
            // Table doesn't exist, create it
            Database::create_tables();
            
            if (resend_is_debug_mode()) {
                error_log('[Resend Plugin] Database table was missing and has been created');
            }
        }
    }

    /**
     * Initialize admin functionality
     *
     * @return void
     */
    private function init_admin(): void
    {
        $settings_page = new Admin\SettingsPage($this->container);
        $settings_page->init();

        $log_page = new Admin\EmailLogPage($this->container);
        $log_page->init();
    }

    /**
     * Plugin activation (handles both single-site and network activation)
     *
     * @return void
     */
    public static function activate(): void
    {
        $is_network_activation = false;

        if (is_multisite()) {
            $networkwide_request = filter_input(INPUT_GET, 'networkwide', FILTER_SANITIZE_NUMBER_INT);

            if (null !== $networkwide_request && 1 === (int) $networkwide_request) {
                $is_network_activation = true;
            }

            if (!$is_network_activation) {
                $network_plugins = get_site_option('active_sitewide_plugins', []);
                $plugin_basename = plugin_basename(RESEND_WP_PLUGIN_FILE);
                $is_network_activation = isset($network_plugins[$plugin_basename]);
            }

            if (!$is_network_activation && function_exists('is_plugin_active_for_network')) {
                $is_network_activation = is_plugin_active_for_network(RESEND_WP_PLUGIN_FILE);
            }
        }

        if ($is_network_activation) {
            self::network_activate();
            return;
        }

        Database::create_tables();
        flush_rewrite_rules();
    }

    /**
     * Network activation (Multisite)
     *
     * @return void
     */
    private static function network_activate(): void
    {
        $blog_ids = get_sites(
            [
                'number' => 0,
                'fields' => 'ids',
            ]
        );

        foreach ($blog_ids as $blog_id) {
            switch_to_blog($blog_id);
            Database::create_tables();
            restore_current_blog();
        }

        flush_rewrite_rules();
    }

    /**
     * Activate plugin for a new site in Multisite
     *
     * @param int $blog_id Blog ID
     * @return void
     */
    public static function activate_new_site(int $blog_id): void
    {
        if (!is_plugin_active_for_network(RESEND_WP_PLUGIN_FILE)) {
            return;
        }

        switch_to_blog($blog_id);
        Database::create_tables();
        restore_current_blog();
    }

}
