<?php
declare(strict_types=1);

/**
 * Resend Mailer
 *
 * @package Resend\WordPress\Mail
 */

namespace Resend\WordPress\Mail;

use Resend\WordPress\Logging\LogRepository;
use Resend\WordPress\ServiceContainer;
use WP_Error;

if (!defined('ABSPATH')) {
    exit;
}

/**
 * Resend API mailer
 */
class ResendMailer {
    /**
     * Resend API endpoint
     */
    private const API_ENDPOINT = 'https://api.resend.com/emails';

    /**
     * Maximum number of retry attempts
     */
    private const MAX_RETRIES = 3;

    /**
     * Initial retry delay in seconds
     */
    private const INITIAL_RETRY_DELAY = 2;

    /**
     * Failure notification rate limit (6 hours in seconds)
     */
    private const FAILURE_NOTIFICATION_RATE_LIMIT = 6 * HOUR_IN_SECONDS;

    /**
     * Duplicate email prevention window (1 minute in seconds)
     */
    private const DUPLICATE_PREVENTION_WINDOW = MINUTE_IN_SECONDS;

    /**
     * Service container
     *
     * @var ServiceContainer
     */
    private ServiceContainer $container;

    /**
     * Log repository
     *
     * @var LogRepository|null
     */
    private ?LogRepository $log_repository = null;

    /**
     * Constructor
     *
     * @param ServiceContainer $container Service container
     */
    public function __construct(ServiceContainer $container) {
        $this->container = $container;
    }

    /**
     * Get log repository
     *
     * @return LogRepository
     */
    private function get_log_repository(): LogRepository {
        if (null === $this->log_repository) {
            $this->log_repository = $this->container->get(LogRepository::class);
        }

        return $this->log_repository;
    }

    /**
     * Send email via Resend API
     *
     * @param array<int, array{0: string, 1: string}> $to_addresses Array of [email, name] pairs
     * @param string $subject Email subject
     * @param string $body Email body (HTML or plain text)
     * @param array<int, array{0: string, 1: string}> $attachments Array of [path, filename] pairs
     * @param array<int, array{0: string, 1: string}> $headers Array of [name, value] pairs
     * @return true|WP_Error True on success, WP_Error on failure
     */
    public function send(array $to_addresses, string $subject, string $body, array $attachments = [], array $headers = []): bool|WP_Error {
        $api_key = resend_get_api_key();

        if (empty($api_key)) {
            return new WP_Error('resend_no_api_key', __('Resend API key is not configured.', 'mail-via-resend'));
        }

        // Trim API key to remove any accidental whitespace
        $api_key = trim($api_key);
        
        // Debug logging for API key validation
        if (resend_is_debug_mode()) {
            $api_key_length = strlen($api_key);
            $api_key_prefix = $api_key_length > 7 ? substr($api_key, 0, 7) . '...' : '***';
            $api_key_suffix = $api_key_length > 10 ? '...' . substr($api_key, -7) : '***';
            $expected_prefix = 're_';
            $has_valid_prefix = strpos($api_key, $expected_prefix) === 0;
            
            // Check for hidden/control characters
            $has_control_chars = preg_match('/[\x00-\x1F\x7F]/', $api_key);
            
            error_log(sprintf(
                '[Resend Mailer] API Key Check - Length: %d, Prefix: %s, Suffix: %s, Valid Prefix: %s, Has Control Chars: %s',
                $api_key_length,
                $api_key_prefix,
                $api_key_suffix,
                $has_valid_prefix ? 'yes' : 'no',
                $has_control_chars ? 'yes' : 'no'
            ));
            
            // Log the raw bytes (hex) of first and last few characters for debugging
            if ($api_key_length > 0) {
                $first_bytes = bin2hex(substr($api_key, 0, min(10, $api_key_length)));
                $last_bytes = $api_key_length > 10 ? bin2hex(substr($api_key, -10)) : '';
                error_log(sprintf(
                    '[Resend Mailer] API Key Bytes - First 10 (hex): %s, Last 10 (hex): %s',
                    $first_bytes,
                    $last_bytes
                ));
            }
            
            if (!$has_valid_prefix && $api_key_length > 0) {
                error_log('[Resend Mailer] Warning: API key does not start with expected prefix "re_"');
            }
            
            if ($has_control_chars) {
                error_log('[Resend Mailer] Warning: API key contains control characters - decryption may have failed!');
            }
        }

        // Parse headers
        $parsed_headers = $this->parse_headers($headers);
        $from_email = $parsed_headers['from_email'] ?? $parsed_headers['sender'] ?? get_option('resend_from_email', get_option('admin_email'));
        $from_name = $parsed_headers['from_name'] ?? get_option('resend_from_name', get_option('blogname'));
        $reply_to = $parsed_headers['reply_to'] ?? null;
        $cc = $parsed_headers['cc'] ?? [];
        $bcc = $parsed_headers['bcc'] ?? [];
        $html_body = $parsed_headers['content_type'] === 'text/html' ? $body : null;
        $text_body = $parsed_headers['content_type'] === 'text/html' ? null : $body;

        // Build recipients
        $to_emails = [];
        foreach ($to_addresses as $address) {
            $to_emails[] = $address[0];
        }

        // Check for duplicate emails (same recipient, subject, and body content)
        // Prevent sending the same email to the same recipient within 1 minute
        $duplicate_key = $this->generate_duplicate_key($to_emails, $subject, $body);
        $duplicate_transient_key = 'resend_duplicate_' . md5($duplicate_key);
        
        if (get_transient($duplicate_transient_key) !== false) {
            // Duplicate detected - log it but don't send
            $this->log_email($to_emails, $subject, $headers, $body, 'failed', null, __('Duplicate email prevented - same email was sent recently.', 'mail-via-resend'));
            return new WP_Error('resend_duplicate_email', __('Duplicate email prevented. The same email was sent to this recipient recently.', 'mail-via-resend'));
        }

        // Build payload
        $payload = [
            'from' => $this->format_email_address($from_email, $from_name),
            'to' => $to_emails,
            'subject' => $subject,
        ];

        if ($html_body) {
            $payload['html'] = $html_body;
        }

        if ($text_body) {
            $payload['text'] = $text_body;
        }

        if ($reply_to) {
            $payload['reply_to'] = $reply_to;
        }

        if (!empty($cc)) {
            $payload['cc'] = $cc;
        }

        if (!empty($bcc)) {
            $payload['bcc'] = $bcc;
        }

        // Handle attachments
        if (!empty($attachments)) {
            $payload['attachments'] = $this->prepare_attachments($attachments);
        }

        // Allow filtering payload before send
        $payload = apply_filters('resend_before_send', $payload, $to_addresses, $subject, $body);

        // Attempt to send with retry logic
        $last_error = null;
        $attempt = 0;
        $delay = self::INITIAL_RETRY_DELAY;

        while ($attempt <= self::MAX_RETRIES) {
            // Build Authorization header
            $auth_header = 'Bearer ' . $api_key;
            
            // Debug logging for Authorization header
            if (resend_is_debug_mode()) {
                $auth_header_length = strlen($auth_header);
                $auth_header_preview = strlen($auth_header) > 20 ? substr($auth_header, 0, 20) . '...' : $auth_header;
                error_log(sprintf(
                    '[Resend Mailer] Authorization Header - Length: %d, Preview: %s',
                    $auth_header_length,
                    $auth_header_preview
                ));
            }
            
            // Make API request
            $response = wp_remote_post(
                self::API_ENDPOINT,
                [
                    'headers' => [
                        'Authorization' => $auth_header,
                        'Content-Type' => 'application/json',
                    ],
                    'body' => wp_json_encode($payload),
                    'timeout' => 30,
                ]
            );

            // Check if request was successful
            if (!is_wp_error($response)) {
                $response_code = wp_remote_retrieve_response_code($response);
                $response_body = wp_remote_retrieve_body($response);
                $response_data = json_decode($response_body, true);

                if ($response_code === 200) {
                    $response_id = $response_data['id'] ?? null;
                    $this->log_email($to_emails, $subject, $headers, $body, 'sent', $response_id);

                    // Set duplicate prevention transient to block duplicates for 1 minute
                    set_transient($duplicate_transient_key, time(), self::DUPLICATE_PREVENTION_WINDOW);

                    // Allow action after successful send
                    do_action('resend_after_send', $response_data, $payload, $to_addresses, $subject, $body);

                    return true;
                }

                // Non-200 response - check if it's a connection/network error that we should retry
                // Don't retry on 4xx errors (client errors), only on 5xx (server errors) or network errors
                if ($response_code < 500 && $response_code >= 400) {
                    $error_message = $response_data['message'] ?? __('Unknown error occurred.', 'mail-via-resend');
                    
                    // Add debug information when debug mode is enabled
                    if (resend_is_debug_mode()) {
                        $debug_info = [
                            'http_status' => $response_code,
                            'api_response' => $response_data,
                            'api_key_length' => strlen($api_key),
                            'api_key_prefix' => substr($api_key, 0, 7) . '...',
                        ];
                        
                        // Log debug information
                        error_log(sprintf(
                            '[Resend Mailer] API Error - Status: %d, Message: %s, Response: %s',
                            $response_code,
                            $error_message,
                            wp_json_encode($response_data)
                        ));
                        
                        $this->log_email($to_emails, $subject, $headers, $body, 'failed', null, $error_message . ' [Debug: ' . wp_json_encode($debug_info) . ']');
                        
                        // Send webhook notification
                        $this->maybe_send_webhook_notification($error_message, $to_emails, $subject, $response_code);
                        
                        return new WP_Error('resend_api_error', $error_message, array_merge($response_data, ['debug' => $debug_info]));
                    }
                    
                    $this->log_email($to_emails, $subject, $headers, $body, 'failed', null, $error_message);
                    
                    // Send webhook notification
                    $this->maybe_send_webhook_notification($error_message, $to_emails, $subject, $response_code);
                    
                    return new WP_Error('resend_api_error', $error_message, array_merge($response_data, ['http_status' => $response_code]));
                }

                // Server error (5xx) - store for retry
                $last_error = new WP_Error('resend_api_error', $response_data['message'] ?? __('Server error occurred.', 'mail-via-resend'), $response_data);
            } else {
                // Check if this is a retryable network/connection error
                $error_code = $response->get_error_code();
                $retryable_codes = ['http_request_failed', 'curl_error', 'timeout'];
                $is_retryable = false;
                
                foreach ($retryable_codes as $retryable_code) {
                    if (strpos($error_code, $retryable_code) !== false) {
                        $is_retryable = true;
                        break;
                    }
                }
                
                if (!$is_retryable) {
                    // Non-retryable error (e.g., SSL error, invalid URL) - fail immediately
                    $error_message = $response->get_error_message();
                    $this->log_email($to_emails, $subject, $headers, $body, 'failed', null, $error_message);
                    
                    // Send webhook notification
                    $this->maybe_send_webhook_notification($error_message, $to_emails, $subject, null);
                    
                    return $response;
                }
                
                // Network/connection error - store for retry
                $last_error = $response;
            }

            // If we've exhausted retries, break
            if ($attempt >= self::MAX_RETRIES) {
                break;
            }

            // Wait before retrying with exponential backoff
            sleep($delay);
            $delay *= 2; // Exponential backoff: 2s, 4s, 8s
            $attempt++;
        }

        // All retries failed - log and send notification if needed
        $error_message = is_wp_error($last_error) ? $last_error->get_error_message() : __('Connection failed after retries.', 'mail-via-resend');
        $this->log_email($to_emails, $subject, $headers, $body, 'failed', null, $error_message);

        // Send failure notification only for connection/network failures (not API errors)
        // Check if the error is a connection failure
        $is_connection_failure = false;
        if (is_wp_error($last_error)) {
            $error_code = $last_error->get_error_code();
            $connection_error_codes = ['http_request_failed', 'curl_error', 'timeout'];
            foreach ($connection_error_codes as $connection_code) {
                if (strpos($error_code, $connection_code) !== false) {
                    $is_connection_failure = true;
                    break;
                }
            }
        } else {
            // If we got here after retries, it's likely a connection failure
            $is_connection_failure = true;
        }

        if ($is_connection_failure) {
            $this->maybe_send_failure_notification($error_message, $to_emails, $subject);
            // Send webhook notification for connection failures
            $this->maybe_send_webhook_notification($error_message, $to_emails, $subject, null);
        }

        return $last_error ?? new WP_Error('resend_connection_failed', $error_message);
    }

    /**
     * Parse email headers
     *
     * @param array<int, array{0: string, 1: string}> $headers Array of [name, value] pairs
     * @return array<string, mixed> Parsed headers
     */
    private function parse_headers(array $headers): array {
        $parsed = [
            'from_email' => null,
            'from_name' => null,
            'sender' => null,
            'reply_to' => null,
            'cc' => [],
            'bcc' => [],
            'content_type' => 'text/plain',
        ];

        foreach ($headers as $header) {
            $name = strtolower($header[0]);
            $value = $header[1];

            switch ($name) {
                case 'from':
                    if (preg_match('/^(.+?)\s*<(.+?)>$/', $value, $matches)) {
                        $parsed['from_name'] = trim($matches[1], '"\'');
                        $parsed['from_email'] = $matches[2];
                    } else {
                        $parsed['from_email'] = $value;
                    }
                    break;

                case 'sender':
                case 'return-path':
                    $parsed['sender'] = $value;
                    break;

                case 'reply-to':
                    $parsed['reply_to'] = $value;
                    break;

                case 'cc':
                    $parsed['cc'][] = $value;
                    break;

                case 'bcc':
                    $parsed['bcc'][] = $value;
                    break;

                case 'content-type':
                    if (strpos($value, 'text/html') !== false) {
                        $parsed['content_type'] = 'text/html';
                    }
                    break;
            }
        }

        return $parsed;
    }

    /**
     * Generate duplicate detection key
     *
     * Creates a unique key based on recipients, subject, and body content
     * to detect duplicate emails within the prevention window.
     *
     * @param array<string> $to_emails Recipient email addresses
     * @param string $subject Email subject
     * @param string $body Email body
     * @return string Duplicate detection key
     */
    private function generate_duplicate_key(array $to_emails, string $subject, string $body): string {
        // Sort emails to ensure consistent key regardless of order
        sort($to_emails);
        
        // Create a hash of the body content (first 1000 chars to handle large emails)
        $body_hash = md5(mb_substr($body, 0, 1000));
        
        // Combine all elements into a unique key
        return implode('|', [
            implode(',', $to_emails),
            md5($subject),
            $body_hash,
        ]);
    }

    /**
     * Format email address with optional name
     *
     * @param string $email Email address
     * @param string|null $name Optional name
     * @return string Formatted address
     */
    private function format_email_address(string $email, ?string $name = null): string {
        if ($name) {
            return sprintf('%s <%s>', $name, $email);
        }

        return $email;
    }

    /**
     * Prepare attachments for Resend API
     *
     * @param array<int, array{0: string, 1: string}> $attachments Array of [path, filename] pairs
     * @return array<int, array{filename: string, content: string}> Prepared attachments
     */
    private function prepare_attachments(array $attachments): array {
        $prepared = [];

        foreach ($attachments as $attachment) {
            $path = $attachment[0];
            $filename = $attachment[1] ?? basename($path);

            if (!file_exists($path) || !is_readable($path)) {
                continue;
            }

            $content = file_get_contents($path);
            if ($content === false) {
                continue;
            }

            $prepared[] = [
                'filename' => $filename,
                'content' => base64_encode($content),
            ];
        }

        return $prepared;
    }

    /**
     * Log email attempt
     *
     * @param array<string> $to_emails Recipient emails
     * @param string $subject Email subject
     * @param array<int, array{0: string, 1: string}> $headers Email headers
     * @param string $body Email body
     * @param string $status Status (sent/failed)
     * @param string|null $response_id Resend response ID
     * @param string|null $error_message Error message if failed
     * @return void
     */
    private function log_email(array $to_emails, string $subject, array $headers, string $body, string $status, ?string $response_id = null, ?string $error_message = null): void {
        $log_repo = $this->get_log_repository();

        // Store first recipient for log (Resend supports multiple recipients)
        $to_email = !empty($to_emails) ? $to_emails[0] : '';

        // Create body excerpt (first 500 characters)
        $body_excerpt = mb_substr(wp_strip_all_tags($body), 0, 500);

        // Serialize headers for storage
        $headers_serialized = wp_json_encode($headers);

        $log_repo->create([
            'to_email' => $to_email,
            'subject' => $subject,
            'headers' => $headers_serialized,
            'body_excerpt' => $body_excerpt,
            'status' => $status,
            'response_id' => $response_id,
            'error_message' => $error_message,
        ]);
    }

    /**
     * Send failure notification email if configured and rate limit allows
     *
     * @param string $error_message Error message
     * @param array<string> $failed_recipients Recipients that failed
     * @param string $failed_subject Subject of failed email
     * @return void
     */
    private function maybe_send_failure_notification(string $error_message, array $failed_recipients, string $failed_subject): void {
        $notification_emails = get_option('resend_failure_notification_emails', '');
        $admin_email = get_option('admin_email');
        
        // If no notification emails configured, use admin email as default
        if (empty($notification_emails)) {
            if (empty($admin_email) || !is_email($admin_email)) {
                return;
            }
            $valid_emails = [$admin_email];
        } else {
            // Parse notification emails (comma separated)
            $emails = explode(',', $notification_emails);
            $valid_emails = [];
            
            foreach ($emails as $email) {
                $email = trim($email);
                if (!empty($email) && is_email($email)) {
                    $valid_emails[] = $email;
                }
            }

            if (empty($valid_emails)) {
                // Fallback to admin email if no valid emails found
                if (!empty($admin_email) && is_email($admin_email)) {
                    $valid_emails = [$admin_email];
                } else {
                    return;
                }
            }
        }

        // Check rate limit - only send one notification per 6 hours
        $transient_key = 'resend_failure_notification_sent';
        $last_notification_time = get_transient($transient_key);
        
        if ($last_notification_time !== false) {
            // Rate limit active, don't send notification
            return;
        }

        // Prepare notification email
        $site_name = get_bloginfo('name');
        $notification_subject = sprintf(
            /* translators: %s: Site name */
            __('[%s] Resend SMTP Connection Failure', 'mail-via-resend'),
            esc_html($site_name)
        );

        $notification_message = sprintf(
            /* translators: %1$s: Site name, %2$s: Error message, %3$s: Failed recipients, %4$s: Failed subject, %5$s: Current time */
            __("A connection failure occurred when trying to send an email via Resend.\n\nSite: %1\$s\nError: %2\$s\n\nFailed Email Details:\nRecipients: %3\$s\nSubject: %4\$s\n\nTime: %5\$s\n\nThis notification will not be sent again for 6 hours to prevent spam.", 'mail-via-resend'),
            esc_html($site_name),
            esc_html($error_message),
            esc_html(implode(', ', $failed_recipients)),
            esc_html($failed_subject),
            esc_html(current_time('mysql'))
        );

        // Send notification using WordPress native mail (to avoid circular dependency)
        // Use a filter flag to bypass Resend mailer for notification emails
        add_filter('resend_bypass_mailer', '__return_true', 999);

        // Send to each notification email
        foreach ($valid_emails as $email) {
            wp_mail(
                $email,
                $notification_subject,
                $notification_message,
                [
                    'From: ' . $site_name . ' <' . $admin_email . '>',
                    'Content-Type: text/plain; charset=UTF-8',
                    'X-Resend-Bypass: true', // Additional header flag
                ]
            );
        }

        // Remove filter
        remove_filter('resend_bypass_mailer', '__return_true', 999);

        // Set rate limit transient (6 hours)
        set_transient($transient_key, time(), self::FAILURE_NOTIFICATION_RATE_LIMIT);
    }

    /**
     * Send webhook notification for email failure
     *
     * @param string $error_message Error message
     * @param array<string> $failed_recipients Recipients that failed
     * @param string $failed_subject Subject of failed email
     * @param int|null $http_status HTTP status code if available
     * @return void
     */
    private function maybe_send_webhook_notification(string $error_message, array $failed_recipients, string $failed_subject, ?int $http_status = null): void {
        $webhook_url = get_option('resend_webhook_url', '');
        
        if (empty($webhook_url)) {
            return;
        }

        // Check rate limit - only send one webhook notification per 5 minutes
        $transient_key = 'resend_webhook_notification_sent';
        $last_notification_time = get_transient($transient_key);
        
        if ($last_notification_time !== false) {
            // Rate limit active, don't send notification
            return;
        }

        // Detect webhook type from URL
        $webhook_type = $this->detect_webhook_type($webhook_url);
        
        if ($webhook_type === 'unknown') {
            if (resend_is_debug_mode()) {
                error_log(sprintf('[Resend Mailer] Unknown webhook type for URL: %s', $webhook_url));
            }
            return;
        }

        // Prepare notification data
        $site_name = get_bloginfo('name');
        $site_url = home_url();
        $timestamp = current_time('mysql');
        
        // Format message based on webhook type
        $payload = $this->format_webhook_payload($webhook_type, [
            'site_name' => $site_name,
            'site_url' => $site_url,
            'error_message' => $error_message,
            'failed_recipients' => $failed_recipients,
            'failed_subject' => $failed_subject,
            'http_status' => $http_status,
            'timestamp' => $timestamp,
        ]);

        if ($payload === null) {
            return;
        }

        // Send webhook request
        $response = wp_remote_post(
            $webhook_url,
            [
                'headers' => [
                    'Content-Type' => 'application/json',
                ],
                'body' => wp_json_encode($payload),
                'timeout' => 10,
            ]
        );

        if (is_wp_error($response)) {
            if (resend_is_debug_mode()) {
                error_log(sprintf(
                    '[Resend Mailer] Webhook notification failed: %s',
                    $response->get_error_message()
                ));
            }
        } else {
            $response_code = wp_remote_retrieve_response_code($response);
            if ($response_code >= 200 && $response_code < 300) {
                // Set rate limit transient (5 minutes)
                set_transient($transient_key, time(), 5 * MINUTE_IN_SECONDS);
            } elseif (resend_is_debug_mode()) {
                error_log(sprintf(
                    '[Resend Mailer] Webhook notification returned status: %d',
                    $response_code
                ));
            }
        }
    }

    /**
     * Detect webhook type from URL
     *
     * @param string $url Webhook URL
     * @return string Webhook type (discord, slack, teams, unknown)
     */
    private function detect_webhook_type(string $url): string {
        $url_lower = strtolower($url);
        
        if (strpos($url_lower, 'discord.com/api/webhooks') !== false || 
            strpos($url_lower, 'discordapp.com/api/webhooks') !== false) {
            return 'discord';
        }
        
        if (strpos($url_lower, 'hooks.slack.com') !== false) {
            return 'slack';
        }
        
        if (strpos($url_lower, 'office.com/webhook') !== false || 
            strpos($url_lower, 'office365.com/webhook') !== false ||
            strpos($url_lower, 'outlook.office.com/webhook') !== false) {
            return 'teams';
        }
        
        return 'unknown';
    }

    /**
     * Format webhook payload based on platform
     *
     * @param string $type Webhook type (discord, slack, teams)
     * @param array<string, mixed> $data Notification data
     * @return array<string, mixed>|null Formatted payload or null on error
     */
    private function format_webhook_payload(string $type, array $data): ?array {
        $recipients_list = implode(', ', $data['failed_recipients']);
        $status_text = $data['http_status'] ? sprintf('HTTP %d', $data['http_status']) : 'Connection Error';
        
        switch ($type) {
            case 'discord':
                $color = $data['http_status'] >= 500 ? 15158332 : // Red for server errors
                        ($data['http_status'] >= 400 ? 15105570 : // Orange for client errors
                        15158332); // Red for connection errors
                
                return [
                    'embeds' => [
                        [
                            'title' => '📧 Email Sending Failed',
                            'color' => $color,
                            'fields' => [
                                [
                                    'name' => 'Site',
                                    'value' => sprintf('[%s](%s)', $data['site_name'], $data['site_url']),
                                    'inline' => true,
                                ],
                                [
                                    'name' => 'Status',
                                    'value' => $status_text,
                                    'inline' => true,
                                ],
                                [
                                    'name' => 'Error',
                                    'value' => substr($data['error_message'], 0, 1024),
                                    'inline' => false,
                                ],
                                [
                                    'name' => 'Recipients',
                                    'value' => !empty($recipients_list) ? $recipients_list : 'N/A',
                                    'inline' => false,
                                ],
                                [
                                    'name' => 'Subject',
                                    'value' => substr($data['failed_subject'], 0, 256),
                                    'inline' => false,
                                ],
                            ],
                            'timestamp' => $data['timestamp'],
                            'footer' => [
                                'text' => 'Mail via Resend Plugin',
                            ],
                        ],
                    ],
                ];

            case 'slack':
                $color = $data['http_status'] >= 500 ? 'danger' : 
                        ($data['http_status'] >= 400 ? 'warning' : 'danger');
                
                return [
                    'attachments' => [
                        [
                            'color' => $color,
                            'title' => '📧 Email Sending Failed',
                            'fields' => [
                                [
                                    'title' => 'Site',
                                    'value' => sprintf('<%s|%s>', $data['site_url'], $data['site_name']),
                                    'short' => true,
                                ],
                                [
                                    'title' => 'Status',
                                    'value' => $status_text,
                                    'short' => true,
                                ],
                                [
                                    'title' => 'Error',
                                    'value' => substr($data['error_message'], 0, 2000),
                                    'short' => false,
                                ],
                                [
                                    'title' => 'Recipients',
                                    'value' => !empty($recipients_list) ? $recipients_list : 'N/A',
                                    'short' => false,
                                ],
                                [
                                    'title' => 'Subject',
                                    'value' => substr($data['failed_subject'], 0, 2000),
                                    'short' => false,
                                ],
                            ],
                            'footer' => 'Mail via Resend Plugin',
                            'ts' => strtotime($data['timestamp']),
                        ],
                    ],
                ];

            case 'teams':
                $theme_color = $data['http_status'] >= 500 ? 'FF0000' : 
                              ($data['http_status'] >= 400 ? 'FFA500' : 'FF0000');
                
                return [
                    '@type' => 'MessageCard',
                    '@context' => 'https://schema.org/extensions',
                    'themeColor' => $theme_color,
                    'summary' => 'Email Sending Failed',
                    'sections' => [
                        [
                            'activityTitle' => '📧 Email Sending Failed',
                            'facts' => [
                                [
                                    'name' => 'Site',
                                    'value' => sprintf('[%s](%s)', $data['site_name'], $data['site_url']),
                                ],
                                [
                                    'name' => 'Status',
                                    'value' => $status_text,
                                ],
                                [
                                    'name' => 'Error',
                                    'value' => substr($data['error_message'], 0, 2000),
                                ],
                                [
                                    'name' => 'Recipients',
                                    'value' => !empty($recipients_list) ? $recipients_list : 'N/A',
                                ],
                                [
                                    'name' => 'Subject',
                                    'value' => substr($data['failed_subject'], 0, 2000),
                                ],
                                [
                                    'name' => 'Time',
                                    'value' => $data['timestamp'],
                                ],
                            ],
                        ],
                    ],
                ];

            default:
                return null;
        }
    }
}

