<?php
declare(strict_types=1);

/**
 * Email Log Repository
 *
 * @package Resend\WordPress\Logging
 */

namespace Resend\WordPress\Logging;

use Resend\WordPress\ServiceContainer;

if (!defined('ABSPATH')) {
    exit;
}

/**
 * Repository for email log operations
 */
class LogRepository {
    /**
     * Service container
     *
     * @var ServiceContainer
     */
    private ServiceContainer $container;

    /**
     * Table name
     *
     * @var string
     */
    private string $table_name;

    /**
     * Constructor
     *
     * @param ServiceContainer $container Service container
     */
    public function __construct(ServiceContainer $container) {
        $this->container = $container;
        global $wpdb;
        $this->table_name = $wpdb->prefix . 'resend_email_log';
    }

    /**
     * Check if table exists, create if missing
     *
     * @return bool True if table exists or was created, false on failure
     */
    private function ensure_table_exists(): bool {
        global $wpdb;
        
        // Check if table exists
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery,WordPress.DB.DirectDatabaseQuery.NoCaching -- Table check requires direct query
        $table_exists = $wpdb->get_var($wpdb->prepare(
            "SHOW TABLES LIKE %s",
            $this->table_name
        )) === $this->table_name;
        
        if (!$table_exists) {
            // Table doesn't exist, try to create it
            \Resend\WordPress\Database::create_tables();
            
            // Verify it was created
            // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery,WordPress.DB.DirectDatabaseQuery.NoCaching -- Table check requires direct query
            $table_exists = $wpdb->get_var($wpdb->prepare(
                "SHOW TABLES LIKE %s",
                $this->table_name
            )) === $this->table_name;
            
            if (!$table_exists && resend_is_debug_mode()) {
                error_log(sprintf(
                    '[Resend LogRepository] Failed to create table: %s. Error: %s',
                    $this->table_name,
                    $wpdb->last_error
                ));
            }
        }
        
        return $table_exists;
    }

    /**
     * Create a log entry
     *
     * @param array<string, mixed> $data Log data
     * @return int|false Log ID on success, false on failure
     */
    public function create(array $data): int|false {
        global $wpdb;

        // Ensure table exists before inserting
        if (!$this->ensure_table_exists()) {
            if (resend_is_debug_mode()) {
                error_log('[Resend LogRepository] Cannot create log entry - table does not exist');
            }
            return false;
        }

        $defaults = [
            'to_email' => '',
            'subject' => '',
            'headers' => null,
            'body_excerpt' => null,
            'status' => 'sent',
            'response_id' => null,
            'error_message' => null,
            'created_at' => current_time('mysql'),
        ];

        $data = wp_parse_args($data, $defaults);

        // Sanitize headers (JSON string) - validate it's valid JSON
        $headers = $data['headers'];
        if ($headers !== null && !empty($headers)) {
            $decoded = json_decode($headers, true);
            if (json_last_error() !== JSON_ERROR_NONE) {
                $headers = null; // Invalid JSON, set to null
            } else {
                // Re-encode to ensure clean JSON
                $headers = wp_json_encode($decoded);
            }
        }

        // Sanitize body_excerpt
        $body_excerpt = $data['body_excerpt'];
        if ($body_excerpt !== null) {
            $body_excerpt = sanitize_textarea_field($body_excerpt);
        }

        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery,WordPress.DB.DirectDatabaseQuery.NoCaching -- Repository class requires direct queries, caching not applicable for inserts
        $result = $wpdb->insert(
            $this->table_name,
            [
                'to_email' => sanitize_email($data['to_email']),
                'subject' => sanitize_text_field($data['subject']),
                'headers' => $headers,
                'body_excerpt' => $body_excerpt,
                'status' => sanitize_text_field($data['status']),
                'response_id' => $data['response_id'] ? sanitize_text_field($data['response_id']) : null,
                'error_message' => $data['error_message'] ? sanitize_textarea_field($data['error_message']) : null,
                'created_at' => $data['created_at'],
            ],
            [
                '%s', // to_email
                '%s', // subject
                '%s', // headers
                '%s', // body_excerpt
                '%s', // status
                '%s', // response_id
                '%s', // error_message
                '%s', // created_at
            ]
        );

        if ($result === false) {
            if (resend_is_debug_mode()) {
                error_log(sprintf(
                    '[Resend LogRepository] Failed to insert log entry. Error: %s',
                    $wpdb->last_error
                ));
            }
            return false;
        }

        return $wpdb->insert_id;
    }

    /**
     * Get log entry by ID
     *
     * @param int $id Log ID
     * @return object|null Log entry or null if not found
     */
    public function get_by_id(int $id): ?object {
        global $wpdb;

        // phpcs:disable WordPress.DB.DirectDatabaseQuery.DirectQuery,WordPress.DB.DirectDatabaseQuery.NoCaching,WordPress.DB.PreparedSQL.InterpolatedNotPrepared,PluginCheck.Security.DirectDB.UnescapedDBParameter -- Repository class requires direct queries. Table name cannot be prepared, but it's safe (from $wpdb->prefix)
        $result = $wpdb->get_row(
            $wpdb->prepare(
                "SELECT * FROM {$this->table_name} WHERE id = %d",
                $id
            )
        );
        // phpcs:enable

        return $result ?: null;
    }

    /**
     * Get paginated log entries
     *
     * @param array<string, mixed> $args Query arguments
     * @return array{items: array<object>, total: int} Log entries and total count
     */
    public function get_paginated(array $args = []): array {
        global $wpdb;

        // Ensure table exists before querying
        if (!$this->ensure_table_exists()) {
            return [
                'items' => [],
                'total' => 0,
            ];
        }

        $defaults = [
            'per_page' => 20,
            'page' => 1,
            'search' => '',
            'status' => '',
            'orderby' => 'created_at',
            'order' => 'DESC',
        ];

        $args = wp_parse_args($args, $defaults);

        $per_page = absint($args['per_page']);
        $page = absint($args['page']);
        $offset = ($page - 1) * $per_page;
        $search = sanitize_text_field($args['search']);
        $status = sanitize_text_field($args['status']);
        
        // Sanitize orderby - this returns false if invalid, so we provide a default
        $orderby_raw = $args['orderby'] . ' ' . $args['order'];
        $orderby = sanitize_sql_orderby($orderby_raw);
        if (!$orderby) {
            $orderby = 'created_at DESC';
        }

        $where = ['1=1'];
        $where_values = [];

        if (!empty($search)) {
            $search_like = '%' . $wpdb->esc_like($search) . '%';
            $where[] = "(to_email LIKE %s OR subject LIKE %s)";
            $where_values[] = $search_like;
            $where_values[] = $search_like;
        }

        if (!empty($status)) {
            $where[] = 'status = %s';
            $where_values[] = $status;
        }

        $where_clause = implode(' AND ', $where);

        // Get total count
        // phpcs:disable WordPress.DB.PreparedSQL.InterpolatedNotPrepared,WordPress.DB.PreparedSQLPlaceholders.UnfinishedPrepare -- Table name and where clause cannot be prepared, but values are sanitized
        if (!empty($where_values)) {
            $total_query = $wpdb->prepare(
                "SELECT COUNT(*) FROM {$this->table_name} WHERE {$where_clause}",
                ...$where_values
            );
        } else {
            $total_query = "SELECT COUNT(*) FROM {$this->table_name} WHERE {$where_clause}";
        }
        // phpcs:ignore WordPress.DB.PreparedSQL.NotPrepared,WordPress.DB.DirectDatabaseQuery.DirectQuery,WordPress.DB.DirectDatabaseQuery.NoCaching,PluginCheck.Security.DirectDB.UnescapedDBParameter -- Query is prepared above or safe (no user input). Repository class requires direct queries. Table name and where clause are safe.
        $total = (int) $wpdb->get_var($total_query);
        // phpcs:enable

        // phpcs:disable WordPress.DB.PreparedSQL.InterpolatedNotPrepared -- Table name and where clause cannot be prepared, but values are sanitized. ORDER BY is sanitized via sanitize_sql_orderby().
        if (!empty($where_values)) {
            $all_values = array_merge($where_values, [$per_page, $offset]);
            $items_query = $wpdb->prepare(
                "SELECT * FROM {$this->table_name} WHERE {$where_clause} ORDER BY {$orderby} LIMIT %d OFFSET %d",
                ...$all_values
            );
        } else {
            $base_query = "SELECT * FROM {$this->table_name} WHERE {$where_clause} ORDER BY {$orderby}";
            $items_query = $wpdb->prepare($base_query . " LIMIT %d OFFSET %d", $per_page, $offset);
        }
        // phpcs:enable
        
        // phpcs:ignore WordPress.DB.PreparedSQL.NotPrepared,WordPress.DB.DirectDatabaseQuery.DirectQuery,WordPress.DB.DirectDatabaseQuery.NoCaching,PluginCheck.Security.DirectDB.UnescapedDBParameter -- Query is prepared above or safe. Repository class requires direct queries. Table name, where clause, and orderby are safe.
        $items = $wpdb->get_results($items_query, OBJECT);
        
        // Ensure we return an array even if query fails
        // get_results returns an array of objects or empty array, but check to be safe
        if (!is_array($items)) {
            $items = [];
        }

        return [
            'items' => $items,
            'total' => $total,
        ];
    }

    /**
     * Delete log entries older than retention days
     *
     * @param int $days Retention days
     * @return int|false Number of deleted rows or false on failure
     */
    public function delete_old_entries(int $days): int|false {
        global $wpdb;

        $date = gmdate('Y-m-d H:i:s', strtotime("-{$days} days"));

        // phpcs:disable WordPress.DB.DirectDatabaseQuery.DirectQuery,WordPress.DB.DirectDatabaseQuery.NoCaching,WordPress.DB.PreparedSQL.InterpolatedNotPrepared,PluginCheck.Security.DirectDB.UnescapedDBParameter -- Repository class requires direct queries. Table name cannot be prepared, but it's safe (from $wpdb->prefix)
        $result = $wpdb->query(
            $wpdb->prepare(
                "DELETE FROM {$this->table_name} WHERE created_at < %s",
                $date
            )
        );
        // phpcs:enable

        return $result;
    }

    /**
     * Delete log entries by IDs
     *
     * @param array<int> $ids Log IDs
     * @return int|false Number of deleted rows or false on failure
     */
    public function delete_by_ids(array $ids): int|false {
        global $wpdb;

        if (empty($ids)) {
            return 0;
        }

        $ids = array_map('absint', $ids);
        $ids = array_filter($ids);
        $ids = array_unique($ids);

        if (empty($ids)) {
            return 0;
        }

        $placeholders = implode(',', array_fill(0, count($ids), '%d'));

        // phpcs:disable WordPress.DB.DirectDatabaseQuery.DirectQuery,WordPress.DB.DirectDatabaseQuery.NoCaching,WordPress.DB.PreparedSQL.InterpolatedNotPrepared,WordPress.DB.PreparedSQLPlaceholders.UnfinishedPrepare,PluginCheck.Security.DirectDB.UnescapedDBParameter -- Repository class requires direct queries. Table name and placeholders cannot be prepared, but table name is safe and IDs are sanitized
        $result = $wpdb->query(
            $wpdb->prepare(
                "DELETE FROM {$this->table_name} WHERE id IN ({$placeholders})",
                ...$ids
            )
        );
        // phpcs:enable

        return $result;
    }

    /**
     * Get recent duplicate emails (same recipient and subject within time window)
     *
     * @param string $to_email Recipient email address
     * @param string $subject Email subject
     * @param int $time_window_seconds Time window in seconds to check for duplicates
     * @return array<object> Array of recent duplicate log entries
     */
    public function get_recent_duplicates(string $to_email, string $subject, int $time_window_seconds): array {
        global $wpdb;

        $cutoff_time = gmdate('Y-m-d H:i:s', time() - $time_window_seconds);

        // phpcs:disable WordPress.DB.DirectDatabaseQuery.DirectQuery,WordPress.DB.DirectDatabaseQuery.NoCaching,WordPress.DB.PreparedSQL.InterpolatedNotPrepared,PluginCheck.Security.DirectDB.UnescapedDBParameter -- Repository class requires direct queries. Table name cannot be prepared, but it's safe (from $wpdb->prefix). Values are sanitized.
        $results = $wpdb->get_results(
            $wpdb->prepare(
                "SELECT * FROM {$this->table_name} 
                WHERE to_email = %s 
                AND subject = %s 
                AND created_at >= %s 
                AND status = 'sent'
                ORDER BY created_at DESC",
                sanitize_email($to_email),
                sanitize_text_field($subject),
                $cutoff_time
            )
        );
        // phpcs:enable

        return is_array($results) ? $results : [];
    }
}

