<?php
declare(strict_types=1);

/**
 * Plugin Hooks
 *
 * @package Resend\WordPress
 */

namespace Resend\WordPress;

if (!defined('ABSPATH')) {
    exit;
}

/**
 * Handle WordPress hooks
 */
class Hooks {
    /**
     * Service container
     *
     * @var ServiceContainer
     */
    private ServiceContainer $container;

    /**
     * Track if Resend successfully sent the email
     * Used to prevent PHPMailer from sending while keeping recipients for validation
     *
     * @var bool
     */
    private static bool $resend_sent = false;

    /**
     * Constructor
     *
     * @param ServiceContainer $container Service container
     */
    public function __construct(ServiceContainer $container) {
        $this->container = $container;
    }

    /**
     * Initialize hooks
     *
     * @return void
     */
    public function init(): void {
        // Replace PHPMailer with Resend mailer
        add_action('phpmailer_init', [$this, 'replace_phpmailer'], 999);
        
        // Prevent PHPMailer from sending if Resend already sent successfully
        // This keeps recipients intact for plugins like Gravity Forms that validate them
        add_filter('phpmailer_pre_send', [$this, 'prevent_phpmailer_send'], 999);
        
        // Schedule cron for log cleanup
        add_action('resend_cleanup_old_logs', [$this, 'cleanup_old_logs']);
        
        if (!wp_next_scheduled('resend_cleanup_old_logs')) {
            wp_schedule_event(time(), 'daily', 'resend_cleanup_old_logs');
        }
    }

    /**
     * Replace PHPMailer with Resend mailer
     *
     * @param \PHPMailer\PHPMailer\PHPMailer $phpmailer PHPMailer instance
     * @return void
     */
    public function replace_phpmailer($phpmailer): void {
        // Check if we should bypass Resend (e.g., for failure notifications)
        if (apply_filters('resend_bypass_mailer', false)) {
            return;
        }

        // Check for bypass header
        $custom_headers = $phpmailer->getCustomHeaders();
        foreach ($custom_headers as $header) {
            if (isset($header[0]) && strtolower($header[0]) === 'x-resend-bypass' && isset($header[1]) && $header[1] === 'true') {
                return;
            }
        }

        $api_key = resend_get_api_key();

        // Only replace if API key is configured
        if (empty($api_key)) {
            return;
        }

        // Skip if no recipients (already sent or empty)
        $to_addresses = $phpmailer->getToAddresses();
        if (empty($to_addresses)) {
            return;
        }

        // Get Resend mailer
        $mailer = $this->container->get(Mail\ResendMailer::class);

        // Get attachments
        $attachments = [];
        foreach ($phpmailer->getAttachments() as $attachment) {
            $attachments[] = [$attachment[0], $attachment[2] ?? basename($attachment[0])];
        }

        // Get custom headers
        $headers = [];
        foreach ($phpmailer->getCustomHeaders() as $header) {
            $headers[] = [$header[0], $header[1]];
        }

        // Ensure From/Return-Path headers reflect the PHPMailer instance (WordPress 6.9 sets Sender by default)
        $has_from_header = false;
        foreach ($headers as $header) {
            if (isset($header[0]) && strtolower($header[0]) === 'from') {
                $has_from_header = true;
                break;
            }
        }

        if (!$has_from_header && !empty($phpmailer->From)) {
            $from_header = $phpmailer->From;
            if (!empty($phpmailer->FromName)) {
                $from_header = sprintf('%s <%s>', $phpmailer->FromName, $phpmailer->From);
            }
            $headers[] = ['From', $from_header];
        }

        if (!empty($phpmailer->Sender)) {
            $headers[] = ['Return-Path', $phpmailer->Sender];
        }

        // Carry over Reply-To/CC/BCC that were set on PHPMailer
        foreach ($phpmailer->getReplyToAddresses() as $reply_to) {
            $formatted = $this->format_address_header($reply_to);
            if ($formatted !== '') {
                $headers[] = ['Reply-To', $formatted];
            }
        }

        foreach ($phpmailer->getCcAddresses() as $cc_address) {
            $formatted = $this->format_address_header($cc_address);
            if ($formatted !== '') {
                $headers[] = ['Cc', $formatted];
            }
        }

        foreach ($phpmailer->getBccAddresses() as $bcc_address) {
            $formatted = $this->format_address_header($bcc_address);
            if ($formatted !== '') {
                $headers[] = ['Bcc', $formatted];
            }
        }

        // Determine content type - PHPMailer sets ContentType property
        $content_type = $phpmailer->ContentType;
        if (empty($content_type) && $phpmailer->isHTML()) {
            $content_type = 'text/html';
        }
        
        if (!empty($content_type)) {
            $headers[] = ['Content-Type', $content_type];
        }

        // Reset flag before attempting send
        self::$resend_sent = false;

        // Send email via Resend
        $result = $mailer->send(
            $to_addresses,
            $phpmailer->Subject,
            $phpmailer->Body,
            $attachments,
            $headers
        );

        // If successful, set flag to prevent PHPMailer from sending
        // We don't clear recipients here to allow plugins like Gravity Forms
        // to validate that recipients exist
        if (!is_wp_error($result)) {
            self::$resend_sent = true;
        }
    }

    /**
     * Prevent PHPMailer from sending if Resend already sent successfully
     *
     * This filter runs after phpmailer_init and allows us to prevent
     * PHPMailer from actually sending while keeping recipients intact
     * for validation by plugins like Gravity Forms.
     *
     * @param \PHPMailer\PHPMailer\PHPMailer $phpmailer PHPMailer instance
     * @return bool|WP_Error False to prevent sending, true to allow, WP_Error to fail
     */
    public function prevent_phpmailer_send($phpmailer) {
        // If Resend already sent successfully, prevent PHPMailer from sending
        if (self::$resend_sent) {
            // Reset flag for next email
            self::$resend_sent = false;
            // Return false to prevent PHPMailer from sending
            // This keeps recipients intact for validation
            return false;
        }

        // Allow normal sending if Resend didn't handle it
        return true;
    }

    /**
     * Format an address array from PHPMailer into a header-safe string
     *
     * @param array<int, string> $address Address array from PHPMailer (email, name)
     * @return string
     */
    private function format_address_header(array $address): string {
        $email = $address[0] ?? '';
        $name = $address[1] ?? '';

        if ($email === '') {
            return '';
        }

        if ($name === '' || $name === $email) {
            return $email;
        }

        return sprintf('%s <%s>', $name, $email);
    }

    /**
     * Cleanup old log entries
     *
     * @return void
     */
    public function cleanup_old_logs(): void {
        $retention_days = get_option('resend_log_retention_days', 30);
        
        if ($retention_days <= 0) {
            return;
        }

        $log_repository = $this->container->get(Logging\LogRepository::class);
        $log_repository->delete_old_entries($retention_days);
    }
}

