<?php
declare(strict_types=1);

/**
 * Encryption Utility
 *
 * @package Resend\WordPress
 */

namespace Resend\WordPress;

if (!defined('ABSPATH')) {
    exit;
}

/**
 * Encryption utility for sensitive data
 */
class Encryption {
    /**
     * Encryption cipher method
     */
    private const CIPHER = 'AES-256-CBC';

    /**
     * Encryption key length
     */
    private const KEY_LENGTH = 32;

    /**
     * IV length for CBC mode
     */
    private const IV_LENGTH = 16;

    /**
     * Get encryption key derived from WordPress salts
     *
     * @return string Encryption key
     */
    private static function get_encryption_key(): string {
        // Use WordPress authentication salt and secure auth salt to derive encryption key
        // These are defined in wp-config.php and are unique per installation
        $salt = wp_salt('auth');
        $secure_salt = wp_salt('secure_auth');

        // Combine and hash to get a consistent 32-byte key
        $combined = $salt . $secure_salt;
        return substr(hash('sha256', $combined, true), 0, self::KEY_LENGTH);
    }

    /**
     * Encrypt data
     *
     * @param string $data Data to encrypt
     * @return string|false Encrypted data (base64 encoded) or false on failure
     */
    public static function encrypt(string $data): string|false {
        if (empty($data)) {
            return false;
        }

        // Safety check: prevent encrypting extremely large data (API keys should be small)
        if (strlen($data) > 10000) {
            return false;
        }

        // Check if OpenSSL is available
        if (!function_exists('openssl_encrypt')) {
            return false;
        }

        $key = self::get_encryption_key();
        $iv = openssl_random_pseudo_bytes(self::IV_LENGTH);

        if ($iv === false) {
            return false;
        }

        $encrypted = openssl_encrypt($data, self::CIPHER, $key, OPENSSL_RAW_DATA, $iv);

        if ($encrypted === false) {
            return false;
        }

        // Prepend IV to encrypted data and base64 encode
        $encrypted_with_iv = $iv . $encrypted;
        return base64_encode($encrypted_with_iv);
    }

    /**
     * Decrypt data
     *
     * @param string $encrypted_data Encrypted data (base64 encoded)
     * @return string|false Decrypted data or false on failure
     */
    public static function decrypt(string $encrypted_data): string|false {
        if (empty($encrypted_data)) {
            return false;
        }

        // Check if OpenSSL is available
        if (!function_exists('openssl_decrypt')) {
            return false;
        }

        $key = self::get_encryption_key();
        $encrypted_with_iv = base64_decode($encrypted_data, true);

        if ($encrypted_with_iv === false) {
            return false;
        }

        // Extract IV (first 16 bytes) and encrypted data
        $iv = substr($encrypted_with_iv, 0, self::IV_LENGTH);
        $encrypted = substr($encrypted_with_iv, self::IV_LENGTH);

        if (strlen($iv) !== self::IV_LENGTH || empty($encrypted)) {
            return false;
        }

        $decrypted = openssl_decrypt($encrypted, self::CIPHER, $key, OPENSSL_RAW_DATA, $iv);

        return $decrypted;
    }

    /**
     * Check if a string appears to be encrypted (base64 encoded with expected structure)
     *
     * @param string $data Data to check
     * @return bool True if appears encrypted, false otherwise
     */
    public static function is_encrypted(string $data): bool {
        if (empty($data)) {
            return false;
        }

        // Check if it's base64 encoded
        $decoded = base64_decode($data, true);
        if ($decoded === false) {
            return false;
        }

        // Check if it has the minimum length (IV + at least some encrypted data)
        if (strlen($decoded) < self::IV_LENGTH + 1) {
            return false;
        }

        // Try to decrypt to verify it's actually encrypted
        $decrypted = self::decrypt($data);
        return $decrypted !== false;
    }
}

