<?php
declare(strict_types=1);

/**
 * Admin Settings Page
 *
 * @package Resend\WordPress\Admin
 */

namespace Resend\WordPress\Admin;

use Resend\WordPress\ServiceContainer;

if (!defined('ABSPATH')) {
    exit;
}

/**
 * Settings page handler
 */
class SettingsPage {
    /**
     * Service container
     *
     * @var ServiceContainer
     */
    private ServiceContainer $container;

    /**
     * Constructor
     *
     * @param ServiceContainer $container Service container
     */
    public function __construct(ServiceContainer $container) {
        $this->container = $container;
    }

    /**
     * Initialize settings page
     *
     * @return void
     */
    public function init(): void {
        add_action('admin_menu', [$this, 'add_settings_page']);
        add_action('admin_init', [$this, 'register_settings']);
        add_action('admin_notices', [$this, 'maybe_show_api_key_notice']);
        add_action('wp_ajax_resend_send_test_email', [$this, 'handle_test_email_ajax']);
        add_action('wp_ajax_resend_dismiss_notice', [$this, 'handle_dismiss_notice']);
        add_action('admin_enqueue_scripts', function($hook) {
            $this->enqueue_scripts($hook);
        });
    }

    /**
     * Add settings page to admin menu
     *
     * @return void
     */
    public function add_settings_page(): void {
        add_options_page(
            __('Resend Settings', 'mail-via-resend'),
            __('Resend', 'mail-via-resend'),
            'manage_options',
            'resend-settings',
            [$this, 'render_settings_page']
        );
    }

    /**
     * Register settings
     *
     * @return void
     */
    public function register_settings(): void {
        register_setting('resend_settings', 'resend_api_key', [
            'type' => 'string',
            'sanitize_callback' => function($value) {
                // Trim whitespace from API key
                // Don't use sanitize_text_field() as it may modify special characters in API keys
                $trimmed = trim((string) $value);
                
                // Get existing encrypted value from database
                $existing_encrypted = get_option('resend_api_key', '');
                
                // Check if value is just asterisks/bullets (placeholder indicating existing key)
                // Remove all asterisks and bullet points to check if anything remains
                $without_placeholders = preg_replace('/[•\*]/', '', $trimmed);
                
                // If empty or only contains placeholder characters, treat as "keep existing"
                if (empty($trimmed) || empty($without_placeholders)) {
                    if (!empty($existing_encrypted)) {
                        // Keep existing encrypted key - return it so WordPress doesn't overwrite
                        return $existing_encrypted;
                    }
                    // No existing key and empty input - delete the option
                    delete_option('resend_api_key');
                    delete_option('resend_api_key_encrypted');
                    return '';
                }
                
                // Log the key being encrypted (for debugging)
                if (resend_is_debug_mode()) {
                    $key_length = strlen($trimmed);
                    $key_prefix = $key_length > 7 ? substr($trimmed, 0, 7) . '...' : '***';
                    error_log(sprintf(
                        '[Resend Settings] Encrypting API key - Length: %d, Prefix: %s',
                        $key_length,
                        $key_prefix
                    ));
                }
                
                // Encrypt the API key (don't save yet - WordPress will save the returned value)
                $encrypted = \Resend\WordPress\Encryption::encrypt($trimmed);
                if ($encrypted === false) {
                    if (resend_is_debug_mode()) {
                        error_log('[Resend Settings] Encryption failed for API key');
                    }
                    // Encryption failed - return existing value to prevent data loss
                    return $existing_encrypted;
                }
                
                // Verify encryption/decryption cycle works
                $test_decrypt = \Resend\WordPress\Encryption::decrypt($encrypted);
                if ($test_decrypt !== $trimmed) {
                    if (resend_is_debug_mode()) {
                        error_log('[Resend Settings] Encryption/decryption cycle verification failed!');
                        error_log(sprintf(
                            '[Resend Settings] Original length: %d, Decrypted length: %d',
                            strlen($trimmed),
                            strlen($test_decrypt)
                        ));
                    }
                    // Encryption/decryption cycle failed - return existing value
                    return $existing_encrypted;
                }
                
                if (resend_is_debug_mode()) {
                    error_log('[Resend Settings] API key encrypted and verified successfully');
                }
                
                // Mark as encrypted
                update_option('resend_api_key_encrypted', true);
                
                // Return encrypted value for WordPress to save
                return $encrypted;
            },
            'default' => '',
        ]);

        register_setting('resend_settings', 'resend_from_email', [
            'type' => 'string',
            'sanitize_callback' => 'sanitize_email',
            'default' => get_option('admin_email'),
        ]);

        register_setting('resend_settings', 'resend_from_name', [
            'type' => 'string',
            'sanitize_callback' => 'sanitize_text_field',
            'default' => get_option('blogname'),
        ]);

        register_setting('resend_settings', 'resend_log_retention_days', [
            'type' => 'integer',
            'sanitize_callback' => 'absint',
            'default' => 30,
        ]);

        register_setting('resend_settings', 'resend_failure_notification_emails', [
            'type' => 'string',
            'sanitize_callback' => [$this, 'sanitize_email_list'],
            'default' => '',
        ]);

        register_setting('resend_settings', 'resend_debug_mode', [
            'type' => 'boolean',
            'sanitize_callback' => function($value) {
                // Checkboxes return '1' when checked, nothing when unchecked
                // If value is not set or empty, return false
                return !empty($value) && $value === '1';
            },
            'default' => false,
        ]);

        register_setting('resend_settings', 'resend_webhook_url', [
            'type' => 'string',
            'sanitize_callback' => function($value) {
                $trimmed = trim((string) $value);
                if (empty($trimmed)) {
                    return '';
                }
                // Validate it's a valid URL
                $url = esc_url_raw($trimmed);
                if (empty($url) || !filter_var($url, FILTER_VALIDATE_URL)) {
                    return '';
                }
                return $url;
            },
            'default' => '',
        ]);

        add_settings_section(
            'resend_main_section',
            __('API Configuration', 'mail-via-resend'),
            [$this, 'render_section_description'],
            'resend-settings'
        );

        add_settings_field(
            'resend_api_key',
            __('API Key', 'mail-via-resend'),
            [$this, 'render_api_key_field'],
            'resend-settings',
            'resend_main_section'
        );

        add_settings_field(
            'resend_from_email',
            __('From Email', 'mail-via-resend'),
            [$this, 'render_from_email_field'],
            'resend-settings',
            'resend_main_section'
        );

        add_settings_field(
            'resend_from_name',
            __('From Name', 'mail-via-resend'),
            [$this, 'render_from_name_field'],
            'resend-settings',
            'resend_main_section'
        );

        add_settings_field(
            'resend_log_retention_days',
            __('Log Retention (days)', 'mail-via-resend'),
            [$this, 'render_log_retention_field'],
            'resend-settings',
            'resend_main_section'
        );

        add_settings_field(
            'resend_failure_notification_emails',
            __('Failure Notification Emails', 'mail-via-resend'),
            [$this, 'render_failure_notification_emails_field'],
            'resend-settings',
            'resend_main_section'
        );

        add_settings_field(
            'resend_debug_mode',
            __('Debug Mode', 'mail-via-resend'),
            [$this, 'render_debug_mode_field'],
            'resend-settings',
            'resend_main_section'
        );

        add_settings_field(
            'resend_webhook_url',
            __('Webhook Failures', 'mail-via-resend'),
            [$this, 'render_webhook_url_field'],
            'resend-settings',
            'resend_main_section'
        );
    }

    /**
     * Render section description
     *
     * @return void
     */
    public function render_section_description(): void {
        echo '<p>';
        esc_html_e('Configure your Resend API settings. You can get your API key from', 'mail-via-resend');
        echo ' <a href="' . esc_url('https://resend.com/api-keys') . '" target="_blank" rel="noopener noreferrer">resend.com/api-keys</a>. ';
        esc_html_e('By using this plugin, you agree to', 'mail-via-resend');
        echo ' <a href="' . esc_url('https://resend.com/legal/terms-of-service') . '" target="_blank" rel="noopener noreferrer">' . esc_html__('Resend\'s Terms of Service', 'mail-via-resend') . '</a>.';
        echo '</p>';
    }

    /**
     * Render API key field
     *
     * @return void
     */
    public function render_api_key_field(): void {
        $has_key = !empty(resend_get_api_key());
        // Show asterisks if key exists, empty string if not
        $value = $has_key ? '••••••••••••••••••••••••••••••••' : '';
        ?>
        <input type="password" 
               id="resend_api_key" 
               name="resend_api_key" 
               value="<?php echo esc_attr($value); ?>" 
               class="regular-text" 
               autocomplete="off" />
        <p class="description">
            <?php esc_html_e('Your Resend API key. This is required to send emails.', 'mail-via-resend'); ?>
        </p>
        <?php
    }

    /**
     * Render from email field
     *
     * @return void
     */
    public function render_from_email_field(): void {
        $value = get_option('resend_from_email', get_option('admin_email'));
        ?>
        <input type="email" 
               id="resend_from_email" 
               name="resend_from_email" 
               value="<?php echo esc_attr($value); ?>" 
               class="regular-text" />
        <p class="description">
            <?php esc_html_e('Default email address to send emails from. Must be verified in your Resend account.', 'mail-via-resend'); ?>
        </p>
        <?php
    }

    /**
     * Render from name field
     *
     * @return void
     */
    public function render_from_name_field(): void {
        $value = get_option('resend_from_name', get_option('blogname'));
        ?>
        <input type="text" 
               id="resend_from_name" 
               name="resend_from_name" 
               value="<?php echo esc_attr($value); ?>" 
               class="regular-text" />
        <p class="description">
            <?php esc_html_e('Default name to send emails from.', 'mail-via-resend'); ?>
        </p>
        <?php
    }

    /**
     * Render log retention field
     *
     * @return void
     */
    public function render_log_retention_field(): void {
        $value = get_option('resend_log_retention_days', 30);
        ?>
        <input type="number" 
               id="resend_log_retention_days" 
               name="resend_log_retention_days" 
               value="<?php echo esc_attr($value); ?>" 
               class="small-text" 
               min="1" 
               step="1" />
        <p class="description">
            <?php esc_html_e('Number of days to keep email logs. Older logs will be automatically deleted.', 'mail-via-resend'); ?>
        </p>
        <?php
    }

    /**
     * Render failure notification emails field
     *
     * @return void
     */
    public function render_failure_notification_emails_field(): void {
        $value = get_option('resend_failure_notification_emails', '');
        ?>
        <input type="text" 
               id="resend_failure_notification_emails" 
               name="resend_failure_notification_emails" 
               value="<?php echo esc_attr($value); ?>" 
               class="regular-text" 
               placeholder="email1@example.com, email2@example.com" />
        <p class="description">
            <?php esc_html_e('Comma-separated email addresses to notify when SMTP connection failures occur. Maximum one notification will be sent per 6 hours. If left empty, notifications will be sent to the WordPress admin email.', 'mail-via-resend'); ?>
        </p>
        <?php
    }

    /**
     * Render debug mode field
     *
     * @return void
     */
    public function render_debug_mode_field(): void {
        $value = get_option('resend_debug_mode', false);
        ?>
        <input type="hidden" name="resend_debug_mode" value="0" />
        <label>
            <input type="checkbox" 
                   id="resend_debug_mode" 
                   name="resend_debug_mode" 
                   value="1" 
                   <?php checked($value, true); ?> />
            <?php esc_html_e('Enable debug mode', 'mail-via-resend'); ?>
        </label>
        <p class="description">
            <?php esc_html_e('When enabled, detailed error messages, API responses, and debug logs will be included in error responses and written to debug.log. This helps diagnose email sending issues.', 'mail-via-resend'); ?>
        </p>
        <?php
    }

    /**
     * Render webhook URL field
     *
     * @return void
     */
    public function render_webhook_url_field(): void {
        $value = get_option('resend_webhook_url', '');
        ?>
        <input type="url" 
               id="resend_webhook_url" 
               name="resend_webhook_url" 
               value="<?php echo esc_url($value); ?>" 
               class="regular-text" 
               placeholder="https://discord.com/api/webhooks/..." />
        <p class="description">
            <?php esc_html_e('Webhook URL to receive notifications when email sending fails. Supports Discord, Slack, and Microsoft Teams webhooks. Leave empty to disable.', 'mail-via-resend'); ?>
        </p>
        <?php
    }

    /**
     * Sanitize email list (comma separated)
     *
     * @param string $input Raw input
     * @return string Sanitized email list
     */
    public function sanitize_email_list(string $input): string {
        if (empty($input)) {
            return '';
        }

        // Split by comma
        $emails = explode(',', $input);
        $valid_emails = [];

        foreach ($emails as $email) {
            $email = trim($email);
            if (!empty($email) && is_email($email)) {
                $valid_emails[] = sanitize_email($email);
            }
        }

        return implode(',', $valid_emails);
    }

    /**
     * Show contextual notice if API key is missing
     *
     * @return void
     */
    public function maybe_show_api_key_notice(): void {
        $screen = get_current_screen();
        if (!$screen || $screen->id !== 'settings_page_resend-settings') {
            return;
        }

        $api_key = resend_get_api_key();
        if (empty($api_key)) {
            $dismissed = get_user_meta(get_current_user_id(), 'resend_api_key_notice_dismissed', true);
            if (!$dismissed) {
                ?>
                <div class="notice notice-warning is-dismissible" data-notice="resend_api_key">
                    <p>
                        <strong><?php esc_html_e('Resend API Key Required', 'mail-via-resend'); ?></strong><br>
                        <?php esc_html_e('Please enter your Resend API key below to enable email sending. You can get your API key from', 'mail-via-resend'); ?>
                        <a href="https://resend.com/api-keys" target="_blank" rel="noopener noreferrer"><?php esc_html_e('your Resend dashboard', 'mail-via-resend'); ?></a>.
                    </p>
                </div>
                <?php
            }
        }
    }

    /**
     * Enqueue admin scripts
     *
     * @param string $hook Current admin page hook
     * @return void
     */
    public function enqueue_scripts(string $hook): void {
        if ($hook !== 'settings_page_resend-settings') {
            return;
        }

        wp_enqueue_script(
            'resend-admin',
            RESEND_WP_PLUGIN_URL . 'assets/admin.js',
            ['jquery'],
            RESEND_WP_VERSION,
            true
        );

        wp_localize_script('resend-admin', 'resendAdmin', [
            'ajaxUrl' => admin_url('admin-ajax.php'),
            'nonce' => wp_create_nonce('resend_send_test_email'),
            'dismissNonce' => wp_create_nonce('resend_dismiss_notice'),
            'strings' => [
                'sending' => __('Sending test email...', 'mail-via-resend'),
                'success' => __('Test email sent successfully!', 'mail-via-resend'),
                'error' => __('Test email failed: ', 'mail-via-resend'),
                'buttonText' => __('Send Test Email', 'mail-via-resend'),
            ],
        ]);
    }

    /**
     * Handle test email AJAX request
     *
     * @return void
     */
    public function handle_test_email_ajax(): void {
        // Check permissions
        if (!current_user_can('manage_options')) {
            wp_send_json_error(['message' => __('You do not have sufficient permissions.', 'mail-via-resend')]);
        }

        // Verify nonce
        if (!isset($_POST['nonce']) || !wp_verify_nonce(sanitize_text_field(wp_unslash($_POST['nonce'])), 'resend_send_test_email')) {
            wp_send_json_error(['message' => __('Security check failed.', 'mail-via-resend')]);
        }

        // Get test email address
        $test_email = isset($_POST['test_email']) ? sanitize_email(wp_unslash($_POST['test_email'])) : get_option('admin_email');

        if (!is_email($test_email)) {
            wp_send_json_error(['message' => __('Invalid email address.', 'mail-via-resend')]);
        }

        // Get mailer
        $mailer = $this->container->get(\Resend\WordPress\Mail\ResendMailer::class);

        // Prepare test email
        $from_email = get_option('resend_from_email', get_option('admin_email'));
        $from_name = get_option('resend_from_name', get_option('blogname'));

        /* translators: %s: Site name */
        $subject = sprintf(__('Test Email from %s', 'mail-via-resend'), esc_html(get_bloginfo('name')));
        $message = sprintf(
            /* translators: %s: Date and time */
            __("This is a test email sent from the Mail via Resend plugin.\n\nIf you received this email, your Resend configuration is working correctly!\n\nSent at: %s", 'mail-via-resend'),
            esc_html(current_time('mysql'))
        );

        $headers = [
            ['From', sprintf('%s <%s>', esc_html($from_name), esc_html($from_email))],
            ['Content-Type', 'text/plain'],
        ];

        // Send test email
        $result = $mailer->send(
            [[$test_email, '']],
            $subject,
            $message,
            [],
            $headers
        );

        // Return response
        if (is_wp_error($result)) {
            $error_message = $result->get_error_message();
            $error_data = $result->get_error_data();
            $error_code = $result->get_error_code();
            
            // Build detailed error response
            $error_response = [
                'message' => sprintf(__('Test email failed: %s', 'mail-via-resend'), esc_html($error_message)),
            ];
            
            // Always include basic debug information for test emails
            $debug_info = [
                'error_code' => $error_code,
            ];
            
            if (!empty($error_data)) {
                if (isset($error_data['http_status'])) {
                    $debug_info['http_status'] = $error_data['http_status'];
                }
                
                // Include API response details if available
                if (isset($error_data['message'])) {
                    $debug_info['api_message'] = $error_data['message'];
                }
                
                // Include full debug data if debug mode is enabled
                if (resend_is_debug_mode()) {
                    $debug_info['full_response'] = $error_data;
                    
                    // Log to error log for debugging
                    error_log(sprintf(
                        '[Resend Test Email] Error: %s, Code: %s, HTTP Status: %s, Data: %s',
                        $error_message,
                        $error_code,
                        $error_data['http_status'] ?? 'N/A',
                        wp_json_encode($error_data)
                    ));
                }
            }
            
            $error_response['debug'] = $debug_info;
            wp_send_json_error($error_response);
        } else {
            wp_send_json_success(['message' => __('Test email sent successfully!', 'mail-via-resend')]);
        }
    }

    /**
     * Handle notice dismissal AJAX request
     *
     * @return void
     */
    public function handle_dismiss_notice(): void {
        // Check permissions
        if (!current_user_can('manage_options')) {
            wp_send_json_error(['message' => __('You do not have sufficient permissions.', 'mail-via-resend')]);
        }

        // Verify nonce
        if (!isset($_POST['nonce']) || !wp_verify_nonce(sanitize_text_field(wp_unslash($_POST['nonce'])), 'resend_dismiss_notice')) {
            wp_send_json_error(['message' => __('Security check failed.', 'mail-via-resend')]);
        }

        // Get notice name
        $notice = isset($_POST['notice']) ? sanitize_text_field(wp_unslash($_POST['notice'])) : '';

        if ($notice === 'resend_api_key') {
            update_user_meta(get_current_user_id(), 'resend_api_key_notice_dismissed', true);
            wp_send_json_success();
        }

        wp_send_json_error(['message' => __('Invalid notice.', 'mail-via-resend')]);
    }

    /**
     * Render settings page
     *
     * @return void
     */
    public function render_settings_page(): void {
        if (!current_user_can('manage_options')) {
            return;
        }

        settings_errors('resend_settings');
        ?>
        <div class="wrap">
            <h1><?php echo esc_html(get_admin_page_title()); ?></h1>
            
            <form action="options.php" method="post">
                <?php
                settings_fields('resend_settings');
                do_settings_sections('resend-settings');
                submit_button(__('Save Settings', 'mail-via-resend'));
                ?>
            </form>

            <hr>

            <h2><?php esc_html_e('Test Email', 'mail-via-resend'); ?></h2>
            <div id="resend-test-email-result" class="notice" style="display: none; margin-top: 10px; margin-bottom: 15px;"></div>
            <p><?php esc_html_e('Send a test email to verify your Resend configuration is working correctly.', 'mail-via-resend'); ?></p>
            
            <table class="form-table" role="presentation">
                <tr>
                    <th scope="row">
                        <label for="test_email"><?php esc_html_e('Send test email to', 'mail-via-resend'); ?></label>
                    </th>
                    <td>
                        <input type="email" 
                               id="test_email" 
                               name="test_email" 
                               value="<?php echo esc_attr(get_option('admin_email')); ?>" 
                               class="regular-text" 
                               required />
                        <p class="description">
                            <?php esc_html_e('The email address to send the test email to.', 'mail-via-resend'); ?>
                        </p>
                    </td>
                </tr>
            </table>

            <p class="submit">
                <button type="button" id="resend-send-test-email" class="button button-secondary">
                    <?php esc_html_e('Send Test Email', 'mail-via-resend'); ?>
                </button>
            </p>

            <hr style="margin: 30px 0 20px;">

            <div style="background: #f0f0f1; border-left: 4px solid #2271b1; padding: 12px 16px; margin-top: 20px;">
                <p style="margin: 0;">
                    <?php
                    printf(
                        /* translators: %1$s: Opening anchor tag for Webscape, %2$s: Closing anchor tag, %3$s: Opening anchor tag for contact, %4$s: Closing anchor tag */
                        esc_html__('Plugin built by %1$sWebscape%2$s. Need help with WordPress? %3$sContact us%4$s.', 'mail-via-resend'),
                        '<a href="https://webscape.co.nz" target="_blank" rel="noopener noreferrer">',
                        '</a>',
                        '<a href="https://webscape.co.nz/contact" target="_blank" rel="noopener noreferrer">',
                        '</a>'
                    );
                    ?>
                </p>
            </div>
        </div>
        <?php
    }
}

