<?php

declare(strict_types=1);

/**
 * Email Log Admin Page
 *
 * @package Resend\WordPress\Admin
 */

namespace Resend\WordPress\Admin;

use Resend\WordPress\Logging\LogRepository;
use Resend\WordPress\ServiceContainer;

if (!defined('ABSPATH')) {
    exit;
}

// Load WP_List_Table if not already loaded
if (!class_exists('WP_List_Table')) {
    require_once ABSPATH . 'wp-admin/includes/class-wp-list-table.php';
}

/**
 * Email log list table
 */
class EmailLogListTable extends \WP_List_Table
{
    /**
     * Log repository
     *
     * @var LogRepository
     */
    private LogRepository $log_repository;

    /**
     * Constructor
     *
     * @param LogRepository $log_repository Log repository
     */
    public function __construct(LogRepository $log_repository)
    {
        parent::__construct([
            'singular' => 'email_log',
            'plural' => 'email_logs',
            'ajax' => false,
        ]);

        $this->log_repository = $log_repository;
    }

    /**
     * Get columns
     *
     * @return array<string, string>
     */
    public function get_columns(): array
    {
        return [
            'cb' => '<input type="checkbox" />',
            'to_email' => __('To', 'mail-via-resend'),
            'subject' => __('Subject', 'mail-via-resend'),
            'status' => __('Status', 'mail-via-resend'),
            'response_id' => __('Response ID', 'mail-via-resend'),
            'created_at' => __('Date', 'mail-via-resend'),
        ];
    }

    /**
     * Get sortable columns
     *
     * @return array<string, array{0: string, 1: bool}>
     */
    protected function get_sortable_columns(): array
    {
        return [
            'to_email' => ['to_email', false],
            'subject' => ['subject', false],
            'status' => ['status', false],
            'created_at' => ['created_at', true],
        ];
    }

    /**
     * Column default
     *
     * @param object $item Log item
     * @param string $column_name Column name
     * @return string
     */
    protected function column_default($item, $column_name): string
    {
        switch ($column_name) {
            case 'to_email':
                return esc_html($item->to_email);

            case 'subject':
                $subject = esc_html($item->subject);
                $view_url = add_query_arg([
                    'page' => 'resend-email-log',
                    'action' => 'view',
                    'log_id' => $item->id,
                ], admin_url('admin.php'));
                return sprintf(
                    '<a href="%s" class="row-title">%s</a>',
                    esc_url($view_url),
                    $subject
                );

            case 'status':
                $status_class = $item->status === 'sent' ? 'success' : 'error';
                return sprintf(
                    '<span class="resend-status resend-status-%s">%s</span>',
                    esc_attr($status_class),
                    esc_html(ucfirst($item->status))
                );

            case 'response_id':
                return $item->response_id ? esc_html($item->response_id) : '—';

            case 'created_at':
                return esc_html(mysql2date(get_option('date_format') . ' ' . get_option('time_format'), $item->created_at));

            default:
                return '';
        }
    }

    /**
     * Column checkbox
     *
     * @param object $item Log item
     * @return string
     */
    protected function column_cb($item): string
    {
        return sprintf(
            '<input type="checkbox" name="log_ids[]" value="%d" />',
            $item->id
        );
    }

    /**
     * Get bulk actions
     *
     * @return array<string, string>
     */
    protected function get_bulk_actions(): array
    {
        return [
            'delete' => __('Delete', 'mail-via-resend'),
        ];
    }

    /**
     * Process bulk actions
     *
     * @return void
     */
    public function process_bulk_action(): void
    {
        $action = $this->current_action();

        if ($action === 'delete') {
            if (!isset($_REQUEST['log_ids']) || !is_array($_REQUEST['log_ids'])) {
                return;
            }

            if (!check_admin_referer('bulk-' . $this->_args['plural'])) {
                wp_die(esc_html__('Security check failed.', 'mail-via-resend'));
            }

            $ids = array_map('absint', $_REQUEST['log_ids']);
            $deleted = $this->log_repository->delete_by_ids($ids);

            if ($deleted !== false) {
                echo '<div class="notice notice-success is-dismissible"><p>' .
                    esc_html(
                        sprintf(
                            /* translators: %d: Number of log entries deleted */
                            _n('%d log entry deleted.', '%d log entries deleted.', $deleted, 'mail-via-resend'),
                            $deleted
                        )
                    ) .
                    '</p></div>';
            }
        }
    }

    /**
     * Prepare items
     *
     * @return void
     */
    public function prepare_items(): void
    {
        $this->process_bulk_action();

        $per_page = $this->get_items_per_page('email_logs_per_page', 20);
        $current_page = $this->get_pagenum();

        // phpcs:disable WordPress.Security.NonceVerification.Recommended -- WP_List_Table handles nonce verification
        $args = [
            'per_page' => $per_page,
            'page' => $current_page,
            'search' => isset($_REQUEST['s']) ? sanitize_text_field(wp_unslash($_REQUEST['s'])) : '',
            'status' => isset($_REQUEST['status']) ? sanitize_text_field(wp_unslash($_REQUEST['status'])) : '',
            'orderby' => isset($_REQUEST['orderby']) ? sanitize_text_field(wp_unslash($_REQUEST['orderby'])) : 'created_at',
            'order' => isset($_REQUEST['order']) ? sanitize_text_field(wp_unslash($_REQUEST['order'])) : 'DESC',
        ];
        // phpcs:enable

        $result = $this->log_repository->get_paginated($args);

        // Ensure items is an array and set it
        $this->items = is_array($result['items']) ? $result['items'] : [];

        $this->set_pagination_args([
            'total_items' => $result['total'],
            'per_page' => $per_page,
            'total_pages' => ceil($result['total'] / $per_page),
        ]);

        // Set column headers (required by WP_List_Table)
        $this->_column_headers = [
            $this->get_columns(),
            [],
            $this->get_sortable_columns(),
        ];
    }

    /**
     * Message to display when there are no items
     *
     * @return void
     */
    public function no_items(): void
    {
        esc_html_e('No email logs found.', 'mail-via-resend');
    }

    /**
     * Extra table nav
     *
     * @param string $which Top or bottom
     * @return void
     */
    protected function extra_tablenav($which): void
    {
        if ($which !== 'top') {
            return;
        }

        // phpcs:ignore WordPress.Security.NonceVerification.Recommended -- WP_List_Table handles nonce verification
        $status = isset($_REQUEST['status']) ? sanitize_text_field(wp_unslash($_REQUEST['status'])) : '';
        // phpcs:ignore WordPress.Security.NonceVerification.Recommended -- WP_List_Table handles nonce verification
        $search = isset($_REQUEST['s']) ? sanitize_text_field(wp_unslash($_REQUEST['s'])) : '';
?>
        <div class="alignleft actions">
            <label for="filter-by-status" class="screen-reader-text"><?php esc_html_e('Filter by status', 'mail-via-resend'); ?></label>
            <select name="status" id="filter-by-status">
                <option value=""><?php esc_html_e('All statuses', 'mail-via-resend'); ?></option>
                <option value="sent" <?php selected($status, 'sent'); ?>><?php esc_html_e('Sent', 'mail-via-resend'); ?></option>
                <option value="failed" <?php selected($status, 'failed'); ?>><?php esc_html_e('Failed', 'mail-via-resend'); ?></option>
            </select>
            <?php if (!empty($search)): ?>
                <input type="hidden" name="s" value="<?php echo esc_attr($search); ?>" />
            <?php endif; ?>
            <input type="hidden" name="page" value="resend-email-log" />
            <input type="submit" name="filter_action" id="post-query-submit" class="button" value="<?php esc_attr_e('Filter', 'mail-via-resend'); ?>">
        </div>
    <?php
    }

    /**
     * Display search box
     *
     * @param string $text The submit button text
     * @param string $input_id The input ID
     * @return void
     */
    public function search_box($text, $input_id): void
    {
        // phpcs:ignore WordPress.Security.NonceVerification.Recommended -- WP_List_Table handles nonce verification
        $search = isset($_REQUEST['s']) ? sanitize_text_field(wp_unslash($_REQUEST['s'])) : '';
        // phpcs:ignore WordPress.Security.NonceVerification.Recommended -- WP_List_Table handles nonce verification
        $status = isset($_REQUEST['status']) ? sanitize_text_field(wp_unslash($_REQUEST['status'])) : '';
    ?>
        <form method="get" action="">
            <p class="search-box">
                <label class="screen-reader-text" for="<?php echo esc_attr($input_id); ?>"><?php echo esc_html($text); ?>:</label>
                <input type="search" id="<?php echo esc_attr($input_id); ?>" name="s" value="<?php echo esc_attr($search); ?>" />
                <?php if (!empty($status)): ?>
                    <input type="hidden" name="status" value="<?php echo esc_attr($status); ?>" />
                <?php endif; ?>
                <input type="hidden" name="page" value="resend-email-log" />
                <?php submit_button($text, '', '', false, ['id' => 'search-submit']); ?>
            </p>
        </form>
    <?php
    }
}

/**
 * Email log page handler
 */
class EmailLogPage
{
    /**
     * Service container
     *
     * @var ServiceContainer
     */
    private ServiceContainer $container;

    /**
     * Constructor
     *
     * @param ServiceContainer $container Service container
     */
    public function __construct(ServiceContainer $container)
    {
        $this->container = $container;
    }

    /**
     * Initialize email log page
     *
     * @return void
     */
    public function init(): void
    {
        add_action('admin_menu', [$this, 'add_email_log_page']);
        add_action('admin_enqueue_scripts', [$this, 'enqueue_styles']);
        // Use priority 20 to run after SettingsPage (which uses default priority 10)
        add_action('admin_enqueue_scripts', [$this, 'enqueue_scripts'], 20);
        add_action('wp_ajax_resend_resend_email', [$this, 'handle_resend_email_ajax']);
    }

    /**
     * Add email log page to admin menu
     *
     * @return void
     */
    public function add_email_log_page(): void
    {
        add_submenu_page(
            'tools.php',
            __('Email Log', 'mail-via-resend'),
            __('Email Log', 'mail-via-resend'),
            'manage_options',
            'resend-email-log',
            [$this, 'render_email_log_page']
        );
    }

    /**
     * Enqueue admin styles
     *
     * @param string $hook Current admin page hook
     * @return void
     */
    public function enqueue_styles(string $hook): void
    {
        if ($hook !== 'tools_page_resend-email-log') {
            return;
        }

        wp_add_inline_style('common', '
            .resend-status {
                display: inline-block;
                padding: 3px 8px;
                border-radius: 3px;
                font-size: 11px;
                font-weight: 600;
                text-transform: uppercase;
            }
            .resend-status-success {
                background: #00a32a;
                color: #fff;
            }
            .resend-status-error {
                background: #d63638;
                color: #fff;
            }
            .resend-log-detail {
                background: #fff;
                border: 1px solid #c3c4c7;
                padding: 20px;
                margin-top: 20px;
            }
            .resend-log-detail h2 {
                margin-top: 0;
            }
            .resend-log-detail dl {
                display: grid;
                grid-template-columns: 200px 1fr;
                gap: 10px 20px;
            }
            .resend-log-detail dt {
                font-weight: 600;
            }
            .resend-log-detail dd {
                margin: 0;
            }
            .resend-resend-button {
                margin-top: 20px;
            }
        ');
    }

    /**
     * Enqueue admin scripts
     *
     * @param string $hook Current admin page hook
     * @return void
     */
    public function enqueue_scripts(string $hook): void
    {
        if ($hook !== 'tools_page_resend-email-log') {
            return;
        }

        wp_enqueue_script(
            'resend-admin',
            RESEND_WP_PLUGIN_URL . 'assets/admin.js',
            ['jquery'],
            RESEND_WP_VERSION,
            true
        );

        // Include all localized data needed for both SettingsPage and EmailLogPage
        // Since we run with priority 20, we'll overwrite SettingsPage's data, so include everything
        wp_localize_script('resend-admin', 'resendAdmin', [
            'ajaxUrl' => admin_url('admin-ajax.php'),
            'nonce' => wp_create_nonce('resend_send_test_email'),
            'dismissNonce' => wp_create_nonce('resend_dismiss_notice'),
            'resendNonce' => wp_create_nonce('resend_resend_email'),
            'strings' => [
                'sending' => __('Sending test email...', 'mail-via-resend'),
                'success' => __('Test email sent successfully!', 'mail-via-resend'),
                'error' => __('Test email failed: ', 'mail-via-resend'),
                'buttonText' => __('Send Test Email', 'mail-via-resend'),
                'resendConfirm' => __('Are you sure you want to resend this email?', 'mail-via-resend'),
                'resending' => __('Resending...', 'mail-via-resend'),
                'resendSuccess' => __('Email resent successfully!', 'mail-via-resend'),
                'resendError' => __('Failed to resend email: ', 'mail-via-resend'),
                'resendButtonText' => __('Resend Email', 'mail-via-resend'),
            ],
        ]);
    }

    /**
     * Render email log page
     *
     * @return void
     */
    public function render_email_log_page(): void
    {
        if (!current_user_can('manage_options')) {
            return;
        }

        // Check if database table exists
        global $wpdb;
        $table_name = $wpdb->prefix . 'resend_email_log';
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery,WordPress.DB.DirectDatabaseQuery.NoCaching -- Table check requires direct query
        $table_exists = $wpdb->get_var($wpdb->prepare(
            "SHOW TABLES LIKE %s",
            $table_name
        )) === $table_name;
        
        if (!$table_exists) {
            // Try to create the table
            \Resend\WordPress\Database::create_tables();
            
            // Check again
            // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery,WordPress.DB.DirectDatabaseQuery.NoCaching -- Table check requires direct query
            $table_exists = $wpdb->get_var($wpdb->prepare(
                "SHOW TABLES LIKE %s",
                $table_name
            )) === $table_name;
            
            if (!$table_exists) {
                // Show error notice
                ?>
                <div class="wrap">
                    <h1><?php echo esc_html(get_admin_page_title()); ?></h1>
                    <div class="notice notice-error">
                        <p>
                            <strong><?php esc_html_e('Database Error', 'mail-via-resend'); ?></strong><br>
                            <?php esc_html_e('The email log database table could not be created. Please check your database permissions and try deactivating and reactivating the plugin.', 'mail-via-resend'); ?>
                            <?php if (resend_is_debug_mode() && !empty($wpdb->last_error)): ?>
                                <br><code><?php echo esc_html($wpdb->last_error); ?></code>
                            <?php endif; ?>
                        </p>
                    </div>
                </div>
                <?php
                return;
            }
        }

        // Handle view action
        // phpcs:ignore WordPress.Security.NonceVerification.Recommended -- View action is read-only, no data modification
        $action = isset($_GET['action']) ? sanitize_text_field(wp_unslash($_GET['action'])) : '';
        // phpcs:ignore WordPress.Security.NonceVerification.Recommended -- View action is read-only, no data modification
        if ($action === 'view' && isset($_GET['log_id'])) {
            $this->render_log_detail();
            return;
        }

        // Render list table
        $log_repository = $this->container->get(LogRepository::class);
        $list_table = new EmailLogListTable($log_repository);
        $list_table->prepare_items();
    ?>
        <div class="wrap">
            <h1 class="wp-heading-inline"><?php esc_html_e('Email Log', 'mail-via-resend'); ?></h1>
            <hr class="wp-header-end">

            <?php $list_table->search_box(__('Search', 'mail-via-resend'), 'resend-email-log'); ?>
            <?php $list_table->display(); ?>

            <hr style="margin: 30px 0 20px;">

            <div style="background: #f0f0f1; border-left: 4px solid #2271b1; padding: 12px 16px; margin-top: 20px;">
                <p style="margin: 0;">
                    <?php
                    printf(
                        /* translators: %1$s: Opening anchor tag for Webscape, %2$s: Closing anchor tag, %3$s: Opening anchor tag for contact, %4$s: Closing anchor tag */
                        esc_html__('Plugin built by %1$sWebscape%2$s. Need help with WordPress? %3$sContact us%4$s.', 'mail-via-resend'),
                        '<a href="https://webscape.co.nz" target="_blank" rel="noopener noreferrer">',
                        '</a>',
                        '<a href="https://webscape.co.nz/contact" target="_blank" rel="noopener noreferrer">',
                        '</a>'
                    );
                    ?>
                </p>
            </div>
        </div>
    <?php
    }

    /**
     * Render log detail view
     *
     * @return void
     */
    private function render_log_detail(): void
    {
        // phpcs:ignore WordPress.Security.NonceVerification.Recommended -- Read-only view, no data modification
        $log_id = isset($_GET['log_id']) ? absint($_GET['log_id']) : 0;

        if (!$log_id) {
            wp_die(esc_html__('Invalid log ID.', 'mail-via-resend'));
        }

        $log_repository = $this->container->get(LogRepository::class);
        $log = $log_repository->get_by_id($log_id);

        if (!$log) {
            wp_die(esc_html__('Log entry not found.', 'mail-via-resend'));
        }

        $headers = !empty($log->headers) ? json_decode($log->headers, true) : [];
    ?>
        <div class="wrap">
            <h1><?php esc_html_e('Email Log Details', 'mail-via-resend'); ?></h1>

            <p>
                <a href="<?php echo esc_url(admin_url('admin.php?page=resend-email-log')); ?>" class="button">
                    <?php esc_html_e('← Back to Email Log', 'mail-via-resend'); ?>
                </a>
            </p>

            <div class="resend-log-detail">
                <h2><?php esc_html_e('Email Details', 'mail-via-resend'); ?></h2>

                <dl>
                    <dt><?php esc_html_e('To:', 'mail-via-resend'); ?></dt>
                    <dd><?php echo esc_html($log->to_email); ?></dd>

                    <dt><?php esc_html_e('Subject:', 'mail-via-resend'); ?></dt>
                    <dd><?php echo esc_html($log->subject); ?></dd>

                    <dt><?php esc_html_e('Status:', 'mail-via-resend'); ?></dt>
                    <dd>
                        <span class="resend-status resend-status-<?php echo esc_attr($log->status === 'sent' ? 'success' : 'error'); ?>">
                            <?php echo esc_html(ucfirst($log->status)); ?>
                        </span>
                    </dd>

                    <dt><?php esc_html_e('Response ID:', 'mail-via-resend'); ?></dt>
                    <dd><?php echo $log->response_id ? esc_html($log->response_id) : '—'; ?></dd>

                    <dt><?php esc_html_e('Date:', 'mail-via-resend'); ?></dt>
                    <dd><?php echo esc_html(mysql2date(get_option('date_format') . ' ' . get_option('time_format'), $log->created_at)); ?></dd>

                    <?php if ($log->error_message) : ?>
                        <dt><?php esc_html_e('Error:', 'mail-via-resend'); ?></dt>
                        <dd><code><?php echo esc_html($log->error_message); ?></code></dd>
                    <?php endif; ?>

                    <?php if (!empty($headers)) : ?>
                        <dt><?php esc_html_e('Headers:', 'mail-via-resend'); ?></dt>
                        <dd>
                            <pre><?php echo esc_html(wp_json_encode($headers, JSON_PRETTY_PRINT | JSON_UNESCAPED_SLASHES)); ?></pre>
                        </dd>
                    <?php endif; ?>

                    <?php if ($log->body_excerpt) : ?>
                        <dt><?php esc_html_e('Body Excerpt:', 'mail-via-resend'); ?></dt>
                        <dd>
                            <pre><?php echo esc_html($log->body_excerpt); ?></pre>
                        </dd>
                    <?php endif; ?>
                </dl>

                <div class="resend-resend-button">
                    <button type="button" 
                            id="resend-resend-email" 
                            class="button button-primary" 
                            data-log-id="<?php echo esc_attr($log->id); ?>">
                        <?php esc_html_e('Resend Email', 'mail-via-resend'); ?>
                    </button>
                    <span id="resend-resend-result" style="display: none; margin-left: 10px;"></span>
                </div>
            </div>
        </div>
<?php
    }

    /**
     * Rate limit for resending emails (5 minutes)
     */
    private const RESEND_RATE_LIMIT_SECONDS = 5 * MINUTE_IN_SECONDS;

    /**
     * Handle resend email AJAX request
     *
     * @return void
     */
    public function handle_resend_email_ajax(): void {
        // Check permissions
        if (!current_user_can('manage_options')) {
            wp_send_json_error(['message' => __('You do not have sufficient permissions.', 'mail-via-resend')]);
        }

        // Verify nonce
        if (!isset($_POST['nonce']) || !wp_verify_nonce(sanitize_text_field(wp_unslash($_POST['nonce'])), 'resend_resend_email')) {
            wp_send_json_error(['message' => __('Security check failed.', 'mail-via-resend')]);
        }

        // Get log ID
        // phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized -- Will be sanitized with absint
        $log_id = isset($_POST['log_id']) ? absint(wp_unslash($_POST['log_id'])) : 0;

        if (!$log_id) {
            wp_send_json_error(['message' => __('Invalid log ID.', 'mail-via-resend')]);
        }

        // Check rate limit - prevent resending the same email too quickly
        $rate_limit_key = 'resend_rate_limit_' . $log_id;
        $last_resend_time = get_transient($rate_limit_key);
        
        if ($last_resend_time !== false) {
            $time_remaining = self::RESEND_RATE_LIMIT_SECONDS - (time() - $last_resend_time);
            if ($time_remaining > 0) {
                $minutes = ceil($time_remaining / 60);
                wp_send_json_error([
                    'message' => sprintf(
                        /* translators: %d: Number of minutes */
                        __('Please wait %d minute(s) before resending this email again.', 'mail-via-resend'),
                        $minutes
                    ),
                ]);
            }
        }

        // Get log entry
        $log_repository = $this->container->get(LogRepository::class);
        $log = $log_repository->get_by_id($log_id);

        if (!$log) {
            wp_send_json_error(['message' => __('Log entry not found.', 'mail-via-resend')]);
        }

        // Parse headers
        $headers = !empty($log->headers) ? json_decode($log->headers, true) : [];
        if (!is_array($headers)) {
            $headers = [];
        }

        // Convert headers to format expected by ResendMailer
        $formatted_headers = [];
        foreach ($headers as $header) {
            if (is_array($header) && count($header) >= 2) {
                $formatted_headers[] = [$header[0], $header[1]];
            }
        }

        // Use body_excerpt as the email body (it's limited to 500 chars, but that's what we have)
        $body = $log->body_excerpt ?: __('[Email body not available]', 'mail-via-resend');

        // Determine content type from headers
        $is_html = false;
        foreach ($formatted_headers as $header) {
            if (strtolower($header[0]) === 'content-type' && strpos(strtolower($header[1]), 'text/html') !== false) {
                $is_html = true;
                break;
            }
        }

        // If no content type header, add one based on body content
        if (empty(array_filter($formatted_headers, function($h) { return strtolower($h[0]) === 'content-type'; }))) {
            $formatted_headers[] = ['Content-Type', $is_html ? 'text/html; charset=UTF-8' : 'text/plain; charset=UTF-8'];
        }

        // Prepare recipient address
        $to_addresses = [[$log->to_email, '']];

        // Get Resend mailer
        $mailer = $this->container->get(\Resend\WordPress\Mail\ResendMailer::class);

        // Send email
        $result = $mailer->send(
            $to_addresses,
            $log->subject,
            $body,
            [], // No attachments available in log
            $formatted_headers
        );

        if (is_wp_error($result)) {
            wp_send_json_error([
                'message' => $result->get_error_message(),
            ]);
        }

        // Set rate limit transient to prevent immediate duplicate resends
        set_transient($rate_limit_key, time(), self::RESEND_RATE_LIMIT_SECONDS);

        wp_send_json_success([
            'message' => __('Email resent successfully!', 'mail-via-resend'),
        ]);
    }
}
