<?php
/**
 * Plugin Name: Mail forwarder to Telegram
 * Plugin URI: https://wp-guruteam.com/mail-forwarder-to-telegram
 * Description: Forward WordPress emails to Telegram
 * Version: 1.0.0
 * Author: Guru Team
 * Author URI: https://wp-guruteam.com/
 * Text Domain: mail-forwarder-to-telegram
 * Domain Path: /languages
 * Requires at least: 5.8
 * Requires PHP: 7.4
 * License: GPL v2 or later
 * License URI: https://www.gnu.org/licenses/gpl-2.0.html
 */

if (!defined('ABSPATH')) {
	exit; // Exit if accessed directly
}

// Define plugin constants
define('MAIL_FORWARDER_TO_TELEGRAM_VERSION', '1.0.0');
define('MAIL_FORWARDER_TO_TELEGRAM_FILE', __FILE__);
define('MAIL_FORWARDER_TO_TELEGRAM_PATH', plugin_dir_path(__FILE__));
define('MAIL_FORWARDER_TO_TELEGRAM_URL', plugin_dir_url(__FILE__));

// Register activation hook
require_once MAIL_FORWARDER_TO_TELEGRAM_PATH . 'includes/class-activator.php';
register_activation_hook(__FILE__, array('MailForwarderToTelegram_Activator', 'activate'));

/**
 * Main plugin class
 */
class MailForwarderToTelegram
{

	/**
	 * Instance of this class
	 */
	private static $instance = null;

	/**
	 * Get instance
	 */
	public static function get_instance()
	{
		if (null === self::$instance) {
			self::$instance = new self();
		}
		return self::$instance;
	}

	/**
	 * Constructor
	 */
	private function __construct()
	{
		$this->load_dependencies();
		$this->init_hooks();
	}

	/**
	 * Load plugin dependencies
	 */
	private function load_dependencies()
	{
		// Load core classes
		require_once MAIL_FORWARDER_TO_TELEGRAM_PATH . 'includes/class-logger.php';
		require_once MAIL_FORWARDER_TO_TELEGRAM_PATH . 'includes/class-telegram-service.php';
		require_once MAIL_FORWARDER_TO_TELEGRAM_PATH . 'includes/class-email-handler.php';

		// Load admin classes
		if (is_admin()) {
			require_once MAIL_FORWARDER_TO_TELEGRAM_PATH . 'includes/admin/class-settings-page.php';
			require_once MAIL_FORWARDER_TO_TELEGRAM_PATH . 'includes/admin/class-logs-page.php';
		}
	}

	/**
	 * Initialize WordPress hooks
	 */
	private function init_hooks()
	{
		add_action('plugins_loaded', array($this, 'init_email_handler'));

		// Initialize admin
		if (is_admin()) {
			add_action('admin_menu', array($this, 'add_admin_menu'));
			add_action('admin_init', array($this, 'register_settings'));
			add_action('admin_enqueue_scripts', array($this, 'enqueue_admin_scripts'));

			// AJAX actions
			add_action('wp_ajax_mail_forwarder_to_telegram_resend_email', array($this, 'ajax_resend_email'));
			add_action('wp_ajax_mail_forwarder_to_telegram_resend_telegram', array($this, 'ajax_resend_telegram'));
			add_action('wp_ajax_mail_forwarder_to_telegram_send_test_email', array($this, 'ajax_send_test_email'));
		}
	}

	/**
	 * Initialize email handler
	 */
	public function init_email_handler()
	{
		new MailForwarderToTelegram_Email_Handler();
	}

	/**
	 * Add admin menu
	 */
	public function add_admin_menu()
	{
		// Main menu
		add_menu_page(
			__('Mail Forwarder for Telegram Settings', 'mail-forwarder-to-telegram'),
			__('Mail Forwarder for Telegram', 'mail-forwarder-to-telegram'),
			'manage_options',
			'mail-forwarder-to-telegram',
			array('MailForwarderToTelegram_Settings_Page', 'render'),
			'dashicons-email-alt',
			80
		);

		// Settings submenu
		add_submenu_page(
			'mail-forwarder-to-telegram',
			__('Settings', 'mail-forwarder-to-telegram'),
			__('Settings', 'mail-forwarder-to-telegram'),
			'manage_options',
			'mail-forwarder-to-telegram',
			array('MailForwarderToTelegram_Settings_Page', 'render')
		);

		// Logs submenu
		add_submenu_page(
			'mail-forwarder-to-telegram',
			__('Email Logs', 'mail-forwarder-to-telegram'),
			__('Logs', 'mail-forwarder-to-telegram'),
			'manage_options',
			'mail-forwarder-to-telegram-logs',
			array('MailForwarderToTelegram_Logs_Page', 'render')
		);
	}

	/**
	 * Register plugin settings
	 */
	public function register_settings()
	{
		MailForwarderToTelegram_Settings_Page::register_settings();
	}

	/**
	 * Enqueue admin scripts and styles
	 */
	public function enqueue_admin_scripts($hook)
	{
		// Load scripts only on Mail Forwarder to Telegram pages
		if (strpos($hook, 'mail-forwarder-to-telegram') === false) {
			return;
		}

		wp_enqueue_style(
			'mail-forwarder-to-telegram-admin',
			MAIL_FORWARDER_TO_TELEGRAM_URL . 'assets/css/admin.css',
			array(),
			MAIL_FORWARDER_TO_TELEGRAM_VERSION
		);

		wp_enqueue_script(
			'mail-forwarder-to-telegram-admin',
			MAIL_FORWARDER_TO_TELEGRAM_URL . 'assets/js/admin.js',
			array('jquery'),
			MAIL_FORWARDER_TO_TELEGRAM_VERSION,
			true
		);

		// Localize script for AJAX
		wp_localize_script(
			'mail-forwarder-to-telegram-admin',
			'mail_forwarder_to_telegram_ajax',
			array(
				'ajax_url' => admin_url('admin-ajax.php'),
				'test_nonce' => wp_create_nonce('mail_forwarder_to_telegram_test'),
			)
		);
	}

	/**
	 * AJAX: Resend email
	 */
	public function ajax_resend_email()
	{
		// Check nonce
		if (!isset($_POST['nonce']) || !wp_verify_nonce(sanitize_text_field(wp_unslash($_POST['nonce'])), 'mail_forwarder_to_telegram_resend')) {
			wp_send_json_error(array('message' => __('Security check failed.', 'mail-forwarder-to-telegram')));
		}

		// Check permissions
		if (!current_user_can('manage_options')) {
			wp_send_json_error(array('message' => __('You do not have permission.', 'mail-forwarder-to-telegram')));
		}

		// Get log ID
		$log_id = isset($_POST['log_id']) ? intval(wp_unslash($_POST['log_id'])) : 0;

		if (!$log_id) {
			wp_send_json_error(array('message' => __('Invalid log ID.', 'mail-forwarder-to-telegram')));
		}

		// Get log
		$logger = new MailForwarderToTelegram_Logger();
		$log = $logger->get_log($log_id);

		if (!$log) {
			wp_send_json_error(array('message' => __('Log not found.', 'mail-forwarder-to-telegram')));
		}

		// Parse recipients
		$to = $log->to_recipients;

		// Send email
		$result = wp_mail($to, $log->subject, $log->message);

		if ($result) {
			// Update log
			$logger->update_log($log_id, array('email_sent' => 1));

			wp_send_json_success(array('message' => __('Email sent successfully!', 'mail-forwarder-to-telegram')));
		} else {
			wp_send_json_error(array('message' => __('Failed to send email.', 'mail-forwarder-to-telegram')));
		}
	}

	/**
	 * AJAX: Resend to Telegram
	 */
	public function ajax_resend_telegram()
	{
		// Check nonce
		if (!isset($_POST['nonce']) || !wp_verify_nonce(sanitize_text_field(wp_unslash($_POST['nonce'])), 'mail_forwarder_to_telegram_resend')) {
			wp_send_json_error(array('message' => __('Security check failed.', 'mail-forwarder-to-telegram')));
		}

		// Check permissions
		if (!current_user_can('manage_options')) {
			wp_send_json_error(array('message' => __('You do not have permission.', 'mail-forwarder-to-telegram')));
		}

		// Get log ID
		$log_id = isset($_POST['log_id']) ? intval($_POST['log_id']) : 0;

		if (!$log_id) {
			wp_send_json_error(array('message' => __('Invalid log ID.', 'mail-forwarder-to-telegram')));
		}

		// Get log
		$logger = new MailForwarderToTelegram_Logger();
		$log = $logger->get_log($log_id);

		if (!$log) {
			wp_send_json_error(array('message' => __('Log not found.', 'mail-forwarder-to-telegram')));
		}

		// Get settings
		$settings = get_option('mail_forwarder_to_telegram_settings', array());

		if (empty($settings['bot_token']) || empty($settings['chat_id'])) {
			wp_send_json_error(array('message' => __('Telegram not configured. Please check your settings.', 'mail-forwarder-to-telegram')));
		}

		// Initialize Telegram service
		$telegram = new MailForwarderToTelegram_Telegram_Service($settings['bot_token'], $settings['chat_id']);

		// Parse email data from log
		$email_data = array(
			'subject' => $log->subject,
			'to' => $log->to_recipients,
			'message' => $log->message,
			'attachments' => maybe_unserialize($log->attachments),
		);

		// Format and send message (ignore filters - force send)
		$message = $telegram->format_email_message($email_data);
		$result = $telegram->send_message($message);

		if (is_wp_error($result)) {
			wp_send_json_error(array('message' => __('Failed to send to Telegram: ', 'mail-forwarder-to-telegram') . $result->get_error_message()));
		}

		// Send attachments if any
		if (!empty($email_data['attachments']) && is_array($email_data['attachments'])) {
			foreach ($email_data['attachments'] as $attachment) {
				if (file_exists($attachment)) {
					$telegram->send_document($attachment, '📎 ' . $log->subject);
				}
			}
		}

		// Update log
		$logger->update_log($log_id, array('sent_to_telegram' => 1));

		wp_send_json_success(array('message' => __('Message sent to Telegram successfully!', 'mail-forwarder-to-telegram')));
	}

	/**
	 * AJAX: Send test email
	 */
	public function ajax_send_test_email()
	{
		// Check nonce
		if (!isset($_POST['nonce']) || !wp_verify_nonce(sanitize_text_field(wp_unslash($_POST['nonce'])), 'mail_forwarder_to_telegram_test')) {
			wp_send_json_error(array('message' => __('Security check failed.', 'mail-forwarder-to-telegram')));
		}

		// Check permissions
		if (!current_user_can('manage_options')) {
			wp_send_json_error(array('message' => __('You do not have permission.', 'mail-forwarder-to-telegram')));
		}

		// Get email address
		$to_email = isset($_POST['email']) ? sanitize_email(wp_unslash($_POST['email'])) : '';

		if (empty($to_email) || !is_email($to_email)) {
			wp_send_json_error(array('message' => __('Invalid email address.', 'mail-forwarder-to-telegram')));
		}

		// Create test attachment
		$upload_dir = wp_upload_dir();
		$plugin_upload_dir = trailingslashit($upload_dir['basedir']) . 'mail-forwarder-to-telegram';
		if (!file_exists($plugin_upload_dir)) {
			// phpcs:ignore WordPress.WP.AlternativeFunctions.file_system_operations_mkdir
			wp_mkdir_p($plugin_upload_dir);
		}
		$test_file = trailingslashit($plugin_upload_dir) . 'mail_forwarder_to_telegram-test.txt';
		$file_content = "Mail Forwarder to Telegram Test Attachment\n\n";
		$file_content .= "This is a test email sent from Mail Forwarder to Telegram plugin.\n";
		$file_content .= "Date: " . gmdate('Y-m-d H:i:s') . "\n";
		$file_content .= "Recipient: " . $to_email . "\n\n";
		$file_content .= "If you received this email and see it in Telegram, your setup is working correctly!\n";

		// Write test file
		// phpcs:ignore WordPress.WP.AlternativeFunctions.file_system_operations_file_put_contents
		file_put_contents($test_file, $file_content);

		// Prepare test email
		$subject = '[Mail Forwarder to Telegram] Test Email with Attachment';
		$message = "<html><body>";
		$message .= "<h2>Mail Forwarder to Telegram Test Email</h2>";
		$message .= "<p>This is a <strong>test email</strong> sent from the Mail Forwarder to Telegram plugin.</p>";
		$message .= "<p><strong>Date:</strong> " . gmdate('Y-m-d H:i:s') . "</p>";
		$message .= "<p><strong>Recipient:</strong> " . esc_html($to_email) . "</p>";
		$message .= "<hr>";
		$message .= "<p>This email includes an attachment to test the full functionality.</p>";
		$message .= "<ul>";
		$message .= "<li>✅ Email logging to database</li>";
		$message .= "<li>✅ HTML formatting support</li>";
		$message .= "<li>✅ Attachment handling</li>";
		$message .= "<li>✅ Telegram forwarding (if configured)</li>";
		$message .= "</ul>";
		$message .= "<p><em>If you see this in Telegram, everything is working perfectly!</em></p>";
		$message .= "</body></html>";

		// Set email headers
		$headers = array('Content-Type: text/html; charset=UTF-8');

		// Send email
		$result = wp_mail($to_email, $subject, $message, $headers, array($test_file));

		// Delete test file after sending
		if (file_exists($test_file)) {
			// phpcs:ignore WordPress.WP.AlternativeFunctions.unlink_unlink
			unlink($test_file);
		}

		if ($result) {
			wp_send_json_success(
				array(
					'message' => sprintf(
						// translators: %s is the email address
						__('Test email sent successfully to %s! Check your inbox and Telegram (if configured).', 'mail-forwarder-to-telegram'),
						$to_email
					),
				)
			);
		} else {
			wp_send_json_error(array('message' => __('Failed to send test email. Please check your WordPress email configuration.', 'mail-forwarder-to-telegram')));
		}
	}
}

/**
 * Initialize the plugin
 */
function mail_forwarder_to_telegram_init()
{
	return MailForwarderToTelegram::get_instance();
}

// Start the plugin
mail_forwarder_to_telegram_init();
