<?php
/**
 * Telegram Service Class
 *
 * @package MailForwarderToTelegram
 */

if (!defined('ABSPATH')) {
	exit;
}

/**
 * Telegram Service
 * Handles communication with Telegram Bot API
 */
class MailForwarderToTelegram_Telegram_Service
{

	/**
	 * Bot token
	 *
	 * @var string
	 */
	private $bot_token;

	/**
	 * Chat ID
	 *
	 * @var string
	 */
	private $chat_id;

	/**
	 * Telegram API base URL
	 *
	 * @var string
	 */
	private $api_url;

	/**
	 * Allowed HTML tags for Telegram
	 * https://core.telegram.org/bots/api#html-style
	 *
	 * @var array
	 */
	private $allowed_tags = array(
		'b' => array(),
		'strong' => array(),
		'i' => array(),
		'em' => array(),
		'u' => array(),
		'ins' => array(),
		's' => array(),
		'strike' => array(),
		'del' => array(),
		'code' => array(),
		'pre' => array(),
		'a' => array('href' => array()),
	);

	/**
	 * Constructor
	 *
	 * @param string $bot_token Bot token
	 * @param string $chat_id Chat ID
	 */
	public function __construct($bot_token = '', $chat_id = '')
	{
		$this->bot_token = $bot_token;
		$this->chat_id = $chat_id;
		$this->api_url = 'https://api.telegram.org/bot' . $this->bot_token . '/';
	}

	/**
	 * Check if service is configured
	 *
	 * @return bool
	 */
	public function is_configured()
	{
		return !empty($this->bot_token) && !empty($this->chat_id);
	}

	/**
	 * Send message to Telegram
	 *
	 * @param string $message Message text
	 * @return array|WP_Error Response or error
	 */
	public function send_message($message)
	{
		if (!$this->is_configured()) {
			return new WP_Error('not_configured', __('Telegram service is not configured', 'mail-forwarder-to-telegram'));
		}

		// Filter HTML for Telegram
		$filtered_message = $this->filter_html($message);

		// Prepare request
		$args = array(
			'body' => array(
				'chat_id' => $this->chat_id,
				'text' => $filtered_message,
				'parse_mode' => 'HTML',
			),
		);

		// Send request
		$response = wp_remote_post($this->api_url . 'sendMessage', $args);

		if (is_wp_error($response)) {
			return $response;
		}

		$body = json_decode(wp_remote_retrieve_body($response), true);

		if (!isset($body['ok']) || !$body['ok']) {
			$error_message = isset($body['description']) ? $body['description'] : __('Unknown error', 'mail-forwarder-to-telegram');
			return new WP_Error('telegram_error', $error_message);
		}

		return $body;
	}

	/**
	 * Send document to Telegram
	 *
	 * @param string $file_path Path to file
	 * @param string $caption Optional caption
	 * @return array|WP_Error Response or error
	 */
	public function send_document($file_path, $caption = '')
	{
		if (!$this->is_configured()) {
			return new WP_Error('not_configured', __('Telegram service is not configured', 'mail-forwarder-to-telegram'));
		}

		if (!file_exists($file_path)) {
			return new WP_Error('file_not_found', __('File not found', 'mail-forwarder-to-telegram'));
		}

		// Prepare multipart form data
		$boundary = wp_generate_password(24, false);
		$headers = array(
			'Content-Type' => 'multipart/form-data; boundary=' . $boundary,
		);

		// Build multipart body
		$body = '';

		// Add chat_id
		$body .= '--' . $boundary . "\r\n";
		$body .= 'Content-Disposition: form-data; name="chat_id"' . "\r\n\r\n";
		$body .= $this->chat_id . "\r\n";

		// Add caption if provided
		if (!empty($caption)) {
			$body .= '--' . $boundary . "\r\n";
			$body .= 'Content-Disposition: form-data; name="caption"' . "\r\n\r\n";
			$body .= $caption . "\r\n";
		}

		// Add file
		$file_content = file_get_contents($file_path);
		$file_name = basename($file_path);

		$body .= '--' . $boundary . "\r\n";
		$body .= 'Content-Disposition: form-data; name="document"; filename="' . $file_name . '"' . "\r\n";
		$body .= 'Content-Type: application/octet-stream' . "\r\n\r\n";
		$body .= $file_content . "\r\n";
		$body .= '--' . $boundary . '--';

		// Send request
		$args = array(
			'headers' => $headers,
			'body' => $body,
			'timeout' => 30,
		);

		$response = wp_remote_post($this->api_url . 'sendDocument', $args);

		if (is_wp_error($response)) {
			return $response;
		}

		$response_body = json_decode(wp_remote_retrieve_body($response), true);

		if (!isset($response_body['ok']) || !$response_body['ok']) {
			$error_message = isset($response_body['description']) ? $response_body['description'] : __('Unknown error', 'mail-forwarder-to-telegram');
			return new WP_Error('telegram_error', $error_message);
		}

		return $response_body;
	}

	/**
	 * Filter HTML for Telegram
	 * Only allows tags supported by Telegram API
	 *
	 * @param string $html HTML content
	 * @return string Filtered HTML
	 */
	public function filter_html($html)
	{
		// Strip all tags except allowed ones
		$filtered = wp_kses($html, $this->allowed_tags);

		// Telegram has a message limit of 4096 characters
		if (strlen($filtered) > 4096) {
			$filtered = substr($filtered, 0, 4090) . '...';
		}

		return $filtered;
	}

	/**
	 * Format email for Telegram
	 *
	 * @param array $email_data Email data
	 * @return string Formatted message
	 */
	public function format_email_message($email_data)
	{
		$message = '';

		// Subject
		if (!empty($email_data['subject'])) {
			$message .= '<b>📧 ' . esc_html($email_data['subject']) . '</b>' . "\n\n";
		}

		// From
		if (!empty($email_data['from'])) {
			$message .= '<b>From:</b> ' . esc_html($email_data['from']) . "\n";
		}

		// To Recipients
		if (!empty($email_data['to'])) {
			$recipients = is_array($email_data['to']) ? implode(', ', $email_data['to']) : $email_data['to'];
			$message .= '<b>To:</b> ' . esc_html($recipients) . "\n";
		}

		// Cc Recipients
		if (!empty($email_data['cc']) && is_array($email_data['cc'])) {
			$cc_recipients = implode(', ', $email_data['cc']);
			$message .= '<b>Cc:</b> ' . esc_html($cc_recipients) . "\n";
		}

		// Bcc Recipients
		if (!empty($email_data['bcc']) && is_array($email_data['bcc'])) {
			$bcc_recipients = implode(', ', $email_data['bcc']);
			$message .= '<b>Bcc:</b> ' . esc_html($bcc_recipients) . "\n";
		}

		// Message body
		if (!empty($email_data['message'])) {
			$message .= "\n" . '<b>Message:</b>' . "\n";

			// Convert email body to Telegram-safe HTML
			$body = $this->convert_email_body($email_data['message']);
			$message .= $body;
		}

		// Attachments info
		if (!empty($email_data['attachments']) && is_array($email_data['attachments'])) {
			$message .= "\n\n" . '<b>📎 Attachments:</b> ' . count($email_data['attachments']) . ' file(s)';
		}

		return $message;
	}

	/**
	 * Convert email body to Telegram-safe HTML
	 *
	 * @param string $body Email body
	 * @return string Converted body
	 */
	private function convert_email_body($body)
	{
		// If body looks like HTML, try to preserve some formatting
		if (preg_match('/<[^>]+>/', $body)) {
			// Convert common HTML tags to Telegram-supported ones
			$body = str_replace(array('<h1>', '</h1>', '<h2>', '</h2>', '<h3>', '</h3>'), array('<b>', '</b>', '<b>', '</b>', '<b>', '</b>'), $body);
			$body = str_replace(array('<br>', '<br/>', '<br />'), "\n", $body);
			$body = str_replace(array('<p>', '</p>'), array('', "\n\n"), $body);
			$body = str_replace(array('<li>', '</li>'), array('• ', "\n"), $body);

			// Strip all other tags and keep only allowed ones
			$body = $this->filter_html($body);
		} else {
			// Plain text - just escape and add some basic formatting
			$body = esc_html($body);
		}

		// Limit length
		if (strlen($body) > 3000) {
			$body = substr($body, 0, 2995) . '...';
		}

		return $body;
	}

	/**
	 * Check if email should be forwarded
	 *
	 * @param array $email_data Email data
	 * @param array $settings Plugin settings
	 * @return bool
	 */
	public function should_forward_email($email_data, $settings)
	{
		// Extract recipient emails
		$to_emails = array();
		if (!empty($email_data['to'])) {
			if (is_array($email_data['to'])) {
				$to_emails = $email_data['to'];
			} else {
				$to_emails = array($email_data['to']);
			}
		}

		// Check ignore list first
		if (!empty($settings['ignore_emails'])) {
			$ignore_emails = array_map('trim', explode(',', $settings['ignore_emails']));

			foreach ($to_emails as $email) {
				if (in_array(trim($email), $ignore_emails, true)) {
					return false; // Email is in ignore list
				}
			}
		}

		// Check forward mode
		if ('specific' === $settings['forward_mode']) {
			if (empty($settings['forward_emails'])) {
				return false; // No specific emails configured
			}

			$forward_emails = array_map('trim', explode(',', $settings['forward_emails']));

			foreach ($to_emails as $email) {
				if (in_array(trim($email), $forward_emails, true)) {
					return true; // Email is in forward list
				}
			}

			return false; // Email not in forward list
		}

		// Default: forward all (if not in ignore list)
		return true;
	}
}
