<?php
/**
 * Logger Class
 *
 * @package MailForwarderToTelegram
 */

if (!defined('ABSPATH')) {
	exit;
}

/**
 * Logger
 * Handles database logging of emails
 */
class MailForwarderToTelegram_Logger
{

	/**
	 * Table name
	 *
	 * @var string
	 */
	private $table_name;

	/**
	 * Constructor
	 */
	public function __construct()
	{
		global $wpdb;
		$this->table_name = $wpdb->prefix . 'mail_forwarder_to_telegram_logs';
	}

	/**
	 * Get table name
	 *
	 * @return string
	 */
	public function get_table_name()
	{
		return $this->table_name;
	}

	/**
	 * Log email to database
	 *
	 * @param array $email_data Email data
	 * @param bool $email_sent Whether email was sent
	 * @param bool $sent_to_telegram Whether sent to Telegram
	 * @return int|false Log ID or false on failure
	 */
	public function log_email($email_data, $email_sent = true, $sent_to_telegram = false)
	{
		global $wpdb;

		// Prepare recipients (to)
		$to = '';
		if (!empty($email_data['to'])) {
			if (is_array($email_data['to'])) {
				$to = implode(', ', $email_data['to']);
			} else {
				$to = $email_data['to'];
			}
		}

		// Add Cc and Bcc to recipients field
		$all_recipients = array($to);

		if (!empty($email_data['cc']) && is_array($email_data['cc'])) {
			$all_recipients[] = 'Cc: ' . implode(', ', $email_data['cc']);
		}

		if (!empty($email_data['bcc']) && is_array($email_data['bcc'])) {
			$all_recipients[] = 'Bcc: ' . implode(', ', $email_data['bcc']);
		}

		$to_field = implode(' | ', array_filter($all_recipients));

		// Prepare attachments
		$attachments = '';
		if (!empty($email_data['attachments']) && is_array($email_data['attachments'])) {
			$attachments = maybe_serialize($email_data['attachments']);
		}

		// Insert log entry
		$result = $wpdb->insert(
			$this->table_name,
			array(
				'date' => current_time('mysql'),
				'subject' => !empty($email_data['subject']) ? $email_data['subject'] : '',
				'to_recipients' => $to_field,
				'message' => !empty($email_data['message']) ? $email_data['message'] : '',
				'attachments' => $attachments,
				'email_sent' => $email_sent ? 1 : 0,
				'sent_to_telegram' => $sent_to_telegram ? 1 : 0,
			),
			array(
				'%s', // date
				'%s', // subject
				'%s', // to_recipients
				'%s', // message
				'%s', // attachments
				'%d', // email_sent
				'%d', // sent_to_telegram
			)
		);

		if ($result) {
			return $wpdb->insert_id;
		}

		return false;
	}

	/**
	 * Update log entry
	 *
	 * @param int $log_id Log ID
	 * @param array $data Data to update
	 * @return bool
	 */
	public function update_log($log_id, $data)
	{
		global $wpdb;

		$result = $wpdb->update(
			$this->table_name,
			$data,
			array('id' => $log_id),
			array('%d', '%d'),
			array('%d')
		);

		return $result !== false;
	}

	/**
	 * Get log entry by ID
	 *
	 * @param int $log_id Log ID
	 * @return object|null
	 */
	public function get_log($log_id) {
		global $wpdb;

		// Table name is safe as it's set in constructor with wpdb prefix
		// Only use prepare() for user-supplied values (log_id)
		$sql = $wpdb->prepare(
			"SELECT * FROM {$this->table_name} WHERE id = %d",
			$log_id
		);

		return $wpdb->get_row($sql);
	}

	/**
	 * Get all logs with pagination
	 *
	 * @param int $per_page Items per page
	 * @param int $page_number Current page number
	 * @param array $filters Optional filters
	 * @return array
	 */
	public function get_logs($per_page = 20, $page_number = 1, $filters = array())
	{
		global $wpdb;

		$offset = ($page_number - 1) * $per_page;

		$where_clauses = array();
		$prepare_values = array();

		// Apply filters
		if (!empty($filters['search'])) {
			$search = '%' . $wpdb->esc_like($filters['search']) . '%';
			$where_clauses[] = '(subject LIKE %s OR to_recipients LIKE %s OR message LIKE %s)';
			$prepare_values[] = $search;
			$prepare_values[] = $search;
			$prepare_values[] = $search;
		}

		if (isset($filters['email_sent']) && '' !== $filters['email_sent']) {
			$where_clauses[] = 'email_sent = %d';
			$prepare_values[] = intval($filters['email_sent']);
		}

		if (isset($filters['sent_to_telegram']) && '' !== $filters['sent_to_telegram']) {
			$where_clauses[] = 'sent_to_telegram = %d';
			$prepare_values[] = intval($filters['sent_to_telegram']);
		}

		// Build WHERE clause with placeholders
		$where_sql = '';
		if (!empty($where_clauses)) {
			$where_sql = 'WHERE ' . implode(' AND ', $where_clauses);
		}

		// Add pagination values to prepare array
		$prepare_values[] = $per_page;
		$prepare_values[] = $offset;

		// Build query with placeholders - table name is safe as it's set in constructor
		$sql = "SELECT * FROM {$this->table_name} {$where_sql} ORDER BY date DESC LIMIT %d OFFSET %d";

		// Prepare and execute query with all values
		$logs = $wpdb->get_results(
			$wpdb->prepare(
				$sql,
				$prepare_values
			)
		);

		return $logs;
	}

	/**
	 * Get total logs count
	 *
	 * @param array $filters Optional filters
	 * @return int
	 */
	public function get_logs_count($filters = array())
	{
		global $wpdb;

		$where_clauses = array();
		$prepare_values = array();

		// Apply filters
		if (!empty($filters['search'])) {
			$search = '%' . $wpdb->esc_like($filters['search']) . '%';
			$where_clauses[] = '(subject LIKE %s OR to_recipients LIKE %s OR message LIKE %s)';
			$prepare_values[] = $search;
			$prepare_values[] = $search;
			$prepare_values[] = $search;
		}

		if (isset($filters['email_sent']) && '' !== $filters['email_sent']) {
			$where_clauses[] = 'email_sent = %d';
			$prepare_values[] = intval($filters['email_sent']);
		}

		if (isset($filters['sent_to_telegram']) && '' !== $filters['sent_to_telegram']) {
			$where_clauses[] = 'sent_to_telegram = %d';
			$prepare_values[] = intval($filters['sent_to_telegram']);
		}

		// Build WHERE clause with placeholders
		$where_sql = '';
		if (!empty($where_clauses)) {
			$where_sql = 'WHERE ' . implode(' AND ', $where_clauses);
		}

		// Build query with placeholders - table name is safe as it's set in constructor
		$sql = "SELECT COUNT(*) FROM {$this->table_name} {$where_sql}";

		// Prepare and execute query with all values
		if (!empty($prepare_values)) {
			return (int)$wpdb->get_var(
				$wpdb->prepare($sql, $prepare_values)
			);
		} else {
			// No dynamic values, execute directly
			return (int)$wpdb->get_var($sql);
		}
	}

	/**
	 * Delete log entry
	 *
	 * @param int $log_id Log ID
	 * @return bool
	 */
	public function delete_log($log_id)
	{
		global $wpdb;

		$result = $wpdb->delete(
			$this->table_name,
			array('id' => $log_id),
			array('%d')
		);

		return $result !== false;
	}

	/**
	 * Delete old logs
	 *
	 * @param int $days Days to keep
	 * @return int Number of deleted rows
	 */
	public function delete_old_logs($days = 30)
	{
		global $wpdb;

		$date = gmdate('Y-m-d H:i:s', strtotime("-{$days} days"));

		// Table name is safe as it's set in constructor with wpdb prefix
		// Only use prepare() for user-supplied values (date)
		$result = $wpdb->query(
			$wpdb->prepare(
				"DELETE FROM {$this->table_name} WHERE date < %s",
				$date
			)
		);

		return $result;
	}
}
