<?php
/**
 * Email Handler Class
 *
 * @package MailForwarderToTelegram
 */

if (!defined('ABSPATH')) {
	exit;
}

/**
 * Email Handler
 * Captures WordPress emails and forwards them to Telegram
 */
class MailForwarderToTelegram_Email_Handler
{

	/**
	 * Telegram service instance
	 *
	 * @var MailForwarderToTelegram_Telegram_Service
	 */
	private $telegram_service;

	/**
	 * Logger instance
	 *
	 * @var MailForwarderToTelegram_Logger
	 */
	private $logger;

	/**
	 * Plugin settings
	 *
	 * @var array
	 */
	private $settings;

	/**
	 * Constructor
	 */
	public function __construct()
	{
		$this->load_settings();
		$this->init_telegram_service();
		$this->init_logger();
		$this->init_hooks();
	}

	/**
	 * Initialize logger
	 */
	private function init_logger()
	{
		$this->logger = new MailForwarderToTelegram_Logger();
	}

	/**
	 * Load plugin settings
	 */
	private function load_settings()
	{
		$this->settings = get_option('mail_forwarder_to_telegram_settings', array());
	}

	/**
	 * Initialize Telegram service
	 */
	private function init_telegram_service()
	{
		$bot_token = isset($this->settings['bot_token']) ? $this->settings['bot_token'] : '';
		$chat_id = isset($this->settings['chat_id']) ? $this->settings['chat_id'] : '';

		$this->telegram_service = new MailForwarderToTelegram_Telegram_Service($bot_token, $chat_id);
	}

	/**
	 * Initialize WordPress hooks
	 */
	private function init_hooks()
	{
		// Hook into wp_mail to capture emails
		add_action('wp_mail_succeeded', array($this, 'handle_mail_succeeded'), 10, 1);
		add_action('wp_mail_failed', array($this, 'handle_mail_failed'), 10, 1);
	}

	/**
	 * Handle successful email
	 *
	 * @param array $mail_data Email data from wp_mail
	 */
	public function handle_mail_succeeded($mail_data)
	{
		// Parse email data
		$email_data = $this->parse_mail_data($mail_data);

		// Check if should forward to Telegram
		$sent_to_telegram = false;

		if ($this->telegram_service->is_configured() && $this->telegram_service->should_forward_email($email_data, $this->settings)) {
			// Send to Telegram
			$sent_to_telegram = $this->send_to_telegram($email_data);
		}

		// Log email to database
		$this->logger->log_email($email_data, true, $sent_to_telegram);
	}

	/**
	 * Handle failed email
	 *
	 * @param WP_Error $error Error object
	 */
	public function handle_mail_failed($error)
	{
		// Extract error data
		$error_data = $error->get_error_data();

		if (empty($error_data)) {
			return;
		}

		// Parse email data
		$email_data = $this->parse_mail_data($error_data);

		// Log email as failed (never sent to Telegram)
		$this->logger->log_email($email_data, false, false);
	}

	/**
	 * Parse mail data
	 *
	 * @param array $mail_data Mail data
	 * @return array Parsed email data
	 */
	private function parse_mail_data($mail_data)
	{
		$email_data = array(
			'to' => isset($mail_data['to']) ? $mail_data['to'] : '',
			'subject' => isset($mail_data['subject']) ? $mail_data['subject'] : '',
			'message' => isset($mail_data['message']) ? $mail_data['message'] : '',
			'headers' => isset($mail_data['headers']) ? $mail_data['headers'] : array(),
			'attachments' => isset($mail_data['attachments']) ? $mail_data['attachments'] : array(),
		);

		// Extract 'from' from headers
		$email_data['from'] = $this->extract_from_header($email_data['headers']);

		// Extract 'cc' from headers
		$email_data['cc'] = $this->extract_cc_header($email_data['headers']);

		// Extract 'bcc' from headers
		$email_data['bcc'] = $this->extract_bcc_header($email_data['headers']);

		return $email_data;
	}

	/**
	 * Extract 'from' email from headers
	 *
	 * @param array|string $headers Email headers
	 * @return string From email
	 */
	private function extract_from_header($headers)
	{
		if (empty($headers)) {
			return '';
		}

		// Convert to array if string
		if (is_string($headers)) {
			$headers = explode("\n", $headers);
		}

		foreach ($headers as $header) {
			if (stripos($header, 'From:') === 0) {
				$from = trim(str_ireplace('From:', '', $header));
				return $from;
			}
		}

		return '';
	}

	/**
	 * Extract 'cc' emails from headers
	 *
	 * @param array|string $headers Email headers
	 * @return array Cc emails
	 */
	private function extract_cc_header($headers)
	{
		if (empty($headers)) {
			return array();
		}

		// Convert to array if string
		if (is_string($headers)) {
			$headers = explode("\n", $headers);
		}

		foreach ($headers as $header) {
			if (stripos($header, 'Cc:') === 0) {
				$cc = trim(str_ireplace('Cc:', '', $header));
				// Split by comma and trim
				$cc_array = array_map('trim', explode(',', $cc));
				return array_filter($cc_array); // Remove empty values
			}
		}

		return array();
	}

	/**
	 * Extract 'bcc' emails from headers
	 *
	 * @param array|string $headers Email headers
	 * @return array Bcc emails
	 */
	private function extract_bcc_header($headers)
	{
		if (empty($headers)) {
			return array();
		}

		// Convert to array if string
		if (is_string($headers)) {
			$headers = explode("\n", $headers);
		}

		foreach ($headers as $header) {
			if (stripos($header, 'Bcc:') === 0) {
				$bcc = trim(str_ireplace('Bcc:', '', $header));
				// Split by comma and trim
				$bcc_array = array_map('trim', explode(',', $bcc));
				return array_filter($bcc_array); // Remove empty values
			}
		}

		return array();
	}

	/**
	 * Log email to database or file
	 *
	 * @param array $email_data Email data
	 * @param bool $success Whether email was sent successfully
	 */
	private function log_email($email_data, $success)
	{
		// For now, just use error_log
		// Later this can be expanded to database logging
		$log_message = sprintf(
			'[Mail Forwarder to Telegram] Email %s - To: %s, Subject: %s',
			$success ? 'sent' : 'failed',
			is_array($email_data['to']) ? implode(', ', $email_data['to']) : $email_data['to'],
			$email_data['subject']
		);
	}

	/**
	 * Send email to Telegram
	 *
	 * @param array $email_data Email data
	 * @return bool Success status
	 */
	private function send_to_telegram($email_data)
	{
		// Format message for Telegram
		$message = $this->telegram_service->format_email_message($email_data);

		// Send message
		$result = $this->telegram_service->send_message($message);

		if (is_wp_error($result)) {
			return false;
		}

		// Send attachments if any
		if (!empty($email_data['attachments']) && is_array($email_data['attachments'])) {
			$this->send_attachments($email_data['attachments'], $email_data['subject']);
		}
		return true;
	}

	/**
	 * Send attachments to Telegram
	 *
	 * @param array $attachments Array of file paths
	 * @param string $caption Optional caption
	 */
	private function send_attachments($attachments, $caption = '')
	{
		foreach ($attachments as $attachment) {
			if (!file_exists($attachment)) {
				continue;
			}

			// Prepare caption
			$file_caption = !empty($caption) ? '📎 ' . $caption : '📎 Attachment';

			// Send document
			$result = $this->telegram_service->send_document($attachment, $file_caption);

			// Small delay between files to avoid rate limiting
			if (count($attachments) > 1) {
				sleep(1);
			}
		}
	}
}
