<?php
/**
 * Settings Page Class
 *
 * @package MailForwarderToTelegram
 */

if (!defined('ABSPATH')) {
	exit;
}

/**
 * MailForwarderToTelegram Settings Page
 */
class MailForwarderToTelegram_Settings_Page
{

	/**
	 * Settings option name
	 */
	const OPTION_NAME = 'mail_forwarder_to_telegram_settings';

	/**
	 * Register settings
	 */
	public static function register_settings()
	{
		register_setting(
			'mail_forwarder_to_telegram_settings_group',
			self::OPTION_NAME,
			array(
				'type' => 'array',
				'sanitize_callback' => array(__CLASS__, 'sanitize_settings'),
				'default' => self::get_default_settings(),
			)
		);

		// General Settings Section
		add_settings_section(
			'mail_forwarder_to_telegram_general_section',
			__('Telegram Settings', 'mail-forwarder-to-telegram'),
			array(__CLASS__, 'render_general_section'),
			'mail-forwarder-to-telegram'
		);

		// Bot Token Field
		add_settings_field(
			'bot_token',
			__('Telegram Bot Token', 'mail-forwarder-to-telegram'),
			array(__CLASS__, 'render_bot_token_field'),
			'mail-forwarder-to-telegram',
			'mail_forwarder_to_telegram_general_section'
		);

		// Chat ID Field
		add_settings_field(
			'chat_id',
			__('Telegram Chat ID', 'mail-forwarder-to-telegram'),
			array(__CLASS__, 'render_chat_id_field'),
			'mail-forwarder-to-telegram',
			'mail_forwarder_to_telegram_general_section'
		);

		// Forwarding Settings Section
		add_settings_section(
			'mail_forwarder_to_telegram_forwarding_section',
			__('Forwarding Settings', 'mail-forwarder-to-telegram'),
			array(__CLASS__, 'render_forwarding_section'),
			'mail-forwarder-to-telegram'
		);

		// Forward Mode Field
		add_settings_field(
			'forward_mode',
			__('Forward Mode', 'mail-forwarder-to-telegram'),
			array(__CLASS__, 'render_forward_mode_field'),
			'mail-forwarder-to-telegram',
			'mail_forwarder_to_telegram_forwarding_section'
		);

		// Forward Emails Field
		add_settings_field(
			'forward_emails',
			__('Forward Specific Emails', 'mail-forwarder-to-telegram'),
			array(__CLASS__, 'render_forward_emails_field'),
			'mail-forwarder-to-telegram',
			'mail_forwarder_to_telegram_forwarding_section'
		);

		// Ignore Emails Field
		add_settings_field(
			'ignore_emails',
			__('Ignore Emails', 'mail-forwarder-to-telegram'),
			array(__CLASS__, 'render_ignore_emails_field'),
			'mail-forwarder-to-telegram',
			'mail_forwarder_to_telegram_forwarding_section'
		);

		// Testing Section
		add_settings_section(
			'mail_forwarder_to_telegram_testing_section',
			__('Testing', 'mail-forwarder-to-telegram'),
			array(__CLASS__, 'render_testing_section'),
			'mail-forwarder-to-telegram'
		);

		// Test Email Field
		add_settings_field(
			'test_email',
			__('Send Test Email', 'mail-forwarder-to-telegram'),
			array(__CLASS__, 'render_test_email_field'),
			'mail-forwarder-to-telegram',
			'mail_forwarder_to_telegram_testing_section'
		);
	}

	/**
	 * Get default settings
	 */
	public static function get_default_settings()
	{
		return array(
			'bot_token' => '',
			'chat_id' => '',
			'forward_mode' => 'all',
			'forward_emails' => '',
			'ignore_emails' => '',
		);
	}

	/**
	 * Get settings
	 */
	public static function get_settings()
	{
		$settings = get_option(self::OPTION_NAME, self::get_default_settings());
		return wp_parse_args($settings, self::get_default_settings());
	}

	/**
	 * Sanitize settings
	 */
	public static function sanitize_settings($input)
	{
		$sanitized = array();

		// Sanitize bot token
		if (isset($input['bot_token'])) {
			$sanitized['bot_token'] = sanitize_text_field($input['bot_token']);
		}

		// Sanitize chat ID
		if (isset($input['chat_id'])) {
			$sanitized['chat_id'] = sanitize_text_field($input['chat_id']);
		}

		// Sanitize forward mode
		if (isset($input['forward_mode'])) {
			$sanitized['forward_mode'] = in_array($input['forward_mode'], array('all', 'specific'), true)
				? $input['forward_mode']
				: 'all';
		}

		// Sanitize forward emails
		if (isset($input['forward_emails'])) {
			$sanitized['forward_emails'] = sanitize_textarea_field($input['forward_emails']);
		}

		// Sanitize ignore emails
		if (isset($input['ignore_emails'])) {
			$sanitized['ignore_emails'] = sanitize_textarea_field($input['ignore_emails']);
		}

		return $sanitized;
	}

	/**
	 * Render settings page
	 */
	public static function render()
	{
		if (!current_user_can('manage_options')) {
			wp_die(esc_html__('You do not have sufficient permissions to access this page.', 'mail-forwarder-to-telegram'));
		}
		?>
		<div class="wrap mail_forwarder_to_telegram-settings">
			<h1><?php echo esc_html(get_admin_page_title()); ?></h1>

			<?php settings_errors('mail_forwarder_to_telegram_settings'); ?>

			<form method="post" action="options.php">
				<?php
				settings_fields('mail_forwarder_to_telegram_settings_group');
				do_settings_sections('mail-forwarder-to-telegram');
				submit_button();
				?>
			</form>

			<div class="mail_forwarder_to_telegram-help">
				<h2><?php esc_html_e('How to Get Your Telegram Bot Token and Chat ID', 'mail-forwarder-to-telegram'); ?></h2>
				<ol>
					<li>
						<strong><?php esc_html_e('Create a Telegram Bot:', 'mail-forwarder-to-telegram'); ?></strong>
						<ul>
							<li><?php esc_html_e('Open Telegram and search for @BotFather', 'mail-forwarder-to-telegram'); ?></li>
							<li><?php esc_html_e('Send /newbot command', 'mail-forwarder-to-telegram'); ?></li>
							<li><?php esc_html_e('Follow the instructions and copy your bot token', 'mail-forwarder-to-telegram'); ?></li>
						</ul>
					</li>
					<li>
						<strong><?php esc_html_e('Get Your Chat ID:', 'mail-forwarder-to-telegram'); ?></strong>
						<ul>
							<li><?php esc_html_e('Start a chat with your bot', 'mail-forwarder-to-telegram'); ?></li>
							<li><?php esc_html_e('Search for @userinfobot on Telegram', 'mail-forwarder-to-telegram'); ?></li>
							<li><?php esc_html_e('Start the bot and it will show your Chat ID', 'mail-forwarder-to-telegram'); ?></li>
						</ul>
					</li>
				</ol>
			</div>
		</div>
		<?php
	}

	/**
	 * Render general section
	 */
	public static function render_general_section()
	{
		echo '<p>' . esc_html__('Configure your Telegram bot credentials.', 'mail-forwarder-to-telegram') . '</p>';
	}

	/**
	 * Render forwarding section
	 */
	public static function render_forwarding_section()
	{
		echo '<p>' . esc_html__('Configure which emails should be forwarded to Telegram.', 'mail-forwarder-to-telegram') . '</p>';
	}

	/**
	 * Render testing section
	 */
	public static function render_testing_section()
	{
		echo '<p>' . esc_html__('Test your email forwarding configuration.', 'mail-forwarder-to-telegram') . '</p>';
	}

	/**
	 * Render bot token field
	 */
	public static function render_bot_token_field()
	{
		$settings = self::get_settings();
		$value = isset($settings['bot_token']) ? $settings['bot_token'] : '';
		?>
		<input
				type="text"
				name="<?php echo esc_attr(self::OPTION_NAME . '[bot_token]'); ?>"
				id="mail_forwarder_to_telegram_bot_token"
				value="<?php echo esc_attr($value); ?>"
				class="regular-text"
				placeholder="123456789:ABCdefGHIjklMNOpqrsTUVwxyz"
		/>
		<p class="description">
			<?php esc_html_e('Enter your Telegram bot token from @BotFather', 'mail-forwarder-to-telegram'); ?>
		</p>
		<?php
	}

	/**
	 * Render chat ID field
	 */
	public static function render_chat_id_field()
	{
		$settings = self::get_settings();
		$value = isset($settings['chat_id']) ? $settings['chat_id'] : '';
		?>
		<input
				type="text"
				name="<?php echo esc_attr(self::OPTION_NAME . '[chat_id]'); ?>"
				id="mail_forwarder_to_telegram_chat_id"
				value="<?php echo esc_attr($value); ?>"
				class="regular-text"
				placeholder="123456789"
		/>
		<p class="description">
			<?php esc_html_e('Enter your Telegram Chat ID', 'mail-forwarder-to-telegram'); ?>
		</p>
		<?php
	}

	/**
	 * Render forward mode field
	 */
	public static function render_forward_mode_field()
	{
		$settings = self::get_settings();
		$value = isset($settings['forward_mode']) ? $settings['forward_mode'] : 'all';
		?>
		<select
				name="<?php echo esc_attr(self::OPTION_NAME . '[forward_mode]'); ?>"
				id="mail_forwarder_to_telegram_forward_mode"
		>
			<option value="all" <?php selected($value, 'all'); ?>>
				<?php esc_html_e('Forward all emails', 'mail-forwarder-to-telegram'); ?>
			</option>
			<option value="specific" <?php selected($value, 'specific'); ?>>
				<?php esc_html_e('Forward only specific emails', 'mail-forwarder-to-telegram'); ?>
			</option>
		</select>
		<p class="description">
			<?php esc_html_e('Choose whether to forward all emails or only specific ones', 'mail-forwarder-to-telegram'); ?>
		</p>
		<?php
	}

	/**
	 * Render forward emails field
	 */
	public static function render_forward_emails_field()
	{
		$settings = self::get_settings();
		$value = isset($settings['forward_emails']) ? $settings['forward_emails'] : '';
		$mode = isset($settings['forward_mode']) ? $settings['forward_mode'] : 'all';
		?>
		<div id="mail_forwarder_to_telegram_forward_emails_wrapper" <?php echo ('all' === $mode) ? 'style="display:none;"' : ''; ?>>
            <textarea
					name="<?php echo esc_attr(self::OPTION_NAME . '[forward_emails]'); ?>"
					id="mail_forwarder_to_telegram_forward_emails"
					rows="5"
					class="large-text"
					placeholder="user@example.com, admin@example.com, contact@example.com"
			><?php echo esc_textarea($value); ?></textarea>
			<p class="description">
				<?php esc_html_e('Enter email addresses separated by commas. Only emails sent to these addresses will be forwarded to Telegram.', 'mail-forwarder-to-telegram'); ?>
			</p>
		</div>
		<?php
	}

	/**
	 * Render ignore emails field
	 */
	public static function render_ignore_emails_field()
	{
		$settings = self::get_settings();
		$value = isset($settings['ignore_emails']) ? $settings['ignore_emails'] : '';
		?>
		<textarea
				name="<?php echo esc_attr(self::OPTION_NAME . '[ignore_emails]'); ?>"
				id="mail_forwarder_to_telegram_ignore_emails"
				rows="5"
				class="large-text"
				placeholder="noreply@example.com, spam@example.com"
		><?php echo esc_textarea($value); ?></textarea>
		<p class="description">
			<?php esc_html_e('Enter email addresses separated by commas. Emails sent to these addresses will NOT be forwarded to Telegram.', 'mail-forwarder-to-telegram'); ?>
		</p>
		<?php
	}

	/**
	 * Render test email field
	 */
	public static function render_test_email_field()
	{
		?>
		<div class="mail_forwarder_to_telegram-test-email">
			<input
					type="email"
					id="mail_forwarder_to_telegram_test_email_address"
					class="regular-text"
					placeholder="test@example.com"
					value="<?php echo esc_attr(get_option('admin_email')); ?>"
			/>
			<button type="button" id="mail_forwarder_to_telegram_send_test_email" class="button button-secondary">
				<?php esc_html_e('Send Test Email with Attachment', 'mail-forwarder-to-telegram'); ?>
			</button>
			<span class="spinner" style="float: none; margin: 0 10px;"></span>
			<p class="description">
				<?php esc_html_e('This will send a test email with an attachment to verify your setup. The email will be logged and forwarded to Telegram according to your settings.', 'mail-forwarder-to-telegram'); ?>
			</p>
			<div id="mail_forwarder_to_telegram_test_result" style="margin-top: 10px;"></div>
		</div>
		<?php
	}
}
