<?php
/**
 * Logs Page Class
 *
 * @package MailForwarderToTelegram
 */

if (!defined('ABSPATH')) {
	exit;
}

/**
 * MailForwarderToTelegram Logs Page
 */
class MailForwarderToTelegram_Logs_Page
{

	/**
	 * Enqueue assets for logs page
	 */
	public static function enqueue_assets()
	{
		// Enqueue CSS
		wp_enqueue_style(
			'mail-forwarder-to-telegram-logs',
			MAIL_FORWARDER_TO_TELEGRAM_URL . 'assets/css/logs-page.css',
			array(),
			MAIL_FORWARDER_TO_TELEGRAM_VERSION
		);

		// Enqueue JS
		wp_enqueue_script(
			'mail-forwarder-to-telegram-logs',
			MAIL_FORWARDER_TO_TELEGRAM_URL . 'assets/js/logs-page.js',
			array('jquery'),
			MAIL_FORWARDER_TO_TELEGRAM_VERSION,
			true
		);

		// Localize script with PHP variables
		wp_localize_script(
			'mail-forwarder-to-telegram-logs',
			'mail_forwarder_to_telegramLogs',
			array(
				'ajaxUrl' => admin_url('admin-ajax.php'),
				'nonce' => wp_create_nonce('mail_forwarder_to_telegram_resend'),
				'confirmResendEmail' => __('Are you sure you want to resend this email?', 'mail-forwarder-to-telegram'),
				'confirmResendTelegram' => __('Are you sure you want to resend this message to Telegram?', 'mail-forwarder-to-telegram'),
				'sendingText' => __('Sending...', 'mail-forwarder-to-telegram'),
				'errorText' => __('Error:', 'mail-forwarder-to-telegram'),
				'ajaxErrorText' => __('AJAX error occurred.', 'mail-forwarder-to-telegram'),
			)
		);
	}

	/**
	 * Render logs page
	 */
	public static function render()
	{
		// Enqueue assets
		self::enqueue_assets();
		if (!current_user_can('manage_options')) {
			wp_die(esc_html__('You do not have sufficient permissions to access this page.', 'mail-forwarder-to-telegram'));
		}

		$logger = new MailForwarderToTelegram_Logger();

		// Handle bulk actions
		if (isset($_POST['action']) && isset($_POST['log_ids']) && check_admin_referer('mail_forwarder_to_telegram_bulk_action')) {
			$action = sanitize_text_field(wp_unslash($_POST['action']));
			$log_ids = array_map('intval', wp_unslash($_POST['log_ids']));

			if ('delete' === $action) {
				foreach ($log_ids as $log_id) {
					$logger->delete_log($log_id);
				}
				echo '<div class="notice notice-success is-dismissible"><p>' . esc_html__('Selected logs deleted.', 'mail-forwarder-to-telegram') . '</p></div>';
			}
		}

		// Handle single delete
		if (isset($_GET['action']) && 'delete' === $_GET['action'] && isset($_GET['log_id']) && check_admin_referer('delete_log_' . intval($_GET['log_id']))) {
			$log_id = intval($_GET['log_id']);
			$logger->delete_log($log_id);
			echo '<div class="notice notice-success is-dismissible"><p>' . esc_html__('Log deleted.', 'mail-forwarder-to-telegram') . '</p></div>';
		}

		// Get filters
		$filters = array();

		if (isset($_GET['search']) && !empty($_GET['search'])) {
			$filters['search'] = sanitize_text_field(wp_unslash($_GET['search']));
		}

		if (isset($_GET['email_sent']) && '' !== $_GET['email_sent']) {
			$filters['email_sent'] = intval($_GET['email_sent']);
		}

		if (isset($_GET['sent_to_telegram']) && '' !== $_GET['sent_to_telegram']) {
			$filters['sent_to_telegram'] = intval($_GET['sent_to_telegram']);
		}

		// Pagination
		$per_page = 20;
		$page_number = isset($_GET['paged']) ? absint($_GET['paged']) : 1;

		// Get logs
		$logs = $logger->get_logs($per_page, $page_number, $filters);
		$total_logs = $logger->get_logs_count($filters);
		$total_pages = ceil($total_logs / $per_page);

		?>
		<div class="wrap mail_forwarder_to_telegram-logs">
			<h1><?php echo esc_html(get_admin_page_title()); ?></h1>

			<div class="tablenav top">
				<div class="alignleft actions">
					<form method="get">
						<input type="hidden" name="page" value="mail-forwarder-to-telegram-logs"/>
						<input type="search" name="search"
							   value="<?php echo isset($filters['search']) ? esc_attr($filters['search']) : ''; ?>"
							   placeholder="<?php esc_attr_e('Search logs...', 'mail-forwarder-to-telegram'); ?>"/>

						<select name="email_sent">
							<option value=""><?php esc_html_e('All Email Status', 'mail-forwarder-to-telegram'); ?></option>
							<option value="1" <?php selected(isset($filters['email_sent']) && 1 === $filters['email_sent']); ?>><?php esc_html_e('Sent', 'mail-forwarder-to-telegram'); ?></option>
							<option value="0" <?php selected(isset($filters['email_sent']) && 0 === $filters['email_sent']); ?>><?php esc_html_e('Failed', 'mail-forwarder-to-telegram'); ?></option>
						</select>

						<select name="sent_to_telegram">
							<option value=""><?php esc_html_e('All Telegram Status', 'mail-forwarder-to-telegram'); ?></option>
							<option value="1" <?php selected(isset($filters['sent_to_telegram']) && 1 === $filters['sent_to_telegram']); ?>><?php esc_html_e('Sent to Telegram', 'mail-forwarder-to-telegram'); ?></option>
							<option value="0" <?php selected(isset($filters['sent_to_telegram']) && 0 === $filters['sent_to_telegram']); ?>><?php esc_html_e('Not Sent', 'mail-forwarder-to-telegram'); ?></option>
						</select>

						<button type="submit" class="button"><?php esc_html_e('Filter', 'mail-forwarder-to-telegram'); ?></button>
						<?php if (!empty($filters)) : ?>
							<a href="?page=mail-forwarder-to-telegram-logs"
							   class="button"><?php esc_html_e('Clear Filters', 'mail-forwarder-to-telegram'); ?></a>
						<?php endif; ?>
					</form>
				</div>

				<?php if ($total_pages > 1) : ?>
					<div class="tablenav-pages">
                        <span class="displaying-num">
                            <?php
							// translators: %s is the number of log items
							echo esc_html(sprintf(_n('%s item', '%s items', $total_logs, 'mail-forwarder-to-telegram'), number_format_i18n($total_logs)));
							?>
                        </span>
						<?php
						echo wp_kses_post(paginate_links(array(
							'base' => add_query_arg('paged', '%#%'),
							'format' => '',
							'prev_text' => '&laquo;',
							'next_text' => '&raquo;',
							'total' => $total_pages,
							'current' => $page_number,
						)));
						?>
					</div>
				<?php endif; ?>
			</div>

			<form method="post">
				<?php wp_nonce_field('mail_forwarder_to_telegram_bulk_action'); ?>

				<table class="wp-list-table widefat fixed striped">
					<thead>
					<tr>
						<td class="manage-column column-cb check-column">
							<input type="checkbox" id="cb-select-all"/>
						</td>
						<th><?php esc_html_e('Date', 'mail-forwarder-to-telegram'); ?></th>
						<th><?php esc_html_e('Subject', 'mail-forwarder-to-telegram'); ?></th>
						<th><?php esc_html_e('To', 'mail-forwarder-to-telegram'); ?></th>
						<th><?php esc_html_e('Attachments', 'mail-forwarder-to-telegram'); ?></th>
						<th><?php esc_html_e('Email Sent', 'mail-forwarder-to-telegram'); ?></th>
						<th><?php esc_html_e('Sent to Telegram', 'mail-forwarder-to-telegram'); ?></th>
						<th><?php esc_html_e('Actions', 'mail-forwarder-to-telegram'); ?></th>
					</tr>
					</thead>
					<tbody>
					<?php if (!empty($logs)) : ?>
						<?php foreach ($logs as $log) : ?>
							<tr>
								<th class="check-column">
									<input type="checkbox" name="log_ids[]" value="<?php echo esc_attr($log->id); ?>"/>
								</th>
								<td><?php echo esc_html($log->date); ?></td>
								<td>
									<strong><?php echo esc_html($log->subject); ?></strong>
									<div class="row-actions">
                                            <span class="view">
                                                <a href="#" class="view-log"
												   data-log-id="<?php echo esc_attr($log->id); ?>"><?php esc_html_e('View', 'mail-forwarder-to-telegram'); ?></a> |
                                            </span>
										<span class="delete">
                                                <a href="<?php echo esc_url(wp_nonce_url(admin_url('admin.php?page=mail-forwarder-to-telegram-logs&action=delete&log_id=' . $log->id), 'delete_log_' . $log->id)); ?>"
												   onclick="return confirm('<?php esc_attr_e('Are you sure you want to delete this log?', 'mail-forwarder-to-telegram'); ?>')"><?php esc_html_e('Delete', 'mail-forwarder-to-telegram'); ?></a>
                                            </span>
									</div>
								</td>
								<td><?php echo esc_html($log->to_recipients); ?></td>
								<td>
									<?php
									$attachments = maybe_unserialize($log->attachments);
									if (!empty($attachments) && is_array($attachments)) {
										echo '<span class="dashicons dashicons-paperclip"></span> ' . count($attachments);
									} else {
										echo '—';
									}
									?>
								</td>
								<td>
									<?php if ($log->email_sent) : ?>
										<span class="dashicons dashicons-yes-alt" style="color: #46b450;"></span>
									<?php else : ?>
										<span class="dashicons dashicons-dismiss" style="color: #dc3232;"></span>
									<?php endif; ?>
								</td>
								<td>
									<?php if ($log->sent_to_telegram) : ?>
										<span class="dashicons dashicons-yes-alt" style="color: #46b450;"></span>
									<?php else : ?>
										<span class="dashicons dashicons-minus" style="color: #999;"></span>
									<?php endif; ?>
								</td>
								<td>
									<button type="button" class="button button-small view-log-details"
											data-log-id="<?php echo esc_attr($log->id); ?>">
										<?php esc_html_e('View', 'mail-forwarder-to-telegram'); ?>
									</button>
									<button type="button" class="button button-small resend-email"
											data-log-id="<?php echo esc_attr($log->id); ?>"
											title="<?php esc_attr_e('Resend Email', 'mail-forwarder-to-telegram'); ?>">
										<span class="dashicons dashicons-email-alt"
											  style="font-size: 14px; line-height: 20px;"></span>
									</button>
									<button type="button" class="button button-small resend-telegram"
											data-log-id="<?php echo esc_attr($log->id); ?>"
											title="<?php esc_attr_e('Resend to Telegram', 'mail-forwarder-to-telegram'); ?>">
										<span class="dashicons dashicons-share"
											  style="font-size: 14px; line-height: 20px;"></span>
									</button>
								</td>
							</tr>

							<!-- Hidden row for log details -->
							<tr id="log-details-<?php echo esc_attr($log->id); ?>" class="log-details"
								style="display: none;">
								<td colspan="8">
									<div class="log-message-content">
										<h3><?php esc_html_e('Message:', 'mail-forwarder-to-telegram'); ?></h3>
										<div class="log-message"><?php echo nl2br(wp_kses_post($log->message)); ?></div>

										<?php if (!empty($attachments) && is_array($attachments)) : ?>
											<h3><?php esc_html_e('Attachments:', 'mail-forwarder-to-telegram'); ?></h3>
											<ul>
												<?php foreach ($attachments as $attachment) : ?>
													<li><code><?php echo esc_html($attachment); ?></code></li>
												<?php endforeach; ?>
											</ul>
										<?php endif; ?>
									</div>
								</td>
							</tr>
						<?php endforeach; ?>
					<?php else : ?>
						<tr>
							<td colspan="8"
								class="no-items"><?php esc_html_e('No logs found.', 'mail-forwarder-to-telegram'); ?></td>
						</tr>
					<?php endif; ?>
					</tbody>
				</table>

				<?php if (!empty($logs)) : ?>
					<div class="tablenav bottom">
						<div class="alignleft actions bulkactions">
							<select name="action">
								<option value="-1"><?php esc_html_e('Bulk Actions', 'mail-forwarder-to-telegram'); ?></option>
								<option value="delete"><?php esc_html_e('Delete', 'mail-forwarder-to-telegram'); ?></option>
							</select>
							<button type="submit"
									class="button action"><?php esc_html_e('Apply', 'mail-forwarder-to-telegram'); ?></button>
						</div>
					</div>
				<?php endif; ?>
			</form>
		</div>
		<?php
	}
}
