/**
 * Mail Forwarder to Telegram Admin JavaScript
 */

(function($) {
    'use strict';

    $(document).ready(function() {

        /**
         * Toggle forward emails field based on forward mode
         */
        function toggleForwardEmailsField() {
            var forwardMode = $('#mail_forwarder_to_telegram_forward_mode').val();
            var forwardEmailsWrapper = $('#mail_forwarder_to_telegram_forward_emails_wrapper');

            if (forwardMode === 'specific') {
                forwardEmailsWrapper.slideDown(300);
            } else {
                forwardEmailsWrapper.slideUp(300);
            }
        }

        // Initial state
        toggleForwardEmailsField();

        // Listen for changes
        $('#mail_forwarder_to_telegram_forward_mode').on('change', function() {
            toggleForwardEmailsField();
        });

        /**
         * Validate email addresses on submit
         */
        $('form').on('submit', function(e) {
            var forwardMode = $('#mail_forwarder_to_telegram_forward_mode').val();

            // Validate forward emails if mode is specific
            if (forwardMode === 'specific') {
                var forwardEmails = $('#mail_forwarder_to_telegram_forward_emails').val().trim();

                if (forwardEmails === '') {
                    alert('Please enter at least one email address to forward.');
                    e.preventDefault();
                    return false;
                }
            }

            // Validate bot token and chat ID
            var botToken = $('#mail_forwarder_to_telegram_bot_token').val().trim();
            var chatId = $('#mail_forwarder_to_telegram_chat_id').val().trim();

            if (botToken === '' || chatId === '') {
                if (!confirm('Bot Token or Chat ID is empty. The plugin will not work without these settings. Do you want to continue?')) {
                    e.preventDefault();
                    return false;
                }
            }
        });

        /**
         * Add visual feedback for email textarea
         */
        function highlightEmailFields() {
            var $emailFields = $('#mail_forwarder_to_telegram_forward_emails, #mail_forwarder_to_telegram_ignore_emails');

            $emailFields.on('blur', function() {
                var emails = $(this).val().trim();

                if (emails !== '') {
                    var emailArray = emails.split(',').map(function(email) {
                        return email.trim();
                    });

                    var invalidEmails = emailArray.filter(function(email) {
                        // Simple email validation regex
                        var regex = /^[^\s@]+@[^\s@]+\.[^\s@]+$/;
                        return !regex.test(email);
                    });

                    if (invalidEmails.length > 0) {
                        $(this).css('border-color', '#dc3232');
                        var errorMsg = $(this).parent().find('.email-validation-error');

                        if (errorMsg.length === 0) {
                            $(this).after('<p class="email-validation-error" style="color: #dc3232; margin-top: 5px;">Invalid email format detected: ' + invalidEmails.join(', ') + '</p>');
                        } else {
                            errorMsg.text('Invalid email format detected: ' + invalidEmails.join(', '));
                        }
                    } else {
                        $(this).css('border-color', '#8c8f94');
                        $(this).parent().find('.email-validation-error').remove();
                    }
                }
            });
        }

        highlightEmailFields();

        /**
         * Character counter for textareas
         */
        function addCharacterCounter() {
            var $textareas = $('#mail_forwarder_to_telegram_forward_emails, #mail_forwarder_to_telegram_ignore_emails');

            $textareas.each(function() {
                var $textarea = $(this);
                var $description = $textarea.siblings('.description');

                $textarea.on('input', function() {
                    var text = $(this).val();
                    var emailCount = text.trim() === '' ? 0 : text.split(',').length;

                    var counterText = ' (' + emailCount + ' email' + (emailCount !== 1 ? 's' : '') + ')';

                    // Update or add counter
                    var $counter = $textarea.siblings('.email-counter');
                    if ($counter.length === 0) {
                        $description.after('<span class="email-counter" style="font-size: 12px; color: #646970;">' + counterText + '</span>');
                    } else {
                        $counter.text(counterText);
                    }
                });

                // Trigger on load
                $textarea.trigger('input');
            });
        }

        addCharacterCounter();

        /**
         * Send test email
         */
        $('#mail_forwarder_to_telegram_send_test_email').on('click', function(e) {
            e.preventDefault();

            var $button = $(this);
            var $spinner = $button.siblings('.spinner');
            var $result = $('#mail_forwarder_to_telegram_test_result');
            var email = $('#mail_forwarder_to_telegram_test_email_address').val().trim();

            // Validate email
            var emailRegex = /^[^\s@]+@[^\s@]+\.[^\s@]+$/;
            if (!emailRegex.test(email)) {
                $result.html('<div class="notice notice-error inline"><p>Please enter a valid email address.</p></div>');
                return;
            }

            // Clear previous results
            $result.html('');

            // Disable button and show spinner
            $button.prop('disabled', true);
            $spinner.addClass('is-active');

            // Send AJAX request
            $.ajax({
                url: mail_forwarder_to_telegram_ajax.ajax_url,
                type: 'POST',
                data: {
                    action: 'mail_forwarder_to_telegram_send_test_email',
                    nonce: mail_forwarder_to_telegram_ajax.test_nonce,
                    email: email
                },
                success: function(response) {
                    if (response.success) {
                        $result.html('<div class="notice notice-success inline"><p>' + response.data.message + '</p></div>');
                    } else {
                        $result.html('<div class="notice notice-error inline"><p>' + response.data.message + '</p></div>');
                    }
                },
                error: function() {
                    $result.html('<div class="notice notice-error inline"><p>An unexpected error occurred. Please try again.</p></div>');
                },
                complete: function() {
                    // Re-enable button and hide spinner
                    $button.prop('disabled', false);
                    $spinner.removeClass('is-active');
                }
            });
        });

    });

})(jQuery);
