<?php
/**
 * Plugin Name: MA Smart Image Cleaner
 * Plugin URI: https://madigitalhub.co.uk/plugins/ma-smart-image-cleaner
 * Description: Safely find and clean unused images in your WordPress Media Library without breaking your website.
 * Version: 1.0.1
 * Author: Moheeb Arif
 * Author URI: https://moheebarif.com
 * License: GPL v2 or later
 * License URI: https://www.gnu.org/licenses/gpl-2.0.html
 * Text Domain: ma-smart-image-cleaner
 * Domain Path: /languages
 * Requires at least: 5.8
 * Requires PHP: 7.2
 *
 * @package MA_Smart_Image_Cleaner
 */

// Prevent direct access
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

// Plugin constants
define( 'MASIC_VERSION', '1.0.1' );
define( 'MASIC_PLUGIN_DIR', plugin_dir_path( __FILE__ ) );
define( 'MASIC_PLUGIN_URL', plugin_dir_url( __FILE__ ) );
define( 'MASIC_PLUGIN_BASENAME', plugin_basename( __FILE__ ) );

/**
 * Main plugin class
 */
class MASIC_Plugin {

	/**
	 * Single instance of the class
	 *
	 * @var MASIC_Plugin
	 */
	private static $instance = null;

	/**
	 * Get single instance of the class
	 *
	 * @return MASIC_Plugin
	 */
	public static function get_instance() {
		if ( null === self::$instance ) {
			self::$instance = new self();
		}
		return self::$instance;
	}

	/**
	 * Constructor
	 */
	private function __construct() {
		$this->load_dependencies();
		$this->init_hooks();
	}

	/**
	 * Load required files
	 */
	private function load_dependencies() {
		require_once MASIC_PLUGIN_DIR . 'includes/helpers.php';
		require_once MASIC_PLUGIN_DIR . 'includes/permissions.php';
		require_once MASIC_PLUGIN_DIR . 'includes/scanner.php';
		require_once MASIC_PLUGIN_DIR . 'includes/cron.php';
		require_once MASIC_PLUGIN_DIR . 'admin/menu.php';
	}

	/**
	 * Initialize hooks
	 */
	private function init_hooks() {
		// Activation and deactivation hooks
		register_activation_hook( __FILE__, array( $this, 'activate' ) );
		register_deactivation_hook( __FILE__, array( $this, 'deactivate' ) );

		// Enqueue admin scripts and styles
		add_action( 'admin_enqueue_scripts', array( $this, 'enqueue_admin_assets' ) );

		// Add settings link to plugins page
		add_filter( 'plugin_action_links_' . MASIC_PLUGIN_BASENAME, array( $this, 'add_settings_link' ) );

		// Register dashboard widget
		add_action( 'wp_dashboard_setup', array( $this, 'add_dashboard_widget' ) );
	}

	/**
	 * Plugin activation
	 */
	public function activate() {
		// Create custom database table for scan results
		$this->create_database_tables();

		// Set default options
		$this->set_default_options();

		// Clear any existing scan data
		delete_option( 'masic_scan_in_progress' );
		delete_option( 'masic_scan_progress' );

		// Flush rewrite rules
		flush_rewrite_rules();
	}

	/**
	 * Plugin deactivation
	 */
	public function deactivate() {
		// Clear scheduled cron events
		wp_clear_scheduled_hook( 'masic_scheduled_scan' );
		wp_clear_scheduled_hook( 'masic_batch_scan' );

		// Clear scan progress
		delete_option( 'masic_scan_in_progress' );
		delete_option( 'masic_scan_progress' );
	}

	/**
	 * Create database tables
	 */
	private function create_database_tables() {
		global $wpdb;

		$charset_collate = $wpdb->get_charset_collate();
		$table_name = $wpdb->prefix . 'masic_scan_results';
		$log_table = $wpdb->prefix . 'masic_delete_logs';

		$sql_scan_results = "CREATE TABLE IF NOT EXISTS $table_name (
			id bigint(20) unsigned NOT NULL AUTO_INCREMENT,
			attachment_id bigint(20) unsigned NOT NULL,
			status varchar(20) NOT NULL DEFAULT 'unknown',
			detected_locations longtext,
			file_size bigint(20) unsigned DEFAULT 0,
			last_scanned_at datetime DEFAULT NULL,
			is_marked_keep tinyint(1) DEFAULT 0,
			PRIMARY KEY (id),
			UNIQUE KEY attachment_id (attachment_id),
			KEY status (status)
		) $charset_collate;";

		$sql_delete_logs = "CREATE TABLE IF NOT EXISTS $log_table (
			id bigint(20) unsigned NOT NULL AUTO_INCREMENT,
			attachment_id bigint(20) unsigned NOT NULL,
			filename varchar(255) NOT NULL,
			file_size bigint(20) unsigned DEFAULT 0,
			deleted_by bigint(20) unsigned NOT NULL,
			deleted_at datetime NOT NULL,
			is_restored tinyint(1) DEFAULT 0,
			restored_at datetime DEFAULT NULL,
			PRIMARY KEY (id),
			KEY attachment_id (attachment_id),
			KEY deleted_at (deleted_at)
		) $charset_collate;";

		require_once ABSPATH . 'wp-admin/includes/upgrade.php';
		dbDelta( $sql_scan_results );
		dbDelta( $sql_delete_logs );

		// Store database version for future upgrades
		update_option( 'masic_db_version', MASIC_VERSION );
	}

	/**
	 * Set default plugin options
	 */
	private function set_default_options() {
		$defaults = array(
			'enable_auto_scan'      => false,
			'scan_frequency'        => 'manual',
			'enable_elementor_scan' => true,
			'excluded_folders'      => '',
			'excluded_file_types'   => '',
			'batch_size'            => 50,
			'trash_retention_days'  => 30,
			'editor_can_view'       => false,
		);

		foreach ( $defaults as $key => $value ) {
			if ( false === get_option( 'masic_' . $key ) ) {
				update_option( 'masic_' . $key, $value );
			}
		}
	}

	/**
	 * Enqueue admin assets
	 *
	 * @param string $hook Current admin page hook.
	 */
	public function enqueue_admin_assets( $hook ) {
		// Only load on our plugin pages
		if ( strpos( $hook, 'masic-cleaner' ) === false && 'index.php' !== $hook ) {
			return;
		}

		wp_enqueue_style(
			'masic-admin',
			MASIC_PLUGIN_URL . 'assets/admin.css',
			array(),
			MASIC_VERSION
		);

		wp_enqueue_script(
			'masic-admin',
			MASIC_PLUGIN_URL . 'assets/admin.js',
			array( 'jquery' ),
			MASIC_VERSION,
			true
		);

		wp_localize_script( 'masic-admin', 'masic_ajax', array(
			'ajax_url'     => admin_url( 'admin-ajax.php' ),
			'nonce'        => wp_create_nonce( 'masic_nonce' ),
			'scan_nonce'   => wp_create_nonce( 'masic_scan_nonce' ),
			'delete_nonce' => wp_create_nonce( 'masic_delete_nonce' ),
			'i18n'         => array(
				'confirm_delete'    => __( 'Are you sure you want to move these images to trash?', 'ma-smart-image-cleaner' ),
				'confirm_permanent' => __( 'Are you sure you want to permanently delete these images? This action cannot be undone!', 'ma-smart-image-cleaner' ),
				'scanning'          => __( 'Scanning...', 'ma-smart-image-cleaner' ),
				'scan_complete'     => __( 'Scan complete!', 'ma-smart-image-cleaner' ),
				'error'             => __( 'An error occurred. Please try again.', 'ma-smart-image-cleaner' ),
				'select_action'     => __( 'Please select an action.', 'ma-smart-image-cleaner' ),
				'select_image'      => __( 'Please select at least one image.', 'ma-smart-image-cleaner' ),
				'restore_confirm'   => __( 'Restore selected images?', 'ma-smart-image-cleaner' ),
				'starting'          => __( 'Starting...', 'ma-smart-image-cleaner' ),
				'scanning_library'  => __( 'Scanning Full Media Library...', 'ma-smart-image-cleaner' ),
				'of'                => __( 'of', 'ma-smart-image-cleaner' ),
				'images_scanned'    => __( 'images scanned', 'ma-smart-image-cleaner' ),
				'initializing'      => __( 'Initializing scan...', 'ma-smart-image-cleaner' ),
				'pause_scan'        => __( 'Pause Scan', 'ma-smart-image-cleaner' ),
				'resume_scan'       => __( 'Resume Scan', 'ma-smart-image-cleaner' ),
				'full_scan'         => __( 'Full Library Scan', 'ma-smart-image-cleaner' ),
				'remaining'         => __( 'remaining', 'ma-smart-image-cleaner' ),
				'scan_complete_msg' => __( 'Scan complete! Refreshing...', 'ma-smart-image-cleaner' ),
				'scan_failed'       => __( 'Scan failed due to an internal error. Please check your database permissions and try again.', 'ma-smart-image-cleaner' ),
				'progress_failed'   => __( 'Scan progress request failed. Please refresh and try again.', 'ma-smart-image-cleaner' ),
				'analyzing'         => __( 'Analyzing:', 'ma-smart-image-cleaner' ),
			),
			'scan_in_progress' => (bool) get_option( 'masic_scan_in_progress', false ),
			'scan_status'      => '',
		) );

		// Pass scan status for dashboard page
		$scan_progress = get_option( 'masic_scan_progress', array() );
		if ( ! empty( $scan_progress ) && isset( $scan_progress['status'] ) ) {
			wp_add_inline_script( 'masic-admin', 'masic_ajax.scan_status = ' . wp_json_encode( $scan_progress['status'] ) . ';', 'after' );
		}
	}

	/**
	 * Add settings link to plugins page
	 *
	 * @param array $links Existing plugin links.
	 * @return array Modified plugin links.
	 */
	public function add_settings_link( $links ) {
		$settings_link = sprintf(
			'<a href="%s">%s</a>',
			admin_url( 'upload.php?page=masic-cleaner-settings' ),
			__( 'Settings', 'ma-smart-image-cleaner' )
		);
		array_unshift( $links, $settings_link );
		return $links;
	}

	/**
	 * Add dashboard widget
	 */
	public function add_dashboard_widget() {
		if ( ! masic_user_can_view() ) {
			return;
		}

		wp_add_dashboard_widget(
			'masic_dashboard_widget',
			__( 'MA Smart Image Cleaner Summary', 'ma-smart-image-cleaner' ),
			array( $this, 'render_dashboard_widget' )
		);
	}

	/**
	 * Render dashboard widget
	 */
	public function render_dashboard_widget() {
		$masic_stats = masic_get_stats();
		?>
		<div class="masic-dashboard-widget">
			<div class="masic-stat-row">
				<span class="masic-stat-label"><?php esc_html_e( 'Total Images:', 'ma-smart-image-cleaner' ); ?></span>
				<span class="masic-stat-value"><?php echo esc_html( number_format( $masic_stats['total_images'] ) ); ?></span>
			</div>
			<div class="masic-stat-row">
				<span class="masic-stat-label"><?php esc_html_e( 'Used Images:', 'ma-smart-image-cleaner' ); ?></span>
				<span class="masic-stat-value masic-status-used"><?php echo esc_html( number_format( $masic_stats['used_images'] ) ); ?></span>
			</div>
			<div class="masic-stat-row">
				<span class="masic-stat-label"><?php esc_html_e( 'Unused Images:', 'ma-smart-image-cleaner' ); ?></span>
				<span class="masic-stat-value masic-status-unused"><?php echo esc_html( number_format( $masic_stats['unused_images'] ) ); ?></span>
			</div>
			<div class="masic-stat-row">
				<span class="masic-stat-label"><?php esc_html_e( 'Potential Savings:', 'ma-smart-image-cleaner' ); ?></span>
				<span class="masic-stat-value"><?php echo esc_html( masic_format_file_size( $masic_stats['potential_savings'] ) ); ?></span>
			</div>
			<?php if ( $masic_stats['last_scan_date'] ) : ?>
			<div class="masic-stat-row">
				<span class="masic-stat-label"><?php esc_html_e( 'Last Scan:', 'ma-smart-image-cleaner' ); ?></span>
				<span class="masic-stat-value"><?php echo esc_html( masic_format_relative_time( $masic_stats['last_scan_date'] ) ); ?></span>
			</div>
			<?php endif; ?>
			<p class="masic-widget-actions">
				<a href="<?php echo esc_url( admin_url( 'upload.php?page=masic-cleaner' ) ); ?>" class="button button-primary">
					<?php esc_html_e( 'View Details', 'ma-smart-image-cleaner' ); ?>
				</a>
				<?php if ( masic_user_can_delete() ) : ?>
				<a href="<?php echo esc_url( admin_url( 'upload.php?page=masic-cleaner&action=scan' ) ); ?>" class="button">
					<?php esc_html_e( 'Run Scan', 'ma-smart-image-cleaner' ); ?>
				</a>
				<?php endif; ?>
			</p>
		</div>
		<?php
	}
}

// Initialize plugin
function masic_init() {
	return MASIC_Plugin::get_instance();
}
masic_init();
