<?php
/**
 * Scanner functionality for MA Smart Image Cleaner
 *
 * @package MA_Smart_Image_Cleaner
 */

// Prevent direct access
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Scanner class
 */
class MASIC_Scanner {

	/**
	 * Batch size for scanning
	 *
	 * @var int
	 */
	private $batch_size;

	/**
	 * Constructor
	 */
	public function __construct() {
		$batch_size = (int) get_option( 'masic_batch_size', 50 );
		if ( $batch_size < 10 ) {
			$batch_size = 50;
		}
		if ( $batch_size > 200 ) {
			$batch_size = 200;
		}

		$this->batch_size = $batch_size;
		$this->init_hooks();
	}

	/**
	 * Initialize hooks
	 */
	private function init_hooks() {
		// AJAX handlers
		add_action( 'wp_ajax_masic_start_scan', array( $this, 'ajax_start_scan' ) );
		add_action( 'wp_ajax_masic_get_scan_progress', array( $this, 'ajax_get_scan_progress' ) );
		add_action( 'wp_ajax_masic_pause_scan', array( $this, 'ajax_pause_scan' ) );
		add_action( 'wp_ajax_masic_resume_scan', array( $this, 'ajax_resume_scan' ) );
	}

	/**
	 * AJAX handler to start scan
	 */
	public function ajax_start_scan() {
		if ( ! masic_user_can_scan() ) {
			wp_send_json_error( array( 'message' => __( 'Permission denied.', 'ma-smart-image-cleaner' ) ) );
		}

		check_ajax_referer( 'masic_scan_nonce', 'nonce' );

		$result = $this->start_scan();

		if ( is_wp_error( $result ) ) {
			wp_send_json_error( array( 'message' => $result->get_error_message() ) );
		}

		wp_send_json_success( $result );
	}

	/**
	 * Start the scanning process
	 *
	 * @return array|WP_Error Scan status or error.
	 */
	public function start_scan() {
		global $wpdb;

		delete_option( 'masic_last_db_error' );

		if ( get_option( 'masic_scan_in_progress' ) ) {
			return new WP_Error( 'scan_in_progress', __( 'A scan is already in progress.', 'ma-smart-image-cleaner' ) );
		}

		if ( function_exists( 'masic_maybe_create_tables' ) ) {
			masic_maybe_create_tables();
		}

		$table_name = $wpdb->prefix . 'masic_scan_results';
		if ( function_exists( 'masic_table_exists' ) && ! masic_table_exists( $table_name ) ) {
			update_option( 'masic_last_db_error', 'Missing table: ' . $table_name );
			return new WP_Error(
				'missing_table',
				__( 'Scan tables are missing. Please deactivate and reactivate the plugin (or check DB permissions) and try again.', 'ma-smart-image-cleaner' )
			);
		}

		$image_mime_like = $wpdb->esc_like( 'image/' ) . '%';
		// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
		$total_images = (int) $wpdb->get_var(
			$wpdb->prepare(
				"SELECT COUNT(*) FROM {$wpdb->posts}
				WHERE post_type = 'attachment'
				AND post_mime_type LIKE %s
				AND post_status = 'inherit'",
				$image_mime_like
			)
		);

		if ( 0 === $total_images ) {
			return new WP_Error( 'no_images', __( 'No images found in the media library.', 'ma-smart-image-cleaner' ) );
		}

		$scan_progress = array(
			'total_images'   => $total_images,
			'scanned_images' => 0,
			'current_offset' => 0,
			'started_at'     => current_time( 'mysql' ),
			'status'         => 'running',
			'current_image'  => '',
		);

		update_option( 'masic_scan_in_progress', true );
		update_option( 'masic_scan_progress', $scan_progress );
		update_option( 'masic_last_batch_run', 0 );

		// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
		$deleted = $wpdb->query( "DELETE FROM {$wpdb->prefix}masic_scan_results" );
		if ( false === $deleted ) {
			$err = $wpdb->last_error ? $wpdb->last_error : 'Unknown database error';
			update_option( 'masic_last_db_error', $err );

			$scan_progress['status'] = 'error';
			/* translators: %s: database error */
			$scan_progress['error_message'] = sprintf( __( 'Database error clearing old scan results: %s', 'ma-smart-image-cleaner' ), $err );
			update_option( 'masic_scan_progress', $scan_progress );
			delete_option( 'masic_scan_in_progress' );
			delete_option( 'masic_last_batch_run' );

			return new WP_Error( 'db_error', $scan_progress['error_message'] );
		}

		wp_schedule_single_event( time(), 'masic_batch_scan' );

		return $scan_progress;
	}

	/**
	 * AJAX handler to get scan progress
	 */
	public function ajax_get_scan_progress() {
		check_ajax_referer( 'masic_scan_nonce', 'nonce' );

		$progress = get_option( 'masic_scan_progress', array() );
		$in_progress = get_option( 'masic_scan_in_progress', false );

		if ( $in_progress && ! empty( $progress ) && isset( $progress['status'] ) && 'running' === $progress['status'] ) {
			$last_run = (int) get_option( 'masic_last_batch_run', 0 );
			if ( time() - $last_run >= 1 ) {
				update_option( 'masic_last_batch_run', time() );
				$this->process_batch( 5 );
				$progress = get_option( 'masic_scan_progress', array() );
				$in_progress = get_option( 'masic_scan_in_progress', false );
			}
		}

		wp_send_json_success( array(
			'in_progress' => $in_progress,
			'progress'    => $progress,
		) );
	}

	/**
	 * AJAX handler to pause scan
	 */
	public function ajax_pause_scan() {
		if ( ! masic_user_can_scan() ) {
			wp_send_json_error( array( 'message' => __( 'Permission denied.', 'ma-smart-image-cleaner' ) ) );
		}

		check_ajax_referer( 'masic_scan_nonce', 'nonce' );

		$progress = get_option( 'masic_scan_progress', array() );
		$progress['status'] = 'paused';
		update_option( 'masic_scan_progress', $progress );

		wp_clear_scheduled_hook( 'masic_batch_scan' );

		wp_send_json_success( array( 'message' => __( 'Scan paused.', 'ma-smart-image-cleaner' ) ) );
	}

	/**
	 * AJAX handler to resume scan
	 */
	public function ajax_resume_scan() {
		if ( ! masic_user_can_scan() ) {
			wp_send_json_error( array( 'message' => __( 'Permission denied.', 'ma-smart-image-cleaner' ) ) );
		}

		check_ajax_referer( 'masic_scan_nonce', 'nonce' );

		$progress = get_option( 'masic_scan_progress', array() );
		$progress['status'] = 'running';
		update_option( 'masic_scan_progress', $progress );

		wp_schedule_single_event( time(), 'masic_batch_scan' );

		wp_send_json_success( array( 'message' => __( 'Scan resumed.', 'ma-smart-image-cleaner' ) ) );
	}

	/**
	 * Process a batch of images
	 *
	 * @param int|null $limit Optional limit for batch size.
	 */
	public function process_batch( $limit = null ) {
		global $wpdb;

		$progress = get_option( 'masic_scan_progress', array() );

		if ( empty( $progress ) || in_array( $progress['status'], array( 'paused', 'error' ), true ) ) {
			return;
		}

		$offset = (int) $progress['current_offset'];

		$batch_limit = max( 1, (int) $this->batch_size );
		if ( null !== $limit ) {
			$batch_limit = max( 1, min( (int) $limit, $batch_limit ) );
		}

		$image_mime_like = $wpdb->esc_like( 'image/' ) . '%';
		// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
		$attachments = $wpdb->get_results(
			$wpdb->prepare(
				"SELECT ID, guid FROM {$wpdb->posts}
				 WHERE post_type = 'attachment'
				 AND post_mime_type LIKE %s
				 AND post_status = 'inherit'
				 ORDER BY ID ASC
				 LIMIT %d OFFSET %d",
				$image_mime_like,
				$batch_limit,
				$offset
			)
		);

		if ( empty( $attachments ) ) {
			$progress['status'] = 'complete';
			$progress['scanned_images'] = $progress['total_images'];
			update_option( 'masic_scan_progress', $progress );
			delete_option( 'masic_scan_in_progress' );
			delete_option( 'masic_last_batch_run' );
			return;
		}

		foreach ( $attachments as $attachment ) {
			if ( masic_is_excluded( $attachment->ID ) ) {
				continue;
			}

			$result = $this->analyze_image( $attachment->ID );

			// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
			$inserted = $wpdb->replace(
				$wpdb->prefix . 'masic_scan_results',
				array(
					'attachment_id'      => $attachment->ID,
					'status'             => $result['status'],
					'detected_locations' => maybe_serialize( $result['locations'] ),
					'file_size'          => $result['file_size'],
					'last_scanned_at'    => current_time( 'mysql' ),
					'is_marked_keep'     => 0,
				),
				array( '%d', '%s', '%s', '%d', '%s', '%d' )
			);

			if ( false === $inserted ) {
				$err = $wpdb->last_error ? $wpdb->last_error : 'Unknown database error';
				update_option( 'masic_last_db_error', $err );

				$progress['status'] = 'error';
				/* translators: %s: database error */
				$progress['error_message'] = sprintf( __( 'Database error writing scan results: %s', 'ma-smart-image-cleaner' ), $err );
				update_option( 'masic_scan_progress', $progress );
				delete_option( 'masic_scan_in_progress' );
				delete_option( 'masic_last_batch_run' );
				return;
			}

			$attached_file = get_attached_file( $attachment->ID );
			if ( $attached_file ) {
				$progress['current_image'] = basename( $attached_file );
			} else {
				$progress['current_image'] = basename( (string) $attachment->guid );
			}
		}

		$progress['current_offset'] = $offset + count( $attachments );
		$progress['scanned_images'] = $progress['current_offset'];
		update_option( 'masic_scan_progress', $progress );

		if ( $progress['scanned_images'] < $progress['total_images'] ) {
			wp_schedule_single_event( time() + 1, 'masic_batch_scan' );
		} else {
			$progress['status'] = 'complete';
			update_option( 'masic_scan_progress', $progress );
			delete_option( 'masic_scan_in_progress' );
			delete_option( 'masic_last_batch_run' );
		}
	}

	/**
	 * Analyze a single image for usage
	 *
	 * @param int $attachment_id Attachment ID.
	 * @return array Analysis result.
	 */
	public function analyze_image( $attachment_id ) {
		global $wpdb;

		$locations = array();
		$file = get_attached_file( $attachment_id );
		$file_size = $file && file_exists( $file ) ? filesize( $file ) : 0;
		$attachment_url = wp_get_attachment_url( $attachment_id );

		// Check 1: Featured image (_thumbnail_id)
		// Note: meta_key queries on postmeta can be slow on very large sites.
		// Consider adding a database index on meta_value if performance is a concern.
		// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
		$featured_posts = $wpdb->get_results(
			$wpdb->prepare(
				"SELECT p.ID, p.post_title, p.post_type
				 FROM {$wpdb->postmeta} pm
				 INNER JOIN {$wpdb->posts} p ON pm.post_id = p.ID
				 WHERE pm.meta_key = '_thumbnail_id'
				 AND pm.meta_value = %d
				 AND p.post_status IN ('publish', 'draft', 'private')",
				$attachment_id
			)
		);

		foreach ( $featured_posts as $post ) {
			$locations[] = array(
				'type'       => 'featured_image',
				'post_id'    => $post->ID,
				'post_title' => $post->post_title,
				'post_type'  => $post->post_type,
			);
		}

		// Check 2: Post content
		$url_patterns = array(
			$attachment_url,
			str_replace( home_url(), '', $attachment_url ),
			'wp-image-' . $attachment_id,
			'"id":' . $attachment_id,
			'"id":"' . $attachment_id . '"',
			'attachment_id="' . $attachment_id . '"',
			'data-id="' . $attachment_id . '"',
		);

		$metadata = wp_get_attachment_metadata( $attachment_id );
		if ( ! empty( $metadata['sizes'] ) ) {
			$base_dir = trailingslashit( dirname( $attachment_url ) );
			foreach ( $metadata['sizes'] as $size_data ) {
				$url_patterns[] = $base_dir . $size_data['file'];
			}
		}

		$url_patterns = array_values( array_unique( array_filter( $url_patterns ) ) );

		foreach ( $url_patterns as $pattern ) {
			// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
			$content_posts = $wpdb->get_results(
				$wpdb->prepare(
					"SELECT ID, post_title, post_type
					 FROM {$wpdb->posts}
					 WHERE post_content LIKE %s
					 AND post_status IN ('publish', 'draft', 'private')
					 AND post_type NOT IN ('revision', 'attachment')",
					'%' . $wpdb->esc_like( $pattern ) . '%'
				)
			);

			foreach ( $content_posts as $post ) {
				$exists = false;
				foreach ( $locations as $loc ) {
					if ( $loc['post_id'] === $post->ID && $loc['type'] === 'post_content' ) {
						$exists = true;
						break;
					}
				}

				if ( ! $exists ) {
					$locations[] = array(
						'type'       => 'post_content',
						'post_id'    => $post->ID,
						'post_title' => $post->post_title,
						'post_type'  => $post->post_type,
					);
				}
			}
		}

		// Check 3: Post meta
		// Note: These LIKE queries on meta_value can be slow on large sites.
		// Consider adding database indexes on meta_value if performance is a concern.
		$serialized_id_pattern = '%i:' . (int) $attachment_id . ';%';
		$quoted_id_pattern = '%' . $wpdb->esc_like( '"' . (int) $attachment_id . '"' ) . '%';
		$private_meta_like = $wpdb->esc_like( '_' ) . '%';

		// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
		$meta_results = $wpdb->get_results(
			$wpdb->prepare(
				"SELECT pm.post_id, p.post_title, p.post_type, pm.meta_key
				 FROM {$wpdb->postmeta} pm
				 INNER JOIN {$wpdb->posts} p ON pm.post_id = p.ID
				 WHERE (pm.meta_value = %d OR pm.meta_value LIKE %s OR pm.meta_value LIKE %s OR pm.meta_value LIKE %s)
				 AND p.post_status IN ('publish', 'draft', 'private')
				 AND p.post_type NOT IN ('revision', 'attachment')
				 AND pm.meta_key NOT LIKE %s",
				$attachment_id,
				'%' . $wpdb->esc_like( $attachment_url ) . '%',
				$serialized_id_pattern,
				$quoted_id_pattern,
				$private_meta_like
			)
		);

		foreach ( $meta_results as $meta ) {
			$exists = false;
			foreach ( $locations as $loc ) {
				if ( $loc['post_id'] === $meta->post_id && $loc['type'] === 'post_meta' ) {
					$exists = true;
					break;
				}
			}

			if ( ! $exists ) {
				$locations[] = array(
					'type'       => 'post_meta',
					'post_id'    => $meta->post_id,
					'post_title' => $meta->post_title,
					'post_type'  => $meta->post_type,
					'meta_key'   => $meta->meta_key,
				);
			}
		}

		// Check 4: Elementor data (if enabled)
		if ( get_option( 'masic_enable_elementor_scan', true ) && defined( 'ELEMENTOR_VERSION' ) ) {
			// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
			$elementor_results = $wpdb->get_results(
				$wpdb->prepare(
					"SELECT pm.post_id, p.post_title, p.post_type
					 FROM {$wpdb->postmeta} pm
					 INNER JOIN {$wpdb->posts} p ON pm.post_id = p.ID
					 WHERE pm.meta_key = '_elementor_data'
					 AND (pm.meta_value LIKE %s OR pm.meta_value LIKE %s)
					 AND p.post_status IN ('publish', 'draft', 'private')",
					'%' . $wpdb->esc_like( '"id":"' . $attachment_id . '"' ) . '%',
					'%' . $wpdb->esc_like( '"id":' . $attachment_id ) . '%'
				)
			);

			foreach ( $elementor_results as $el_post ) {
				$exists = false;
				foreach ( $locations as $loc ) {
					if ( $loc['post_id'] === $el_post->post_id && $loc['type'] === 'elementor' ) {
						$exists = true;
						break;
					}
				}

				if ( ! $exists ) {
					$locations[] = array(
						'type'       => 'elementor',
						'post_id'    => $el_post->post_id,
						'post_title' => $el_post->post_title,
						'post_type'  => $el_post->post_type,
					);
				}
			}
		}

		$status = empty( $locations ) ? 'unused' : 'used';

		return array(
			'status'    => $status,
			'locations' => $locations,
			'file_size' => $file_size,
		);
	}
}

// Initialize scanner
new MASIC_Scanner();

// Hook for batch processing
add_action( 'masic_batch_scan', function() {
	$scanner = new MASIC_Scanner();
	$scanner->process_batch();
} );
