<?php
/**
 * Permission handling for MA Smart Image Cleaner
 *
 * @package MA_Smart_Image_Cleaner
 */

// Prevent direct access
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Check if current user can view scan results
 *
 * @return bool Whether the user can view.
 */
function masic_user_can_view() {
	// Administrators can always view
	if ( current_user_can( 'manage_options' ) ) {
		return true;
	}

	// Check if editors can view
	$editor_can_view = get_option( 'masic_editor_can_view', false );
	if ( $editor_can_view && current_user_can( 'edit_others_posts' ) ) {
		return true;
	}

	return false;
}

/**
 * Check if current user can delete images
 *
 * @return bool Whether the user can delete.
 */
function masic_user_can_delete() {
	// Only administrators can delete images
	return current_user_can( 'manage_options' );
}

/**
 * Check if current user can run scans
 *
 * @return bool Whether the user can run scans.
 */
function masic_user_can_scan() {
	// Only administrators can run scans
	return current_user_can( 'manage_options' );
}

/**
 * Check if current user can modify settings
 *
 * @return bool Whether the user can modify settings.
 */
function masic_user_can_settings() {
	// Only administrators can modify settings
	return current_user_can( 'manage_options' );
}

/**
 * Verify nonce for AJAX requests
 *
 * @param string $action Nonce action.
 * @param string $nonce_name Nonce name in request.
 * @return bool Whether the nonce is valid.
 */
function masic_verify_nonce( $action, $nonce_name = 'nonce' ) {
	$nonce = isset( $_REQUEST[ $nonce_name ] ) ? sanitize_text_field( wp_unslash( $_REQUEST[ $nonce_name ] ) ) : '';
	return wp_verify_nonce( $nonce, $action );
}

/**
 * Die with error if permission check fails
 *
 * @param string $capability Required capability.
 */
function masic_check_permission( $capability = 'manage_options' ) {
	if ( ! current_user_can( $capability ) ) {
		wp_die(
			esc_html__( 'You do not have sufficient permissions to access this page.', 'ma-smart-image-cleaner' ),
			esc_html__( 'Permission Denied', 'ma-smart-image-cleaner' ),
			array( 'response' => 403 )
		);
	}
}

/**
 * Die with error if nonce check fails
 *
 * @param string $action Nonce action.
 * @param string $nonce_name Nonce name in request.
 */
function masic_check_nonce( $action, $nonce_name = 'nonce' ) {
	if ( ! masic_verify_nonce( $action, $nonce_name ) ) {
		wp_die(
			esc_html__( 'Security check failed. Please refresh the page and try again.', 'ma-smart-image-cleaner' ),
			esc_html__( 'Security Error', 'ma-smart-image-cleaner' ),
			array( 'response' => 403 )
		);
	}
}
