<?php
/**
 * Helper functions for MA Smart Image Cleaner
 *
 * @package MA_Smart_Image_Cleaner
 */

// Prevent direct access
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Check whether a database table exists.
 *
 * @param string $table_name Full table name including prefix.
 * @return bool
 */
function masic_table_exists( $table_name ) {
	global $wpdb;
	// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
	$found = $wpdb->get_var( $wpdb->prepare( 'SHOW TABLES LIKE %s', $table_name ) );
	return (string) $found === (string) $table_name;
}

/**
 * Ensure plugin tables exist (fallback if activation hook didn't run or dbDelta failed).
 *
 * @return void
 */
function masic_maybe_create_tables() {
	static $did_run = false;
	if ( $did_run ) {
		return;
	}
	$did_run = true;

	global $wpdb;

	$scan_table = $wpdb->prefix . 'masic_scan_results';
	$log_table  = $wpdb->prefix . 'masic_delete_logs';

	if ( masic_table_exists( $scan_table ) && masic_table_exists( $log_table ) ) {
		return;
	}

	$charset_collate = $wpdb->get_charset_collate();

	$sql_scan_results = "CREATE TABLE IF NOT EXISTS {$scan_table} (
		id bigint(20) unsigned NOT NULL AUTO_INCREMENT,
		attachment_id bigint(20) unsigned NOT NULL,
		status varchar(20) NOT NULL DEFAULT 'unknown',
		detected_locations longtext,
		file_size bigint(20) unsigned DEFAULT 0,
		last_scanned_at datetime DEFAULT NULL,
		is_marked_keep tinyint(1) DEFAULT 0,
		PRIMARY KEY (id),
		UNIQUE KEY attachment_id (attachment_id),
		KEY status (status)
	) {$charset_collate};";

	$sql_delete_logs = "CREATE TABLE IF NOT EXISTS {$log_table} (
		id bigint(20) unsigned NOT NULL AUTO_INCREMENT,
		attachment_id bigint(20) unsigned NOT NULL,
		filename varchar(255) NOT NULL,
		file_size bigint(20) unsigned DEFAULT 0,
		deleted_by bigint(20) unsigned NOT NULL,
		deleted_at datetime NOT NULL,
		is_restored tinyint(1) DEFAULT 0,
		restored_at datetime DEFAULT NULL,
		PRIMARY KEY (id),
		KEY attachment_id (attachment_id),
		KEY deleted_at (deleted_at)
	) {$charset_collate};";

	require_once ABSPATH . 'wp-admin/includes/upgrade.php';
	dbDelta( $sql_scan_results );
	dbDelta( $sql_delete_logs );
}

/**
 * Format file size to human readable format
 *
 * @param int $bytes File size in bytes.
 * @return string Formatted file size.
 */
function masic_format_file_size( $bytes ) {
	if ( 0 === $bytes ) {
		return '0 B';
	}

	$units = array( 'B', 'KB', 'MB', 'GB', 'TB' );
	$power = floor( log( $bytes, 1024 ) );
	$power = min( $power, count( $units ) - 1 );

	return round( $bytes / pow( 1024, $power ), 2 ) . ' ' . $units[ $power ];
}

/**
 * Format date to relative time
 *
 * @param string|DateTime $date Date to format.
 * @return string Relative time string.
 */
function masic_format_relative_time( $date ) {
	if ( is_string( $date ) ) {
		$date = new DateTime( $date );
	}

	$now = new DateTime();
	$diff = $now->diff( $date );

	if ( $diff->days === 0 ) {
		return __( 'Today', 'ma-smart-image-cleaner' );
	} elseif ( $diff->days === 1 ) {
		return __( 'Yesterday', 'ma-smart-image-cleaner' );
	} elseif ( $diff->days < 7 ) {
		/* translators: %d: number of days */
		return sprintf( __( '%d days ago', 'ma-smart-image-cleaner' ), $diff->days );
	} elseif ( $diff->days < 30 ) {
		$weeks = floor( $diff->days / 7 );
		/* translators: %d: number of weeks */
		return sprintf( _n( '%d week ago', '%d weeks ago', $weeks, 'ma-smart-image-cleaner' ), $weeks );
	} elseif ( $diff->days < 365 ) {
		$months = floor( $diff->days / 30 );
		/* translators: %d: number of months */
		return sprintf( _n( '%d month ago', '%d months ago', $months, 'ma-smart-image-cleaner' ), $months );
	} else {
		$years = floor( $diff->days / 365 );
		/* translators: %d: number of years */
		return sprintf( _n( '%d year ago', '%d years ago', $years, 'ma-smart-image-cleaner' ), $years );
	}
}

/**
 * Get days until permanent deletion
 *
 * @param string $deleted_at Date when item was deleted.
 * @param int    $retention_days Number of days to retain in trash.
 * @return int Days remaining until permanent deletion.
 */
function masic_get_days_until_deletion( $deleted_at, $retention_days = 30 ) {
	$deleted_date = new DateTime( $deleted_at );
	$now = new DateTime();
	$diff = $now->diff( $deleted_date );

	return max( 0, $retention_days - $diff->days );
}

/**
 * Get plugin statistics
 *
 * @return array Statistics array.
 */
function masic_get_stats() {
	global $wpdb;

	masic_maybe_create_tables();

	$scan_table = $wpdb->prefix . 'masic_scan_results';

	$image_mime_like = $wpdb->esc_like( 'image/' ) . '%';
	// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
	$total_images = (int) $wpdb->get_var(
		$wpdb->prepare(
			"SELECT COUNT(*) FROM {$wpdb->posts}
			WHERE post_type = 'attachment'
			AND post_mime_type LIKE %s",
			$image_mime_like
		)
	);

	// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
	$used_images = (int) $wpdb->get_var(
		$wpdb->prepare(
			"SELECT COUNT(*) FROM {$scan_table} WHERE status = %s",
			'used'
		)
	);

	// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
	$unused_images = (int) $wpdb->get_var(
		$wpdb->prepare(
			"SELECT COUNT(*) FROM {$scan_table} WHERE status = %s",
			'unused'
		)
	);

	// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
	$potential_savings = (int) $wpdb->get_var(
		$wpdb->prepare(
			"SELECT COALESCE(SUM(file_size), 0) FROM {$scan_table} WHERE status = %s",
			'unused'
		)
	);

	$total_storage = 0;
	// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
	$attachments = $wpdb->get_col(
		$wpdb->prepare(
			"SELECT ID FROM {$wpdb->posts}
			WHERE post_type = 'attachment'
			AND post_mime_type LIKE %s",
			$image_mime_like
		)
	);

	foreach ( $attachments as $attachment_id ) {
		$file = get_attached_file( $attachment_id );
		if ( $file && file_exists( $file ) ) {
			$total_storage += filesize( $file );
		}
	}

	// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
	$last_scan = $wpdb->get_var(
		"SELECT MAX(last_scanned_at) FROM {$scan_table}"
	);

	return array(
		'total_images'      => $total_images,
		'used_images'       => $used_images,
		'unused_images'     => $unused_images,
		'total_storage'     => $total_storage,
		'potential_savings' => $potential_savings,
		'last_scan_date'    => $last_scan,
	);
}

/**
 * Get scan results with filters - uses conditional query building with safe placeholders
 *
 * @param array $args Filter arguments.
 * @return array Array of scan results.
 */
function masic_get_scan_results( $args = array() ) {
	global $wpdb;

	masic_maybe_create_tables();

	$defaults = array(
		'status'     => 'all',
		'min_size'   => 0,
		'older_than' => 0,
		'file_type'  => 'all',
		'orderby'    => 'file_size',
		'order'      => 'DESC',
		'per_page'   => 20,
		'paged'      => 1,
		'search'     => '',
	);

	$args = wp_parse_args( $args, $defaults );

	// Sanitize all inputs
	$status     = sanitize_key( $args['status'] );
	$min_size   = absint( $args['min_size'] ) * 1024 * 1024;
	$older_than = absint( $args['older_than'] );
	$file_type  = sanitize_key( $args['file_type'] );
	$orderby    = sanitize_key( $args['orderby'] );
	$order      = strtoupper( sanitize_key( $args['order'] ) );
	$per_page   = max( 1, absint( $args['per_page'] ) );
	$paged      = max( 1, absint( $args['paged'] ) );
	$search     = sanitize_text_field( $args['search'] );
	$offset     = ( $paged - 1 ) * $per_page;

	// Validate order direction
	if ( 'ASC' !== $order ) {
		$order = 'DESC';
	}

	// Validate status
	if ( ! in_array( $status, array( 'all', 'used', 'unused' ), true ) ) {
		$status = 'all';
	}

	// Table references
	$scan_table  = $wpdb->prefix . 'masic_scan_results';
	$posts_table = $wpdb->posts;

	// Build filter flags
	$has_status    = 'all' !== $status;
	$has_min_size  = $min_size > 0;
	$has_older     = $older_than > 0;
	$has_file_type = 'all' !== $file_type && ! empty( $file_type );
	$has_search    = ! empty( $search );

	// Prepare search value if needed
	$search_like = $has_search ? '%' . $wpdb->esc_like( $search ) . '%' : '';
	$mime_type   = $has_file_type ? 'image/' . $file_type : '';

	// Whitelisted ORDER BY clause - no dynamic column interpolation
	switch ( $orderby ) {
		case 'post_date':
			$order_sql = ( 'ASC' === $order ) ? 'ORDER BY p.post_date ASC' : 'ORDER BY p.post_date DESC';
			break;
		case 'post_title':
			$order_sql = ( 'ASC' === $order ) ? 'ORDER BY p.post_title ASC' : 'ORDER BY p.post_title DESC';
			break;
		case 'status':
			$order_sql = ( 'ASC' === $order ) ? 'ORDER BY sr.status ASC' : 'ORDER BY sr.status DESC';
			break;
		case 'file_size':
		default:
			$order_sql = ( 'ASC' === $order ) ? 'ORDER BY sr.file_size ASC' : 'ORDER BY sr.file_size DESC';
			break;
	}

	// Build WHERE conditions and parameters dynamically but safely
	$where_conditions = array();
	$where_values     = array();

	if ( $has_status ) {
		$where_conditions[] = 'sr.status = %s';
		$where_values[]     = $status;
	}

	if ( $has_min_size ) {
		$where_conditions[] = 'sr.file_size >= %d';
		$where_values[]     = $min_size;
	}

	if ( $has_older ) {
		$where_conditions[] = 'p.post_date < DATE_SUB(NOW(), INTERVAL %d MONTH)';
		$where_values[]     = $older_than;
	}

	if ( $has_file_type ) {
		$where_conditions[] = 'p.post_mime_type = %s';
		$where_values[]     = $mime_type;
	}

	if ( $has_search ) {
		$where_conditions[] = 'p.post_title LIKE %s';
		$where_values[]     = $search_like;
	}

	// Build WHERE clause
	$where_sql = '';
	if ( ! empty( $where_conditions ) ) {
		$where_sql = 'WHERE ' . implode( ' AND ', $where_conditions );
	}

	// Build the SELECT columns (fixed, no interpolation)
	$select_cols = 'sr.*, p.ID as attachment_id, p.post_title, p.post_mime_type, p.post_date, p.guid';
	$from_clause = "{$scan_table} sr INNER JOIN {$posts_table} p ON sr.attachment_id = p.ID";

	// Build full query with LIMIT/OFFSET
	$query_values = array_merge( $where_values, array( $per_page, $offset ) );

	if ( ! empty( $where_sql ) ) {
		// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.InterpolatedNotPrepared
		$results = $wpdb->get_results(
			$wpdb->prepare(
				"SELECT {$select_cols} FROM {$from_clause} {$where_sql} {$order_sql} LIMIT %d OFFSET %d",
				$query_values
			)
		);
	} else {
		// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.InterpolatedNotPrepared
		$results = $wpdb->get_results(
			$wpdb->prepare(
				"SELECT {$select_cols} FROM {$from_clause} {$order_sql} LIMIT %d OFFSET %d",
				$per_page,
				$offset
			)
		);
	}

	// Count query
	if ( ! empty( $where_values ) ) {
		// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.InterpolatedNotPrepared
		$total = (int) $wpdb->get_var(
			$wpdb->prepare(
				"SELECT COUNT(*) FROM {$from_clause} {$where_sql}",
				$where_values
			)
		);
	} else {
		// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
		$total = (int) $wpdb->get_var(
			"SELECT COUNT(*) FROM {$from_clause}"
		);
	}

	// Enhance results with additional data
	foreach ( $results as &$result ) {
		$result->thumbnail_url       = wp_get_attachment_image_url( $result->attachment_id, 'thumbnail' );
		$result->filename            = basename( get_attached_file( $result->attachment_id ) );
		$result->detected_locations  = maybe_unserialize( $result->detected_locations );
	}

	return array(
		'items'        => $results,
		'total'        => $total,
		'total_pages'  => ceil( $total / $per_page ),
		'current_page' => $paged,
	);
}

/**
 * Get trashed images
 *
 * @param array $args Filter arguments.
 * @return array Array of trashed images.
 */
function masic_get_trashed_images( $args = array() ) {
	global $wpdb;

	$defaults = array(
		'per_page' => 20,
		'paged'    => 1,
	);

	$args = wp_parse_args( $args, $defaults );

	$per_page = max( 1, absint( $args['per_page'] ) );
	$paged    = max( 1, absint( $args['paged'] ) );
	$offset   = ( $paged - 1 ) * $per_page;

	$image_mime_like = $wpdb->esc_like( 'image/' ) . '%';

	// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
	$results = $wpdb->get_results(
		$wpdb->prepare(
			"SELECT * FROM {$wpdb->posts}
			 WHERE post_type = 'attachment'
			 AND post_mime_type LIKE %s
			 AND post_status = 'trash'
			 ORDER BY post_modified DESC
			 LIMIT %d OFFSET %d",
			$image_mime_like,
			$per_page,
			$offset
		)
	);

	// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
	$total = (int) $wpdb->get_var(
		$wpdb->prepare(
			"SELECT COUNT(*) FROM {$wpdb->posts}
			 WHERE post_type = 'attachment'
			 AND post_mime_type LIKE %s
			 AND post_status = 'trash'",
			$image_mime_like
		)
	);

	foreach ( $results as &$result ) {
		$result->thumbnail_url  = wp_get_attachment_image_url( $result->ID, 'thumbnail' );
		$result->filename       = basename( get_attached_file( $result->ID ) );
		$file_path              = get_attached_file( $result->ID );
		$result->file_size      = $file_path && file_exists( $file_path ) ? filesize( $file_path ) : 0;
		$result->deleted_at     = $result->post_modified;
		$result->days_remaining = masic_get_days_until_deletion( $result->post_modified );
	}

	return array(
		'items'        => $results,
		'total'        => $total,
		'total_pages'  => ceil( $total / $per_page ),
		'current_page' => $paged,
	);
}

/**
 * Get delete logs
 *
 * @param array $args Filter arguments.
 * @return array Array of delete logs.
 */
function masic_get_delete_logs( $args = array() ) {
	global $wpdb;

	$defaults = array(
		'per_page' => 20,
		'paged'    => 1,
	);

	$args = wp_parse_args( $args, $defaults );

	$per_page  = max( 1, absint( $args['per_page'] ) );
	$paged     = max( 1, absint( $args['paged'] ) );
	$offset    = ( $paged - 1 ) * $per_page;
	$log_table = $wpdb->prefix . 'masic_delete_logs';

	// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
	$results = $wpdb->get_results(
		$wpdb->prepare(
			"SELECT l.*, u.display_name as deleted_by_name
			 FROM {$log_table} l
			 LEFT JOIN {$wpdb->users} u ON l.deleted_by = u.ID
			 ORDER BY l.deleted_at DESC
			 LIMIT %d OFFSET %d",
			$per_page,
			$offset
		)
	);

	// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
	$total = (int) $wpdb->get_var( "SELECT COUNT(*) FROM {$log_table}" );

	return array(
		'items'        => $results,
		'total'        => $total,
		'total_pages'  => ceil( $total / $per_page ),
		'current_page' => $paged,
	);
}

/**
 * Log a delete action
 *
 * @param int    $attachment_id Attachment ID.
 * @param string $filename File name.
 * @param int    $file_size File size in bytes.
 * @return bool Whether the log was inserted successfully.
 */
function masic_log_delete( $attachment_id, $filename, $file_size ) {
	global $wpdb;

	// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery
	return $wpdb->insert(
		$wpdb->prefix . 'masic_delete_logs',
		array(
			'attachment_id' => absint( $attachment_id ),
			'filename'      => sanitize_file_name( $filename ),
			'file_size'     => absint( $file_size ),
			'deleted_by'    => get_current_user_id(),
			'deleted_at'    => current_time( 'mysql' ),
			'is_restored'   => 0,
		),
		array( '%d', '%s', '%d', '%d', '%s', '%d' )
	);
}

/**
 * Log a restore action
 *
 * @param int $attachment_id Attachment ID.
 * @return bool Whether the log was updated successfully.
 */
function masic_log_restore( $attachment_id ) {
	global $wpdb;

	// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
	return $wpdb->update(
		$wpdb->prefix . 'masic_delete_logs',
		array(
			'is_restored' => 1,
			'restored_at' => current_time( 'mysql' ),
		),
		array( 'attachment_id' => absint( $attachment_id ) ),
		array( '%d', '%s' ),
		array( '%d' )
	);
}

/**
 * Check if an attachment is excluded from scanning
 *
 * @param int $attachment_id Attachment ID.
 * @return bool Whether the attachment is excluded.
 */
function masic_is_excluded( $attachment_id ) {
	$file = get_attached_file( absint( $attachment_id ) );
	if ( ! $file ) {
		return false;
	}

	// Check excluded folders
	$excluded_folders = get_option( 'masic_excluded_folders', '' );
	if ( ! empty( $excluded_folders ) ) {
		$folders = array_map( 'trim', explode( ',', $excluded_folders ) );
		foreach ( $folders as $folder ) {
			if ( ! empty( $folder ) && strpos( $file, $folder ) !== false ) {
				return true;
			}
		}
	}

	// Check excluded file types
	$excluded_types = get_option( 'masic_excluded_file_types', '' );
	if ( ! empty( $excluded_types ) ) {
		$types     = array_map( 'trim', explode( ',', $excluded_types ) );
		$extension = strtolower( pathinfo( $file, PATHINFO_EXTENSION ) );
		if ( in_array( $extension, $types, true ) ) {
			return true;
		}
	}

	return false;
}
