<?php
/**
 * Cron functionality for MA Smart Image Cleaner
 *
 * @package MA_Smart_Image_Cleaner
 */

// Prevent direct access
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Cron handler class
 */
class MASIC_Cron {

	/**
	 * Constructor
	 */
	public function __construct() {
		$this->init_hooks();
	}

	/**
	 * Initialize hooks
	 */
	private function init_hooks() {
		// Add custom cron schedules
		add_filter( 'cron_schedules', array( $this, 'add_cron_schedules' ) );

		// Scheduled scan action
		add_action( 'masic_scheduled_scan', array( $this, 'run_scheduled_scan' ) );

		// Settings update hook
		add_action( 'update_option_masic_scan_frequency', array( $this, 'reschedule_scan' ), 10, 2 );
		add_action( 'update_option_masic_enable_auto_scan', array( $this, 'toggle_auto_scan' ), 10, 2 );
	}

	/**
	 * Add custom cron schedules
	 *
	 * @param array $schedules Existing schedules.
	 * @return array Modified schedules.
	 */
	public function add_cron_schedules( $schedules ) {
		$schedules['masic_daily'] = array(
			'interval' => DAY_IN_SECONDS,
			'display'  => __( 'Once Daily', 'ma-smart-image-cleaner' ),
		);

		$schedules['masic_weekly'] = array(
			'interval' => WEEK_IN_SECONDS,
			'display'  => __( 'Once Weekly', 'ma-smart-image-cleaner' ),
		);

		return $schedules;
	}

	/**
	 * Run scheduled scan
	 */
	public function run_scheduled_scan() {
		// Check if auto scan is enabled
		if ( ! get_option( 'masic_enable_auto_scan', false ) ) {
			return;
		}

		// Check if a scan is already in progress
		if ( get_option( 'masic_scan_in_progress', false ) ) {
			return;
		}

		// Start the scan
		$scanner = new MASIC_Scanner();
		$scanner->start_scan();
	}

	/**
	 * Reschedule scan when frequency changes
	 *
	 * @param mixed $old_value Old option value.
	 * @param mixed $new_value New option value.
	 */
	public function reschedule_scan( $old_value, $new_value ) {
		// Clear existing schedule
		wp_clear_scheduled_hook( 'masic_scheduled_scan' );

		// Don't schedule if auto scan is disabled or manual
		if ( ! get_option( 'masic_enable_auto_scan', false ) || 'manual' === $new_value ) {
			return;
		}

		// Schedule based on frequency
		$schedule = 'masic_' . $new_value;
		if ( ! wp_next_scheduled( 'masic_scheduled_scan' ) ) {
			wp_schedule_event( time() + HOUR_IN_SECONDS, $schedule, 'masic_scheduled_scan' );
		}
	}

	/**
	 * Toggle auto scan scheduling
	 *
	 * @param mixed $old_value Old option value.
	 * @param mixed $new_value New option value.
	 */
	public function toggle_auto_scan( $old_value, $new_value ) {
		if ( $new_value ) {
			// Enable auto scan
			$frequency = get_option( 'masic_scan_frequency', 'manual' );
			if ( 'manual' !== $frequency ) {
				$schedule = 'masic_' . $frequency;
				if ( ! wp_next_scheduled( 'masic_scheduled_scan' ) ) {
					wp_schedule_event( time() + HOUR_IN_SECONDS, $schedule, 'masic_scheduled_scan' );
				}
			}
		} else {
			// Disable auto scan
			wp_clear_scheduled_hook( 'masic_scheduled_scan' );
		}
	}

	/**
	 * Get next scheduled scan time
	 *
	 * @return int|false Timestamp of next scan or false.
	 */
	public static function get_next_scheduled_scan() {
		return wp_next_scheduled( 'masic_scheduled_scan' );
	}

	/**
	 * Check if auto scan is active
	 *
	 * @return bool Whether auto scan is active.
	 */
	public static function is_auto_scan_active() {
		return get_option( 'masic_enable_auto_scan', false )
			   && 'manual' !== get_option( 'masic_scan_frequency', 'manual' );
	}
}

// Initialize cron handler
new MASIC_Cron();
