/**
 * MA Smart Image Cleaner - Admin JavaScript
 *
 * @package MA_Smart_Image_Cleaner
 */

(function($) {
	'use strict';

	window.MASIC = window.MASIC || {};

	var scanStartTime = null;
	var pollInterval = null;

	/**
	 * Initialize
	 */
	MASIC.init = function() {
		this.bindDashboardEvents();
		this.bindScanResultsEvents();
		this.bindTrashEvents();
		this.bindLogsEvents();
		this.bindSettingsEvents();

		// Auto-start polling if scan is in progress
		if (masic_ajax.scan_in_progress && masic_ajax.scan_status === 'running') {
			MASIC.startProgressPolling();
		}
	};

	/**
	 * Show a notification
	 */
	MASIC.showNotice = function(message, type) {
		type = type || 'success';
		var $notice = $('<div class="notice notice-' + type + ' is-dismissible"><p>' + message + '</p></div>');
		$('.wrap h1').first().after($notice);
		setTimeout(function() {
			$notice.fadeOut(function() { $(this).remove(); });
		}, 5000);
		$notice.find('.notice-dismiss').on('click', function() {
			$notice.fadeOut(function() { $(this).remove(); });
		});
	};

	/**
	 * Format file size
	 */
	MASIC.formatFileSize = function(bytes) {
		if (bytes === 0) return '0 B';
		var units = ['B', 'KB', 'MB', 'GB'];
		var i = Math.floor(Math.log(bytes) / Math.log(1024));
		return parseFloat((bytes / Math.pow(1024, i)).toFixed(2)) + ' ' + units[i];
	};

	/**
	 * Update selected count helper
	 */
	function updateSelectedCount() {
		var count = $('.masic-item-check:checked').length;
		$('#masic-selected-num').text(count);
	}

	// =========================================================================
	// Dashboard
	// =========================================================================

	MASIC.showProgressUI = function(totalImages) {
		var i18n = masic_ajax.i18n;
		var progressHTML =
			'<div class="masic-scan-progress">' +
				'<div class="masic-scan-header">' +
					'<span class="dashicons dashicons-update masic-spinning"></span>' +
					'<h3>' + i18n.scanning_library + '</h3>' +
					'<span class="masic-progress-percent" id="masic-progress-percent">0%</span>' +
				'</div>' +
				'<div class="masic-progress-bar">' +
					'<div class="masic-progress-fill" id="masic-progress-fill" style="width: 0%"></div>' +
				'</div>' +
				'<div class="masic-progress-stats">' +
					'<span id="masic-progress-text">' +
						'<strong>0</strong> ' + i18n.of + ' <strong>' + totalImages + '</strong> ' + i18n.images_scanned +
					'</span>' +
					'<span class="masic-estimated-time" id="masic-estimated-time"></span>' +
				'</div>' +
				'<div class="masic-progress-current" id="masic-progress-current">' +
					i18n.initializing +
				'</div>' +
				'<div class="masic-scan-actions">' +
					'<button type="button" class="button masic-btn-large" id="masic-pause-scan">' +
						'<span class="dashicons dashicons-controls-pause"></span> ' + i18n.pause_scan +
					'</button>' +
				'</div>' +
			'</div>';
		$('#masic-scan-card').html(progressHTML);
		MASIC.bindPauseButton();
	};

	MASIC.startProgressPolling = function() {
		scanStartTime = Date.now();
		var i18n = masic_ajax.i18n;

		pollInterval = setInterval(function() {
			$.ajax({
				url: masic_ajax.ajax_url,
				type: 'POST',
				data: {
					action: 'masic_get_scan_progress',
					nonce: masic_ajax.scan_nonce
				},
				success: function(response) {
					if (!response || response.success !== true) {
						clearInterval(pollInterval);
						var msg = (response && response.data && response.data.message) ? response.data.message : i18n.progress_failed;
						$('#masic-progress-current').html('⚠️ ' + msg);
						return;
					}

					var progress = response.data.progress;

					if (progress && progress.status === 'error') {
						clearInterval(pollInterval);
						var errMsg = progress.error_message || i18n.scan_failed;
						$('#masic-progress-current').html('⚠️ ' + errMsg);
						return;
					}

					if (!response.data.in_progress || progress.status === 'complete') {
						clearInterval(pollInterval);
						$('#masic-progress-percent').text('100%');
						$('#masic-progress-fill').css('width', '100%');
						$('#masic-progress-current').html('✅ ' + i18n.scan_complete_msg);
						setTimeout(function() { location.reload(); }, 1500);
						return;
					}

					if (progress.status !== 'paused') {
						var percent = Math.round((progress.scanned_images / progress.total_images) * 100);
						$('#masic-progress-fill').css('width', percent + '%');
						$('#masic-progress-percent').text(percent + '%');
						$('#masic-progress-text').html(
							'<strong>' + progress.scanned_images + '</strong> ' +
							i18n.of + ' <strong>' + progress.total_images + '</strong> ' + i18n.images_scanned
						);

						if (progress.current_image) {
							$('#masic-progress-current').html('📁 ' + i18n.analyzing + ' <code>' + progress.current_image + '</code>');
						}

						var elapsedTime = (Date.now() - scanStartTime) / 1000;
						var imagesPerSecond = progress.scanned_images / elapsedTime;
						var remainingImages = progress.total_images - progress.scanned_images;

						if (imagesPerSecond > 0 && remainingImages > 0) {
							var remainingSeconds = remainingImages / imagesPerSecond;
							var minutes = Math.floor(remainingSeconds / 60);
							var seconds = Math.floor(remainingSeconds % 60);

							if (minutes > 0) {
								$('#masic-estimated-time').text('~' + minutes + 'm ' + seconds + 's ' + i18n.remaining);
							} else {
								$('#masic-estimated-time').text('~' + seconds + 's ' + i18n.remaining);
							}
						}
					}
				},
				error: function() {
					// Don't stop polling on network errors
				}
			});
		}, 1000);
	};

	MASIC.bindPauseButton = function() {
		$('#masic-pause-scan').off('click').on('click', function() {
			var $btn = $(this);
			var i18n = masic_ajax.i18n;
			$btn.prop('disabled', true);

			$.ajax({
				url: masic_ajax.ajax_url,
				type: 'POST',
				data: { action: 'masic_pause_scan', nonce: masic_ajax.scan_nonce },
				success: function(response) {
					if (response.success) {
						clearInterval(pollInterval);
						$btn.attr('id', 'masic-resume-scan')
							.addClass('button-primary')
							.html('<span class="dashicons dashicons-controls-play"></span> ' + i18n.resume_scan)
							.prop('disabled', false);
						MASIC.bindResumeButton();
					}
				},
				error: function() { $btn.prop('disabled', false); }
			});
		});
	};

	MASIC.bindResumeButton = function() {
		$('#masic-resume-scan').off('click').on('click', function() {
			var $btn = $(this);
			var i18n = masic_ajax.i18n;
			$btn.prop('disabled', true);

			$.ajax({
				url: masic_ajax.ajax_url,
				type: 'POST',
				data: { action: 'masic_resume_scan', nonce: masic_ajax.scan_nonce },
				success: function(response) {
					if (response.success) {
						$btn.attr('id', 'masic-pause-scan')
							.removeClass('button-primary')
							.html('<span class="dashicons dashicons-controls-pause"></span> ' + i18n.pause_scan)
							.prop('disabled', false);
						MASIC.bindPauseButton();
						MASIC.startProgressPolling();
					}
				},
				error: function() { $btn.prop('disabled', false); }
			});
		});
	};

	MASIC.bindDashboardEvents = function() {
		// Start scan button
		$('#masic-start-scan').on('click', function() {
			var $btn = $(this);
			var i18n = masic_ajax.i18n;
			$btn.prop('disabled', true).text(i18n.starting);

			$.ajax({
				url: masic_ajax.ajax_url,
				type: 'POST',
				data: { action: 'masic_start_scan', nonce: masic_ajax.scan_nonce },
				success: function(response) {
					if (response.success) {
						MASIC.showProgressUI(response.data.total_images);
						MASIC.startProgressPolling();
					} else {
						alert(response.data.message);
						$btn.prop('disabled', false).html('<span class="dashicons dashicons-controls-play"></span> ' + i18n.full_scan);
					}
				},
				error: function() {
					alert(i18n.error);
					$btn.prop('disabled', false).html('<span class="dashicons dashicons-controls-play"></span> ' + i18n.full_scan);
				}
			});
		});

		// Initial bindings for pause/resume
		MASIC.bindPauseButton();
		MASIC.bindResumeButton();
	};

	// =========================================================================
	// Scan Results
	// =========================================================================

	MASIC.bindScanResultsEvents = function() {
		$('#masic-select-all').on('change', function() {
			$('.masic-item-check').prop('checked', $(this).prop('checked'));
			updateSelectedCount();
		});

		$('.masic-item-check').on('change', function() {
			updateSelectedCount();
		});

		// Bulk action (scan results page - trash only)
		$('#masic-apply-bulk').on('click', function() {
			var action = $('#masic-bulk-action').val();
			var ids = $('.masic-item-check:checked').map(function() { return $(this).val(); }).get();

			if (!action) { alert(masic_ajax.i18n.select_action); return; }
			if (ids.length === 0) { alert(masic_ajax.i18n.select_image); return; }

			// Determine AJAX action based on bulk action value
			var ajaxAction, confirmMsg;
			if (action === 'trash') {
				ajaxAction = 'masic_delete_images';
				confirmMsg = masic_ajax.i18n.confirm_delete;
			} else if (action === 'restore') {
				ajaxAction = 'masic_restore_images';
				confirmMsg = masic_ajax.i18n.restore_confirm;
			} else if (action === 'delete') {
				ajaxAction = 'masic_permanent_delete';
				confirmMsg = masic_ajax.i18n.confirm_permanent;
			} else {
				return;
			}

			if (!confirm(confirmMsg)) { return; }

			$.ajax({
				url: masic_ajax.ajax_url,
				type: 'POST',
				data: { action: ajaxAction, nonce: masic_ajax.delete_nonce, ids: ids },
				success: function(response) {
					if (response.success) { location.reload(); }
					else { alert(response.data.message); }
				}
			});
		});

		// Single delete
		$('.masic-delete-single').on('click', function() {
			var id = $(this).data('id');
			if (!confirm(masic_ajax.i18n.confirm_delete)) { return; }

			$.ajax({
				url: masic_ajax.ajax_url,
				type: 'POST',
				data: { action: 'masic_delete_images', nonce: masic_ajax.delete_nonce, ids: [id] },
				success: function(response) {
					if (response.success) {
						$('tr[data-id="' + id + '"]').fadeOut(function() { $(this).remove(); });
					} else { alert(response.data.message); }
				}
			});
		});

		// Mark to keep
		$('.masic-mark-keep').on('click', function() {
			var $btn = $(this);
			var id = $btn.data('id');
			var keep = $btn.data('keep');

			$.ajax({
				url: masic_ajax.ajax_url,
				type: 'POST',
				data: { action: 'masic_mark_keep', nonce: masic_ajax.nonce, id: id, keep: keep },
				success: function(response) {
					if (response.success) { location.reload(); }
					else { alert(response.data.message); }
				}
			});
		});
	};

	// =========================================================================
	// Trash
	// =========================================================================

	MASIC.bindTrashEvents = function() {
		// Single restore
		$('.masic-restore-single').on('click', function() {
			var id = $(this).data('id');

			$.ajax({
				url: masic_ajax.ajax_url,
				type: 'POST',
				data: { action: 'masic_restore_images', nonce: masic_ajax.delete_nonce, ids: [id] },
				success: function(response) {
					if (response.success) {
						$('tr[data-id="' + id + '"]').fadeOut(function() { $(this).remove(); });
					} else { alert(response.data.message); }
				}
			});
		});

		// Permanent delete
		$('.masic-permanent-delete').on('click', function() {
			var id = $(this).data('id');
			if (!confirm(masic_ajax.i18n.confirm_permanent)) { return; }

			$.ajax({
				url: masic_ajax.ajax_url,
				type: 'POST',
				data: { action: 'masic_permanent_delete', nonce: masic_ajax.delete_nonce, ids: [id] },
				success: function(response) {
					if (response.success) {
						$('tr[data-id="' + id + '"]').fadeOut(function() { $(this).remove(); });
					} else { alert(response.data.message); }
				}
			});
		});
	};

	// =========================================================================
	// Logs
	// =========================================================================

	MASIC.bindLogsEvents = function() {
		$('.masic-restore-from-log').on('click', function() {
			var $btn = $(this);
			var id = $btn.data('id');
			$btn.prop('disabled', true);

			$.ajax({
				url: masic_ajax.ajax_url,
				type: 'POST',
				data: { action: 'masic_restore_images', nonce: masic_ajax.delete_nonce, ids: [id] },
				success: function(response) {
					if (response.success) { location.reload(); }
					else { alert(response.data.message); $btn.prop('disabled', false); }
				},
				error: function() {
					alert(masic_ajax.i18n.error);
					$btn.prop('disabled', false);
				}
			});
		});
	};

	// =========================================================================
	// Settings
	// =========================================================================

	MASIC.bindSettingsEvents = function() {
		// Toggle frequency based on auto scan checkbox
		function toggleFrequency() {
			if ($('#enable_auto_scan').is(':checked')) {
				$('.masic-frequency-row').show();
			} else {
				$('.masic-frequency-row').hide();
			}
		}

		$('#enable_auto_scan').on('change', toggleFrequency);
		toggleFrequency();

		// Save settings
		$('#masic-settings-form').on('submit', function(e) {
			e.preventDefault();

			var $btn = $('#masic-save-settings');
			var $spinner = $('#masic-settings-spinner');

			$btn.prop('disabled', true);
			$spinner.addClass('is-active');

			$.ajax({
				url: masic_ajax.ajax_url,
				type: 'POST',
				data: {
					action: 'masic_save_settings',
					nonce: masic_ajax.nonce,
					enable_auto_scan: $('#enable_auto_scan').is(':checked') ? 1 : 0,
					scan_frequency: $('#scan_frequency').val(),
					enable_elementor_scan: $('#enable_elementor_scan').is(':checked') ? 1 : 0,
					excluded_folders: $('#excluded_folders').val(),
					excluded_file_types: $('#excluded_file_types').val(),
					batch_size: $('#batch_size').val(),
					editor_can_view: $('#editor_can_view').is(':checked') ? 1 : 0
				},
				success: function(response) {
					if (response.success) {
						$('<div class="notice notice-success is-dismissible"><p>' + response.data.message + '</p></div>')
							.insertAfter('.wp-heading-inline')
							.delay(3000)
							.fadeOut();
					} else {
						alert(response.data.message);
					}
				},
				error: function() {
					alert(masic_ajax.i18n.error);
				},
				complete: function() {
					$btn.prop('disabled', false);
					$spinner.removeClass('is-active');
				}
			});
		});
	};

	// Initialize when document is ready
	$(document).ready(function() {
		MASIC.init();
	});

})(jQuery);
