<?php
/**
 * Scan results page for MA Smart Image Cleaner
 *
 * @package MA_Smart_Image_Cleaner
 */

// Prevent direct access
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

// Get filter parameters
// phpcs:disable WordPress.Security.NonceVerification.Recommended -- These are read-only filter parameters for page display
$masic_current_status = isset( $_GET['status'] ) ? sanitize_text_field( wp_unslash( $_GET['status'] ) ) : 'all';
$masic_current_page = isset( $_GET['paged'] ) ? absint( $_GET['paged'] ) : 1;
$masic_orderby = isset( $_GET['orderby'] ) ? sanitize_text_field( wp_unslash( $_GET['orderby'] ) ) : 'file_size';
$masic_order = isset( $_GET['order'] ) ? sanitize_text_field( wp_unslash( $_GET['order'] ) ) : 'DESC';
$masic_search = isset( $_GET['s'] ) ? sanitize_text_field( wp_unslash( $_GET['s'] ) ) : '';
$masic_min_size = isset( $_GET['min_size'] ) ? floatval( $_GET['min_size'] ) : 0;
$masic_file_type = isset( $_GET['file_type'] ) ? sanitize_text_field( wp_unslash( $_GET['file_type'] ) ) : 'all';
// phpcs:enable WordPress.Security.NonceVerification.Recommended

// Get results
$masic_results = masic_get_scan_results( array(
	'status'    => $masic_current_status,
	'paged'     => $masic_current_page,
	'orderby'   => $masic_orderby,
	'order'     => $masic_order,
	'search'    => $masic_search,
	'min_size'  => $masic_min_size,
	'file_type' => $masic_file_type,
	'per_page'  => 20,
) );
?>

<div class="wrap masic-wrap">
	<h1 class="wp-heading-inline">
		<span class="dashicons dashicons-images-alt2"></span>
		<?php esc_html_e( 'Scan Results', 'ma-smart-image-cleaner' ); ?>
	</h1>

	<!-- Navigation Tabs -->
	<nav class="nav-tab-wrapper masic-nav-tabs">
		<a href="<?php echo esc_url( admin_url( 'upload.php?page=masic-cleaner' ) ); ?>" class="nav-tab">
			<?php esc_html_e( 'Dashboard', 'ma-smart-image-cleaner' ); ?>
		</a>
		<a href="<?php echo esc_url( admin_url( 'upload.php?page=masic-cleaner-results' ) ); ?>" class="nav-tab nav-tab-active">
			<?php esc_html_e( 'Scan Results', 'ma-smart-image-cleaner' ); ?>
		</a>
		<a href="<?php echo esc_url( admin_url( 'upload.php?page=masic-cleaner-trash' ) ); ?>" class="nav-tab">
			<?php esc_html_e( 'Trash', 'ma-smart-image-cleaner' ); ?>
		</a>
		<a href="<?php echo esc_url( admin_url( 'upload.php?page=masic-cleaner-logs' ) ); ?>" class="nav-tab">
			<?php esc_html_e( 'Activity Log', 'ma-smart-image-cleaner' ); ?>
		</a>
		<?php if ( masic_user_can_settings() ) : ?>
		<a href="<?php echo esc_url( admin_url( 'upload.php?page=masic-cleaner-settings' ) ); ?>" class="nav-tab">
			<?php esc_html_e( 'Settings', 'ma-smart-image-cleaner' ); ?>
		</a>
		<?php endif; ?>
	</nav>

	<!-- Filters -->
	<div class="masic-filters">
		<form method="get" action="">
			<input type="hidden" name="page" value="masic-cleaner-results">

			<div class="masic-filter-row">
				<div class="masic-filter-group">
					<label for="status"><?php esc_html_e( 'Status:', 'ma-smart-image-cleaner' ); ?></label>
					<select name="status" id="status">
						<option value="all" <?php selected( $masic_current_status, 'all' ); ?>><?php esc_html_e( 'All', 'ma-smart-image-cleaner' ); ?></option>
						<option value="used" <?php selected( $masic_current_status, 'used' ); ?>><?php esc_html_e( 'Used', 'ma-smart-image-cleaner' ); ?></option>
						<option value="unused" <?php selected( $masic_current_status, 'unused' ); ?>><?php esc_html_e( 'Unused', 'ma-smart-image-cleaner' ); ?></option>
					</select>
				</div>

				<div class="masic-filter-group">
					<label for="file_type"><?php esc_html_e( 'File Type:', 'ma-smart-image-cleaner' ); ?></label>
					<select name="file_type" id="file_type">
						<option value="all" <?php selected( $masic_file_type, 'all' ); ?>><?php esc_html_e( 'All Types', 'ma-smart-image-cleaner' ); ?></option>
						<option value="jpeg" <?php selected( $masic_file_type, 'jpeg' ); ?>>JPEG</option>
						<option value="png" <?php selected( $masic_file_type, 'png' ); ?>>PNG</option>
						<option value="gif" <?php selected( $masic_file_type, 'gif' ); ?>>GIF</option>
						<option value="webp" <?php selected( $masic_file_type, 'webp' ); ?>>WebP</option>
						<option value="svg+xml" <?php selected( $masic_file_type, 'svg+xml' ); ?>>SVG</option>
					</select>
				</div>

				<div class="masic-filter-group">
					<label for="min_size"><?php esc_html_e( 'Min Size (MB):', 'ma-smart-image-cleaner' ); ?></label>
					<input type="number" name="min_size" id="min_size" value="<?php echo esc_attr( $masic_min_size ); ?>" step="0.1" min="0" style="width: 80px;">
				</div>

				<div class="masic-filter-group">
					<label for="s"><?php esc_html_e( 'Search:', 'ma-smart-image-cleaner' ); ?></label>
					<input type="search" name="s" id="s" value="<?php echo esc_attr( $masic_search ); ?>" placeholder="<?php esc_attr_e( 'Search images...', 'ma-smart-image-cleaner' ); ?>">
				</div>

				<button type="submit" class="button"><?php esc_html_e( 'Filter', 'ma-smart-image-cleaner' ); ?></button>
				<a href="<?php echo esc_url( admin_url( 'upload.php?page=masic-cleaner-results' ) ); ?>" class="button"><?php esc_html_e( 'Reset', 'ma-smart-image-cleaner' ); ?></a>
			</div>
		</form>
	</div>

	<!-- Bulk Actions -->
	<?php if ( masic_user_can_delete() && ! empty( $masic_results['items'] ) ) : ?>
	<div class="masic-bulk-actions">
		<form method="post" id="masic-bulk-form">
			<?php wp_nonce_field( 'masic_bulk_action', 'masic_bulk_nonce' ); ?>
			<select name="bulk_action" id="masic-bulk-action">
				<option value=""><?php esc_html_e( 'Bulk Actions', 'ma-smart-image-cleaner' ); ?></option>
				<option value="trash"><?php esc_html_e( 'Move to Trash', 'ma-smart-image-cleaner' ); ?></option>
			</select>
			<button type="button" class="button" id="masic-apply-bulk"><?php esc_html_e( 'Apply', 'ma-smart-image-cleaner' ); ?></button>
			<span class="masic-selected-count">
				<span id="masic-selected-num">0</span> <?php esc_html_e( 'selected', 'ma-smart-image-cleaner' ); ?>
			</span>
		</form>
	</div>
	<?php endif; ?>

	<!-- Results Table -->
	<div class="masic-results-table">
		<?php if ( empty( $masic_results['items'] ) ) : ?>
			<div class="masic-no-results">
				<span class="dashicons dashicons-format-gallery"></span>
				<p><?php esc_html_e( 'No images found. Run a scan to analyze your media library.', 'ma-smart-image-cleaner' ); ?></p>
				<a href="<?php echo esc_url( admin_url( 'upload.php?page=masic-cleaner' ) ); ?>" class="button button-primary">
					<?php esc_html_e( 'Go to Dashboard', 'ma-smart-image-cleaner' ); ?>
				</a>
			</div>
		<?php else : ?>
			<table class="wp-list-table widefat fixed striped">
				<thead>
					<tr>
						<?php if ( masic_user_can_delete() ) : ?>
						<td class="check-column">
							<input type="checkbox" id="masic-select-all">
						</td>
						<?php endif; ?>
						<th class="column-thumbnail"><?php esc_html_e( 'Image', 'ma-smart-image-cleaner' ); ?></th>
						<th class="column-filename">
							<a href="<?php echo esc_url( add_query_arg( array( 'orderby' => 'post_title', 'order' => 'post_title' === $masic_orderby && 'ASC' === $masic_order ? 'DESC' : 'ASC' ) ) ); ?>">
								<?php esc_html_e( 'Filename', 'ma-smart-image-cleaner' ); ?>
								<?php if ( 'post_title' === $masic_orderby ) : ?>
									<span class="dashicons dashicons-arrow-<?php echo 'ASC' === $masic_order ? 'up' : 'down'; ?>-alt2"></span>
								<?php endif; ?>
							</a>
						</th>
						<th class="column-filesize">
							<a href="<?php echo esc_url( add_query_arg( array( 'orderby' => 'file_size', 'order' => 'file_size' === $masic_orderby && 'DESC' === $masic_order ? 'ASC' : 'DESC' ) ) ); ?>">
								<?php esc_html_e( 'Size', 'ma-smart-image-cleaner' ); ?>
								<?php if ( 'file_size' === $masic_orderby ) : ?>
									<span class="dashicons dashicons-arrow-<?php echo 'ASC' === $masic_order ? 'up' : 'down'; ?>-alt2"></span>
								<?php endif; ?>
							</a>
						</th>
						<th class="column-date">
							<a href="<?php echo esc_url( add_query_arg( array( 'orderby' => 'post_date', 'order' => 'post_date' === $masic_orderby && 'DESC' === $masic_order ? 'ASC' : 'DESC' ) ) ); ?>">
								<?php esc_html_e( 'Upload Date', 'ma-smart-image-cleaner' ); ?>
								<?php if ( 'post_date' === $masic_orderby ) : ?>
									<span class="dashicons dashicons-arrow-<?php echo 'ASC' === $masic_order ? 'up' : 'down'; ?>-alt2"></span>
								<?php endif; ?>
							</a>
						</th>
						<th class="column-status"><?php esc_html_e( 'Status', 'ma-smart-image-cleaner' ); ?></th>
						<th class="column-usage"><?php esc_html_e( 'Used In', 'ma-smart-image-cleaner' ); ?></th>
						<th class="column-actions"><?php esc_html_e( 'Actions', 'ma-smart-image-cleaner' ); ?></th>
					</tr>
				</thead>
				<tbody>
					<?php foreach ( $masic_results['items'] as $masic_item ) : ?>
					<tr data-id="<?php echo esc_attr( $masic_item->attachment_id ); ?>">
						<?php if ( masic_user_can_delete() ) : ?>
						<th class="check-column">
							<input type="checkbox" class="masic-item-check" value="<?php echo esc_attr( $masic_item->attachment_id ); ?>">
						</th>
						<?php endif; ?>
						<td class="column-thumbnail">
							<?php if ( $masic_item->thumbnail_url ) : ?>
								<img src="<?php echo esc_url( $masic_item->thumbnail_url ); ?>" alt="<?php echo esc_attr( $masic_item->filename ); ?>" class="masic-thumbnail">
							<?php else : ?>
								<span class="dashicons dashicons-format-image masic-no-thumb"></span>
							<?php endif; ?>
						</td>
						<td class="column-filename">
							<strong><?php echo esc_html( $masic_item->filename ); ?></strong>
							<?php if ( $masic_item->is_marked_keep ) : ?>
								<span class="masic-keep-badge" title="<?php esc_attr_e( 'Marked to keep', 'ma-smart-image-cleaner' ); ?>">
									<span class="dashicons dashicons-heart"></span>
								</span>
							<?php endif; ?>
							<div class="row-actions">
								<span class="view">
									<a href="<?php echo esc_url( get_edit_post_link( $masic_item->attachment_id ) ); ?>" target="_blank">
										<?php esc_html_e( 'View', 'ma-smart-image-cleaner' ); ?>
									</a>
								</span>
							</div>
						</td>
						<td class="column-filesize">
							<?php echo esc_html( masic_format_file_size( $masic_item->file_size ) ); ?>
						</td>
						<td class="column-date">
							<?php echo esc_html( date_i18n( get_option( 'date_format' ), strtotime( $masic_item->post_date ) ) ); ?>
						</td>
						<td class="column-status">
							<span class="masic-status-badge masic-status-<?php echo esc_attr( $masic_item->status ); ?>">
								<?php
								$masic_status_labels = array(
									'used'   => __( 'Used', 'ma-smart-image-cleaner' ),
									'unused' => __( 'Unused', 'ma-smart-image-cleaner' ),
								);
								echo esc_html( isset( $masic_status_labels[ $masic_item->status ] ) ? $masic_status_labels[ $masic_item->status ] : $masic_item->status );
								?>
							</span>
						</td>
						<td class="column-usage">
							<?php if ( ! empty( $masic_item->detected_locations ) ) : ?>
								<ul class="masic-usage-list">
									<?php foreach ( array_slice( $masic_item->detected_locations, 0, 3 ) as $masic_location ) : ?>
										<li>
											<a href="<?php echo esc_url( get_edit_post_link( $masic_location['post_id'] ) ); ?>" target="_blank">
												<?php echo esc_html( $masic_location['post_title'] ); ?>
											</a>
											<span class="masic-usage-type">(<?php echo esc_html( $masic_location['type'] ); ?>)</span>
										</li>
									<?php endforeach; ?>
									<?php if ( count( $masic_item->detected_locations ) > 3 ) : ?>
										<li class="masic-more">
											<?php
											printf(
												/* translators: %d: number of additional locations */
												esc_html__( '+%d more', 'ma-smart-image-cleaner' ),
												count( $masic_item->detected_locations ) - 3
											);
											?>
										</li>
									<?php endif; ?>
								</ul>
							<?php else : ?>
								<span class="masic-no-usage"><?php esc_html_e( 'Not used', 'ma-smart-image-cleaner' ); ?></span>
							<?php endif; ?>
						</td>
						<td class="column-actions">
							<?php if ( masic_user_can_delete() ) : ?>
								<button type="button" class="button masic-mark-keep" data-id="<?php echo esc_attr( $masic_item->attachment_id ); ?>" data-keep="<?php echo $masic_item->is_marked_keep ? '0' : '1'; ?>">
									<span class="dashicons dashicons-<?php echo $masic_item->is_marked_keep ? 'yes' : 'heart'; ?>"></span>
									<?php echo $masic_item->is_marked_keep ? esc_html__( 'Unmark', 'ma-smart-image-cleaner' ) : esc_html__( 'Keep', 'ma-smart-image-cleaner' ); ?>
								</button>
								<button type="button" class="button masic-delete-single" data-id="<?php echo esc_attr( $masic_item->attachment_id ); ?>">
									<span class="dashicons dashicons-trash"></span>
									<?php esc_html_e( 'Trash', 'ma-smart-image-cleaner' ); ?>
								</button>
							<?php endif; ?>
						</td>
					</tr>
					<?php endforeach; ?>
				</tbody>
			</table>

			<!-- Pagination -->
			<?php if ( $masic_results['total_pages'] > 1 ) : ?>
			<div class="masic-pagination">
				<?php
				// phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped -- paginate_links() returns safe HTML
				echo paginate_links( array(
					'base'      => add_query_arg( 'paged', '%#%' ),
					'format'    => '',
					'current'   => $masic_current_page,
					'total'     => $masic_results['total_pages'],
					'prev_text' => '&laquo; ' . __( 'Previous', 'ma-smart-image-cleaner' ),
					'next_text' => __( 'Next', 'ma-smart-image-cleaner' ) . ' &raquo;',
				) );
				?>
				<span class="masic-total-items">
					<?php
					printf(
						/* translators: %s: total number of items */
						esc_html__( '%s items total', 'ma-smart-image-cleaner' ),
						number_format( $masic_results['total'] )
					);
					?>
				</span>
			</div>
			<?php endif; ?>
		<?php endif; ?>
	</div>
</div>
