<?php
/**
 * Admin menu registration for MA Smart Image Cleaner
 *
 * @package MA_Smart_Image_Cleaner
 */

// Prevent direct access
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Admin menu class
 */
class MASIC_Admin_Menu {

	/**
	 * Constructor
	 */
	public function __construct() {
		add_action( 'admin_menu', array( $this, 'register_menu' ) );
		$this->init_ajax_handlers();
	}

	/**
	 * Register admin menu
	 */
	public function register_menu() {
		if ( ! masic_user_can_view() ) {
			return;
		}

		// Main menu under Media
		add_media_page(
			__( 'Image Cleaner', 'ma-smart-image-cleaner' ),
			__( 'Image Cleaner', 'ma-smart-image-cleaner' ),
			'upload_files',
			'masic-cleaner',
			array( $this, 'render_dashboard' )
		);

		// Submenu pages
		add_submenu_page(
			'upload.php',
			__( 'Scan Results', 'ma-smart-image-cleaner' ),
			__( 'Scan Results', 'ma-smart-image-cleaner' ),
			'upload_files',
			'masic-cleaner-results',
			array( $this, 'render_scan_results' )
		);

		add_submenu_page(
			'upload.php',
			__( 'Trash', 'ma-smart-image-cleaner' ),
			__( 'Trash', 'ma-smart-image-cleaner' ),
			'upload_files',
			'masic-cleaner-trash',
			array( $this, 'render_trash' )
		);

		add_submenu_page(
			'upload.php',
			__( 'Activity Logs', 'ma-smart-image-cleaner' ),
			__( 'Activity Logs', 'ma-smart-image-cleaner' ),
			'upload_files',
			'masic-cleaner-logs',
			array( $this, 'render_logs' )
		);

		if ( masic_user_can_settings() ) {
			add_submenu_page(
				'upload.php',
				__( 'Settings', 'ma-smart-image-cleaner' ),
				__( 'Settings', 'ma-smart-image-cleaner' ),
				'manage_options',
				'masic-cleaner-settings',
				array( $this, 'render_settings' )
			);
		}
	}

	/**
	 * Initialize AJAX handlers
	 */
	private function init_ajax_handlers() {
		add_action( 'wp_ajax_masic_delete_images', array( $this, 'ajax_delete_images' ) );
		add_action( 'wp_ajax_masic_restore_images', array( $this, 'ajax_restore_images' ) );
		add_action( 'wp_ajax_masic_permanent_delete', array( $this, 'ajax_permanent_delete' ) );
		add_action( 'wp_ajax_masic_mark_keep', array( $this, 'ajax_mark_keep' ) );
		add_action( 'wp_ajax_masic_save_settings', array( $this, 'ajax_save_settings' ) );
	}

	/**
	 * Render dashboard page
	 */
	public function render_dashboard() {
		require_once MASIC_PLUGIN_DIR . 'admin/dashboard.php';
	}

	/**
	 * Render scan results page
	 */
	public function render_scan_results() {
		require_once MASIC_PLUGIN_DIR . 'admin/scan-results.php';
	}

	/**
	 * Render trash page
	 */
	public function render_trash() {
		require_once MASIC_PLUGIN_DIR . 'admin/trash.php';
	}

	/**
	 * Render logs page
	 */
	public function render_logs() {
		require_once MASIC_PLUGIN_DIR . 'admin/logs.php';
	}

	/**
	 * Render settings page
	 */
	public function render_settings() {
		require_once MASIC_PLUGIN_DIR . 'admin/settings.php';
	}

	/**
	 * AJAX handler for deleting images (move to trash)
	 */
	public function ajax_delete_images() {
		if ( ! masic_user_can_delete() ) {
			wp_send_json_error( array( 'message' => __( 'Permission denied.', 'ma-smart-image-cleaner' ) ) );
		}

		check_ajax_referer( 'masic_delete_nonce', 'nonce' );

		$ids = isset( $_POST['ids'] ) ? array_map( 'absint', (array) $_POST['ids'] ) : array();

		if ( empty( $ids ) ) {
			wp_send_json_error( array( 'message' => __( 'No images selected.', 'ma-smart-image-cleaner' ) ) );
		}

		$deleted = 0;
		foreach ( $ids as $id ) {
			$attachment = get_post( $id );
			if ( $attachment && 'attachment' === $attachment->post_type ) {
				$file = get_attached_file( $id );
				$file_size = $file && file_exists( $file ) ? filesize( $file ) : 0;
				masic_log_delete( $id, basename( $file ), $file_size );

				wp_trash_post( $id );
				$deleted++;
			}
		}

		wp_send_json_success( array(
			'message' => sprintf(
				/* translators: %d: number of images deleted */
				_n( '%d image moved to trash.', '%d images moved to trash.', $deleted, 'ma-smart-image-cleaner' ),
				$deleted
			),
			'count' => $deleted,
		) );
	}

	/**
	 * AJAX handler for restoring images from trash
	 */
	public function ajax_restore_images() {
		if ( ! masic_user_can_delete() ) {
			wp_send_json_error( array( 'message' => __( 'Permission denied.', 'ma-smart-image-cleaner' ) ) );
		}

		check_ajax_referer( 'masic_delete_nonce', 'nonce' );

		$ids = isset( $_POST['ids'] ) ? array_map( 'absint', (array) $_POST['ids'] ) : array();

		if ( empty( $ids ) ) {
			wp_send_json_error( array( 'message' => __( 'No images selected.', 'ma-smart-image-cleaner' ) ) );
		}

		$restored = 0;
		foreach ( $ids as $id ) {
			$attachment = get_post( $id );
			if ( $attachment && 'attachment' === $attachment->post_type && 'trash' === $attachment->post_status ) {
				wp_untrash_post( $id );
				masic_log_restore( $id );
				$restored++;
			}
		}

		wp_send_json_success( array(
			'message' => sprintf(
				/* translators: %d: number of images restored */
				_n( '%d image restored.', '%d images restored.', $restored, 'ma-smart-image-cleaner' ),
				$restored
			),
			'count' => $restored,
		) );
	}

	/**
	 * AJAX handler for permanent deletion
	 */
	public function ajax_permanent_delete() {
		if ( ! masic_user_can_delete() ) {
			wp_send_json_error( array( 'message' => __( 'Permission denied.', 'ma-smart-image-cleaner' ) ) );
		}

		check_ajax_referer( 'masic_delete_nonce', 'nonce' );

		$ids = isset( $_POST['ids'] ) ? array_map( 'absint', (array) $_POST['ids'] ) : array();

		if ( empty( $ids ) ) {
			wp_send_json_error( array( 'message' => __( 'No images selected.', 'ma-smart-image-cleaner' ) ) );
		}

		$deleted = 0;
		foreach ( $ids as $id ) {
			$attachment = get_post( $id );
			if ( $attachment && 'attachment' === $attachment->post_type ) {
				wp_delete_attachment( $id, true );
				$deleted++;
			}
		}

		wp_send_json_success( array(
			'message' => sprintf(
				/* translators: %d: number of images deleted */
				_n( '%d image permanently deleted.', '%d images permanently deleted.', $deleted, 'ma-smart-image-cleaner' ),
				$deleted
			),
			'count' => $deleted,
		) );
	}

	/**
	 * AJAX handler for marking images to keep
	 */
	public function ajax_mark_keep() {
		if ( ! masic_user_can_delete() ) {
			wp_send_json_error( array( 'message' => __( 'Permission denied.', 'ma-smart-image-cleaner' ) ) );
		}

		check_ajax_referer( 'masic_nonce', 'nonce' );

		$id = isset( $_POST['id'] ) ? absint( $_POST['id'] ) : 0;
		$keep = isset( $_POST['keep'] ) ? (bool) $_POST['keep'] : true;

		if ( ! $id ) {
			wp_send_json_error( array( 'message' => __( 'Invalid image ID.', 'ma-smart-image-cleaner' ) ) );
		}

		global $wpdb;

		// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
		$result = $wpdb->update(
			$wpdb->prefix . 'masic_scan_results',
			array( 'is_marked_keep' => $keep ? 1 : 0 ),
			array( 'attachment_id' => $id ),
			array( '%d' ),
			array( '%d' )
		);

		if ( false === $result ) {
			wp_send_json_error( array( 'message' => __( 'Failed to update image.', 'ma-smart-image-cleaner' ) ) );
		}

		wp_send_json_success( array(
			'message' => $keep
				? __( 'Image marked to keep.', 'ma-smart-image-cleaner' )
				: __( 'Keep mark removed.', 'ma-smart-image-cleaner' ),
		) );
	}

	/**
	 * AJAX handler for saving settings
	 */
	public function ajax_save_settings() {
		if ( ! masic_user_can_settings() ) {
			wp_send_json_error( array( 'message' => __( 'Permission denied.', 'ma-smart-image-cleaner' ) ) );
		}

		check_ajax_referer( 'masic_nonce', 'nonce' );

		$settings = array(
			'enable_auto_scan'      => isset( $_POST['enable_auto_scan'] ) ? (bool) $_POST['enable_auto_scan'] : false,
			'scan_frequency'        => isset( $_POST['scan_frequency'] ) ? sanitize_text_field( wp_unslash( $_POST['scan_frequency'] ) ) : 'manual',
			'enable_elementor_scan' => isset( $_POST['enable_elementor_scan'] ) ? (bool) $_POST['enable_elementor_scan'] : true,
			'excluded_folders'      => isset( $_POST['excluded_folders'] ) ? sanitize_text_field( wp_unslash( $_POST['excluded_folders'] ) ) : '',
			'excluded_file_types'   => isset( $_POST['excluded_file_types'] ) ? sanitize_text_field( wp_unslash( $_POST['excluded_file_types'] ) ) : '',
			'batch_size'            => isset( $_POST['batch_size'] ) ? absint( $_POST['batch_size'] ) : 50,
			'editor_can_view'       => isset( $_POST['editor_can_view'] ) ? (bool) $_POST['editor_can_view'] : false,
		);

		$allowed_frequencies = array( 'manual', 'daily', 'weekly' );
		if ( ! in_array( $settings['scan_frequency'], $allowed_frequencies, true ) ) {
			$settings['scan_frequency'] = 'manual';
		}

		if ( $settings['batch_size'] < 10 ) {
			$settings['batch_size'] = 10;
		} elseif ( $settings['batch_size'] > 200 ) {
			$settings['batch_size'] = 200;
		}

		foreach ( $settings as $key => $value ) {
			update_option( 'masic_' . $key, $value );
		}

		wp_send_json_success( array( 'message' => __( 'Settings saved successfully.', 'ma-smart-image-cleaner' ) ) );
	}
}

// Initialize admin menu
new MASIC_Admin_Menu();
