<?php
/**
 * Plugin Name: Lumiverse Security Watchdog Lite
 * Plugin URI:  https://lumiverse.gr/
 * Description: Modular background scanner: tracks new plugins, new admin users and scans changed .js files for malware signatures. Sends email alerts. Options to harden your installation.
 * Version:     1.1.3
 * Author:      Lumiverse Dynamic
 * License:     GPLv2+
 * Text Domain: lumiverse-security-watchdog-lite
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

final class LVSW_Security_Watchdog_Lite {

	/* ---------- New (prefixed) keys ---------- */
	const OPTION_KEY     = 'lvsw_options_v1';
	const HASH_OPTION    = 'lvsw_js_hashes_v1';
	const PLUGINS_OPTION = 'lvsw_plugins_list_v1';
	const ADMINS_OPTION  = 'lvsw_admins_list_v1';
	const LOG_OPTION     = 'lvsw_log_v1';
	const CRON_HOOK      = 'lvsw_cron_event';
	const CRON_SCHEDULE  = 'lvsw_daily';

	/* ---------- Old keys (for migration) ---------- */
	const OLD_OPTION_KEY     = 'wp_sw_options_v1';
	const OLD_HASH_OPTION    = 'wp_sw_js_hashes_v1';
	const OLD_PLUGINS_OPTION = 'wp_sw_plugins_list_v1';
	const OLD_ADMINS_OPTION  = 'wp_sw_admins_list_v1';
	const OLD_LOG_OPTION     = 'wp_sw_log_v1';
	const OLD_CRON_HOOK      = 'wp_sw_cron_event';
	const OLD_PRE_USER_OPT   = 'wp_sw_prev_pre_user_opts';

	/**
	 * Default options.
	 * NOTE: 'enabled' default is 0 (OFF).
	 */
	private static $defaults = array(
		'enabled'             => 0,
		'scan_interval_hours' => 24, // stored for compatibility; cadence is fixed to daily in Lite
		'notify_email'        => '',
		'scan_js'             => 1,
		'scan_plugins'        => 1,
		'scan_admins'         => 1,

		// Hardening toggles
		'block_xmlrpc'        => 0,

		// Lite: ONLY block file editing (not installs/updates)
		'block_file_mods'     => 0,

		// Lite: block comments sitewide
		'block_comments'      => 0,

		// internal (no UI editing in Lite)
		'signatures'          => null,
		'last_run'            => 0,
	);

	/* ------------------ Bootstrapping ------------------ */

	public static function init() {
		add_action( 'init', array( __CLASS__, 'ensure_initialized' ), 20 );

		add_action( 'admin_menu', array( __CLASS__, 'admin_menu' ) );
		add_action( 'admin_init', array( __CLASS__, 'admin_init' ) );

		add_filter( 'cron_schedules', array( __CLASS__, 'add_cron_intervals' ) );
		add_action( self::CRON_HOOK, array( __CLASS__, 'cron_handler' ) );

		register_activation_hook( __FILE__, array( __CLASS__, 'activate' ) );
		register_deactivation_hook( __FILE__, array( __CLASS__, 'deactivate' ) );
		register_uninstall_hook( __FILE__, array( __CLASS__, 'uninstall' ) );

		// Hardening
		add_action( 'init', array( __CLASS__, 'maybe_block_xmlrpc' ) );
		add_action( 'init', array( __CLASS__, 'maybe_block_file_mods' ) );
		add_action( 'init', array( __CLASS__, 'maybe_block_comments' ) );
	}

	/* ------------------ Migration ------------------ */

	private static function maybe_migrate_from_old_keys() {
		// If new exists, do nothing.
		$new = get_option( self::OPTION_KEY, null );
		if ( null !== $new ) {
			return;
		}

		$old = get_option( self::OLD_OPTION_KEY, null );
		if ( null === $old ) {
			return;
		}

		// Migrate options
		$old = wp_parse_args( (array) $old, self::$defaults );
		if ( empty( $old['signatures'] ) ) {
			$old['signatures'] = self::default_signatures();
		}
		update_option( self::OPTION_KEY, $old );

		// Migrate baselines/logs if present
		$val = get_option( self::OLD_HASH_OPTION, null );
		if ( null !== $val ) {
			update_option( self::HASH_OPTION, $val );
		}

		$val = get_option( self::OLD_PLUGINS_OPTION, null );
		if ( null !== $val ) {
			update_option( self::PLUGINS_OPTION, $val );
		}

		$val = get_option( self::OLD_ADMINS_OPTION, null );
		if ( null !== $val ) {
			update_option( self::ADMINS_OPTION, $val );
		}

		$val = get_option( self::OLD_LOG_OPTION, null );
		if ( null !== $val ) {
			update_option( self::LOG_OPTION, $val );
		}

		// Migrate pre_user tracking option name
		$val = get_option( self::OLD_PRE_USER_OPT, null );
		if ( null !== $val ) {
			update_option( self::pre_user_option_key(), $val );
		}

		// Clear old cron hook (if any) so it doesn't keep running in parallel
		wp_clear_scheduled_hook( self::OLD_CRON_HOOK );

		self::log_event( 'Migrated data from legacy keys' );
	}

	private static function pre_user_option_key() {
		return 'lvsw_prev_pre_user_opts_v1';
	}

	/* ------------------ Initialize ------------------ */

	public static function ensure_initialized() {
		self::maybe_migrate_from_old_keys();

		$opts = get_option( self::OPTION_KEY );

		if ( ! $opts ) {
			$opts = array_merge( self::$defaults, array( 'signatures' => self::default_signatures() ) );
			update_option( self::OPTION_KEY, $opts );
		} else {
			$opts = wp_parse_args( $opts, self::$defaults );

			// Ensure signatures exist (internal)
			if ( empty( $opts['signatures'] ) || ! is_array( $opts['signatures'] ) ) {
				$opts['signatures'] = self::default_signatures();
			}

			update_option( self::OPTION_KEY, $opts );
		}

		self::ensure_cron_scheduled();
	}

	public static function activate() {
		self::maybe_migrate_from_old_keys();

		$opts = wp_parse_args( get_option( self::OPTION_KEY, array() ), self::$defaults );
		if ( empty( $opts['signatures'] ) || ! is_array( $opts['signatures'] ) ) {
			$opts['signatures'] = self::default_signatures();
		}
		update_option( self::OPTION_KEY, $opts );

		self::initialize_baselines();

		self::reschedule_cron( true );
	}

	public static function deactivate() {
		wp_clear_scheduled_hook( self::CRON_HOOK );
		// also clear old hook just in case
		wp_clear_scheduled_hook( self::OLD_CRON_HOOK );
	}

	public static function uninstall() {
		delete_option( self::OPTION_KEY );
		delete_option( self::HASH_OPTION );
		delete_option( self::PLUGINS_OPTION );
		delete_option( self::ADMINS_OPTION );
		delete_option( self::LOG_OPTION );
		delete_option( self::pre_user_option_key() );

		// legacy cleanup (safe)
		delete_option( self::OLD_OPTION_KEY );
		delete_option( self::OLD_HASH_OPTION );
		delete_option( self::OLD_PLUGINS_OPTION );
		delete_option( self::OLD_ADMINS_OPTION );
		delete_option( self::OLD_LOG_OPTION );
		delete_option( self::OLD_PRE_USER_OPT );

		wp_clear_scheduled_hook( self::CRON_HOOK );
		wp_clear_scheduled_hook( self::OLD_CRON_HOOK );
	}

	/* ------------------ Hardening helpers ------------------ */

	public static function maybe_block_xmlrpc() {
		$opts = wp_parse_args( get_option( self::OPTION_KEY, array() ), self::$defaults );
		if ( empty( $opts['block_xmlrpc'] ) ) {
			return;
		}

		add_filter( 'xmlrpc_enabled', '__return_false' );
		add_action( 'xmlrpc_call', function () {
			wp_die(
				esc_html__( 'XML-RPC is disabled for security reasons.', 'lumiverse-security-watchdog-lite' ),
				esc_html__( 'XML-RPC Disabled', 'lumiverse-security-watchdog-lite' ),
				array( 'response' => 403 )
			);
		} );
	}

	/**
	 * Lite: block ONLY file editing (editor). Does NOT block plugin/theme installs/updates.
	 */
	public static function maybe_block_file_mods() {
		$opts = wp_parse_args( get_option( self::OPTION_KEY, array() ), self::$defaults );
		if ( empty( $opts['block_file_mods'] ) ) {
			return;
		}

		add_filter( 'user_has_cap', function ( $allcaps ) {
			$remove = array( 'edit_themes', 'edit_plugins', 'edit_files' );
			foreach ( $remove as $cap ) {
				$allcaps[ $cap ] = false;
			}
			return $allcaps;
		}, PHP_INT_MAX );

		add_filter( 'pre_option_disallow_file_edit', '__return_true' );
		// IMPORTANT: do NOT set disallow_file_mods (installs/updates) in Lite
	}

	/**
	 * Block comments sitewide (front-end + admin UI).
	 */
	public static function maybe_block_comments() {
		$opts = wp_parse_args( get_option( self::OPTION_KEY, array() ), self::$defaults );
		if ( empty( $opts['block_comments'] ) ) {
			return;
		}

		add_filter( 'comments_open', '__return_false', 20, 2 );
		add_filter( 'pings_open', '__return_false', 20, 2 );
		add_filter( 'comments_array', '__return_empty_array', 10, 2 );

		add_action( 'admin_menu', function () {
			remove_menu_page( 'edit-comments.php' );
		}, 999 );

		add_action( 'wp_before_admin_bar_render', function () {
			global $wp_admin_bar;
			if ( is_object( $wp_admin_bar ) ) {
				$wp_admin_bar->remove_node( 'comments' );
			}
		}, 999 );

		add_action( 'admin_init', function () {
			global $pagenow;
			if ( 'edit-comments.php' === $pagenow ) {
				wp_safe_redirect( admin_url() );
				exit;
			}
		} );

		add_action( 'init', function () {
			$post_types = get_post_types( array( 'public' => true ), 'names' );
			foreach ( $post_types as $post_type ) {
				if ( post_type_supports( $post_type, 'comments' ) ) {
					remove_post_type_support( $post_type, 'comments' );
					remove_post_type_support( $post_type, 'trackbacks' );
				}
			}
		}, 100 );
	}

	/* ------------------ Cron ------------------ */

	public static function add_cron_intervals( $schedules ) {
		$schedules[ self::CRON_SCHEDULE ] = array(
			'interval' => 24 * 60 * 60,
			'display'  => esc_html__( 'Every 24 hours', 'lumiverse-security-watchdog-lite' ),
		);

		// Keep old interval key defined to avoid edge cases if another plugin scheduled with it.
		if ( ! isset( $schedules['six_hours'] ) ) {
			$schedules['six_hours'] = array(
				'interval' => 24 * 60 * 60,
				'display'  => esc_html__( 'Every 24 hours', 'lumiverse-security-watchdog-lite' ),
			);
		} else {
			$schedules['six_hours']['interval'] = 24 * 60 * 60;
			$schedules['six_hours']['display']  = esc_html__( 'Every 24 hours', 'lumiverse-security-watchdog-lite' );
		}

		return $schedules;
	}

	private static function ensure_cron_scheduled() {
		$event = function_exists( 'wp_get_scheduled_event' ) ? wp_get_scheduled_event( self::CRON_HOOK ) : null;

		if ( ! $event ) {
			self::reschedule_cron( true );
			return;
		}

		if ( ! empty( $event->schedule ) && $event->schedule !== self::CRON_SCHEDULE ) {
			self::reschedule_cron( true );
		}
	}

	private static function reschedule_cron( $force = false ) {
		if ( $force ) {
			wp_clear_scheduled_hook( self::CRON_HOOK );
		}

		if ( ! wp_next_scheduled( self::CRON_HOOK ) ) {
			wp_schedule_event( time() + 60, self::CRON_SCHEDULE, self::CRON_HOOK );
			self::log_event( 'Cron scheduled (every 24 hours)' );
		}
	}

	public static function cron_handler() {
		try {
			self::do_scan();
		} catch ( Throwable $e ) {
			self::log_event( 'cron_handler exception: ' . $e->getMessage() );
		}
	}

	/* ------------------ Admin UI ------------------ */

	public static function admin_menu() {
		add_menu_page(
			esc_html__( 'Security Watchdog', 'lumiverse-security-watchdog-lite' ),
			esc_html__( 'Security Watchdog', 'lumiverse-security-watchdog-lite' ),
			'manage_options',
			'lumiverse-security-watchdog-lite',
			array( __CLASS__, 'settings_page' ),
			'dashicons-shield-alt',
			80
		);
	}

	public static function admin_init() {
		register_setting(
			'lvsw_settings',
			self::OPTION_KEY,
			array(
				'type'              => 'array',
				'sanitize_callback' => array( __CLASS__, 'validate_options' ),
			)
		);

		add_settings_section(
			'lvsw_main',
			esc_html__( 'Settings', 'lumiverse-security-watchdog-lite' ),
			'__return_false',
			'lumiverse-security-watchdog-lite'
		);

		add_settings_field( 'enabled', esc_html__( 'Enable scanning and alerts', 'lumiverse-security-watchdog-lite' ), array( __CLASS__, 'field_enabled' ), 'lumiverse-security-watchdog-lite', 'lvsw_main' );
		add_settings_field( 'notify_email', esc_html__( 'Notification Email', 'lumiverse-security-watchdog-lite' ), array( __CLASS__, 'field_notify_email' ), 'lumiverse-security-watchdog-lite', 'lvsw_main' );
		add_settings_field( 'scan_js', esc_html__( 'Scan JS files', 'lumiverse-security-watchdog-lite' ), array( __CLASS__, 'field_scan_js' ), 'lumiverse-security-watchdog-lite', 'lvsw_main' );
		add_settings_field( 'scan_plugins', esc_html__( 'Detect new plugins', 'lumiverse-security-watchdog-lite' ), array( __CLASS__, 'field_scan_plugins' ), 'lumiverse-security-watchdog-lite', 'lvsw_main' );
		add_settings_field( 'scan_admins', esc_html__( 'Detect new / suspicious admins', 'lumiverse-security-watchdog-lite' ), array( __CLASS__, 'field_scan_admins' ), 'lumiverse-security-watchdog-lite', 'lvsw_main' );

		add_settings_field( 'block_xmlrpc', esc_html__( 'Block XML-RPC', 'lumiverse-security-watchdog-lite' ), array( __CLASS__, 'field_block_xmlrpc' ), 'lumiverse-security-watchdog-lite', 'lvsw_main' );
		add_settings_field( 'block_file_mods', esc_html__( 'Block File Editing', 'lumiverse-security-watchdog-lite' ), array( __CLASS__, 'field_block_file_mods' ), 'lumiverse-security-watchdog-lite', 'lvsw_main' );
		add_settings_field( 'block_comments', esc_html__( 'Block Comments (sitewide)', 'lumiverse-security-watchdog-lite' ), array( __CLASS__, 'field_block_comments' ), 'lumiverse-security-watchdog-lite', 'lvsw_main' );
	}

	public static function field_enabled() {
		$opts = wp_parse_args( get_option( self::OPTION_KEY, array() ), self::$defaults );
		echo '<label><input type="checkbox" name="' . esc_attr( self::OPTION_KEY ) . '[enabled]" value="1" ' . checked( 1, $opts['enabled'] ?? 0, false ) . ' /> ' . esc_html__( 'Enable automatic scanning & email alerts', 'lumiverse-security-watchdog-lite' ) . '</label>';
		echo '<p class="description">' . esc_html__( 'When enabled the plugin will run scheduled checks and send alerts to the configured email. Runs every 24 hours.', 'lumiverse-security-watchdog-lite' ) . '</p>';
	}

	public static function field_notify_email() {
		$opts = wp_parse_args( get_option( self::OPTION_KEY, array() ), self::$defaults );
		echo '<input type="email" name="' . esc_attr( self::OPTION_KEY ) . '[notify_email]" value="' . esc_attr( $opts['notify_email'] ?? '' ) . '" class="regular-text" />';
	}

	public static function field_scan_js() {
		$opts = wp_parse_args( get_option( self::OPTION_KEY, array() ), self::$defaults );
		echo '<input type="checkbox" name="' . esc_attr( self::OPTION_KEY ) . '[scan_js]" value="1" ' . checked( 1, $opts['scan_js'] ?? 0, false ) . ' />';
	}

	public static function field_scan_plugins() {
		$opts = wp_parse_args( get_option( self::OPTION_KEY, array() ), self::$defaults );
		echo '<input type="checkbox" name="' . esc_attr( self::OPTION_KEY ) . '[scan_plugins]" value="1" ' . checked( 1, $opts['scan_plugins'] ?? 0, false ) . ' />';
	}

	public static function field_scan_admins() {
		$opts = wp_parse_args( get_option( self::OPTION_KEY, array() ), self::$defaults );
		echo '<input type="checkbox" name="' . esc_attr( self::OPTION_KEY ) . '[scan_admins]" value="1" ' . checked( 1, $opts['scan_admins'] ?? 0, false ) . ' />';
	}

	public static function field_block_xmlrpc() {
		$opts = wp_parse_args( get_option( self::OPTION_KEY, array() ), self::$defaults );
		echo '<input type="checkbox" name="' . esc_attr( self::OPTION_KEY ) . '[block_xmlrpc]" value="1" ' . checked( 1, $opts['block_xmlrpc'] ?? 0, false ) . ' />';
		echo '<p class="description">' . esc_html__( 'Blocks remote XML-RPC access (recommended for sites that do not use pingbacks / remote publishing).', 'lumiverse-security-watchdog-lite' ) . '</p>';
	}

	public static function field_block_file_mods() {
		$opts = wp_parse_args( get_option( self::OPTION_KEY, array() ), self::$defaults );
		echo '<input type="checkbox" name="' . esc_attr( self::OPTION_KEY ) . '[block_file_mods]" value="1" ' . checked( 1, $opts['block_file_mods'] ?? 0, false ) . ' />';
		echo '<p class="description">' . esc_html__( 'Disables plugin/theme editing in wp-admin (does not block installs/updates).', 'lumiverse-security-watchdog-lite' ) . '</p>';
	}

	public static function field_block_comments() {
		$opts = wp_parse_args( get_option( self::OPTION_KEY, array() ), self::$defaults );
		echo '<input type="checkbox" name="' . esc_attr( self::OPTION_KEY ) . '[block_comments]" value="1" ' . checked( 1, $opts['block_comments'] ?? 0, false ) . ' />';
		echo '<p class="description">' . esc_html__( 'Disables comments sitewide (front-end + admin UI).', 'lumiverse-security-watchdog-lite' ) . '</p>';
	}

	public static function settings_page() {
		if ( ! current_user_can( 'manage_options' ) ) {
			wp_die( esc_html__( 'Insufficient permissions', 'lumiverse-security-watchdog-lite' ) );
		}
		?>
		<div class="wrap">
			<h1><?php esc_html_e( 'Lumiverse Security Watchdog Lite', 'lumiverse-security-watchdog-lite' ); ?></h1>

			<div style="margin: 16px 0 24px;">
				<img
					src="<?php echo esc_url( plugins_url( 'wp-security-watchdog-lite.png', __FILE__ ) ); ?>"
					alt="<?php esc_attr_e( 'Lumiverse Security Watchdog Lite', 'lumiverse-security-watchdog-lite' ); ?>"
					style="max-width:800px; height:auto; border-radius:12px; box-shadow:0 2px 8px rgba(0,0,0,0.08);"
				/>
			</div>

			<div style="background:#fff; padding:15px; border:1px solid #000; margin-bottom:10px;">
				<h2 style="margin:0 0 6px;">
					<?php esc_html_e( 'Coming Soon: Security Watchdog PRO', 'lumiverse-security-watchdog-lite' ); ?>
				</h2>
				<p style="margin:0;">
					<?php esc_html_e( 'The PRO version is coming soon with additional security features and deeper monitoring.', 'lumiverse-security-watchdog-lite' ); ?>
				</p>
			</div>

			<div style="background:#fff; padding:15px; border:1px solid #ddd; margin-bottom:10px;">
				<h2 style="margin:0 0 6px;">
					<?php esc_html_e( 'Need help cleaning an infected site?', 'lumiverse-security-watchdog-lite' ); ?>
				</h2>
				<p style="margin:0 0 10px;">
					<?php esc_html_e( 'If your site is infected, we can help with malware cleanup and hardening. This is a paid service (cleanup is not included for free).', 'lumiverse-security-watchdog-lite' ); ?>
				</p>
				<p style="margin:0;">
					<?php
					// translators: %s: the support website / contact reference shown to the user.
					printf(
						esc_html__( 'Plugin users may be eligible for a better rate. Contact us at %s.', 'lumiverse-security-watchdog-lite' ),
						'<strong>lumiverse.gr</strong>'
					);
					?>
				</p>
			</div>

			<div style="background:#fff; padding:15px; border:1px solid #ddd;">
				<form method="post" action="options.php">
					<?php
					settings_fields( 'lvsw_settings' );
					do_settings_sections( 'lumiverse-security-watchdog-lite' );
					submit_button();
					?>
				</form>
			</div>

			<h2><?php esc_html_e( 'Recent Logs', 'lumiverse-security-watchdog-lite' ); ?></h2>
			<div style="background:#fff; padding:15px; border:1px solid #ddd;">
				<?php
				$logs = get_option( self::LOG_OPTION, array() );
				if ( empty( $logs ) ) {
					echo '<p>' . esc_html__( 'No logs yet.', 'lumiverse-security-watchdog-lite' ) . '</p>';
				} else {
					echo '<ul>';
					$cnt = 0;
					foreach ( array_reverse( $logs ) as $entry ) {
						$cnt++;
						if ( $cnt > 30 ) {
							break;
						}
						$time    = isset( $entry['time'] ) ? $entry['time'] : '';
						$summary = isset( $entry['summary'] ) ? $entry['summary'] : '';
						echo '<li><strong>' . esc_html( $time ) . '</strong>: ' . esc_html( $summary ) . '</li>';
					}
					echo '</ul>';
				}
				?>
			</div>
		</div>
		<?php
	}

	/* ------------------ Validation ------------------ */

	public static function validate_options( $input ) {
		$input = (array) $input;

		$existing = wp_parse_args( get_option( self::OPTION_KEY, array() ), self::$defaults );

		$clean = array();
		$clean['enabled']         = ! empty( $input['enabled'] ) ? 1 : 0;
		$clean['scan_js']         = ! empty( $input['scan_js'] ) ? 1 : 0;
		$clean['scan_plugins']    = ! empty( $input['scan_plugins'] ) ? 1 : 0;
		$clean['scan_admins']     = ! empty( $input['scan_admins'] ) ? 1 : 0;

		$clean['block_xmlrpc']    = ! empty( $input['block_xmlrpc'] ) ? 1 : 0;
		$clean['block_file_mods'] = ! empty( $input['block_file_mods'] ) ? 1 : 0;
		$clean['block_comments']  = ! empty( $input['block_comments'] ) ? 1 : 0;

		$clean['notify_email']        = isset( $input['notify_email'] ) ? sanitize_email( $input['notify_email'] ) : '';
		$clean['scan_interval_hours'] = 24;

		// IMPORTANT (WP.org review): do NOT accept user-provided signatures in Lite.
		$clean['signatures'] = ! empty( $existing['signatures'] ) && is_array( $existing['signatures'] )
			? $existing['signatures']
			: self::default_signatures();

		$clean['last_run'] = ! empty( $existing['last_run'] ) ? (int) $existing['last_run'] : 0;

		self::reschedule_cron( true );

		return wp_parse_args( $clean, self::$defaults );
	}

	/* ------------------ Baseline init ------------------ */

	public static function initialize_baselines() {
		update_option( self::PLUGINS_OPTION, self::list_plugin_directories() );
		update_option( self::ADMINS_OPTION, self::list_admin_users_full() );
		update_option( self::HASH_OPTION, self::hash_all_js_files() );
		update_option( self::LOG_OPTION, array() );
		self::log_event( 'Baselines initialized' );
	}

	/* ------------------ Main scan routine ------------------ */

	public static function do_scan() {
		$opts = wp_parse_args( get_option( self::OPTION_KEY, array() ), self::$defaults );

		if ( empty( $opts['enabled'] ) ) {
			self::log_event( 'Scan skipped: plugin disabled in settings' );
			return;
		}

		if ( empty( $opts['signatures'] ) || ! is_array( $opts['signatures'] ) ) {
			$opts['signatures'] = self::default_signatures();
		}

		$now     = time();
		$details = array();

		/* 1) JS file changes */
		if ( ! empty( $opts['scan_js'] ) ) {
			$prev_hashes    = get_option( self::HASH_OPTION, array() );
			$current_hashes = self::hash_all_js_files();

			$changed = array();
			foreach ( $current_hashes as $path => $hash ) {
				if ( self::should_skip_js_file( $path ) ) {
					continue;
				}
				if ( ! isset( $prev_hashes[ $path ] ) || $prev_hashes[ $path ] !== $hash ) {
					$changed[] = $path;
				}
			}

			$suspected = array();
			foreach ( $changed as $f ) {
				if ( self::should_skip_js_file( $f ) ) {
					continue;
				}
				$content = @file_get_contents( $f );
				if ( false === $content ) {
					continue;
				}
				$reasons = self::analyse_content_for_signatures( $content, $opts['signatures'] );
				if ( ! empty( $reasons ) ) {
					$suspected[ $f ] = $reasons;
				}
			}

			update_option( self::HASH_OPTION, $current_hashes );

			if ( ! empty( $suspected ) ) {
				$details['js_suspected'] = $suspected;
				self::log_event( count( $suspected ) . ' suspicious JS files detected' );
			} else {
				self::log_event( count( $changed ) . ' JS files changed, none suspicious' );
			}
		}

		/* 2) New / deleted plugins */
		if ( ! empty( $opts['scan_plugins'] ) ) {
			$prev_plugins = get_option( self::PLUGINS_OPTION, array() );
			$now_plugins  = self::list_plugin_directories();

			$new_plugins     = array_values( array_diff( $now_plugins, $prev_plugins ) );
			$deleted_plugins = array_values( array_diff( $prev_plugins, $now_plugins ) );

			if ( ! empty( $new_plugins ) ) {
				$details['new_plugins'] = $new_plugins;
			}
			if ( ! empty( $deleted_plugins ) ) {
				$details['deleted_plugins'] = $deleted_plugins;
			}

			update_option( self::PLUGINS_OPTION, $now_plugins );
			self::log_event( count( $new_plugins ) . ' new plugins, ' . count( $deleted_plugins ) . ' deleted plugins' );
		}

		/* 3) Scan plugins for suspicious names & malware patterns */
		if ( ! empty( $opts['scan_plugins'] ) ) {
			$suspicious_names = array(
				'httpsd2-cache-engine',
				'wp-compat',
				'wp-default',
				'wp-console',
			);

			$plugin_malware_results = array();
			$now_plugins            = self::list_plugin_directories();

			foreach ( $now_plugins as $plugin_dir ) {
				$plugin_path = trailingslashit( WP_PLUGIN_DIR ) . $plugin_dir;

				if ( in_array( $plugin_dir, $suspicious_names, true ) ) {
					$plugin_malware_results[ $plugin_dir ]['suspicious_name'] = true;
				}

				$php_files = self::list_php_files_safe( $plugin_path );

				foreach ( $php_files as $file ) {
					if ( stripos( $file, plugin_basename( __FILE__ ) ) !== false || basename( $file ) === basename( __FILE__ ) ) {
						continue;
					}

					$content = @file_get_contents( $file );
					if ( false === $content ) {
						continue;
					}

					$hits     = array();
					$patterns = array(
						'/wp_insert_user\s*\(\s*\$params/i',
						'/class\s+WP_GHOST/i',
						'/killRedirect/i',
						'/shellaccess/i',
						'/makeadminuser/i',
						'/emergency_login/i',
						'/brazilc\.com\/ads/i',
						'/admin-ajax\.php\?action=ajjs_run/i',
						'/__AJJS_LOADED__/i',
					);

					foreach ( $patterns as $pattern ) {
						if ( @preg_match( $pattern, $content ) ) {
							$hits[] = $pattern;
						}
					}

					if ( ! empty( $hits ) ) {
						$plugin_malware_results[ $plugin_dir ]['files'][ $file ] = $hits;
					}
				}
			}

			if ( ! empty( $plugin_malware_results ) ) {
				$details['plugin_malware'] = $plugin_malware_results;
				self::log_event( 'Plugin malware scan: ' . count( $plugin_malware_results ) . ' suspicious plugin(s) found' );
			} else {
				self::log_event( 'Plugin malware scan: no suspicious plugins found' );
			}
		}

		/* 4) New + suspicious admins */
		if ( ! empty( $opts['scan_admins'] ) ) {
			$prev_admins    = get_option( self::ADMINS_OPTION, array() );
			$current_admins = self::list_admin_users_full();

			$new_admins = array();
			foreach ( $current_admins as $id => $data ) {
				if ( ! isset( $prev_admins[ $id ] ) ) {
					$new_admins[ $id ] = $data;
				}
			}

			if ( ! empty( $new_admins ) ) {
				$details['new_admins'] = $new_admins;
				self::log_event( count( $new_admins ) . ' new admin(s) detected' );
			} else {
				self::log_event( 'No new admin users' );
			}

			$patterns          = array( '/^wphadm/i', '/^adminbackup/i' );
			$suspicious_admins = array();

			foreach ( $current_admins as $id => $data ) {
				$username = $data['user_login'] ?? '';
				$email    = $data['user_email'] ?? '';

				$is_suspicious = false;
				foreach ( $patterns as $p ) {
					if ( preg_match( $p, $username ) ) {
						$is_suspicious = true;
						break;
					}
				}
				if ( empty( $email ) || ! is_email( $email ) ) {
					$is_suspicious = true;
				}

				if ( $is_suspicious ) {
					$suspicious_admins[ $id ] = $data;
				}
			}

			if ( ! empty( $suspicious_admins ) ) {
				$details['suspicious_admins'] = $suspicious_admins;
				self::log_event( count( $suspicious_admins ) . ' suspicious existing admin(s) detected' );
			}

			update_option( self::ADMINS_OPTION, $current_admins );
		}

		/* 5) Suspicious _pre_user_id options (optional alert) */
		if ( ! empty( $opts['scan_admins'] ) ) {
			$prev_pre_opts = get_option( self::pre_user_option_key(), array() );
			$now_pre_opts  = self::list_pre_user_options();

			$prev_map = array();
			foreach ( $prev_pre_opts as $r ) {
				if ( isset( $r['option_name'] ) ) {
					$prev_map[ $r['option_name'] ] = $r;
				}
			}

			$new_pre_opts = array();
			foreach ( $now_pre_opts as $row ) {
				$name = $row['option_name'];
				if ( ! isset( $prev_map[ $name ] ) ) {
					$new_pre_opts[ $name ] = $row;
				}
			}

			if ( ! empty( $new_pre_opts ) ) {
				$details['new_pre_user_id_entries'] = $new_pre_opts;
				self::log_event( count( $new_pre_opts ) . ' new _pre_user_id option(s) detected' );
			} else {
				self::log_event( 'No new _pre_user_id options detected' );
			}

			update_option( self::pre_user_option_key(), $now_pre_opts );
		}

		if ( ! empty( $opts['notify_email'] ) ) {
			self::send_email_report( $opts['notify_email'], $details );
		}

		$opts['last_run'] = $now;
		update_option( self::OPTION_KEY, $opts );
	}

	/* ------------------ Filesystem helpers ------------------ */

	public static function list_plugin_directories() {
		$all     = array();
		$basedir = untrailingslashit( WP_PLUGIN_DIR );
		if ( ! is_dir( $basedir ) ) {
			return $all;
		}

		$dirs = @glob( $basedir . '/*', GLOB_ONLYDIR );
		if ( $dirs && is_array( $dirs ) ) {
			foreach ( $dirs as $d ) {
				$all[] = basename( $d );
			}
			sort( $all );
			return $all;
		}

		try {
			$it = new DirectoryIterator( $basedir );
			foreach ( $it as $file ) {
				if ( $file->isDir() && ! $file->isDot() ) {
					$all[] = $file->getFilename();
				}
			}
			sort( $all );
			return $all;
		} catch ( Throwable $e ) {
			return $all;
		}
	}

	private static function list_php_files_safe( $dir ) {
		$files = array();
		$dir   = rtrim( $dir, '/\\' );

		if ( ! is_dir( $dir ) ) {
			return $files;
		}

		try {
			$iterator = new RecursiveIteratorIterator(
				new RecursiveDirectoryIterator( $dir, FilesystemIterator::SKIP_DOTS )
			);
			foreach ( $iterator as $file ) {
				if ( $file->isFile() && strtolower( $file->getExtension() ) === 'php' ) {
					$files[] = $file->getPathname();
				}
			}
			return $files;
		} catch ( Throwable $e ) {
			// fallback below
		}

		$stack = array( $dir );
		while ( $stack ) {
			$current = array_pop( $stack );
			$entries = @glob( $current . '/*' );
			if ( ! $entries ) {
				continue;
			}
			foreach ( $entries as $e ) {
				if ( is_dir( $e ) ) {
					$stack[] = $e;
				} elseif ( strtolower( pathinfo( $e, PATHINFO_EXTENSION ) ) === 'php' ) {
					$files[] = $e;
				}
			}
		}

		return $files;
	}

	public static function hash_all_js_files() {
		$hashes = array();
		$dirs   = array( WP_CONTENT_DIR );

		foreach ( $dirs as $dir ) {
			if ( ! is_dir( $dir ) ) {
				continue;
			}

			try {
				$iterator = new RecursiveIteratorIterator(
					new RecursiveDirectoryIterator( $dir, FilesystemIterator::SKIP_DOTS )
				);
				foreach ( $iterator as $file ) {
					if ( $file->isFile() && strtolower( $file->getExtension() ) === 'js' ) {
						$path = $file->getPathname();
						if ( self::should_skip_js_file( $path ) ) {
							continue;
						}
						$content = @file_get_contents( $path );
						if ( false === $content ) {
							continue;
						}
						$hashes[ $path ] = md5( $content );
					}
				}
				continue;
			} catch ( Throwable $e ) {
				// fallback below
			}

			$stack = array( $dir );
			while ( $stack ) {
				$current = array_pop( $stack );
				$entries = @glob( $current . '/*' );
				if ( ! $entries ) {
					continue;
				}

				foreach ( $entries as $e ) {
					if ( is_dir( $e ) ) {
						$stack[] = $e;
					} else {
						if ( strtolower( pathinfo( $e, PATHINFO_EXTENSION ) ) === 'js' ) {
							if ( self::should_skip_js_file( $e ) ) {
								continue;
							}
							$content = @file_get_contents( $e );
							if ( false === $content ) {
								continue;
							}
							$hashes[ $e ] = md5( $content );
						}
					}
				}
			}
		}

		ksort( $hashes );
		return $hashes;
	}

	/**
	 * Build skip directories without hardcoding wp-content paths.
	 *
	 * @return string[] Absolute directories to skip (normalized).
	 */
	private static function get_skip_js_dirs() {
		static $dirs = null;

		if ( null !== $dirs ) {
			return $dirs;
		}

		$dirs = array();

		// Uploads base dir (may differ per setup / wp-content renamed).
		$upload = wp_upload_dir();
		$base   = ! empty( $upload['basedir'] ) ? $upload['basedir'] : '';

		if ( $base ) {
			$dirs[] = trailingslashit( $base ) . 'wflogs';
			$dirs[] = trailingslashit( $base ) . 'fusion-scripts';
			$dirs[] = trailingslashit( $base ) . 'elementor';
			$dirs[] = trailingslashit( $base ) . 'simply-static';
		}

		// WordPress content dir variations.
		$dirs[] = trailingslashit( WP_CONTENT_DIR ) . 'wflogs';
		$dirs[] = trailingslashit( WP_CONTENT_DIR ) . 'cache';

		// Plugin dir variations.
		$dirs[] = trailingslashit( WP_PLUGIN_DIR ) . 'wordfence';

		// Normalize.
		$dirs = array_map(
			function ( $d ) {
				return strtolower( untrailingslashit( wp_normalize_path( $d ) ) );
			},
			$dirs
		);

		return $dirs;
	}

	public static function should_skip_js_file( $path ) {
		$path = strtolower( wp_normalize_path( (string) $path ) );

		// Skip common heavy/vendor dirs based on real paths (no hardcoded /wp-content/...).
		$skip_dirs = self::get_skip_js_dirs();
		foreach ( $skip_dirs as $dir ) {
			$dir = trailingslashit( $dir );
			if ( 0 === strpos( $path, $dir ) ) {
				return true;
			}
		}

		// Generic skip: any node_modules anywhere.
		if ( false !== strpos( $path, '/node_modules/' ) ) {
			return true;
		}

		// Keep previous behavior: skip jquery files.
		if ( false !== strpos( $path, 'jquery' ) ) {
			return true;
		}

		return false;
	}

	/* ------------------ Signatures & Analysis ------------------ */

	public static function analyse_content_for_signatures( $content, $signatures ) {
		$hits = array();
		if ( empty( $content ) || empty( $signatures ) || ! is_array( $signatures ) ) {
			return $hits;
		}

		foreach ( $signatures as $sig ) {
			if ( is_array( $sig ) && isset( $sig['pattern'] ) ) {
				$pattern = $sig['pattern'];
				$reason  = isset( $sig['reason'] ) ? $sig['reason'] : $pattern;
			} elseif ( is_string( $sig ) ) {
				$pattern = $sig;
				$reason  = $sig;
			} else {
				continue;
			}

			try {
				if ( @preg_match( $pattern, $content ) ) {
					$hits[] = $reason;
				}
			} catch ( Throwable $e ) {
				continue;
			}
		}

		return $hits;
	}

	/* ------------------ Email / Reporting ------------------ */

	public static function send_email_report( $email, $details ) {
		$subject = '[Lumiverse Security Watchdog] Report for ' . get_bloginfo( 'name' );

		$lines   = array();
		$lines[] = 'Security Watchdog scan report for: ' . home_url();
		$lines[] = 'Generated: ' . gmdate( 'Y-m-d H:i:s' );
		$lines[] = '';

		if ( empty( $details ) ) {
			return;
		}

		if ( ! empty( $details['new_plugins'] ) ) {
			$lines[] = 'New plugins detected:';
			foreach ( $details['new_plugins'] as $p ) {
				$lines[] = ' - ' . $p;
			}
			$lines[] = '';
		}

		if ( ! empty( $details['deleted_plugins'] ) ) {
			$lines[] = 'Deleted plugins detected:';
			foreach ( $details['deleted_plugins'] as $p ) {
				$lines[] = ' - ' . $p;
			}
			$lines[] = '';
		}

		if ( ! empty( $details['plugin_malware'] ) ) {
			$lines[] = 'Suspicious / infected plugins:';
			foreach ( $details['plugin_malware'] as $plugin => $info ) {
				$lines[] = ' - ' . $plugin;

				if ( ! empty( $info['suspicious_name'] ) ) {
					$lines[] = '    * Suspicious plugin folder/name detected';
				}

				if ( ! empty( $info['files'] ) ) {
					$lines[] = '    * Files with indicators:';
					foreach ( $info['files'] as $file => $matches ) {
						$lines[] = '        - ' . $file;
						foreach ( $matches as $m ) {
							$lines[] = '            match: ' . $m;
						}
					}
				}

				$lines[] = '';
			}
		}

		if ( ! empty( $details['js_suspected'] ) ) {
			$lines[] = 'Suspicious JS files:';
			foreach ( $details['js_suspected'] as $f => $reasons ) {
				$lines[] = ' - ' . $f;
				foreach ( $reasons as $r ) {
					$lines[] = '    * ' . $r;
				}
			}
			$lines[] = '';
		}

		if ( ! empty( $details['new_admins'] ) ) {
			$lines[] = 'New administrator accounts:';
			foreach ( $details['new_admins'] as $id => $d ) {
				$lines[] = sprintf(
					' - ID:%s user:%s display:%s email:%s created:%s',
					$id,
					$d['user_login'],
					$d['display_name'] ?? '',
					$d['user_email'] ?? '',
					gmdate( 'c', $d['registered'] ?? 0 )
				);
			}
			$lines[] = '';
		}

		if ( ! empty( $details['suspicious_admins'] ) ) {
			$lines[] = 'Suspicious existing administrator accounts (please check wp_users):';
			foreach ( $details['suspicious_admins'] as $id => $d ) {
				$lines[] = sprintf(
					' - ID:%s user:%s display:%s email:%s',
					$id,
					$d['user_login'],
					$d['display_name'] ?? '',
					$d['user_email'] ?? ''
				);
			}
			$lines[] = '';
		}

		if ( ! empty( $details['new_pre_user_id_entries'] ) ) {
			$lines[] = '_pre_user_id* options found (possible hidden admin indicators):';
			foreach ( $details['new_pre_user_id_entries'] as $name => $row ) {
				$lines[] = ' - ' . $name;
			}
			$lines[] = '';
		}

		$lines[] = 'Regards,';
		$lines[] = 'Lumiverse Security Watchdog';

		wp_mail( $email, $subject, implode( "\n", $lines ) );
		self::log_event( 'Report email sent to ' . $email );
	}

	/* ------------------ Utilities ------------------ */

	public static function log_event( $summary ) {
		$logs   = get_option( self::LOG_OPTION, array() );
		$logs[] = array(
			'time'    => gmdate( 'Y-m-d H:i:s' ),
			'summary' => (string) $summary,
		);

		if ( count( $logs ) > 200 ) {
			$logs = array_slice( $logs, -200 );
		}

		update_option( self::LOG_OPTION, $logs );
	}

	public static function default_signatures() {
		return array(
			'/aHR0cHM6Ly9nZXRmaXgud2lu/i',
			'/getfix\.win/i',
			'/cptchdm\.icu/i',
			'/__INJECT_MARKER__/i',
			'/document\.location\s*=\s*["\']https?:\/\/[a-z0-9\-]+\.(icu|win)/i',

			'/\(function\(\)\{try\{var _0x[0-9a-f]+\s*=/i',
			'/_0x[a-f0-9]{4,}\s*=\s*function/i',
			'/String\.fromCharCode\(\d{3,}(,\d{3,})+\)/i',

			'/gzinflate\s*\(\s*base64_decode\s*\(/i',

			'/(eval|system|exec|shell_exec|passthru)\s*\(\s*\$_(POST|GET|REQUEST)/i',
			'/assert\s*\(\s*\$_(POST|GET|REQUEST)/i',

			'/Plugin Name:\s*HTTP2 Basic Cache Engine/i',
			'/Author:\s*Gregg Palmer/i',
			'/unset\(\$plugins\[\$current_plugin_file\]\)/i',
			'/ajjs_run/i',
			'/\\\\x68\\\\x74\\\\x74\\\\x70\\\\x73/i',
			'/getfix\.win\/jstest/i',
			'/eval\(jsCode\)/i',

			'/Plugin Name:\s*WP Default/i',
			'/Plugin Name:\s*Cache Optimization Engine/i',
			'/Plugin Name:\s*Slick Popup/i',
		);
	}

	public static function list_admin_users_full() {
		$users = get_users(
			array(
				'role'    => 'administrator',
				'orderby' => 'ID',
			)
		);

		$out = array();
		foreach ( $users as $u ) {
			$out[ $u->ID ] = array(
				'user_login'   => $u->user_login,
				'display_name' => $u->display_name,
				'user_email'   => $u->user_email,
				'registered'   => isset( $u->user_registered ) ? strtotime( $u->user_registered ) : 0,
			);
		}

		return $out;
	}

	private static function list_pre_user_options() {
		global $wpdb;

		$cache_key   = 'lvsw_pre_user_opts_v1';
		$cache_group = 'lvsw';
		$cached      = wp_cache_get( $cache_key, $cache_group );

		if ( false !== $cached && is_array( $cached ) ) {
			return $cached;
		}

		$results = $wpdb->get_results(
			$wpdb->prepare(
				"SELECT option_name, option_value FROM {$wpdb->options} WHERE option_name LIKE %s",
				'_pre_user_id%'
			),
			ARRAY_A
		);

		$results = is_array( $results ) ? $results : array();
		wp_cache_set( $cache_key, $results, $cache_group, 300 );

		return $results;
	}
}

/* Bootstrap plugin */
LVSW_Security_Watchdog_Lite::init();