<?php
/**
 * Plugin Name: Lueira Embed
 * Plugin URI: https://lueira.com
 * Description: Provides a shortcode that outputs a button opening a modal with an embedded activity iframe.
 * Version: 1.0.2
 * Author: Maladeta Studio
 * Author URI: https://maladeta.studio
 * License: GPL-2.0-or-later
 * License URI: https://www.gnu.org/licenses/gpl-2.0.html
 * Text Domain: lueira
 * Domain Path: /languages
 * Requires at least: 5.0
 * Tested up to: 6.8
 * Requires PHP: 7.4
 */

if (!defined('ABSPATH')) {
    exit;
}

final class Lueira_Embed_Modal {
    private const OPTION_HOST = 'lueira_embed_host';
    private const OPTION_DEFAULT_HOST = 'https://store.lueira.com';
    private const PLUGIN_VERSION = '1.0.2';
    private const ASSET_HANDLE = 'lueira-embed';

    private static ?Lueira_Embed_Modal $instance = null;

    private string $host;

    public static function instance(): Lueira_Embed_Modal {
        if (self::$instance === null) {
            self::$instance = new self();
        }

        return self::$instance;
    }

    private function __construct() {
        $stored_host = get_option(self::OPTION_HOST);
        if (!$stored_host) {
            $stored_host = self::OPTION_DEFAULT_HOST;
        }

        $this->host = $this->sanitize_host($stored_host);

        add_action('init', [$this, 'register_shortcode']);
        add_action('wp_enqueue_scripts', [$this, 'register_assets']);
        add_action('admin_menu', [$this, 'register_settings_page']);
        add_action('admin_init', [$this, 'register_setting']);
    }

    public static function activate(): void {
        if (!get_option(self::OPTION_HOST)) {
            add_option(self::OPTION_HOST, self::OPTION_DEFAULT_HOST);
        }
    }

    public function register_shortcode(): void {
        add_shortcode('lueira', [$this, 'render_shortcode']);
    }

    public function register_assets(): void {
        wp_register_style(
            self::ASSET_HANDLE,
            plugins_url('assets/css/modal.css', __FILE__),
            [],
            self::PLUGIN_VERSION
        );

        wp_register_script(
            self::ASSET_HANDLE,
            plugins_url('assets/js/modal.js', __FILE__),
            [],
            self::PLUGIN_VERSION,
            true
        );
    }

    public function register_settings_page(): void {
        add_options_page(
            __('Lueira Embed Settings', 'lueira'),
            __('Lueira Embed', 'lueira'),
            'manage_options',
            'lueira',
            [$this, 'render_settings_page']
        );
    }

    public function register_setting(): void {
        register_setting(
            'lueira_embed_settings',
            self::OPTION_HOST,
            [
                'type' => 'string',
                'sanitize_callback' => [$this, 'sanitize_host'],
                'default' => self::OPTION_DEFAULT_HOST,
            ]
        );
    }

    public function render_settings_page(): void {
        if (!current_user_can('manage_options')) {
            return;
        }

        ?>
        <div class="wrap">
            <h1><?php esc_html_e('Lueira Embed Settings', 'lueira'); ?></h1>
            <form action="options.php" method="post">
                <?php
                settings_fields('lueira_embed_settings');
                ?><table class="form-table" role="presentation">
                    <tbody>
                    <tr>
                        <th scope="row">
                            <label for="<?php echo esc_attr(self::OPTION_HOST); ?>">
                                <?php esc_html_e('Embed Host URL', 'lueira'); ?>
                            </label>
                        </th>
                        <td>
                            <input
                                name="<?php echo esc_attr(self::OPTION_HOST); ?>"
                                id="<?php echo esc_attr(self::OPTION_HOST); ?>"
                                type="url"
                                class="regular-text"
                                value="<?php echo esc_attr(get_option(self::OPTION_HOST, self::OPTION_DEFAULT_HOST)); ?>"
                                placeholder="https://store.lueira.com"
                                required
                            />
                            <p class="description">
                                <?php esc_html_e('Enter the base URL for the activities marketplace (include https://).', 'lueira'); ?>
                            </p>
                        </td>
                    </tr>
                    </tbody>
                </table>
                <?php submit_button(); ?>
            </form>
        </div>
        <?php
    }

    public function render_shortcode(array $atts): string {
        $atts = shortcode_atts(
            [
                'id' => '',
                'module' => '',
                'product_type' => '',
                'lang' => 'es',
                'button_text' => __('View activity', 'lueira'),
                'class' => '',
            ],
            $atts,
            'lueira'
        );

        $activity_id = trim((string) $atts['id']);
        $module = trim((string) $atts['module']);
        $product_type = trim((string) $atts['product_type']);
        $lang = trim((string) $atts['lang']);

        // Validate required parameters
        if ($activity_id === '' || $module === '') {
            return '';
        }

        $modal_id = 'lueira-embed-' . wp_generate_uuid4();
        $iframe_src = $this->build_iframe_url($activity_id, $module, $product_type, $lang);

        wp_enqueue_style(self::ASSET_HANDLE);
        wp_enqueue_script(self::ASSET_HANDLE);

        ob_start();
        ?>
        <div class="lueira-embed-button-wrapper">
            <button
                type="button"
                class="lueira-embed-trigger <?php echo esc_attr($atts['class']); ?>"
                data-lueira-embed-target="<?php echo esc_attr($modal_id); ?>"
            >
                <?php echo esc_html($atts['button_text']); ?>
            </button>
        </div>
        <div
            id="<?php echo esc_attr($modal_id); ?>"
            class="lueira-embed"
            data-lueira-embed
            aria-hidden="true"
            hidden
        >
            <div class="lueira-embed__backdrop" data-lueira-embed-dismiss></div>
            <div class="lueira-embed__dialog" role="dialog" aria-modal="true">
                <button
                    type="button"
                    class="lueira-embed__close"
                    aria-label="<?php esc_attr_e('Close modal', 'lueira'); ?>"
                    data-lueira-embed-dismiss
                >
                    &times;
                </button>
                <iframe
                    src="<?php echo esc_url($iframe_src); ?>"
                    data-lueira-embed-src="<?php echo esc_url($iframe_src); ?>"
                    title="<?php echo esc_attr(sprintf(
                        /* translators: %s: activity ID being embedded. */
                        __('Activity %s', 'lueira'),
                        $activity_id
                    )); ?>"
                    loading="lazy"
                    sandbox="allow-same-origin allow-scripts allow-popups allow-forms allow-top-navigation-by-user-activation"
                    allowfullscreen
                ></iframe>
            </div>
        </div>
        <?php
        return (string) ob_get_clean();
    }

    private function build_iframe_url(string $activity_id, string $module, string $product_type, string $lang): string {
        $parts = [
            rtrim($this->host, '/'),
            rawurlencode($lang),
            'embed',
            rawurlencode($module),
        ];

        if ($product_type !== '') {
            $parts[] = rawurlencode($product_type);
        }

        $parts[] = rawurlencode($activity_id);

        return implode('/', $parts);
    }

    public function sanitize_host(string $host): string {
        $host = trim($host);
        if ($host === '') {
            return self::OPTION_DEFAULT_HOST;
        }

        $sanitized = esc_url_raw($host);
        if (!$sanitized) {
            return self::OPTION_DEFAULT_HOST;
        }

        return $sanitized;
    }
}

Lueira_Embed_Modal::instance();

register_activation_hook(__FILE__, ['Lueira_Embed_Modal', 'activate']);
