<?php
/*
Plugin Name: LoyCart-POS
Description: A responsive POS system inside WordPress admin that displays WooCommerce products and handles the cart.
Version: 1.0.20
Author: David Herbert
License: GPLv2 or later
License URI: https://www.gnu.org/licenses/gpl-2.0.html
Text Domain: loycart-pos
Domain Path: /languages
*/



if (!defined('ABSPATH')) {
    exit;
}

// --- Auto-redirect to POS after login ---
function loycart_pos_redirect_admin_to_pos() {
    // Only run if option is enabled
    $enabled = get_option('loycart_pos_auto_redirect_pos', 'no');
    if ($enabled !== 'yes') {
        return;
    }
    // Only redirect admins with WooCommerce access
    if (current_user_can('manage_woocommerce') && current_user_can('administrator')) {
        // Only redirect if login flag is set
        if (get_transient('loycart_pos_redirect_after_login')) {
            delete_transient('loycart_pos_redirect_after_login');
            // Prevent redirect loop: if already on POS page, do not redirect
            // phpcs:ignore WordPress.Security.NonceVerification.Recommended -- Safe read-only check, not processing form data or actions
            if (is_admin() && isset($_GET['page']) && $_GET['page'] === 'loycart-pos') {
                return;
            }
            $pos_page_url = admin_url('admin.php?page=loycart-pos');
            wp_safe_redirect($pos_page_url);
            exit;
        }
    }
}

// Set redirect flag on login
function loycart_pos_set_redirect_flag($user_login, $user) {
    $enabled = get_option('loycart_pos_auto_redirect_pos', 'no');
    if ($enabled === 'yes' && user_can($user, 'manage_woocommerce') && user_can($user, 'administrator')) {
        set_transient('loycart_pos_redirect_after_login', 1, 60);
    }
}
add_action('wp_login', 'loycart_pos_set_redirect_flag', 10, 2);
add_action('admin_init', 'loycart_pos_redirect_admin_to_pos');


register_activation_hook(__FILE__, function() {
    add_rewrite_endpoint('store-credit', EP_ROOT | EP_PAGES);
    flush_rewrite_rules();
});
register_deactivation_hook(__FILE__, function() {
    flush_rewrite_rules();
});


function loycart_check_woocommerce_setup() {
    if ( ! is_plugin_active( 'woocommerce/woocommerce.php' ) || ! function_exists( 'WC' ) ) {
        add_action( 'admin_notices', 'loycart_show_wc_inactive_notice' );
        return;
    }

    $product_count = wp_count_posts( 'product' );
    $published_products = isset( $product_count->publish ) ? $product_count->publish : 0;
    if ( $published_products === 0 ) {
        add_action( 'admin_notices', 'loycart_show_no_products_notice' );
    }
}

add_action( 'admin_init', 'loycart_check_woocommerce_setup' );
require_once( plugin_dir_path( __FILE__ ) . 'loycart-pos-ajax-handlers.php' );
require_once( plugin_dir_path( __FILE__ ) . 'loycart-pos-data-functions.php' );
require_once( plugin_dir_path( __FILE__ ) . 'loycart-pos-store-credit.php' );

function loycart_show_wc_inactive_notice() {
    ?>
    <div class="notice notice-error is-dismissible">
        <p>
            <strong><?php esc_html_e( 'LoyCart POS Requires WooCommerce!', 'loycart-pos' ); ?></strong><br>
            <?php esc_html_e( 'Please install, activate and fully configure WooCommerce for LoyCart POS to function. Once  WooCommerce is activated, LoyCart POS will appear in the WooCommerce menu', 'loycart-pos' ); ?>
        </p>
    </div>
    <?php
}


function loycart_show_no_products_notice() {
    ?>
    <div class="notice notice-warning is-dismissible">
         <p>
            <strong><?php esc_html_e( 'Get Started with LoyCart POS', 'loycart-pos' ); ?></strong><br>
            <?php esc_html_e( 'WooCommerce is active, but you haven\'t added any products yet. LoyCart POS needs products to function.', 'loycart-pos' ); ?>
            <a href="<?php echo esc_url( admin_url( 'post-new.php?post_type=product' ) ); ?>" style="margin-left: 10px;"><?php esc_html_e( 'Add Your First Product', 'loycart-pos' ); ?></a>
        </p>
    </div>
    <?php
}

function loycart_pos_clear_product_transients() {
    global $wpdb;

    // phpcs:disable WordPress.DB.DirectDatabaseQuery
    $wpdb->query(
        $wpdb->prepare(
            "DELETE FROM {$wpdb->options} WHERE option_name LIKE %s OR option_name LIKE %s",
            $wpdb->esc_like('_transient_loycart_products_') . '%',
            $wpdb->esc_like('_transient_timeout_loycart_products_') . '%'
        )
    );
    // phpcs:enable WordPress.DB.DirectDatabaseQuery
}



function loycart_pos_clear_product_transients_on_save($post_id, $post, $update) {
    if (wp_is_post_revision($post_id)) {
        return;
    }
    if (defined('DOING_AUTOSAVE') && DOING_AUTOSAVE) {
        return;
    }
    if (isset($post->post_type) && 'product' === $post->post_type && !current_user_can('edit_post', $post_id)) {
        return;
    }
    if (!isset($post->post_type) || 'product' !== $post->post_type) {
        return;
    }
    loycart_pos_clear_product_transients();
}


add_action('woocommerce_reduce_order_stock', 'loycart_pos_clear_product_transients', 10, 1);
add_action('woocommerce_restore_order_stock', 'loycart_pos_clear_product_transients', 10, 1);
add_action('woocommerce_product_set_stock', 'loycart_pos_clear_product_transients', 10, 1);
add_action('woocommerce_variation_set_stock', 'loycart_pos_clear_product_transients', 10, 1);
add_action('save_post_product', 'loycart_pos_clear_product_transients_on_save', 10, 3);
add_action('admin_menu', 'loycart_pos_register_admin_page');
add_action('admin_bar_menu', 'loycart_pos_add_dark_mode_toggle', 999);





function loycart_pos_register_admin_page()
{
    add_submenu_page(
        'woocommerce',
        'LoyCart POS',
        'LoyCart POS',
        'manage_woocommerce',
        'loycart-pos',
        'loycart_pos_render_app',
        5
    );
}

function loycart_pos_add_dark_mode_toggle( $admin_bar ) {
    // Only show on LoyCart POS page in admin area
    if ( ! is_admin() ) {
        return;
    }
    
    $screen = get_current_screen();
    if ( ! $screen || $screen->id !== 'woocommerce_page_loycart-pos' ) {
        return;
    }

    $admin_bar->add_menu( array(
        'id'    => 'loycart-dark-mode',
        'title' => '🌙 ' . esc_html__( 'Dark Mode', 'loycart-pos' ),
        'href'  => '#',
        'meta'  => array(
            'onclick' => 'window.toggleDarkMode(); return false;',
            'class'   => 'loycart-dark-mode-toggle',
        )
    ) );

    // Add a centered notification container to the admin bar
    $admin_bar->add_menu( array(
        'id'    => 'loycart-pos-notifications',
        'title' => '<div id="loycart-pos-adminbar-notifications"></div>',
        'parent'=> 'top-secondary',
        'meta'  => array(
            'class' => 'loycart-pos-adminbar-notifications',
            'title' => '',
        ),
        'href'  => false,
    ) );
}


function loycart_pos_enqueue_admin_scripts( $hook_suffix ) {
        // Add custom style for admin bar notification container to prevent stray content
        add_action('admin_head', function() {
            echo '<style>#loycart-pos-adminbar-notifications { min-height: 32px; display: flex; align-items: center; justify-content: center; } .loycart-pos-adminbar-notifications { min-width: 300px; }</style>';
        });
    $items_per_page_default = 24;
    if ( 'woocommerce_page_loycart-pos' !== $hook_suffix ) {
        return;
    }

    // If auto-redirect is enabled, inject JS to collapse the admin menu by default
    $enabled = get_option('loycart_pos_auto_redirect_pos', 'no');
    if ($enabled === 'yes') {
        add_action('admin_footer', function() {
            ?>
            <script>
            (function() {
                try {
                    // Only run if menu is not already collapsed
                    if (!document.body.classList.contains('folded')) {
                        // Set localStorage so menu stays collapsed for this session
                        localStorage.setItem('wp_auto_fold', '1');
                        // Collapse the menu
                        document.body.classList.add('folded');
                        // Also trigger the menu collapse event for WP
                        var evt = document.createEvent('HTMLEvents');
                        evt.initEvent('adminmenu', true, false);
                        document.dispatchEvent(evt);
                    }
                } catch(e) {}
            })();
            </script>
            <?php
        });
    }
    $version = '1.0.20';
    
    wp_enqueue_style( 'loycart-pos-css', plugin_dir_url( __FILE__ ) . 'assets/css/loycart-pos-style.css', [], $version );
    wp_enqueue_style( 'dashicons' );
    wp_enqueue_script( 'decimal-js', plugin_dir_url( __FILE__ ) . 'assets/js/decimal.min.js', [ 'jquery' ], '0.12.1', true );
    wp_enqueue_script( 'quagga-js', plugin_dir_url( __FILE__ ) . 'assets/js/quagga.min.js', [ 'jquery' ], '0.12.1', true );
    wp_enqueue_script( 'jsbarcode', plugin_dir_url( __FILE__ ) . 'assets/js/JsBarcode.all.min.js', [], '3.11.5', true );
    wp_enqueue_script( 
        'loycart-pos-utils-js', 
        plugin_dir_url( __FILE__ ) . 'assets/js/loycart-pos-utils.js', 
        [ 'jquery', 'decimal-js' ],
        $version, 
        true 
    );
    wp_enqueue_script( 'loycart-pos-api-js', plugin_dir_url( __FILE__ ) . 'assets/js/loycart-pos-api.js', [ 'jquery' ], $version, true );
    
    // Initialize loycart namespace before modules
    wp_enqueue_script( 'loycart-pos-init-js', plugin_dir_url( __FILE__ ) . 'assets/js/loycart-pos-init.js', [ 'jquery' ], $version, true );
    
    // Enqueue module scripts in dependency order
    wp_enqueue_script( 'loycart-pos-modals-js', plugin_dir_url( __FILE__ ) . 'assets/js/loycart-pos-modals.js', [ 'jquery', 'loycart-pos-utils-js', 'loycart-pos-api-js', 'loycart-pos-init-js' ], $version, true );
    wp_enqueue_script( 'loycart-pos-labelManager-js', plugin_dir_url( __FILE__ ) . 'assets/js/loycart-pos-labelManager.js', [ 'jquery', 'loycart-pos-utils-js', 'loycart-pos-api-js', 'loycart-pos-init-js' ], $version, true );
    wp_enqueue_script( 'loycart-pos-scanner-js', plugin_dir_url( __FILE__ ) . 'assets/js/loycart-pos-scanner.js', [ 'jquery', 'loycart-pos-utils-js', 'loycart-pos-api-js', 'loycart-pos-init-js' ], $version, true );
    wp_enqueue_script( 'loycart-pos-refundManager-js', plugin_dir_url( __FILE__ ) . 'assets/js/loycart-pos-refundManager.js', [ 'jquery', 'loycart-pos-utils-js', 'loycart-pos-api-js', 'loycart-pos-init-js' ], $version, true );
    wp_enqueue_script( 'loycart-pos-customerManager-js', plugin_dir_url( __FILE__ ) . 'assets/js/loycart-pos-customerManager.js', [ 'jquery', 'loycart-pos-utils-js', 'loycart-pos-api-js', 'loycart-pos-init-js' ], $version, true );
    wp_enqueue_script( 'loycart-pos-shippingManager-js', plugin_dir_url( __FILE__ ) . 'assets/js/loycart-pos-shippingManager.js', [ 'jquery', 'loycart-pos-utils-js', 'loycart-pos-api-js', 'loycart-pos-init-js' ], $version, true );
    wp_enqueue_script( 'loycart-pos-cartManager-js', plugin_dir_url( __FILE__ ) . 'assets/js/loycart-pos-cartManager.js', [ 'jquery', 'loycart-pos-utils-js', 'loycart-pos-api-js', 'loycart-pos-init-js' ], $version, true );
    wp_enqueue_script( 'loycart-pos-paymentManager-js', plugin_dir_url( __FILE__ ) . 'assets/js/loycart-pos-paymentManager.js', [ 'jquery', 'loycart-pos-utils-js', 'loycart-pos-api-js', 'loycart-pos-init-js' ], $version, true );
    wp_enqueue_script( 'loycart-pos-productManager-js', plugin_dir_url( __FILE__ ) . 'assets/js/loycart-pos-productManager.js', [ 'jquery', 'loycart-pos-utils-js', 'loycart-pos-api-js', 'loycart-pos-init-js' ], $version, true );
    wp_enqueue_script( 'loycart-pos-heldSalesManager-js', plugin_dir_url( __FILE__ ) . 'assets/js/loycart-pos-heldSalesManager.js', [ 'jquery', 'loycart-pos-utils-js', 'loycart-pos-api-js', 'loycart-pos-init-js' ], $version, true );
    wp_enqueue_script( 'loycart-pos-couponManager-js', plugin_dir_url( __FILE__ ) . 'assets/js/loycart-pos-couponManager.js', [ 'jquery', 'loycart-pos-utils-js', 'loycart-pos-api-js', 'loycart-pos-init-js' ], $version, true );
    wp_enqueue_script( 
        'loycart-pos-js', 
        plugin_dir_url( __FILE__ ) . 'assets/js/loycart-pos-app.js', 
        [ 
            'jquery', 
            'jquery-migrate', 
            'loycart-pos-utils-js', 
            'loycart-pos-api-js',
            'loycart-pos-init-js',
            'loycart-pos-modals-js',
            'loycart-pos-labelManager-js',
            'loycart-pos-scanner-js',
            'loycart-pos-refundManager-js',
            'loycart-pos-customerManager-js',
            'loycart-pos-shippingManager-js',
            'loycart-pos-cartManager-js',
            'loycart-pos-paymentManager-js',
            'loycart-pos-productManager-js',
            'loycart-pos-heldSalesManager-js',
            'loycart-pos-couponManager-js',
            'quagga-js',
            'decimal-js'
        ], 
        $version, 
        true 
    );
    
    wp_enqueue_media();

    $localized_strings = [
        'disabledNote'   => esc_html__( 'Limit 1', 'loycart-pos' ),

        // showNotification related
        'errorFormattingPrice'   => esc_html__( 'Error formatting price:', 'loycart-pos' ),
        // translators: %s is the error message from the print attempt.
        'errorTriggeringPrint'   => esc_html__( 'Error triggering print: %s', 'loycart-pos' ),
        'failedLoadReceiptPrint' => esc_html__( 'Failed to load receipt for printing after timeout.', 'loycart-pos' ),
        // translators: %s is the specific error message for receipt generation.
        'errorGeneratingReceipt' => esc_html__( 'Error generating receipt: %s', 'loycart-pos' ),
        // loadProducts related
        'loadingProducts'        => esc_html__( 'Loading products...', 'loycart-pos' ),
        'noProductsFoundCriteria' => esc_html__( 'No products found matching your criteria.', 'loycart-pos' ),
        'noImage'               => esc_html__( 'No Image', 'loycart-pos' ),
        // translators: %s is the specific error message for product loading failure.
        'failedLoadProducts'     => esc_html__( 'Failed to load products: %s', 'loycart-pos' ),
        'unknownError'           => esc_html__( 'Unknown error', 'loycart-pos' ),
        'errorLoadProducts'      => esc_html__( 'Error loading products.', 'loycart-pos' ),
        'errorLoadProductsConnection' => esc_html__( 'Error loading products. Please check connection.', 'loycart-pos' ),
        // updateProductStock related
        'stockPrefix'            => esc_html__( 'Stock: ', 'loycart-pos' ),
        'outOfStock'             => esc_html__( 'Out of Stock', 'loycart-pos' ),
        // translators: %1$s is the product name, %2$s is the product ID.
        'stockInfoUnavailable'   => esc_html__( 'Stock for %1$s (ID: %2$s) might have changed, but quantity not tracked client-side.', 'loycart-pos' ),
        // updateCart related
        'cartIsEmpty'            => esc_html__( 'Cart is empty', 'loycart-pos' ),
        // addToCart related
        // translators: %s is the product name added to the cart.
        // translators: %s is the product name added to the cart.
        'productAddedToCart'     => __( '%s added to cart.', 'loycart-pos' ),
        // processSale related
        'cartEmptyCannotProcess' => __( 'Cart is empty. Please add items.', 'loycart-pos' ),
        'processingPreviousRequest' => __( 'Processing previous request. Please wait.', 'loycart-pos' ),
        // submitSale related
        'failedCompleteSale'     => __( 'Failed to complete sale.', 'loycart-pos' ),
        'errorCompleteSale'      => __( 'Error completing sale.', 'loycart-pos' ),
        // showProductOptionsModal related
        'selectOptions'          => esc_html__( 'Select options for price and stock availability', 'loycart-pos' ),
        // translators: %s is the attribute name (e.g., Color, Size).
        'selectAttribute'        => esc_html__( 'Select %s', 'loycart-pos' ),
        'noVariationsDefined'    => esc_html__( 'No variations defined for this product.', 'loycart-pos' ),
        // updateVariationDisplay related
        'inStock'                => esc_html__( 'In Stock', 'loycart-pos' ),
        'onBackorder'            => esc_html__( 'On Backorder', 'loycart-pos' ),
        'variationNotAvailable'  => esc_html__( 'Variation not available', 'loycart-pos' ),
        // handleBarcodeScan related
        'posIsBusy'              => __( 'POS is busy.', 'loycart-pos' ),
        // translators: %s is the product name that is out of stock.
        'productOutOfStock'      => __( '%s is out of stock.', 'loycart-pos' ),
        // translators: %s is the barcode that was not found.
        'productNotFoundBarcode' => __( 'Product with code [%s] not found.', 'loycart-pos' ),
        'errorSearchingBarcode'  => __( 'Error searching product by barcode.', 'loycart-pos' ),
        // productsList click handlers
        'productDataMissing'     => __( 'Error: Product data is missing or invalid.', 'loycart-pos' ),
        'variableOptionsMissing' => __( 'Variable product options seem to be missing. Trying to reload...', 'loycart-pos' ),
        // customerSearch
        'noCustomersFound'       => esc_html__( 'No customers found.', 'loycart-pos' ),
        // translators: %s is the selected customer's name.
        'customerSelected'       => __( 'Customer %s selected.', 'loycart-pos' ),
        'itemRemoved'            => __( 'Item removed.', 'loycart-pos' ),
        // translators: %s is the maximum discount percentage.
        'maxDiscountReached'     => __( 'Max discount is %s%%.', 'loycart-pos' ),
        // translators: %s is the maximum discount percentage.
        'invalidDiscountRange'   => __( 'Invalid discount. Must be 0-%s%%.', 'loycart-pos' ),
        'discountApplied'        => __( 'Discount applied.', 'loycart-pos' ),
        'discountRemoved'        => __( 'Discount removed.', 'loycart-pos' ),
        'amountTenderedLow'      => __( 'Amount tendered is less than total.', 'loycart-pos' ),
        'cartEmptyCannotHold'    => __( 'Cart is empty. Cannot hold.', 'loycart-pos' ),
        'promptHoldSaleName'     => __( 'Enter a name for this held sale (optional):', 'loycart-pos' ),
        // translators: %s is the customer name.
        'defaultHeldOrderFor'    => __( 'Order for %s', 'loycart-pos' ),
        // translators: %s is the time the sale was held.
        'defaultHeldAtTime'      => __( 'Held At %s', 'loycart-pos' ),
        // translators: %s is the name or identifier of the held sale.
        'saleHeldSuccess'        => __( 'Sale held: %s', 'loycart-pos' ),
        'failedHoldSale'         => __( 'Failed to hold sale.', 'loycart-pos' ),
        'errorHoldingSale'       => __( 'Error holding sale.', 'loycart-pos' ),
        'retrieveSalePosBusy'    => __( 'POS is busy.', 'loycart-pos' ),
        'loadingHeldSales'       => esc_html__( 'Loading held sales...', 'loycart-pos' ),
        'noSalesOnHold'          => esc_html__( 'No sales currently on hold.', 'loycart-pos' ),
        'failedLoadHeldSales'    => esc_html__( 'Failed to load held sales.', 'loycart-pos' ),
        'errorRetrievingHeldSales' => __( 'Error retrieving held sales.', 'loycart-pos' ),
        'completeOrHoldCurrentSale' => __( 'Please complete or hold current sale first.', 'loycart-pos' ),
        'invalidHeldCartData'    => __( 'Error: Held cart data was invalid.', 'loycart-pos' ),
        'saleResumedSuccess'     => __( 'Sale resumed successfully!', 'loycart-pos' ),
        'failedResumeSale'       => __( 'Failed to resume sale.', 'loycart-pos' ),
        'errorResumingSaleConnection' => __( 'Error resuming sale. Please check connection.', 'loycart-pos' ),
        'confirmDeleteHeldSale'  => __( 'Are you sure you want to delete this held sale? This cannot be undone.', 'loycart-pos' ),
        'heldSaleDeletedSuccess' => __( 'Held sale deleted.', 'loycart-pos' ),
        'failedDeleteHeldSale'   => __( 'Failed to delete held sale.', 'loycart-pos' ),
        'errorDeletingHeldSale'  => __( 'Error deleting held sale.', 'loycart-pos' ),
        'quantityMinOne'         => __( 'Quantity must be at least 1.', 'loycart-pos' ),
        // translators: %s is the quantity available in stock.
        'notEnoughStock'         => __( 'Not enough stock. Only %s available.', 'loycart-pos' ),
        'selectedOptionOutOfStock' => __( 'Selected option is out of stock.', 'loycart-pos' ),
        'selectAllOptions'       => __( 'Please select all product options.', 'loycart-pos' ),
        // translators: %s is the current page number.
        'pageText'               => __( 'Page: %s', 'loycart-pos' ),
        'checkoutDefaultText'    => esc_html__( 'Checkout', 'loycart-pos' ),
        'searchForCustomer'      => esc_html__( 'Start typing to find a customer.', 'loycart-pos' ),
        'searching'              => esc_html__( 'Searching...', 'loycart-pos' ),
        'select'                 => esc_html__( 'Select', 'loycart-pos' ),
        'printShippingLabel'    => esc_html__( 'Print Shipping Label', 'loycart-pos' ),
        'printCollectionLabel'  => esc_html__( 'Print Collection Label', 'loycart-pos' ),
        'collectionAddressTitle'=> esc_html__( 'Collection Point', 'loycart-pos' ),
        // Refund/Return mode additions
        'refundModeTitle'        => esc_html__( 'Refund / Return Mode', 'loycart-pos' ),
        'refundModeSubtitle'     => esc_html__( 'Select items from the customer\'s previous order to refund', 'loycart-pos' ),
        'refundSelectOrderPlaceholder' => esc_html__( 'Please select an order to see refundable items.', 'loycart-pos' ),
        'refundLoadingOrders'    => esc_html__( 'Loading orders...', 'loycart-pos' ),
        'refundStatusFully'      => esc_html__( 'Fully Refunded', 'loycart-pos' ),
        'refundStatusPartial'    => esc_html__( 'Partial Refund', 'loycart-pos' ),
        'refundStatusEligible'   => esc_html__( 'Eligible for Refund', 'loycart-pos' ),
        'refundModeActivated'    => esc_html__( 'Return Mode Activated.', 'loycart-pos' ),
        'refundModeCanceled'     => esc_html__( 'Return Mode Canceled', 'loycart-pos' ),
        // translators: %s is the order ID number.
        'verifyingOrder'         => esc_html__( 'Verifying Order #%s...', 'loycart-pos' ),
        // translators: %s is the order ID number.
        'orderLoaded'            => esc_html__( 'Order #%s loaded.', 'loycart-pos' ),
        'serverConnectionError'  => esc_html__( 'Server connection error. Please check your connection and try again.', 'loycart-pos' ),
        'noItemsFound'           => esc_html__( 'No items found.', 'loycart-pos' ),
        'refundableShipping'     => esc_html__( 'Refundable Shipping', 'loycart-pos' ),
        'shippingFullyRefunded'  => esc_html__( 'Shipping Fully Refunded', 'loycart-pos' ),
        'itemRefunded'           => esc_html__( 'Item Refunded', 'loycart-pos' ),
        // translators: %s is the refundable quantity remaining for the item.
        'refundableQty'          => esc_html__( 'Refundable: %s', 'loycart-pos' ),
        'printingRefundReceipt'  => esc_html__( 'Refund receipt printing...', 'loycart-pos' ),
        // Mobile printing and notifications
        'tapHereToPrint'         => esc_html__( 'Tap Here to Print', 'loycart-pos' ),
        'iosSharePrintHint'      => esc_html__( 'If the button does nothing, use the iOS Share icon to Print.', 'loycart-pos' ),
        'printContentOpenedShare'=> esc_html__( 'Print content opened. Tap the Share icon to Print/Save.', 'loycart-pos' ),
        'popupBlockedEnable'     => esc_html__( 'Pop-up blocked. Please enable pop-ups in your app settings.', 'loycart-pos' ),
        'errorTryingToPrint'     => esc_html__( 'Error trying to print.', 'loycart-pos' ),

        // Receipt labels
        'receiptCashierDiscountsLabel' => esc_html__( 'Cashier Discounts:', 'loycart-pos' ),
        // translators: %s is the coupon code
        'receiptCouponLabel'     => esc_html__( 'Coupon (%s):', 'loycart-pos' ),
        'receiptShippingLabel'   => esc_html__( 'Shipping:', 'loycart-pos' ),
        'receiptTotalRefundLabel'=> esc_html__( 'Total Refund:', 'loycart-pos' ),
        'receiptShippingToLabel' => esc_html__( 'SHIPPING TO:', 'loycart-pos' ),
        'receiptCopyLabelRefund' => esc_html__( 'Refund Copy', 'loycart-pos' ),
        'receiptCopyLabelCustomer' => esc_html__( 'Customer Copy', 'loycart-pos' ),

        // Discount label formats
        // translators: %1$s is a percent number, %2$s is a formatted currency amount
        'cashierDiscountLabelFormat' => esc_html__( 'Cashier Discount %1$s%% Saving %2$s', 'loycart-pos' ),
        // translators: %1$s is a percent number, %2$s is a formatted currency amount
        'saleDiscountLabelFormat'    => esc_html__( 'Sale Discount %1$s%% Saving %2$s', 'loycart-pos' ),
        // translators: %1$s is a percent number, %2$s is a formatted currency amount
        'cartDiscountLabelFormat'    => esc_html__( 'Cart Discount %1$s%% Saving %2$s', 'loycart-pos' ),

        // Label and shipping titles
        'printLabelTitle'        => esc_html__( 'Print Label', 'loycart-pos' ),
        // translators: %s is the order number
        'shippingLabelTitleWithOrder' => esc_html__( 'Shipping Label - Order %s', 'loycart-pos' ),
        'shippingDateLabel'      => esc_html__( 'SHIPPING DATE:', 'loycart-pos' ),

        // Scanner
        'scannerQuaggaNotLoaded' => esc_html__( 'Error: QuaggaJS library not loaded.', 'loycart-pos' ),
        // translators: %s is the scanned barcode
        'scannedPrefix'          => esc_html__( 'Scanned: %s', 'loycart-pos' ),

        // Modals fallbacks
        'noDescriptionAvailable' => esc_html__( 'No description available.', 'loycart-pos' ),

        // Coupon flow strings
        'couponRemoved'          => esc_html__( 'Coupon removed.', 'loycart-pos' ),
        'couponSelectionCleared' => esc_html__( 'Coupon selection cleared.', 'loycart-pos' ),
        'fetchingAvailableCoupons' => esc_html__( 'Fetching available coupons...', 'loycart-pos' ),
        'noActiveCouponsFound'   => esc_html__( 'No active coupons found.', 'loycart-pos' ),
        'failedLoadCoupons'      => esc_html__( 'Failed to load coupons.', 'loycart-pos' ),
        // translators: %s is the coupon code
        'couponSelected'         => esc_html__( 'Coupon %s selected.', 'loycart-pos' ),
        'uiErrorCouponModalMissing' => esc_html__( 'UI Error: Coupon Modal missing.', 'loycart-pos' ),
        'applyCouponAriaAvailable' => esc_html__( 'Apply Coupon (Store credit available)', 'loycart-pos' ),
        'applyCouponAria'        => esc_html__( 'Apply Coupon', 'loycart-pos' ),

        // Product alt and aria labels
        'productAltFallback'     => esc_html__( 'Product', 'loycart-pos' ),
        'printProductLabelAria'  => esc_html__( 'Print product label', 'loycart-pos' ),

        // Keyboard shortcuts and UI
        'shortcutRefundModeNotification' => esc_html__( 'Refund Mode activated (Ctrl+R)', 'loycart-pos' ),
        'shortcutHoldSaleNotification'   => esc_html__( 'Hold Sale (Ctrl+H)', 'loycart-pos' ),
        'shortcutApplyCouponNotification'=> esc_html__( 'Apply Coupon (Ctrl+D)', 'loycart-pos' ),
        'compactLabel'                   => esc_html__( 'Compact', 'loycart-pos' ),
        'defaultLabel'                   => esc_html__( 'Default', 'loycart-pos' ),
        'keyboardShortcutsTitle'         => esc_html__( 'LoyCart POS - Keyboard Shortcuts', 'loycart-pos' ),
        'shortcutCompleteSale'           => esc_html__( 'Complete Sale / Checkout', 'loycart-pos' ),
        'shortcutEnterRefundMode'        => esc_html__( 'Enter Refund Mode', 'loycart-pos' ),
        'shortcutHoldSale'               => esc_html__( 'Hold Sale', 'loycart-pos' ),
        'shortcutApplyCouponDiscount'    => esc_html__( 'Apply Coupon/Discount', 'loycart-pos' ),
        'shortcutToggleCompactLayout'    => esc_html__( 'Toggle Compact Layout', 'loycart-pos' ),
        'shortcutShowHelp'               => esc_html__( 'Show this help', 'loycart-pos' ),
        'shortcutHintWorksEverywhere'    => esc_html__( 'Shortcuts work from anywhere except while typing in input fields.', 'loycart-pos' ),
    ];

    $all_gateways = WC()->payment_gateways->payment_gateways();
    $bacs_enabled = isset( $all_gateways['bacs'] ) && 'yes' === $all_gateways['bacs']->enabled;
    $bacs_details_html = '';
    if ($bacs_enabled) {
        $bacs_gateway = $all_gateways['bacs'];
        $account_details = $bacs_gateway->account_details;
        if ( ! empty( $account_details ) ) {
            $bacs_details_html .= '<h4>' . esc_html__( 'Our Bank Details', 'loycart-pos' ) . '</h4>';
            foreach ( $account_details as $i => $detail ) {
                if ( $i > 0 ) {
                    $bacs_details_html .= '<hr style="border:0; border-top: 1px dashed #ccc; margin: 10px 0;">';
                }
                $bacs_details_html .= '<ul style="list-style-type: none; padding-left: 0; margin-bottom: 0;">';
                $fields_to_display = [
                    'account_name'   => __( 'Account Name', 'loycart-pos' ),
                    'bank_name'      => __( 'Bank', 'loycart-pos' ),
                    'account_number' => __( 'Account Number', 'loycart-pos' ),
                    'sort_code'      => __( 'Sort Code', 'loycart-pos' ),
                    'iban'           => __( 'IBAN', 'loycart-pos' ),
                    'bic'            => __( 'BIC / SWIFT', 'loycart-pos' ),
                ];
                foreach ($fields_to_display as $key => $label) {
                    if ( ! empty( $detail[ $key ] ) ) {
                        $bacs_details_html .= '<li><strong>' . esc_html( $label ) . ':</strong> ' . esc_html( $detail[ $key ] ) . '</li>';
                    }
                }
                $bacs_details_html .= '</ul>';
            }
        }
    }

    $cheque_enabled = isset( $all_gateways['cheque'] ) && 'yes' === $all_gateways['cheque']->enabled;
    $cheque_instructions = '';
    if ($cheque_enabled) {
        $cheque_gateway = $all_gateways['cheque'];
        if ( ! empty( $cheque_gateway->instructions ) ) {
            $cheque_instructions = wpautop( wptexturize( $cheque_gateway->instructions ) );
        }
    }

    $final_store_address = '';
    $pos_address_textarea = get_option( 'woocommerce_pos_store_address' );
    if ( ! empty( trim( $pos_address_textarea ) ) ) {
        $final_store_address = nl2br( esc_textarea( $pos_address_textarea ) );
    } else {
        $address_array = [
            'address_1' => get_option( 'woocommerce_store_address', '' ),
            'address_2' => get_option( 'woocommerce_store_address_2', '' ),
            'city'      => get_option( 'woocommerce_store_city', '' ),
            'postcode'  => get_option( 'woocommerce_store_postcode', '' ),
            'country'   => get_option( 'woocommerce_store_country', '' ),
            'state'     => get_option( 'woocommerce_store_state', '' ),
        ];
        $final_store_address = WC()->countries->get_formatted_address( $address_array);
    }
    $store_options = [
        'store_name'     => get_option( 'woocommerce_pos_store_name', get_bloginfo( 'name' ) ),
        'store_address'  => $final_store_address,
        'store_phone'    => get_option( 'woocommerce_pos_store_phone', '' ),
        'store_logo_url' => wp_get_attachment_image_url( get_theme_mod( 'custom_logo' ), 'full' ),
        'footer_message' => get_option( 'woocommerce_pos_receipt_footer', __( 'Thank you for your purchase!', 'loycart-pos' ) ),
        'items_per_page_default' => $items_per_page_default,
    ];



    wp_localize_script( 'loycart-pos-js', 'loycartPOS', array_merge(
        [
            'ajax_url'           => admin_url( 'admin-ajax.php' ),
            'nonce'              => wp_create_nonce( 'loycart_pos_nonce' ),
            'plugin_url'         => plugin_dir_url( __FILE__ ),
            'logout_url'         => wp_logout_url( home_url() ),
            'currency_symbol'    => html_entity_decode( get_woocommerce_currency_symbol() ),
            'currency_pos'       => get_option( 'woocommerce_currency_pos' ),
            'price_decimals'     => wc_get_price_decimals(),
            'price_thousand_sep' => wc_get_price_thousand_separator(),
            'price_decimal_sep'  => wc_get_price_decimal_separator(),
            'placeholder_image_url' => wc_placeholder_img_src('thumbnail'),
            'is_rtl'             => is_rtl(),
            'is_bacs_enabled'    => $bacs_enabled,
            'is_cheque_enabled'  => $cheque_enabled,
            'bacs_instructions'  => $bacs_details_html,
            'cheque_instructions'=> $cheque_instructions,
            'tax_enabled'        => wc_tax_enabled() ? 'true' : 'false',
            'tax_display_mode'    => get_option( 'woocommerce_tax_display_shop' ),
            'tax_label'          => WC()->countries->tax_or_vat(),
            'tax_classes'        => function_exists('wc_get_product_tax_class_options') ? wc_get_product_tax_class_options() : [],
            'tax_statuses'       => [
                'taxable'  => esc_html__( 'Taxable', 'loycart-pos' ),
                'shipping' => esc_html__( 'Shipping Only', 'loycart-pos' ),
                'none'     => esc_html__( 'None', 'loycart-pos' ),
            ],
        ],
        $store_options,
        [ 'strings' => $localized_strings ]
    ));

}


add_action( 'admin_enqueue_scripts', 'loycart_pos_enqueue_admin_scripts' );
add_action('wp_ajax_loycart_pos_get_products', 'loycart_ajax_get_products');
add_action('wp_ajax_loycart_search_customers', 'loycart_ajax_search_customers');
add_action('wp_ajax_loycart_complete_sale', 'loycart_ajax_complete_sale');
add_action('wp_ajax_loycart_hold_sale', 'loycart_ajax_hold_sale');
add_action('wp_ajax_loycart_get_held_carts', 'loycart_ajax_get_held_carts');
add_action('wp_ajax_loycart_resume_held_cart', 'loycart_ajax_resume_held_cart');
add_action('wp_ajax_loycart_delete_held_cart', 'loycart_ajax_delete_held_cart');
add_action('wp_ajax_loycart_pos_process_refund', 'loycart_ajax_process_refund');
add_action('wp_ajax_loycart_ajax_get_label_data', 'loycart_ajax_get_label_data');
add_action('wp_ajax_loycart_refresh_nonce', 'loycart_ajax_refresh_nonce');
add_action('wp_ajax_loycart_pos_create_custom_product', 'loycart_ajax_create_custom_product');
add_action('wp_ajax_loycart_pos_get_shipping_options', 'loycart_ajax_get_shipping_options');

add_action('add_meta_boxes', 'loycart_pos_add_order_meta_box');
function loycart_pos_add_order_meta_box() {  
    add_meta_box('pos_order_info', __( 'POS Order Information', 'loycart-pos' ), 'loycart_pos_render_order_meta_box', 'shop_order', 'side', 'core');
}

function loycart_pos_render_order_meta_box($post) {
    $order = wc_get_order($post->ID);
    if ($order && $order->get_meta('_pos_order') === 'yes') {
        echo '<div class="pos-order-info" style="padding:10px;">';
        echo '<p><strong>' . esc_html__( 'Order Source:', 'loycart-pos' ) . '</strong> ' . esc_html__( 'POS System', 'loycart-pos' ) . '</p>';
        $user_name = $order->get_meta('_pos_user_name') ?: 'N/A';
        echo '<p><strong>' . esc_html__( 'POS User:', 'loycart-pos' ) . '</strong> ' . esc_html($user_name) . '</p>';
        echo '<p><strong>' . esc_html__( 'Terminal:', 'loycart-pos' ) . '</strong> ' . esc_html($order->get_meta('_pos_terminal') ?: 'N/A') . '</p>';
        echo '<p><strong>' . esc_html__( 'Sale Time:', 'loycart-pos' ) . '</strong> ' . esc_html($order->get_meta('_pos_sale_time') ?: 'N/A') . '</p>';
        echo '<p><strong>' . esc_html__( 'Payment Method:', 'loycart-pos' ) . '</strong> ' . esc_html($order->get_payment_method_title()) . '</p>';
        if ($order->get_payment_method() === 'cash' && $order->get_meta('_pos_cash_tendered')) {
            echo '<p><strong>' . esc_html__( 'Amount Tendered:', 'loycart-pos' ) . '</strong> ' . wp_kses_post(wc_price(floatval($order->get_meta('_pos_cash_tendered')))) . '</p>';
            echo '<p><strong>' . esc_html__( 'Change Given:', 'loycart-pos' ) . '</strong> ' . wp_kses_post(wc_price(floatval($order->get_meta('_pos_cash_change')))) . '</p>';
        }
        echo '</div>';
    } else { echo '<p style="padding:10px;">' . esc_html__( 'Not a POS order.', 'loycart-pos' ) . '</p>'; }
}


add_action('wp_ajax_loycart_calculate_cart_totals', 'loycart_ajax_calculate_cart_totals');
add_action('wp_ajax_loycart_pos_get_customer_orders', 'loycart_ajax_get_customer_orders');
add_action('wp_ajax_loycart_pos_get_refundable_order_items', 'loycart_ajax_get_refundable_order_items');
add_action('wp_ajax_loycart_pos_get_customer_address', 'loycart_ajax_get_customer_address');
add_action('wp_ajax_loycart_pos_save_data', 'loycart_ajax_save_cart_state');
add_action('wp_ajax_loycart_pos_load_data', 'loycart_ajax_load_cart_state');
add_action('wp_ajax_loycart_pos_clear_data', 'loycart_ajax_clear_cart_state');
add_action('wp_ajax_loycart_ajax_validate_cart', 'loycart_ajax_validate_cart');
add_action('wp_ajax_loycart_pos_clear_cart_state', 'loycart_ajax_clear_cart_state');
add_action('wp_ajax_loycart_ajax_get_available_coupons', 'loycart_ajax_get_available_coupons');

function loycart_pos_render_app() {
    if ( ! function_exists( 'WC' ) || ! is_plugin_active( 'woocommerce/woocommerce.php' ) ) {
        $message = sprintf(
            '<p>%s</p><a href="%s" class="button button-primary" style="margin-top: 1em;">%s</a>',
            esc_html__( 'LoyCart POS requires WooCommerce to be installed and activated.', 'loycart-pos' ),
            esc_url( admin_url( 'plugins.php' ) ),
            esc_html__( 'Go to Plugins Page', 'loycart-pos' )
        );
        
        wp_die( wp_kses_post( $message ), esc_html__( 'WooCommerce Not Active', 'loycart-pos' ), [ 'response' => 500 ] );
    }

    if ( is_admin() && ! function_exists( 'wc_get_tax_class_options' ) && function_exists('WC') ) {
    include_once WC()->plugin_path() . '/includes/admin/wc-admin-functions.php';
    }
    
    if (!current_user_can('manage_woocommerce')) { wp_die(esc_html__('Access Denied', 'loycart-pos')); }
    $categories = loycart_get_categories();
    ?>
    
<?php
    include_once( plugin_dir_path( __FILE__ ) . 'loycart-pos-app-layout.php' );
    ?>



<?php
    include_once( plugin_dir_path( __FILE__ ) . 'loycart-pos-modals.php' );
    ?>

<?php
}


add_action('wp_ajax_loycart_pos_create_customer', 'loycart_ajax_create_customer');

function loycart_pos_setup_usb_permissions() {
    if ( current_user_can( 'manage_woocommerce' ) ) {
        wp_localize_script( 'loycart-pos-app', 'loycartHardware', [
            'isAuthorized' => true
        ]);
    }
}
add_action( 'wp_enqueue_scripts', 'loycart_pos_setup_usb_permissions' );


?>