<?php
if (!defined('ABSPATH')) { exit; }

/**
 * LoyCart POS - Store Credit via Coupons
 * Implements per-customer store credit using WooCommerce coupons (fixed_cart),
 * restricted by the customer's email. Coupon amount is treated as a running balance
 * and updated after orders and refunds. No expiry by default.
 */

// --- Settings ---
function loycart_pos_get_store_credit_settings() {
    // Back-compat: if old single option exists, use it as default for both new options
    $old_exclude = get_option('loycart_pos_credit_exclude_sale_items', null);
    $default_exclude = (null !== $old_exclude) ? $old_exclude : 'yes';
    return [
        'enabled'                       => get_option('loycart_pos_credit_enabled', 'no') === 'yes',
        'earn_percent'                  => floatval(get_option('loycart_pos_credit_earn_percent', 2)), // default 2%
        'apply_online'                  => get_option('loycart_pos_credit_apply_online', 'no') === 'yes',
        'exclude_sale_items_earning'    => get_option('loycart_pos_credit_exclude_sale_items_earning', $default_exclude) === 'yes',
        'exclude_sale_items_redeeming'  => get_option('loycart_pos_credit_exclude_sale_items_redeeming', $default_exclude) === 'yes',
    ];
}

// Settings UI moved into POS Settings modal; Woo General settings section removed.

// --- Core helpers ---
// Unpublish all store credit coupons
function loycart_pos_unpublish_store_credit_coupons() {
    // This query is safe: store credit coupons are few, only run in admin context, not on every page.
    // phpcs:disable WordPress.DB.SlowDBQuery.slow_db_query_meta_key,WordPress.DB.SlowDBQuery.slow_db_query_meta_value,WordPress.DB.SlowDBQuery.slow_db_query_meta_query
    $coupon_ids = get_posts([
        'post_type'      => 'shop_coupon',
        'post_status'    => ['publish', 'pending', 'private'],
        'posts_per_page' => -1,
        'meta_key'       => '_loycart_store_credit',
        'meta_value'     => 'yes',
        'fields'         => 'ids',
    ]);
    // phpcs:enable
    if (!empty($coupon_ids)) {
        foreach ($coupon_ids as $cid) {
            wp_update_post([
                'ID' => $cid,
                'post_status' => 'draft',
            ]);
        }
    }
}

// Publish all store credit coupons
function loycart_pos_publish_store_credit_coupons() {
    // This query is safe: store credit coupons are few, only run in admin context, not on every page.
    // phpcs:disable WordPress.DB.SlowDBQuery.slow_db_query_meta_key,WordPress.DB.SlowDBQuery.slow_db_query_meta_value,WordPress.DB.SlowDBQuery.slow_db_query_meta_query
    $coupon_ids = get_posts([
        'post_type'      => 'shop_coupon',
        'post_status'    => ['draft', 'pending', 'private'],
        'posts_per_page' => -1,
        'meta_key'       => '_loycart_store_credit',
        'meta_value'     => 'yes',
        'fields'         => 'ids',
    ]);
    // phpcs:enable
    if (!empty($coupon_ids)) {
        foreach ($coupon_ids as $cid) {
            wp_update_post([
                'ID' => $cid,
                'post_status' => 'publish',
            ]);
        }
    }
}
function loycart_pos_credit_coupon_code($user_id) {
    return 'STORE-CREDIT-' . intval($user_id);
}

function loycart_pos_get_or_create_credit_coupon($user_id) {
    $user = get_userdata($user_id);
    if (!$user) return 0;
    $settings = loycart_pos_get_store_credit_settings();
    $code = loycart_pos_credit_coupon_code($user_id);

    // Try to find existing coupon by code (any status)
    $existing = wc_get_coupon_id_by_code($code);
    if ($existing) {
        // Always update meta and status as needed
        update_post_meta($existing, 'customer_email', [ $user->user_email ]);
        update_post_meta($existing, '_loycart_store_credit', 'yes');
        update_post_meta($existing, '_loycart_customer_id', intval($user_id));
        update_post_meta($existing, 'exclude_sale_items', $settings['exclude_sale_items_redeeming'] ? 'yes' : 'no');
        // Set description to full name and email
        $full_name = trim($user->first_name . ' ' . $user->last_name);
        $desc = $full_name ? $full_name . ' <' . $user->user_email . '>' : $user->user_email;
        wp_update_post([
            'ID' => $existing,
            'post_excerpt' => $desc,
        ]);
        $desired_status = $settings['enabled'] ? 'publish' : 'draft';
        $current_status = get_post_status($existing);
        if ($current_status !== $desired_status) {
            wp_update_post([
                'ID' => $existing,
                'post_status' => $desired_status,
            ]);
        }
        return $existing;
    }

    // Create new coupon post if none exists
    $coupon_status = $settings['enabled'] ? 'publish' : 'draft';
    $coupon_id = wp_insert_post([
        'post_title'  => $code,
        'post_type'   => 'shop_coupon',
        'post_status' => $coupon_status,
        'post_author' => get_current_user_id(),
    ]);
    if (is_wp_error($coupon_id) || !$coupon_id) return 0;

    // Initialize coupon meta
    update_post_meta($coupon_id, 'discount_type', 'fixed_cart');
    update_post_meta($coupon_id, 'coupon_amount', 0);
    update_post_meta($coupon_id, 'individual_use', 'no');
    update_post_meta($coupon_id, 'usage_limit', '');
    update_post_meta($coupon_id, 'usage_limit_per_user', '');
    update_post_meta($coupon_id, 'limit_usage_to_x_items', '');
    update_post_meta($coupon_id, 'free_shipping', 'no');
    update_post_meta($coupon_id, 'exclude_sale_items', $settings['exclude_sale_items_redeeming'] ? 'yes' : 'no');
    update_post_meta($coupon_id, 'product_ids', '');
    update_post_meta($coupon_id, 'exclude_product_ids', '');
    update_post_meta($coupon_id, 'product_categories', '');
    update_post_meta($coupon_id, 'exclude_product_categories', '');
    update_post_meta($coupon_id, 'minimum_amount', '');
    update_post_meta($coupon_id, 'maximum_amount', '');
    update_post_meta($coupon_id, 'customer_email', [ $user->user_email ]); // email restriction
    delete_post_meta($coupon_id, 'date_expires'); // no expiry
    update_post_meta($coupon_id, '_loycart_store_credit', 'yes');
    update_post_meta($coupon_id, '_loycart_customer_id', intval($user_id));
    // Set description to full name and email
    $full_name = trim($user->first_name . ' ' . $user->last_name);
    $desc = $full_name ? $full_name . ' <' . $user->user_email . '>' : $user->user_email;
    wp_update_post([
        'ID' => $coupon_id,
        'post_excerpt' => $desc,
    ]);

    return $coupon_id;
}

function loycart_pos_adjust_credit_balance($user_id, $delta_amount) {
    $coupon_id = loycart_pos_get_or_create_credit_coupon($user_id);
    if (!$coupon_id) return false;
    $current = floatval(get_post_meta($coupon_id, 'coupon_amount', true));
    $new = max(0, round($current + floatval($delta_amount), 2));
    update_post_meta($coupon_id, 'coupon_amount', $new);
    return true;
}

// --- Earn credit on order completion ---
add_action('woocommerce_order_status_completed', function ($order_id) {
    $settings = loycart_pos_get_store_credit_settings();
    if (!$settings['enabled']) return;

    $order = wc_get_order($order_id);
    if (!$order) return;

    $is_pos_order = ($order->get_meta('_pos_order') === 'yes');
    if (!$is_pos_order && !$settings['apply_online']) return;

    $user_id = $order->get_user_id();
    if (!$user_id) return; // Only for known customers

    $earn_percent = max(0, $settings['earn_percent']);
    if ($earn_percent <= 0) return;

    $eligible_subtotal = 0.0;
    foreach ($order->get_items('line_item') as $item) {
        $product = $item->get_product();
        if (!$product) continue;
        // Exclude sale items from earning credit if enabled
        if ($settings['exclude_sale_items_earning'] && $product->is_on_sale()) continue;
        $eligible_subtotal += floatval($item->get_subtotal()); // pre-discount, excl tax
    }

    if ($eligible_subtotal <= 0) return;
    $credit = round(($earn_percent / 100.0) * $eligible_subtotal, 2);
    if ($credit <= 0) return;

    loycart_pos_adjust_credit_balance($user_id, $credit);
    // Add an order note to clarify how store credit was calculated
    $rate_str = number_format($earn_percent, (floor($earn_percent) == $earn_percent ? 0 : 2));
    $note = sprintf(
        /* translators: 1: earned credit amount, 2: earn rate %, 3: eligible subtotal, 4: note re: sale items exclusion */
        __('Store credit earned: %1$s at %2$s%% of eligible subtotal %3$s%4$s', 'loycart-pos'),
        wc_price($credit),
        $rate_str,
        wc_price($eligible_subtotal),
        $settings['exclude_sale_items_earning'] ? __(' (sale items excluded from earning)', 'loycart-pos') : ''
    );
    $order->add_order_note($note);
    $order->update_meta_data('_loycart_credit_earned', $credit);
    $order->save();
}, 10);

// --- Decrement credit by redeemed amount (actual discount used) ---
add_action('woocommerce_order_status_completed', function ($order_id) {
    $settings = loycart_pos_get_store_credit_settings();
    if (!$settings['enabled']) return;
    $order = wc_get_order($order_id);
    if (!$order) return;

    // For any order (POS or online) where store credit coupons are used, decrement
    foreach ($order->get_items('coupon') as $coupon_item) {
        $code = $coupon_item->get_code();
        $coupon_id = wc_get_coupon_id_by_code($code);
        if (!$coupon_id) continue;
        $is_store_credit = get_post_meta($coupon_id, '_loycart_store_credit', true) === 'yes';
        if (!$is_store_credit) continue;

        $discount_used = floatval($coupon_item->get_discount());
        if ($discount_used <= 0) continue;

        // Find mapped customer
        $user_id = intval(get_post_meta($coupon_id, '_loycart_customer_id', true));
        if ($user_id) {
            loycart_pos_adjust_credit_balance($user_id, -$discount_used);
        } else {
            // Fallback: if coupon is store credit but no mapping, prevent negative, just reduce amount
            $current = floatval(get_post_meta($coupon_id, 'coupon_amount', true));
            $new = max(0, round($current - $discount_used, 2));
            update_post_meta($coupon_id, 'coupon_amount', $new);
        }

        // Store meta on order for reporting
        $prev = floatval($order->get_meta('_loycart_credit_redeemed'));
        $order->update_meta_data('_loycart_credit_redeemed', round($prev + $discount_used, 2));
    }
    $order->save();
}, 11);

// --- Handle refunds: restore redeemed credit and remove earned credit proportionally ---
add_action('woocommerce_order_refunded', function ($order_id, $refund_id) {
    $settings = loycart_pos_get_store_credit_settings();
    if (!$settings['enabled']) return;
    $order = wc_get_order($order_id);
    $refund = wc_get_order($refund_id);
    if (!$order || !$refund) return;

    $order_total = floatval($order->get_total());
    $refund_total = floatval($refund->get_total()); // negative number in Woo for refunds
    $refund_amount = abs($refund_total);
    if ($order_total <= 0 || $refund_amount <= 0) return;

    $ratio = min(1.0, $refund_amount / $order_total);

    // 1) Restore proportion of redeemed credit
    $redeemed_total = floatval($order->get_meta('_loycart_credit_redeemed'));
    $restore = round($redeemed_total * $ratio, 2);
    if ($restore > 0) {
        // We need a customer context; use order customer
        $user_id = $order->get_user_id();
        if ($user_id) {
            loycart_pos_adjust_credit_balance($user_id, $restore);
        }
    }

    // 2) Remove proportion of earned credit
    $earned_total = floatval($order->get_meta('_loycart_credit_earned'));
    $remove = round($earned_total * $ratio, 2);
    if ($remove > 0) {
        $user_id = $order->get_user_id();
        if ($user_id) {
            loycart_pos_adjust_credit_balance($user_id, -$remove);
        }
    }
}, 10, 2);

// --- Utility: ensure coupon stays tied to latest customer email ---
add_action('profile_update', function ($user_id, $old_user_data) {
    $settings = loycart_pos_get_store_credit_settings();
    if (!$settings['enabled']) return;
    // Update email restriction on coupon when user changes email
    $coupon_id = loycart_pos_get_or_create_credit_coupon($user_id);
    if ($coupon_id) {
        $user = get_userdata($user_id);
        if ($user) {
            update_post_meta($coupon_id, 'customer_email', [ $user->user_email ]);
        }
    }
}, 10, 2);

// =========================
// WooCommerce My Account: Store Credit endpoint
// =========================

// 1) Register endpoint
add_action('init', function() {
    add_rewrite_endpoint('store-credit', EP_ROOT | EP_PAGES);
});

// 2) Add menu item
add_filter('woocommerce_account_menu_items', function($items) {
    // Insert after Orders if present
    $new = [];
    foreach ($items as $key => $label) {
        $new[$key] = $label;
        if ('orders' === $key) {
            $new['store-credit'] = __('Store Credit', 'loycart-pos');
        }
    }
    if (!isset($new['store-credit'])) {
        $new['store-credit'] = __('Store Credit', 'loycart-pos');
    }
    return $new;
});

// 3) Render endpoint content
add_action('woocommerce_account_store-credit_endpoint', function() {
    if (!is_user_logged_in()) {
        echo '<p>' . esc_html__('Please log in to view your store credit.', 'loycart-pos') . '</p>';
        return;
    }

    $settings = loycart_pos_get_store_credit_settings();
    if (!$settings['enabled']) {
        echo '<p>' . esc_html__('Store credit is not enabled.', 'loycart-pos') . '</p>';
        return;
    }

    $user_id = get_current_user_id();
    $coupon_id = loycart_pos_get_or_create_credit_coupon($user_id);
    $amount = $coupon_id ? floatval(get_post_meta($coupon_id, 'coupon_amount', true)) : 0.0;
    $code = $coupon_id ? get_the_title($coupon_id) : '';

    echo '<div class="loycart-store-credit">';
    echo '<h3>' . esc_html__('Store Credit Balance', 'loycart-pos') . '</h3>';
    echo '<p style="font-size:1.4em; font-weight:700;">' . wp_kses_post(wc_price($amount)) . '</p>';
    if ($code) {
        /* translators: %s: Customer's store credit coupon code */
        echo '<p>' . sprintf( esc_html__('Your credit coupon code: %s', 'loycart-pos'), esc_html( strtoupper($code) ) ) . '</p>';
    }

    // Recent activity
    echo '<h4 style="margin-top:1em;">' . esc_html__('Recent Activity', 'loycart-pos') . '</h4>';
    $orders = wc_get_orders([
        'customer' => $user_id,
        'limit'    => 5,
        'orderby'  => 'date',
        'order'    => 'DESC',
        'return'   => 'objects',
        'status'   => ['completed', 'processing', 'on-hold', 'refunded'],
    ]);

    $has_activity = false;
    if (!empty($orders)) {
        echo '<ul class="store-credit-history" style="list-style: none; padding-left:0;">';
        foreach ($orders as $order) {
            $earned   = floatval($order->get_meta('_loycart_credit_earned'));
            $redeemed = floatval($order->get_meta('_loycart_credit_redeemed'));
            if ($earned <= 0 && $redeemed <= 0) { continue; }
            $has_activity = true;
            $date = $order->get_date_created() ? $order->get_date_created()->date_i18n(get_option('date_format')) : '';
            $parts = [];
            if ($earned > 0) {
                /* translators: %s: Amount of store credit earned */
                $earned_label = __('Earned %s', 'loycart-pos');
                $parts[] = sprintf($earned_label, wc_price($earned));
            }
            if ($redeemed > 0) {
                /* translators: %s: Amount of store credit redeemed */
                $redeemed_label = __('Redeemed %s', 'loycart-pos');
                $parts[] = sprintf($redeemed_label, wc_price($redeemed));
            }
            $desc = implode(' • ', $parts);
            echo '<li style="padding:6px 0; border-bottom: 1px dashed var(--border);">' . esc_html($date) . ' — ' . wp_kses_post($desc) . ' (' . esc_html__('Order #', 'loycart-pos') . esc_html($order->get_order_number()) . ')</li>';
        }
        echo '</ul>';
    }
    if (!$has_activity) {
        echo '<p>' . esc_html__('No recent credit activity yet.', 'loycart-pos') . '</p>';
    }
    echo '</div>';
});

// Admin Store Credit Reports submenu removed; POS now provides reports via modal.

?>
