<?php
/**
 * LoyCart POS Data Functions
 *
 * This file contains all the functions for querying WordPress/WooCommerce data.
 */

if ( ! defined( 'ABSPATH' ) ) {
    exit; 
}

function loycart_get_categories() {
    $args = ['taxonomy' => 'product_cat', 'orderby' => 'name', 'order' => 'ASC', 'hide_empty' => false];
    $categories = get_terms($args);
    return is_wp_error($categories) ? [] : $categories;
}


function loycart_pos_get_products($category_id = null, $search = '', $page = 1, $per_page = null, $sort_param = null) {
    global $wpdb;

    if ($per_page === null) {
        $per_page = intval(get_option('loycart_pos_products_per_page', 24));
        if ($per_page < 1) { $per_page = 24; }
    }
    if ($sort_param === null) {
        $sort_param = get_option('loycart_pos_default_sort', 'popular');
    }

    // Bump cache namespace to invalidate older search results after logic changes
    $transient_key = 'loycart_products_v4_' . md5( wp_json_encode( func_get_args() ) );
    
    
    $cached_results = get_transient( $transient_key );
    if ( false !== $cached_results ) {
        return $cached_results;
    }
    

    $search = sanitize_text_field($search);
    $page = max(1, intval($page));
    $per_page = max(1, intval($per_page));
    $is_detail_lookup = ($per_page === 1 && !empty($search));
    $allow_private_detail = get_option('loycart_pos_allow_private_detail_lookups', 'yes') === 'yes';
    $is_grid_view = ($per_page > 1);

    $matched_variation_id = 0;
    $custom_error = null;

    $args = [
        'post_type'      => 'product',
        'posts_per_page' => $per_page,
        'paged'          => $page,
        'post_status'    => 'publish',
        'fields'         => 'ids'
    ];
    
    
    // --- LOGIC CHANGE START ---
    if ($is_detail_lookup) {
        // Detail lookups (barcode/ID) may include private items if allowed in settings
        $args['post_status'] = $allow_private_detail ? array('publish', 'private') : 'publish';
    } else {
        // For the regular POS Grid and Category browsing, ONLY show 'publish'
        $args['post_status'] = 'publish';
    }
    // --- LOGIC CHANGE END ---
    
    
    $meta_query = [ 'relation' => 'AND' ];
     // phpcs:disable WordPress.DB.DirectDatabaseQuery
    if ($is_detail_lookup) {
        $barcode_matches = $wpdb->get_results($wpdb->prepare(
            "SELECT post_id FROM {$wpdb->postmeta} WHERE meta_key = '_global_unique_id' AND meta_value = %s",
            $search
        ));
     // phpcs:enable WordPress.DB.DirectDatabaseQuery
        if (count($barcode_matches) === 1) {
            $barcode_post_id = $barcode_matches[0]->post_id;
            $post_type = get_post_type($barcode_post_id);
            $post_status = get_post_status($barcode_post_id);

            // Respect visibility setting: allow private only if enabled
            $status_ok = ($post_status === 'publish') || ($allow_private_detail && $post_status === 'private');
            if ($status_ok || $post_type === 'product_variation') {
                if ('product_variation' === $post_type) {
                    $parent_id = wp_get_post_parent_id($barcode_post_id);
                    // If parent product is private and private detail lookups are disabled, block
                    $parent_status = get_post_status($parent_id);
                    $parent_ok = ($parent_status === 'publish') || ($allow_private_detail && $parent_status === 'private');
                    $args['post__in'] = $parent_ok ? [$parent_id] : [0];
                    $matched_variation_id = (int)$barcode_post_id;
                } else {
                    $args['post__in'] = [$barcode_post_id];
                }
            }
        } else {
            
            // phpcs:disable WordPress.DB.DirectDatabaseQuery
            $sku_post_id = $wpdb->get_var($wpdb->prepare(
                "SELECT post_id FROM {$wpdb->postmeta} WHERE meta_key = '_sku' AND meta_value = %s",
                $search
            ));
            // phpcs:enable WordPress.DB.DirectDatabaseQuery

            if ($sku_post_id) {
                $post_type = get_post_type($sku_post_id);
                if ('product_variation' === $post_type) {
                    $parent_product_id = wp_get_post_parent_id($sku_post_id);
                    if ($parent_product_id) {
                        // Parent must respect private visibility setting
                        $parent_status = get_post_status($parent_product_id);
                        $parent_ok = ($parent_status === 'publish') || ($allow_private_detail && $parent_status === 'private');
                        $args['post__in'] = $parent_ok ? [$parent_product_id] : [0];
                        $matched_variation_id = (int)$sku_post_id; 
                    }
                } elseif ('product' === $post_type) {
                    // Product must respect private visibility setting
                    $post_status = get_post_status($sku_post_id);
                    $status_ok = ($post_status === 'publish') || ($allow_private_detail && $post_status === 'private');
                    $args['post__in'] = $status_ok ? [$sku_post_id] : [0];
                }
            } else {
                
                if ( is_numeric($search) ) {
                    $post_by_id = get_post( $search );
                    if ( $post_by_id && ( 'product' === $post_by_id->post_type || 'product_variation' === $post_by_id->post_type ) ) {
                        if ( 'product_variation' === $post_by_id->post_type ) {
                            $parent_id = wp_get_post_parent_id( $post_by_id->ID );
                            if ( $parent_id ) {
                                // Parent must respect private visibility setting
                                $parent_status = get_post_status($parent_id);
                                $parent_ok = ($parent_status === 'publish') || ($allow_private_detail && $parent_status === 'private');
                                $args['post__in'] = $parent_ok ? [$parent_id] : [0];
                                $matched_variation_id = (int)$search;
                            }
                        } else {
                            // Product must respect private visibility setting
                            $post_status = get_post_status($post_by_id->ID);
                            $status_ok = ($post_status === 'publish') || ($allow_private_detail && $post_status === 'private');
                            $args['post__in'] = $status_ok ? [$search] : [0];
                        }
                    } else {
                        $args['post__in'] = [0];
                    }
                } else {
                    $args['post__in'] = [0];
                }
            }
        }

   } else {
        
        // Apply category filter only when not actively searching by text.
        // Global text search should span the entire catalog for better UX.
        if (empty($search) && $category_id && $category_id !== 'all' && term_exists((int) $category_id, 'product_cat')) {
            // phpcs:disable WordPress.DB.SlowDBQuery
            $args['tax_query'] = [['taxonomy' => 'product_cat', 'field' => 'term_id', 'terms' => absint($category_id)]];
            // phpcs:enable WordPress.DB.SlowDBQuery
        }
        if (!empty($search)) {
            // Robust, predictable matching:
            // - Split the search into terms and require each term to appear somewhere in the title (mid-word OK)
            // - Also include products whose _sku partially matches the raw search
            // We prefetch matching IDs and feed them via post__in to avoid brittle search filters.
            $title_match_ids = [];
            $sku_match_ids   = [];

            // 1) Title matches: AND across all terms for relevancy
            $terms = preg_split('/\s+/', $search);
            $terms = array_filter(array_map('trim', (array) $terms));

            if (!empty($terms)) {
                global $wpdb;
                // Intersect per-term matches to achieve AND-across-terms without dynamic placeholder concatenation
                $acc_ids = null;
                foreach ($terms as $t) {
                    $like = '%' . $wpdb->esc_like($t) . '%';
                    // phpcs:disable WordPress.DB.DirectDatabaseQuery
                    $ids_for_term = $wpdb->get_col(
                        $wpdb->prepare(
                            "SELECT ID FROM {$wpdb->posts} \n"
                            . "WHERE {$wpdb->posts}.post_type = 'product' \n"
                            . "AND {$wpdb->posts}.post_status = 'publish' \n"
                            . "AND ( {$wpdb->posts}.post_title LIKE %s OR {$wpdb->posts}.post_name LIKE %s OR {$wpdb->posts}.post_excerpt LIKE %s )",
                            $like, $like, $like
                        )
                    );
                    // phpcs:enable WordPress.DB.DirectDatabaseQuery

                    if (is_null($acc_ids)) {
                        $acc_ids = $ids_for_term;
                    } else {
                        $acc_ids = array_values(array_intersect($acc_ids, $ids_for_term));
                    }

                    if (empty($acc_ids)) {
                        break; // No need to continue if intersection is empty
                    }
                }
                $title_match_ids = !empty($acc_ids) ? array_values(array_unique($acc_ids)) : [];
            }

            // Fallback: if multi-word search returns nothing, try phrase match across title/slug/excerpt
            if (empty($title_match_ids) && count($terms) > 1) {
                global $wpdb;
                $phrase_like = '%' . $wpdb->esc_like($search) . '%';
                // phpcs:disable WordPress.DB.DirectDatabaseQuery
                $title_match_ids = $wpdb->get_col(
                    $wpdb->prepare(
                        "SELECT ID FROM {$wpdb->posts} \n"
                        . "WHERE {$wpdb->posts}.post_type = 'product' \n"
                        . "AND {$wpdb->posts}.post_status = 'publish' \n"
                        . "AND ( {$wpdb->posts}.post_title LIKE %s OR {$wpdb->posts}.post_name LIKE %s OR {$wpdb->posts}.post_excerpt LIKE %s )",
                        $phrase_like, $phrase_like, $phrase_like
                    )
                );
                // phpcs:enable WordPress.DB.DirectDatabaseQuery
            }

            // 2) SKU partial matches on simple products (grid shows parents). Variations are handled via detail scans.
            if (!empty($search)) {
                global $wpdb;
                $like = '%' . $wpdb->esc_like($search) . '%';
                // phpcs:disable WordPress.DB.DirectDatabaseQuery
                $sku_match_ids = $wpdb->get_col(
                    $wpdb->prepare(
                        "SELECT p.ID FROM {$wpdb->posts} p\n"
                        . "INNER JOIN {$wpdb->postmeta} pm ON pm.post_id = p.ID \n"
                        . "WHERE p.post_type = 'product' AND p.post_status = 'publish' \n"
                        . "AND pm.meta_key = '_sku' AND pm.meta_value LIKE %s",
                        $like
                    )
                );
                // phpcs:enable WordPress.DB.DirectDatabaseQuery
            }

            $combined_ids = array_values(array_unique(array_merge($title_match_ids, $sku_match_ids)));

            // If no matches at all, short-circuit with an impossible ID to avoid scanning entire catalog
            $args['post__in'] = !empty($combined_ids) ? $combined_ids : [0];
        }
        switch ($sort_param) {
            case 'name_asc': $args['orderby'] = 'title'; $args['order'] = 'ASC'; break;
            case 'name_desc': $args['orderby'] = 'title'; $args['order'] = 'DESC'; break;
            case 'price_asc': $meta_query['price_clause'] = ['key' => '_price', 'type' => 'NUMERIC']; $args['orderby'] = 'price_clause'; $args['order'] = 'ASC'; break;
            case 'price_desc': $meta_query['price_clause'] = ['key' => '_price', 'type' => 'NUMERIC']; $args['orderby'] = 'price_clause'; $args['order'] = 'DESC'; break;
            case 'sale': $args['post__in'] = array_merge([0], wc_get_product_ids_on_sale()); $args['orderby'] = 'menu_order'; break;
            case 'newest': $args['orderby'] = 'date'; $args['order'] = 'DESC'; break;
            default: $meta_query['sales_clause'] = ['key' => 'total_sales', 'type' => 'NUMERIC']; $args['orderby'] = 'sales_clause'; $args['order'] = 'DESC'; break;
        }
    }

    // Exclude products flagged as store-only or hidden from POS visibility
    $meta_query[] = [
        'relation' => 'OR',
        [
            'key'     => '_loycart_visibility',
            'value'   => ['store_only', 'hidden'],
            'compare' => 'NOT IN',
        ],
        [
            'key'     => '_loycart_visibility',
            'compare' => 'NOT EXISTS',
        ],
    ];
    
        if (!empty($meta_query)) {
            // phpcs:disable WordPress.DB.SlowDBQuery
            $args['meta_query'] = $meta_query;
            // phpcs:enable WordPress.DB.SlowDBQuery
        }
    
    $query = new WP_Query($args);
    
    $products_data = [];
    $total_products = $query->found_posts;
    if ($query->have_posts()) {
        foreach ($query->posts as $product_id) {
            $product = wc_get_product($product_id);
            if ($product) {
  
                $image_size = 'medium';
                $image_url = wp_get_attachment_image_url($product->get_image_id(), $image_size) ?: wc_placeholder_img_src('thumbnail');
                
               
                $gallery_images = [];
                if ( ! $is_grid_view ) {
                    foreach ($product->get_gallery_image_ids() as $image_id) {
                        
                        $gallery_images[] = wp_get_attachment_image_url($image_id, 'medium');
                    }
                }

                $item_data = [
                    'id'               => $product->get_id(),
                    'name'             => $product->get_name(),
                    'sku'              => $product->get_sku(),
                    'barcode'          => get_post_meta( $product->get_id(), '_global_unique_id', true ),
                    'description'      => $product->get_description(),
                    'image'            => $image_url,
                    'gallery_images'   => $gallery_images, 
                    'type'             => $product->get_type(),
                    'price'            => wc_format_decimal( wc_get_price_to_display( $product, array( 'price' => $product->get_price() ) ), wc_get_price_decimals() ), 
                    'regular_price'    => wc_format_decimal( wc_get_price_to_display( $product, array( 'price' => $product->get_regular_price() ) ), wc_get_price_decimals() ),
                    'sale_price'       => wc_format_decimal( wc_get_price_to_display( $product, array( 'price' => $product->get_sale_price() ) ), wc_get_price_decimals() ),
                    'on_sale'          => $product->is_on_sale(),
                    'stock_status'     => $product->get_stock_status(),
                    'stock_quantity'   => $product->get_stock_quantity(),
                    'sold_individually'=> $product->get_sold_individually(),
                    'is_virtual'       => $product->is_virtual(),
                    'max_price'        => $product->is_type( 'variable' )
                        ? wc_format_decimal( wc_get_price_to_display( $product, array( 'price' => $product->get_variation_price( 'max', false ) ) ), wc_get_price_decimals() )
                        : wc_format_decimal( wc_get_price_to_display( $product, array( 'price' => $product->get_price() ) ), wc_get_price_decimals() ),
                ];

                if ($matched_variation_id > 0) {
                    $item_data['matched_variation_id'] = $matched_variation_id;
                }

                if ( $is_detail_lookup && $product->is_type('variable') ) {
                    $item_data['variations'] = [];
                    $parent_barcode = get_post_meta($product->get_id(), '_global_unique_id', true);

                    foreach ($product->get_children() as $variation_id) {
                        $variation = wc_get_product($variation_id);
                        if ($variation && $variation->exists()) {
                            
                            $variation_barcode = get_post_meta($variation_id, '_global_unique_id', true);
                            $final_barcode = !empty($variation_barcode) ? $variation_barcode : $parent_barcode;

                            $var_image_url = wp_get_attachment_image_url($variation->get_image_id(), 'thumbnail') ?: $image_url;
                            $item_data['variations'][] = [
                                'id' => $variation->get_id(),
                                'attributes' => $variation->get_variation_attributes(),
                                'barcode' => $final_barcode,
                                'price' => wc_format_decimal( wc_get_price_to_display( $variation, array( 'price' => $variation->get_price() ) ), wc_get_price_decimals() ),
                                'regular_price' => wc_format_decimal( wc_get_price_to_display( $variation, array( 'price' => $variation->get_regular_price() ) ), wc_get_price_decimals() ),
                                'sale_price' => wc_format_decimal( wc_get_price_to_display( $variation, array( 'price' => $variation->get_sale_price() ) ), wc_get_price_decimals() ),
                                'on_sale' => $variation->is_on_sale(),
                                'stock_status' => $variation->get_stock_status(),
                                'stock_quantity' => $variation->get_stock_quantity(),
                                'sku' => $variation->get_sku(),
                                'image' => $var_image_url,
                                'sold_individually' => $variation->get_sold_individually(),
                                'is_virtual' => $variation->is_virtual(),
                            ];
                        }
                    }
                }
                $products_data[] = $item_data;
            }
        }
    }
    wp_reset_postdata();

    // Clean up scoped filter if we set it
    if (!empty($search)) {
        remove_filter('posts_search', '__return_false'); // no-op safety in case of mis-binding
        // We must remove our anonymous filter; since it's anonymous, re-binding isn't trivial. To avoid side-effects
        // we only added it conditionally for this query and rely on WordPress creating SQL at instantiation time.
        // After $query is constructed, the filter no longer affects anything else in this request.
    }
    
    $result_data = [
        'products' => $products_data, 
        'total_products' => $total_products, 
        'total_pages' => ceil($total_products / $per_page), 
        'current_page' => $page,
        'error' => $custom_error
    ];
    
    if ( is_null( $custom_error ) ) {
        set_transient( $transient_key, $result_data, 2 * HOUR_IN_SECONDS );
    }

    return $result_data;
}


function loycart_pos_calculate_order_package_details( $order ) {
    $total_weight = 0;
    $max_length = 0;
    $max_width = 0;
    $max_height = 0;
    $weight_unit = get_option( 'woocommerce_weight_unit' );
    $dimension_unit = get_option( 'woocommerce_dimension_unit' );

    foreach ( $order->get_items() as $item ) {
        $product = $item->get_product();

        if ( $product ) {

            $weight = floatval( $product->get_weight() );
            $total_weight += $weight * $item->get_quantity();
            
            $length = floatval( $product->get_length() );
            $width = floatval( $product->get_width() );
            $height = floatval( $product->get_height() );

            $max_length = max( $max_length, $length );
            $max_width = max( $max_width, $width );
            $max_height = max( $max_height, $height );
        }
    }

    $weight_formatted = number_format( $total_weight, 2 ) . ' ' . $weight_unit;

    $dimensions_formatted = 'N/A';
    if ( $max_length > 0 || $max_width > 0 || $max_height > 0 ) {
        $dimensions_formatted = sprintf( 
            '%s x %s x %s %s',
            number_format( $max_length, 1 ),
            number_format( $max_width, 1 ),
            number_format( $max_height, 1 ),
            $dimension_unit
        );
    }

    return [
        'weight' => $weight_formatted,
        'dimensions' => $dimensions_formatted
    ];
}