<?php
/**
 * LoyCart POS AJAX Handlers
 *
 * This file contains all the callback functions for AJAX requests.
 */

if ( ! defined( 'ABSPATH' ) ) {
    exit; 
}



function loycart_ajax_process_refund() {
    global $wpdb;

    check_ajax_referer('loycart_pos_nonce', 'nonce');

    // Capability check: only WooCommerce managers can process refunds
    if ( ! current_user_can( 'manage_woocommerce' ) ) {
        wp_send_json_error( __( 'Unauthorized', 'loycart-pos' ), 403 );
        return;
    }

    $original_order_id = isset($_POST['original_order_id']) ? absint(wp_unslash($_POST['original_order_id'])) : 0;
    $cart_raw = isset($_POST['cart']) ? sanitize_textarea_field(wp_unslash($_POST['cart'])) : '[]';
    
    $sanitized_reason = isset($_POST['reason']) ? sanitize_textarea_field(wp_unslash($_POST['reason'])) : '';
    $refund_reason = !empty(trim($sanitized_reason)) 
        ? $sanitized_reason 
        : __('Return processed via POS', 'loycart-pos');
    
    $refund_items_from_pos = json_decode($cart_raw, true);
    
    // Read shipping refund parameters from POST (unslash and sanitize)
    $refund_shipping_raw = isset($_POST['refund_shipping']) ? sanitize_text_field( wp_unslash( $_POST['refund_shipping'] ) ) : '';
    $refund_shipping = in_array( strtolower( (string) $refund_shipping_raw ), [ 'true', '1', 'yes', 'on' ], true );
    $shipping_cost = isset($_POST['shipping_cost']) ? floatval( sanitize_text_field( wp_unslash( $_POST['shipping_cost'] ) ) ) : 0;
    
    // Also check cart array for shipping items (backwards compatibility)
    if (!$refund_shipping && is_array($refund_items_from_pos)) {
        foreach ($refund_items_from_pos as $item) {
            if (isset($item['is_shipping_refund']) && $item['is_shipping_refund']) {
                $refund_shipping = true;
                break;
            }
        }
    }

    $order = wc_get_order($original_order_id);
    if ( ! $order ) {
        wp_send_json_error( __('Original order not found.', 'loycart-pos') );
        return;
    }

    try {
        $line_items_to_refund = [];
        $grand_total_refund_amount = 0;
        $precision = wc_get_price_decimals();
        
        $receipt_subtotal = 0;
        $receipt_tax_total = 0;
        $receipt_shipping_total = 0;
        $receipt_shipping_tax = 0;

        $original_order_items = $order->get_items(['line_item', 'shipping']);

        foreach ($original_order_items as $item_id => $item) {
            if ( 'line_item' === $item->get_type() ) {
                if (is_array($refund_items_from_pos)) {
                    foreach ($refund_items_from_pos as $refund_item) {
                        if (isset($refund_item['original_item_id']) && absint($refund_item['original_item_id']) === $item_id) {
                            $qty_to_refund = absint($refund_item['quantity']);
                            $original_qty = $item->get_quantity();
                            
                            // ── FIX: Use subtotal (pre-discount) for proper calculations ──
                            $subtotal_per_unit = (float) $item->get_subtotal() / $original_qty;
                            $subtotal_to_refund = $subtotal_per_unit * $qty_to_refund;
                            
                            // ── FIX: Calculate discount amount proportionally ──
                            $original_total = (float) $item->get_total();
                            $original_subtotal = (float) $item->get_subtotal();
                            $total_discount = $original_subtotal - $original_total;
                            $discount_per_unit = $total_discount / $original_qty;
                            $discount_to_refund = $discount_per_unit * $qty_to_refund;
                            
                            // ── FIX: Refund total = subtotal - discount (what customer actually paid) ──
                            $refund_subtotal_amount = $subtotal_to_refund - $discount_to_refund;
                            
                            $refund_tax_arr = [];
                            foreach ($item->get_taxes()['total'] as $tax_rate_id => $tax_amount) {
                                $tax_per_item = (float) $tax_amount / $original_qty;
                                $refund_tax_arr[$tax_rate_id] = $tax_per_item * $qty_to_refund;
                            }

                            $grand_total_refund_amount -= $refund_subtotal_amount + array_sum($refund_tax_arr);
  
                            // ── FIX: Receipt shows subtotal before discount, then discount is shown separately in receipt ──
                            $receipt_subtotal += $subtotal_to_refund;
                            $receipt_tax_total += array_sum($refund_tax_arr);
                            
                            $line_items_to_refund[$item_id] = [ 
                                'qty' => $qty_to_refund, 
                                'refund_total' => wc_format_decimal($refund_subtotal_amount, $precision), 
                                'refund_tax' => array_map(function($tax) use ($precision) { return wc_format_decimal($tax, $precision); }, $refund_tax_arr) 
                            ];
                            
                            break;
                        }
                    }
                }
            }

            if ( 'shipping' === $item->get_type() && $refund_shipping && $shipping_cost > 0 ) {
                $original_shipping_total = abs($shipping_cost); // Use the shipping cost sent from frontend

                $total_shipping_refunded = 0;
                if ( $order->get_refunds() ) {
                    foreach ( $order->get_refunds() as $refund ) { $total_shipping_refunded += abs($refund->get_shipping_total()); }
                }
                
                $remaining_shipping = (float) $order->get_shipping_total() - $total_shipping_refunded;
                
                if ($remaining_shipping > 0 && $original_shipping_total <= $remaining_shipping) { 
                    $shipping_total_to_refund = $original_shipping_total;
                    $shipping_tax_to_refund_arr = [];
                    foreach($item->get_taxes()['total'] as $tax_rate_id => $tax_amount) {
                        $shipping_tax_to_refund_arr[$tax_rate_id] = (float) $tax_amount;
                    }
                    
                    $grand_total_refund_amount -= ($shipping_total_to_refund + array_sum($shipping_tax_to_refund_arr));
                    $line_items_to_refund[$item_id] = [ 
                        'qty' => 1, 
                        'refund_total' => wc_format_decimal($shipping_total_to_refund, $precision), 
                        'refund_tax' => array_map(function($tax) use ($precision) { return wc_format_decimal($tax, $precision); }, $shipping_tax_to_refund_arr) 
                    ];
                    
                    // Store shipping and shipping tax separately
                    $receipt_shipping_total += $shipping_total_to_refund;
                    $receipt_shipping_tax += array_sum($shipping_tax_to_refund_arr);
                }
            }
        }
        
 
        $grand_total_refund_amount = abs($grand_total_refund_amount);


           // IMPORTANT: Do not cap against WooCommerce get_total()/get_total_refunded here,
           // as some configurations may exclude shipping tax from get_total(), causing shipping-only
           // refunds to be limited to the shipping cost without tax. Our computed amount already
           // respects item quantities and remaining shipping via the checks above and includes tax.
           // Simply format the calculated total for refund creation and receipt output.
           $grand_total_refund_amount = wc_format_decimal( $grand_total_refund_amount, $precision );
      


        $refund = wc_create_refund([
            'amount'         => $grand_total_refund_amount,
            'reason'         => $refund_reason,
            'order_id'       => $original_order_id,
            'line_items'     => $line_items_to_refund,
            'refund_payment' => false,
            'restock_items'  => true,
        ]);

        if ( is_wp_error($refund) ) {
            /* translators: %s: The specific error message returned when a refund fails. */
            wp_send_json_error( sprintf(__('Error creating refund: %s', 'loycart-pos'), $refund->get_error_message()) );
            return;
        }

        $refund->update_meta_data('_pos_order', 'yes');
        $refund->update_meta_data('_pos_user_id', get_current_user_id());
        $refund->save();
        

        foreach ($line_items_to_refund as $item_id => $details) {
             $item = $order->get_item($item_id);
            if ( $item && $item->is_type('line_item') && isset($details['qty']) && $details['qty'] > 0) {
                wc_update_order_item_meta($item_id, '_qty_refunded', (int) wc_get_order_item_meta($item_id, '_qty_refunded', true) + $details['qty']);
            }
        }

        /* translators: %s: The formatted total amount of the refund. */
        $note = sprintf(__('POS refund of %s processed.', 'loycart-pos'), $refund->get_formatted_order_total());
        $note .= ' ' . __('Reason:', 'loycart-pos') . ' ' . $refund_reason;
        $order->add_order_note($note);
        
        // Calculate separate cashier and coupon discounts for receipt
        $receipt_cashier_discount = 0;
        $receipt_coupon_discount = 0;
        
        // Get order-level data for calculations
        $order_coupon_discount = abs($order->get_discount_total());
        $order_subtotal_before_discount = 0;
        $order_total_cashier_discount = 0;
        
        foreach ($order->get_items() as $order_item) {
            $order_subtotal_before_discount += $order_item->get_subtotal();
            $unit_cashier_discount = floatval($order_item->get_meta('_pos_cashier_discount_per_unit', true));
            $order_total_cashier_discount += ($unit_cashier_discount * $order_item->get_quantity());
        }
        
        $order_subtotal_after_cashier = $order_subtotal_before_discount - $order_total_cashier_discount;
        
        // Calculate cashier discount for refunded items
        if (is_array($refund_items_from_pos)) {
            foreach ($refund_items_from_pos as $refund_item) {
                $item_id = isset($refund_item['original_item_id']) ? absint($refund_item['original_item_id']) : 0;
                if ($item_id > 0) {
                    $item = $order->get_item($item_id);
                    if ($item && $item->is_type('line_item')) {
                        $qty_to_refund = absint($refund_item['quantity']);
                        $unit_cashier_discount = floatval($item->get_meta('_pos_cashier_discount_per_unit', true));
                        $receipt_cashier_discount += ($unit_cashier_discount * $qty_to_refund);
                    }
                }
            }
        }
        
        // Calculate proportional coupon discount
        if ($order_coupon_discount > 0 && $order_subtotal_after_cashier > 0) {
            $refund_subtotal_after_cashier = $receipt_subtotal - $receipt_cashier_discount;
            $refund_ratio = $refund_subtotal_after_cashier / $order_subtotal_after_cashier;
            $receipt_coupon_discount = $order_coupon_discount * $refund_ratio;
        }
        
        // Get coupon info for receipt
        $coupon_codes = $order->get_coupon_codes();
        $coupon_code = !empty($coupon_codes) ? $coupon_codes[0] : '';
        
        // Prepare cart items with correct prices for receipt
        $receipt_cart_items = [];
        if (is_array($refund_items_from_pos)) {
            foreach ($refund_items_from_pos as $refund_item) {
                $item_id = isset($refund_item['original_item_id']) ? absint($refund_item['original_item_id']) : 0;
                if ($item_id > 0) {
                    $item = $order->get_item($item_id);
                    if ($item && $item->is_type('line_item')) {
                        $original_qty = $item->get_quantity();
                        $unit_cashier_discount = floatval($item->get_meta('_pos_cashier_discount_per_unit', true));
                        
                        // Get ORIGINAL price before cashier discount (if it exists in metadata)
                        $unit_original_price = floatval($item->get_meta('_original_price', true));
                        if (!$unit_original_price) {
                            // Fallback: calculate from subtotal + cashier discount
                            $unit_original_price = ($item->get_subtotal() / $original_qty) + $unit_cashier_discount;
                        }
                        
                        // Build receipt item with original prices
                        $receipt_item = $refund_item;
                        $receipt_item['regular_price'] = $unit_original_price; // Original price before cashier discount
                        $receipt_item['price'] = $unit_original_price;
                        $receipt_item['discount_amount'] = $unit_cashier_discount;
                        
                        $receipt_cart_items[] = $receipt_item;
                    }
                }
            }
        }
  
        wp_send_json_success([
            'message' => __('Refund processed successfully.', 'loycart-pos'),
            'is_refund' => true,
            'order_id' => $original_order_id,
            'date_created' => $refund->get_date_created()->format('Y-m-d H:i:s'),
            'subtotal' => wc_format_decimal($receipt_subtotal, $precision),
            'cashier_discount' => wc_format_decimal($receipt_cashier_discount, $precision),
            'coupon_discount' => wc_format_decimal($receipt_coupon_discount, $precision),
            'coupon_code' => $coupon_code,
            'shipping_total' => wc_format_decimal($receipt_shipping_total, $precision),
            'shipping_tax' => wc_format_decimal($receipt_shipping_tax, $precision),
            'total_tax' => wc_format_decimal($receipt_tax_total, $precision),
            'total' => $grand_total_refund_amount * -1, // Use the amount we calculated, not WooCommerce's (which might not include tax yet)
            'payment_method_title' => 'Refunded',
            'cart_items' => $receipt_cart_items
        ]);

    } catch (Exception $e) {
        wp_send_json_error($e->getMessage());
    }
}


function loycart_ajax_get_products() {
    check_ajax_referer('loycart_pos_nonce', 'nonce');
    
    $category_id = isset($_POST['category_id']) && $_POST['category_id'] !== 'all' ? intval(wp_unslash($_POST['category_id'])) : null;
    $search = isset($_POST['search']) ? sanitize_text_field(wp_unslash($_POST['search'])) : '';
    $page = isset($_POST['page']) ? max(1, intval(wp_unslash($_POST['page']))) : 1;
    $per_page = isset($_POST['per_page']) ? max(1, intval(wp_unslash($_POST['per_page']))) : intval(get_option('loycart_pos_items_per_page', 24));
    
    $sort_param = isset($_POST['sort']) ? sanitize_text_field(wp_unslash($_POST['sort'])) : 'popular';
    
    $result = loycart_pos_get_products($category_id, $search, $page, $per_page, $sort_param);

    if ( ! empty( $result['error'] ) ) {
        wp_send_json_error( $result['error'] );
        return;
    }

    wp_send_json([
        'success' => true,
        'data' => $result['products'],
        'pagination' => [
            'total_items' => $result['total_products'],
            'total_pages' => $result['total_pages'],
            'current_page' => $result['current_page'],
            'items_per_page' => $per_page,
            'items_from' => (($page - 1) * $per_page) + 1,
            'items_to' => min($page * $per_page, $result['total_products'])
        ]
    ]);
}


function loycart_ajax_search_customers() { 
    check_ajax_referer('loycart_pos_nonce', 'nonce');
    $query = isset($_POST['query']) ? sanitize_text_field(wp_unslash($_POST['query'])) : '';
    
    $current_user_id = get_current_user_id();
    $args = [
        'role__in' => ['customer', 'subscriber', 'administrator', 'shop_manager'],
        'number'   => 20,
        'orderby'  => 'display_name',
        'order'    => 'ASC',
        // Rationale: Excluding the current operator from search results is intentional UX.
        // The query is limited to 20 users and only available to authorized admins in the POS,
        // so the VIP performance sniff is safe to suppress here.
        // phpcs:ignore WordPressVIPMinimum.Performance.WPQueryParams.PostNotIn_exclude
        'exclude'  => [$current_user_id]
    ];

    if (is_numeric($query) && intval($query) > 0) {
        $args['include'] = [intval($query)];
    } elseif (!empty($query)) {
        $args['search'] = '*' . esc_attr($query) . '*';
        $args['search_columns'] = ['user_login', 'user_email', 'user_nicename', 'display_name'];
        $meta_query = [
            'relation' => 'OR',
            ['key' => 'first_name', 'value' => $query, 'compare' => 'LIKE'],
            ['key' => 'last_name', 'value' => $query, 'compare' => 'LIKE'],
            ['key' => 'billing_phone', 'value' => $query, 'compare' => 'LIKE'],
            ['key' => 'billing_email', 'value' => $query, 'compare' => 'LIKE']
        ];
        if (isset($args['meta_query'])) {
            // Rationale: Meta queries with LIKE can be slow on large datasets, but this search
            // is limited to 20 results and scoped to privileged POS users. We need to search
            // first/last name and billing fields for a practical customer lookup experience.
            // phpcs:ignore WordPress.DB.SlowDBQuery.slow_db_query_meta_query
            $args['meta_query'] = array_merge($args['meta_query'], $meta_query);
        } else {
            // Rationale: See note above regarding controlled scope and necessity of meta queries
            // for customer search across user meta fields in the POS.
            // phpcs:ignore WordPress.DB.SlowDBQuery.slow_db_query_meta_query
            $args['meta_query'] = $meta_query;
        }
    }
    
    $users = get_users($args);
    $customers = [];
    foreach ($users as $user) {
        $customers[] = [
            'id' => $user->ID,
            'name' => $user->display_name,
            'email' => $user->user_email,
            'phone' => get_user_meta($user->ID, 'billing_phone', true),
            'total_orders' => wc_get_customer_order_count($user->ID),
            'total_spent' => wc_price(wc_get_customer_total_spent($user->ID))
        ];
    }
    wp_send_json_success($customers);
}

function loycart_ajax_refresh_nonce() {
    if (!current_user_can('manage_woocommerce')) {
        wp_send_json_error(['message' => __('Unauthorized', 'loycart-pos')], 403);
        return;
    }
    wp_send_json_success(['nonce' => wp_create_nonce('loycart_pos_nonce')]);
}

function loycart_ajax_create_custom_product() {
    check_ajax_referer('loycart_pos_nonce', 'nonce');
    if ( ! current_user_can( 'manage_woocommerce' ) ) {
        wp_send_json_error( __( 'Unauthorized', 'loycart-pos' ), 403 );
        return;
    }

    $name                = isset($_POST['name']) ? sanitize_text_field(wp_unslash($_POST['name'])) : '';
    $regular_price_input = isset($_POST['regular_price']) ? sanitize_text_field(wp_unslash($_POST['regular_price'])) : '';
    $sale_price_input    = isset($_POST['sale_price']) ? sanitize_text_field(wp_unslash($_POST['sale_price'])) : '';
    $sale_from_input     = isset($_POST['sale_from']) ? sanitize_text_field(wp_unslash($_POST['sale_from'])) : '';
    $sale_to_input       = isset($_POST['sale_to']) ? sanitize_text_field(wp_unslash($_POST['sale_to'])) : '';
    $regular_price = '' !== $regular_price_input ? wc_format_decimal($regular_price_input) : '';
    $sale_price    = '' !== $sale_price_input ? wc_format_decimal($sale_price_input) : '';
    $sale_from     = $sale_from_input;
    $sale_to       = $sale_to_input;
    $category_id = isset($_POST['category_id']) ? absint(wp_unslash($_POST['category_id'])) : 0;
    $image_id    = isset($_POST['image_id']) ? absint(wp_unslash($_POST['image_id'])) : 0;
    $tax_status  = isset($_POST['tax_status']) ? sanitize_text_field(wp_unslash($_POST['tax_status'])) : 'none';
    $tax_class   = isset($_POST['tax_class']) ? sanitize_text_field(wp_unslash($_POST['tax_class'])) : '';
    $sku         = isset($_POST['sku']) ? sanitize_text_field(wp_unslash($_POST['sku'])) : '';
    $barcode     = isset($_POST['barcode']) ? sanitize_text_field(wp_unslash($_POST['barcode'])) : '';
    $stock       = isset($_POST['stock']) && $_POST['stock'] !== '' ? absint(wp_unslash($_POST['stock'])) : null;
    $description = isset($_POST['description']) ? sanitize_textarea_field(wp_unslash($_POST['description'])) : '';
    $visibility  = isset($_POST['visibility']) ? sanitize_text_field(wp_unslash($_POST['visibility'])) : 'pos_only';


    if (empty(trim($name))) { wp_send_json_error(__('Item Name is required.', 'loycart-pos')); return; }
    if ('' === $regular_price) { wp_send_json_error(__('A valid Regular Price is required.', 'loycart-pos')); return; }
    if ($regular_price <= 0) { wp_send_json_error(__('The Price must be a positive number.', 'loycart-pos')); return; }
    if ('' !== $sale_price && $sale_price >= $regular_price) { wp_send_json_error(__('Sale Price must be less than Regular Price.', 'loycart-pos')); return; }
    
    try {
        $product = new WC_Product_Simple();
        $product->set_name($name);
        $product->set_regular_price($regular_price);
        
     
        if ('' !== $sale_price) {
            $product->set_sale_price($sale_price);
            $product->set_price($sale_price); 
            if ($sale_from) $product->set_date_on_sale_from(strtotime($sale_from));
            if ($sale_to) $product->set_date_on_sale_to(strtotime($sale_to));
        } else {
            $product->set_price($regular_price);
        }
        
        $product->set_description(!empty($description) ? $description : 'Custom POS Sale Item');
        $product->set_sku(!empty($sku) ? $sku : 'POS-CUSTOM-' . time());

        // Determine visibility settings based on visibility option
        $visibility_meta = 'hidden'; // default
        
        switch ($visibility) {
            case 'public':
                // POS & Store - visible in both
                $product->set_status('publish');
                $visibility_meta = 'visible';
                break;

            case 'store_only':
                // Store only (exclude from POS lists via meta filter)
                $product->set_status('publish');
                $visibility_meta = 'visible';
                break;

            case 'hidden':
                // Hidden from both POS & Store
                $product->set_status('private');
                $visibility_meta = 'hidden';
                break;

            case 'pos_only':
            default:
                // POS only (hidden from storefront catalogs/search)
                $product->set_status('publish');
                $visibility_meta = 'hidden';
                break;
        }
        
        // Set catalog visibility on the product object
        $product->set_catalog_visibility($visibility_meta);

        if (is_null($stock)) {
            $product->set_manage_stock(false); $product->set_virtual(false);
        } else {
            $product->set_manage_stock(true); $product->set_stock_quantity($stock); $product->set_virtual(false);
        }


        if (wc_tax_enabled()) {
            $product->set_tax_status($tax_status); $product->set_tax_class($tax_class);
        }


        if ( $category_id > 0 && term_exists( $category_id, 'product_cat' ) ) {
            $product->set_category_ids([$category_id]);
        } else {
            $category_term = term_exists('pos-custom-products', 'product_cat');
            if (0 === $category_term || null === $category_term) {
                $category_term = wp_insert_term('POS_custom_products', 'product_cat', ['slug' => 'pos-custom-products']);
            }
            if (!is_wp_error($category_term)) {
                $product->set_category_ids([$category_term['term_id']]);
            }
        }
        

        $new_product_id = $product->save();
        if (is_wp_error($new_product_id)) throw new Exception('Could not create product: ' . $new_product_id->get_error_message());

        $new_product = wc_get_product($new_product_id);
        $final_barcode = !empty($barcode) ? $barcode : time();
        update_post_meta($new_product_id, '_global_unique_id', $final_barcode);
        update_post_meta($new_product_id, '_loycart_visibility', $visibility);
        // Explicitly set WooCommerce visibility meta
        update_post_meta($new_product_id, '_visibility', $visibility_meta);

        if ( $image_id > 0 ) { $new_product->set_image_id($image_id); $new_product->save(); }

        $image_url = $new_product->get_image_id() ? wp_get_attachment_image_url($new_product->get_image_id(), 'thumbnail') : wc_placeholder_img_src('thumbnail');

        $response_data = [
            'id' => $new_product->get_id(), 'name' => $new_product->get_name(), 'price' => $new_product->get_price(), 'image' => $image_url, 'is_virtual' => $new_product->is_virtual(),
            'manage_stock' => $new_product->managing_stock(), 'stock_quantity' => $new_product->get_stock_quantity(), 'on_sale' => $new_product->is_on_sale(),
            'regular_price' => $new_product->get_regular_price(), 'sale_price' => $new_product->get_sale_price(),
        ];
        wp_send_json_success($response_data);
    } catch (Exception $e) { wp_send_json_error($e->getMessage()); }
}

function loycart_ajax_get_shipping_options() {
    check_ajax_referer('loycart_pos_nonce', 'nonce');
    // Capability check: restrict shipping option calculation to WooCommerce managers
    if ( ! current_user_can( 'manage_woocommerce' ) ) {
        wp_send_json_error( __( 'Unauthorized', 'loycart-pos' ), 403 );
        return;
    }
    $customer_id = isset($_POST['customer_id']) ? intval(wp_unslash($_POST['customer_id'])) : 0;
    $cart_raw = isset($_POST['cart']) ? sanitize_textarea_field(wp_unslash($_POST['cart'])) : '[]';
    $guest_address = isset($_POST['guest_address']) && is_array($_POST['guest_address'])
        ? array_map('sanitize_text_field', wp_unslash($_POST['guest_address']))
        : null;
    $cart = json_decode($cart_raw, true);

    if (!is_array($cart)) { wp_send_json_error(__('Invalid cart data.', 'loycart-pos')); return; }

    $shipping_address = [];
    $guest_email = null; 
    $guest_phone = null;

    if (is_array($guest_address)) {
        $shipping_address = [
            'country'   => $guest_address['country'] ?? '', 'state' => $guest_address['state'] ?? '',
            'postcode'  => $guest_address['postcode'] ?? '','city' => $guest_address['city'] ?? '',
            'address'   => $guest_address['address_1'] ?? '','address_2' => $guest_address['address_2'] ?? '',
            'first_name' => $guest_address['first_name'] ?? '', 'last_name' => $guest_address['last_name'] ?? '',
        ];
        if (isset($guest_address['email'])) {
            $guest_email = sanitize_email($guest_address['email']); 
        }
        if (isset($guest_address['phone'])) {
            $guest_phone = sanitize_text_field($guest_address['phone']); 
        }

        if (empty($shipping_address['country']) || empty($shipping_address['postcode'])) {
             wp_send_json_error(__('Country and Postcode are required for shipping calculation.', 'loycart-pos'));
             return;
        }
    } elseif ($customer_id > 0) {
        $customer = new WC_Customer($customer_id);
        if (!$customer->get_shipping_country() || !$customer->get_shipping_postcode()) {
            wp_send_json_error(__('The selected customer does not have a complete shipping address.', 'loycart-pos'));
            return;
        }
        $shipping_address = $customer->get_shipping();
        $guest_email = $customer->get_billing_email(); 
        $guest_phone = $customer->get_billing_phone(); 
    } else {
        wp_send_json_error(__('An address must be provided to calculate shipping.', 'loycart-pos'));
        return;
    }

    if (null === WC()->session) {
        WC()->session = new WC_Session_Handler();
        WC()->session->init();
    }
    if (null === WC()->customer) {
        WC()->customer = new WC_Customer(get_current_user_id(), true);
    }
    

    if ($guest_email) {
        WC()->customer->set_billing_email($guest_email);
    }
    if ($guest_phone) {
        WC()->customer->set_billing_phone($guest_phone);
    }

    foreach ($shipping_address as $key => $value) {
        if (is_callable([WC()->customer, "set_shipping_{$key}"])) {
            WC()->customer->{"set_shipping_{$key}"}($value);
        }
    }
    
    WC()->cart->empty_cart();

    if (!empty($cart)) {
        foreach ($cart as $item) {
            $product_id = intval($item['id']);
            $quantity = intval($item['quantity']);
            $variation_id = !empty($item['variation_id']) ? intval($item['variation_id']) : 0;
            $variation_attributes = !empty($item['attributes']) ? $item['attributes'] : [];
            
            $cart_item_key = WC()->cart->add_to_cart($product_id, $quantity, $variation_id, $variation_attributes);

            if ($cart_item_key && !empty($item['discount_percent'])) {
                $cart_item = WC()->cart->get_cart_item($cart_item_key);
                $original_price = $cart_item['data']->get_price('edit');
                $discounted_price = $original_price * (1 - (floatval($item['discount_percent']) / 100));
                $cart_item['data']->set_price($discounted_price);
            }
        }
    }

    $packages = WC()->cart->get_shipping_packages();
    
 
    WC()->shipping->calculate_shipping($packages);

    $rated_packages = WC()->shipping->get_packages();
    
    $shipping_methods = [];


    if (empty($rated_packages) || empty($rated_packages[0]['rates'])) {
        WC()->cart->empty_cart();
        wp_send_json_error(__('No shipping options found for this address. Check WooCommerce shipping zone setup.', 'loycart-pos'));
        return; 
    }

    
    if (!empty($rated_packages)) {
        foreach ($rated_packages as $package) {
            if (!empty($package['rates'])) {
                foreach ($package['rates'] as $rate) {
                    $shipping_methods[] = [
                        'id'    => $rate->get_id(),
                        'label' => $rate->get_label(),
                        'cost'  => wc_format_decimal($rate->get_cost()),
                    ];
                }
            }
        }
    }
    
    WC()->cart->empty_cart();

    wp_send_json_success($shipping_methods);
}


function loycart_ajax_complete_sale() {
    // Security check
    check_ajax_referer('loycart_pos_nonce', 'nonce');
    if ( ! current_user_can( 'manage_woocommerce' ) ) {
        wp_send_json_error( __( 'Unauthorized', 'loycart-pos' ), 403 );
        return;
    }

    $cart_raw = isset($_POST['cart']) ? sanitize_textarea_field(wp_unslash($_POST['cart'])) : '';
    $cart = json_decode($cart_raw, true);
    if (!is_array($cart)) {
        wp_send_json_error(__('Invalid cart data.', 'loycart-pos'));
        return;
    }

    $payments_raw = isset($_POST['payments']) ? sanitize_textarea_field(wp_unslash($_POST['payments'])) : '[]';
    $payments = json_decode($payments_raw, true);
    if (!is_array($payments) || empty($payments)) {
        wp_send_json_error(__('No payment information received.', 'loycart-pos'));
        return;
    }

    // Stock check loop
    foreach ($cart as $item) {
        $product_id = isset($item['variation_id']) && $item['variation_id'] ? $item['variation_id'] : $item['id'];
        $product = wc_get_product($product_id);
        if ($product && $product->managing_stock()) {
            if (!$product->has_enough_stock($item['quantity'])) {
                /* translators: %s: Product name */
                wp_send_json_error( sprintf( __( 'Not enough stock for %s.', 'loycart-pos' ), $product->get_name() ) ); 
                return;
            }
        }
    }

    $customer_id = isset($_POST['customer_id']) ? intval(wp_unslash($_POST['customer_id'])) : 0;
    $shipping_method_id = isset($_POST['shipping_method_id']) ? sanitize_text_field(wp_unslash($_POST['shipping_method_id'])) : null;
    $shipping_cost = isset($_POST['shipping_cost']) ? floatval(wp_unslash($_POST['shipping_cost'])) : null;
    $shipping_label = isset($_POST['shipping_label']) ? sanitize_text_field(wp_unslash($_POST['shipping_label'])) : 'Shipping';
    $guest_address = isset($_POST['guest_address']) && is_array($_POST['guest_address'])
        ? array_map('sanitize_text_field', wp_unslash($_POST['guest_address']))
        : null;
        
    $order_note = isset($_POST['order_note']) ? sanitize_textarea_field(wp_unslash($_POST['order_note'])) : '';

    if (empty($cart) && $shipping_method_id === null) {
        wp_send_json_error(__('Cannot process an empty order.', 'loycart-pos'));
        return;
    }
    
    try {
        $order = wc_create_order(['status' => 'pending']);
        
        // Customer and address logic
        $customer = null;
        if ($customer_id > 0) {
            $customer = new WC_Customer($customer_id);
            if ($customer && $customer->get_id() > 0) {
                $order->set_customer_id($customer_id);
                $order->set_address($customer->get_billing(), 'billing');
            }
        } elseif ($guest_address !== null) {
            $order->set_address($guest_address, 'billing');
        }
        
        $is_local_pickup = false;
        if ($shipping_method_id !== null) {
            $final_shipping_address_data = [];
            if (strpos($shipping_method_id, 'local_pickup') !== false) {
                $is_local_pickup = true;
                $final_shipping_address_data = [
                    'address_1' => get_option('woocommerce_store_address'),
                    'address_2' => get_option('woocommerce_store_address_2'),
                    'city'      => get_option('woocommerce_store_city'),
                    'postcode'  => get_option('woocommerce_store_postcode'),
                    'country'   => get_option('woocommerce_default_country'),
                ];
            } else {
                if ($guest_address !== null) {
                    $final_shipping_address_data = $guest_address;
                } elseif ($customer) { 
                    $final_shipping_address_data = $customer->get_shipping();
                }
            }
            $order->set_address($final_shipping_address_data, 'shipping');
        }

        $first_payment_method_id = $payments[0]['method'];
        $first_payment_method_title = $payments[0]['title'];
        $payment_method_title_for_order = (count($payments) > 1) ? 'POS - Split Payment' : 'POS - ' . $first_payment_method_title;
        $order->set_payment_method($first_payment_method_id);
        $order->set_payment_method_title($payment_method_title_for_order);

        $total_manual_cashier_discount = 0; 

        if (!empty($cart)) {
            foreach ($cart as $item) {
                $product_to_add = wc_get_product(isset($item['variation_id']) && $item['variation_id'] ? $item['variation_id'] : $item['id']);
                
                if ($product_to_add) {
                    $qty = intval($item['quantity']);
                    
                    $fixed_discount_per_unit = floatval($item['discount_amount'] ?? 0);
                    $original_unit_price = wc_get_price_excluding_tax($product_to_add, array('qty' => 1));
                    $net_unit_price = max(0, $original_unit_price - $fixed_discount_per_unit);

                    $total_manual_cashier_discount += ($fixed_discount_per_unit * $qty);

                    $order_item_id = $order->add_product($product_to_add, $qty, array(
                        'subtotal' => $net_unit_price * $qty, 
                        'total'    => $net_unit_price * $qty,
                    ));

                    $order_item = $order->get_item($order_item_id);

                    // 2. Add Audit Meta for line item transparency
                    if ($fixed_discount_per_unit > 0) {
                        // Store meta for receipt generation
                        $order_item->add_meta_data('_cashier_discount_amount', $fixed_discount_per_unit, true);
                        $order_item->add_meta_data('_original_price', $original_unit_price, true);
                        // Store display meta for order admin
                        $order_item->add_meta_data('_pos_cashier_discount_per_unit', $fixed_discount_per_unit, true);
                        $order_item->add_meta_data(__('POS Cashier Discount', 'loycart-pos'), '-' . wc_price($fixed_discount_per_unit), false);
                        $order_item->add_meta_data(__('Original Price', 'loycart-pos'), wc_price($original_unit_price), false);
                    }

                    // 3. Finalize the item
                    $order_item->set_subtotal($net_unit_price * $qty);
                    $order_item->set_total($net_unit_price * $qty);
                    $order_item->save();

                    if (isset($item['is_custom']) && $item['is_custom']) {
                        wc_add_order_item_meta($order_item_id, '_pos_custom_item_name', $item['custom_name']);
                    }
                }
            }
        }

        $coupon_code = isset($_POST['applied_coupon']) ? sanitize_text_field(wp_unslash($_POST['applied_coupon'])) : '';
        if (!empty($coupon_code)) {
            $order->apply_coupon($coupon_code);
            /* translators: %s: Coupon Code */
            $order->add_order_note( sprintf( __( 'Coupon "%s" applied via POS.', 'loycart-pos' ), $coupon_code ) );
        }

       
        if ($shipping_method_id !== null && $shipping_cost !== null) {
            $shipping_item = new WC_Order_Item_Shipping();
            $shipping_item->set_method_title($shipping_label);
            $rate_id_parts = explode(':', $shipping_method_id);
            $shipping_item->set_method_id($rate_id_parts[0]);
            $shipping_item->set_total($shipping_cost);
            $order->add_item($shipping_item);
        }


        $order->calculate_totals(true);
        $order->update_meta_data('_pos_order', 'yes');
        $order->update_meta_data('_pos_user_id', get_current_user_id());
        $order->update_meta_data('_pos_is_local_pickup', $is_local_pickup ? 'yes' : 'no');
 
        // Tender and status logic
        $total_tendered = 0;
        $all_pending = true;
        foreach ($payments as $payment) {
            $amount = floatval($payment['amount']);
            $title = sanitize_text_field($payment['title']);
            $method_id = sanitize_text_field($payment['method']);
            /* translators: 1: Amount, 2: Payment method title */
            $order->add_order_note( sprintf(__('POS Payment: %1$s received via %2$s.', 'loycart-pos'), wc_price($amount), $title) );
            if ($method_id === 'cash' || $method_id === 'card') {
                $total_tendered += $amount;
                $all_pending = false; 
            }
        }

        $order_status_to_set = $all_pending ? 'on-hold' : 'completed';
        $order_id = $order->save(); 
        if ($order_id) {
            $order->update_status($order_status_to_set);
        }
        

        if (!empty($order_note)) {
            $order->add_order_note($order_note);
        }
        
        $final_total = $order->get_total();
        $change_due = max(0, $total_tendered - $final_total);


        if ($total_manual_cashier_discount > 0) {
            $order->add_order_note(
                /* translators: %s: total amount of manual price overrides applied on this order (formatted currency). */
                sprintf(__('Audit: Cashier performed manual price overrides totaling %s on this order.', 'loycart-pos'), wc_price($total_manual_cashier_discount)),
                false 
            );
        }

        $order->save();

        $response_data = [
            'order_id'       => $order_id,
            'status'         => $order->get_status(),
            'total'          => $final_total,
            'amount_tendered'=> $total_tendered,
            'change'         => $change_due, 
            'subtotal'       => $order->get_subtotal(),
            'discount_total' => $order->get_discount_total(),
            'total_tax'      => $order->get_total_tax(),
            'date_created'   => $order->get_date_created()->format('Y-m-d H:i:s'),
        ];
        
        wp_send_json_success($response_data);

    } catch (Exception $e) {
        /* translators: %s: The specific error message returned when completing a sale fails. */
        wp_send_json_error(sprintf(__('Error completing sale: %s', 'loycart-pos'), $e->getMessage()));
    }
}



function loycart_ajax_hold_sale() {
    check_ajax_referer('loycart_pos_nonce', 'nonce');
    $cart_data = isset($_POST['cart']) ? sanitize_textarea_field(wp_unslash($_POST['cart'])) : '';
    $customer_id = isset($_POST['customer_id']) ? intval(wp_unslash($_POST['customer_id'])) : 0;
    $cart_total = isset($_POST['cart_total']) ? sanitize_text_field(wp_unslash($_POST['cart_total'])) : wc_price(0);
    $held_name = isset($_POST['held_name']) ? sanitize_text_field(wp_unslash($_POST['held_name'])) : '';
    if (empty($cart_data) || $cart_data === '[]') { wp_send_json_error(__('Cannot hold an empty cart.', 'loycart-pos')); return; }
    $decoded_cart = json_decode($cart_data, true);
    if (json_last_error() !== JSON_ERROR_NONE || !is_array($decoded_cart)) { wp_send_json_error(__('Invalid cart data format for holding.', 'loycart-pos')); return; }
    $current_user = wp_get_current_user();
    $operator_name = $current_user->exists() ? $current_user->display_name : 'Unknown Operator';
    $customer_name_for_title = 'Guest';
    if ($customer_id > 0) { $customer_user = get_user_by('id', $customer_id); if ($customer_user) { $customer_name_for_title = $customer_user->display_name; } }
    /* translators: 1: Customer name, 2: Operator name, 3: Cart total */
    $title = !empty($held_name) ? $held_name : sprintf(__('Held: %1$s by %2$s (%3$s)', 'loycart-pos'), $customer_name_for_title, $operator_name, $cart_total);
    $held_cart_post_id = wp_insert_post(['post_type' => 'loycart_held_cart', 'post_title' => $title, 'post_status' => 'publish', 'post_author' => ($current_user->exists() ? $current_user->ID : 0)]);
    if (is_wp_error($held_cart_post_id)) { /* translators: %s: Error message when holding a sale fails. */ wp_send_json_error(sprintf(__('Failed to hold sale: %s', 'loycart-pos'), $held_cart_post_id->get_error_message())); return; }
    update_post_meta($held_cart_post_id, '_loycart_held_cart_data', $cart_data);
    update_post_meta($held_cart_post_id, '_loycart_held_customer_id', $customer_id);
    update_post_meta($held_cart_post_id, '_loycart_held_total', $cart_total);
    update_post_meta($held_cart_post_id, '_loycart_held_operator_id', ($current_user->exists() ? $current_user->ID : 0));
    update_post_meta($held_cart_post_id, '_loycart_held_operator_name', $operator_name);
    update_post_meta($held_cart_post_id, '_loycart_held_name', $held_name);
    wp_send_json_success(['message' => __('Sale held successfully.', 'loycart-pos'), 'held_cart_id' => $held_cart_post_id, 'title' => $title]);
}
function loycart_ajax_get_held_carts() {
    check_ajax_referer('loycart_pos_nonce', 'nonce');
    $args = ['post_type' => 'loycart_held_cart', 'post_status' => 'publish', 'posts_per_page' => -1, 'orderby' => 'date', 'order' => 'DESC'];
    $held_cart_posts = get_posts($args);
    $held_carts = [];
    foreach ($held_cart_posts as $post) {
        $customer_id = get_post_meta($post->ID, '_loycart_held_customer_id', true);
        $customer_name = 'Guest';
        if ($customer_id > 0) { $customer_user = get_user_by('id', $customer_id); if ($customer_user) { $customer_name = $customer_user->display_name; } }
        $held_carts[] = ['id' => $post->ID, 'title' => get_the_title($post->ID), 'held_name' => get_post_meta($post->ID, '_loycart_held_name', true), 'customer_id' => intval($customer_id), 'customer_name' => $customer_name, 'operator_name' => get_post_meta($post->ID, '_loycart_held_operator_name', true) ?: 'N/A', 'total' => get_post_meta($post->ID, '_loycart_held_total', true), 'date_held' => get_the_date('Y-m-d H:i', $post->ID),];
    }
    wp_send_json_success($held_carts);
}


function loycart_ajax_resume_held_cart() {
    check_ajax_referer('loycart_pos_nonce', 'nonce');
    $held_cart_id = isset($_POST['held_cart_id']) ? intval(wp_unslash($_POST['held_cart_id'])) : 0;

    if (!$held_cart_id || get_post_type($held_cart_id) !== 'loycart_held_cart') {
        wp_send_json_error(__('Invalid held cart ID.', 'loycart-pos'));
        return;
    }
    
    if (!current_user_can('manage_woocommerce')) {
        wp_send_json_error(__('Permission denied to resume held cart.', 'loycart-pos'));
        return;
    }


    $cart_data_json = get_post_meta($held_cart_id, '_loycart_held_cart_data', true);
    $customer_id = get_post_meta($held_cart_id, '_loycart_held_customer_id', true);
    
    $stale_cart_array = [];
    if (!empty($cart_data_json)) {

        $decoded_data = json_decode($cart_data_json, true);
        if (json_last_error() === JSON_ERROR_NONE && is_array($decoded_data)) {
            $stale_cart_array = $decoded_data;
        }
    }

    $validation_results = loycart_pos_validate_stale_cart_data($stale_cart_array);


    wp_delete_post($held_cart_id, true);

    wp_send_json_success([
        'cart'        => $validation_results['cart'],
        'warnings'    => $validation_results['warnings'], 
        'customer_id' => intval($customer_id)
    ]);
}

function loycart_ajax_delete_held_cart() {
    check_ajax_referer('loycart_pos_nonce', 'nonce');
    $held_cart_id = isset($_POST['held_cart_id']) ? intval(wp_unslash($_POST['held_cart_id'])) : 0;
    if (!$held_cart_id || get_post_type($held_cart_id) !== 'loycart_held_cart') {
        wp_send_json_error(__('Invalid held cart ID.', 'loycart-pos')); return;
    }
    if (!current_user_can('manage_woocommerce')) {
        wp_send_json_error(__('Permission denied to delete held cart.', 'loycart-pos')); return;
    }
    $deleted = wp_delete_post($held_cart_id, true);
    if (is_wp_error($deleted) || !$deleted) {
        wp_send_json_error(__('Failed to delete held cart.', 'loycart-pos')); return;
    }
    wp_send_json_success(['message' => __('Held cart deleted successfully.', 'loycart-pos')]);
}


function loycart_ajax_calculate_cart_totals() {
    check_ajax_referer('loycart_pos_nonce', 'nonce');

    if ( function_exists( 'WC' ) && WC()->session ) {
        remove_action( 'shutdown', array( WC()->session, 'write_session' ), 99 );
        remove_action( 'wp_loaded', 'wc_setup_cart', 10 );
    }

    if (!isset($_POST['cart'])) { wp_send_json_error(__('Invalid cart data.', 'loycart-pos')); return; }

    $cart_data = json_decode(sanitize_textarea_field(wp_unslash($_POST['cart'])), true);
    $customer_id = isset($_POST['customer_id']) ? intval(wp_unslash($_POST['customer_id'])) : 0;
    $precision = wc_get_price_decimals();
    $original_order_id = isset($_POST['original_order_id']) ? absint(wp_unslash($_POST['original_order_id'])) : 0;
    
    $coupon_code = isset($_POST['coupon_code']) ? sanitize_text_field(wp_unslash($_POST['coupon_code'])) : '';

    if ($customer_id > 0) { 
        WC()->customer = new WC_Customer($customer_id); 
    } else { 
        WC()->customer->set_billing_location(WC()->countries->get_base_country(), '', '', ''); 
    }

    // --- 1. REFUND LOGIC ---
    // Treat as refund whenever original_order_id is provided. This ensures shipping-only refunds
    // (no line items, only selectedShipping) are handled by the refund pathway.
    $is_refund = ($original_order_id > 0);

    if ($is_refund) {
        $order = wc_get_order($original_order_id);
        if (!$order) {
            wp_send_json_error(__('Order not found.', 'loycart-pos'));
            return;
        }

        // Get order-level discounts
        $order_cashier_discount = abs(floatval($order->get_meta('_cashier_discount_amount', true)));
        $order_coupon_discount = abs($order->get_discount_total());

        // Calculate refund totals from selected items
        $refund_subtotal = 0;
        $refund_tax = 0;
        $refund_shipping = 0;
        $refund_cashier_discount = 0;
        $refund_coupon_discount = 0;
        
        // Get all order items for reference
        $order_items = $order->get_items();
        $order_subtotal_before_discount = 0;
        $order_subtotal_after_cashier = 0;
        $order_total_cashier_discount = 0;
        
        foreach ($order_items as $order_item) {
            $item_subtotal = $order_item->get_subtotal(); // Before any discount
            $item_qty = $order_item->get_quantity();
            $unit_cashier_discount = floatval($order_item->get_meta('_pos_cashier_discount_per_unit', true));
            
            $order_subtotal_before_discount += $item_subtotal;
            $order_total_cashier_discount += ($unit_cashier_discount * $item_qty);
        }
        
        // Calculate order subtotal after cashier discounts (for coupon ratio calculation)
        $order_subtotal_after_cashier = $order_subtotal_before_discount - $order_total_cashier_discount;

        // Process cart items (refund selection)
        foreach ($cart_data as $cart_item) {
            $qty_refunding = abs((float)$cart_item['quantity']);
            $original_item_id = $cart_item['original_item_id'];
            
            // Find matching order item
            $order_item = isset($order_items[$original_item_id]) ? $order_items[$original_item_id] : null;
            
            if ($order_item) {
                // Get cashier discount from metadata (if it exists)
                $unit_cashier_discount_meta = floatval($order_item->get_meta('_pos_cashier_discount_per_unit', true));
                
                // Calculate per-unit prices from order item
                $unit_subtotal = $order_item->get_subtotal() / $order_item->get_quantity(); // Before ANY discounts
                $unit_tax = $order_item->get_total_tax() / $order_item->get_quantity();
                
                // Add to refund totals
                $refund_subtotal += ($unit_subtotal * $qty_refunding);
                $refund_tax += ($unit_tax * $qty_refunding);
                $refund_cashier_discount += ($unit_cashier_discount_meta * $qty_refunding);
            }
        }

        // Calculate subtotal after cashier discounts (for coupon calculation)
        $refund_subtotal_after_cashier = $refund_subtotal - $refund_cashier_discount;

        // Calculate proportional coupon discount if applicable
        if ($order_coupon_discount > 0 && $order_subtotal_after_cashier > 0) {
            $refund_ratio = $refund_subtotal_after_cashier / $order_subtotal_after_cashier;
            $refund_coupon_discount = $order_coupon_discount * $refund_ratio;
        }

        // Handle shipping refund
        $refund_shipping = 0;
        $refund_shipping_tax = 0;
        $shipping_method_id = isset($_POST['shipping_method_id']) ? sanitize_text_field(wp_unslash($_POST['shipping_method_id'])) : null;
        $shipping_cost = isset($_POST['shipping_cost']) ? floatval(wp_unslash($_POST['shipping_cost'])) : 0;
        
        if ($shipping_cost > 0) {
            $refund_shipping = abs($shipping_cost);
            // Get shipping tax from original order
            $order_shipping_tax = $order->get_shipping_tax();
            $refund_shipping_tax = abs($order_shipping_tax);
            // Add shipping tax to total tax
            $refund_tax += $refund_shipping_tax;
        }
        
        // Calculate final refund total
        // Note: refund_subtotal already excludes cashier discounts (they were applied when order was created)
        // Only subtract the coupon discount which is calculated proportionally
        $refund_total = $refund_subtotal - $refund_coupon_discount + $refund_tax + $refund_shipping;

        wp_send_json_success([
            'subtotal'       => wc_format_decimal(abs($refund_subtotal), $precision),
            'total_tax'      => wc_format_decimal(abs($refund_tax), $precision),
            'shipping_total' => wc_format_decimal(abs($refund_shipping), $precision),
            'shipping_tax'   => wc_format_decimal(abs($refund_shipping_tax), $precision),
            'cashier_discount' => wc_format_decimal(abs($refund_cashier_discount), $precision),
            'coupon_discount' => wc_format_decimal(abs($refund_coupon_discount), $precision),
            'total'          => wc_format_decimal(abs($refund_total), $precision),
            'gross_subtotal_before_discount' => wc_format_decimal(abs($refund_subtotal), $precision)
        ]);
        return;
    }

 
    $order = wc_create_order();
    if ($customer_id > 0) {
        $order->set_customer_id($customer_id);
        // Ensure billing context is present so email-restricted coupons validate
        try {
            $tmp_customer = new WC_Customer($customer_id);
            if ($tmp_customer && is_a($tmp_customer, 'WC_Customer')) {
                $order->set_address($tmp_customer->get_billing(), 'billing');
            }
        } catch (Exception $e) {
            // No-op: fallback to WC()->customer context set above
        }
    }

    $gross_subtotal_before_discount = 0;
    $manual_discount_total = 0;

    foreach ($cart_data as $item) {
    $product_id = isset($item['variation_id']) && $item['variation_id'] ? $item['variation_id'] : $item['id'];
    $product = wc_get_product($product_id);
    
    if ($product) {
        $qty = intval($item['quantity']);
        $fixed_discount = floatval($item['discount_amount'] ?? 0);
        $original_price = wc_get_price_excluding_tax($product, ['qty' => 1]);
        
        $net_price = max(0, $original_price - $fixed_discount);

        $item_id = $order->add_product($product, $qty, [
            'subtotal' => $net_price * $qty, 
            'total'    => $net_price * $qty 
        ]);
        
        $order_item = $order->get_item($item_id);
        $order_item->set_subtotal($net_price * $qty);
        $order_item->set_total($net_price * $qty);
        
        // Store cashier discount meta for receipt generation
        if ($fixed_discount > 0) {
            $order_item->add_meta_data('_cashier_discount_amount', $fixed_discount, true);
            $order_item->add_meta_data('_original_price', $original_price, true);
            $order_item->add_meta_data('_pos_cashier_discount_per_unit', $fixed_discount, true);
        }

        $order_item->save();
        
        $manual_discount_total += ($fixed_discount * $qty);
        $gross_subtotal_before_discount += ($original_price * $qty);
    }
}

if (!empty($coupon_code)) {
    $order->apply_coupon($coupon_code);
}

    $shipping_method_id = isset($_POST['shipping_method_id']) ? sanitize_text_field(wp_unslash($_POST['shipping_method_id'])) : null;
    $is_local_pickup = false;
    if ($shipping_method_id) {
        $shipping_cost = isset($_POST['shipping_cost']) ? floatval(wp_unslash($_POST['shipping_cost']) ) : 0;
        if (strpos($shipping_method_id, 'local_pickup') !== false) { $is_local_pickup = true; }
        $shipping_item = new WC_Order_Item_Shipping();
        $shipping_item->set_method_id($shipping_method_id);
        $shipping_item->set_total($shipping_cost);
        $order->add_item($shipping_item);
    }
    
    $order->calculate_totals(true);

    $coupon_discount_total = $order->get_discount_total(); 


    $response_data = [
        'subtotal'         => wc_format_decimal($order->get_subtotal(), $precision), 
        'total'            => wc_format_decimal($order->get_total(), $precision),
        'total_tax'        => wc_format_decimal($order->get_total_tax(), $precision), 
        'shipping_total'   => wc_format_decimal($order->get_shipping_total(), $precision),
        'cashier_discount' => wc_format_decimal($manual_discount_total, $precision),
        'coupon_discount'  => wc_format_decimal($coupon_discount_total, $precision),
        'gross_subtotal_before_discount' => wc_format_decimal($gross_subtotal_before_discount, $precision),
        'applied_coupons'    => $order->get_coupon_codes(),
        'shipping_method_id' => $shipping_method_id,
        'is_local_pickup'    => $is_local_pickup,
    ];


    remove_action( 'before_delete_post', 'woocommerce_delete_order_item_data' );
    $order->delete(true); 

    wp_send_json_success($response_data);
}

function loycart_ajax_get_customer_orders() {
    check_ajax_referer('loycart_pos_nonce', 'nonce');
    $cid = isset($_POST['customer_id']) ? absint(wp_unslash($_POST['customer_id'])) : 0;
    

    $search_query   = isset($_POST['search_query']) ? sanitize_text_field(wp_unslash($_POST['search_query'])) : '';
    $status_filter  = isset($_POST['status_filter']) ? sanitize_text_field(wp_unslash($_POST['status_filter'])) : 'all';
    $date_filter    = isset($_POST['date_filter']) ? sanitize_text_field(wp_unslash($_POST['date_filter'])) : ''; 
 
    $page = isset($_POST['page']) ? max(1, intval(wp_unslash($_POST['page']))) : 1;
    $orders_per_page = 6;

    if ($cid <= 0) {
        wp_send_json_success(['orders' => [], 'pagination' => ['current_page' => 1, 'total_pages' => 0]]);
        return;
    }
    

    $args = [
        'customer_id'    => $cid,
        'limit'          => $orders_per_page,
        'paged'          => $page,
        'orderby'        => 'date',
        'order'          => 'DESC',
        'status'         => ['completed', 'processing', 'on-hold', 'refunded'], 
        'return'         => 'objects', 
        // Rationale: Meta query scaffolding is initialized here and optionally populated below.
        // Although meta queries can be slower, this endpoint paginates to 6 orders and is used
        // by authorized staff only, making the suppression acceptable and localized.
        // phpcs:ignore WordPress.DB.SlowDBQuery.slow_db_query_meta_query
        'meta_query'     => [],
        'date_query'     => [], 
    ];
    

    if (!empty($search_query)) {
        $args['s'] = $search_query;
    }
    

    if (!empty($date_filter) && preg_match('/^\d{4}-\d{2}-\d{2}$/', $date_filter)) {
        $args['date_query'][] = [
            'after'     => $date_filter . ' 00:00:00',
            'before'    => $date_filter . ' 23:59:59',
            'inclusive' => true,
            'column'    => 'post_date_gmt', 
        ];
    }
    

    switch ($status_filter) {
    case 'not_refunded':
         $args['meta_query'][] = [
            'key' => '_refund_amount',
            'value' => 0,
            'compare' => '=',
            'type' => 'NUMERIC'
        ];
        break;
        
    case 'fully_refunded':
        $args['status'] = 'refunded';
        break;
    case 'partially_refunded':
         $args['meta_query'][] = [
            'key' => '_refund_amount',
            'value' => 0,
            'compare' => '>',
            'type' => 'NUMERIC'
        ];
        break;
    case 'all':
    default:

        $args['meta_query'][] = [
            'relation' => 'OR',
            [
                'key' => '_refund_amount',
                'compare' => 'EXISTS', 
            ],
            [
                'key' => '_refund_amount',
                'compare' => 'NOT EXISTS',
            ]
        ];
        break;
}


    $orders_query = wc_get_orders($args);
    $total_orders = $orders_query->total;
    
    $data = [];
    foreach ($orders_query as $order) {
        $status = $order->get_status();
        $total_refunded = (float) $order->get_total_refunded();
        $order_total_raw = (float) $order->get_total();


        if ($status_filter === 'partially_refunded') {

            if (!($total_refunded > 0 && $total_refunded < $order_total_raw)) {
                continue;
            }
        }
        

        if (in_array($status, ['failed', 'cancelled']) && $total_refunded <= 0) {
            continue;
        }
        
        $data[] = [
            'id'             => $order->get_id(),
            'date_created'   => $order->get_date_created()->format('Y-m-d'),
            'total'          => $order->get_formatted_order_total(), 
            'item_count'     => count($order->get_items()),
            'total_raw'      => wc_format_decimal($order_total_raw, 2),
            'total_refunded' => wc_format_decimal($total_refunded, 2),
            'status'         => $status,
        ];
    }
    

    $total_pages = ceil($total_orders / $orders_per_page);
    
    wp_send_json_success([
        'orders' => $data,
        'pagination' => [
            'current_page' => $page,
            'total_pages'  => $total_pages > 0 ? $total_pages : 1,
            'total_items'  => $total_orders
        ]
    ]);
}

function loycart_ajax_get_refundable_order_items() {
    check_ajax_referer('loycart_pos_nonce', 'nonce');
    $oid = isset($_POST['order_id']) ? absint(wp_unslash($_POST['order_id'])) : 0;
    $order = wc_get_order($oid);
    if (!$order) { 
        wp_send_json_error(__('Order not found.', 'loycart-pos')); 
        return; 
    }
    
    $items = [];
    $order_total_shipping = $order->get_shipping_total();
    $refunds = $order->get_refunds();

    $previous_refunds = [];
    foreach ($refunds as $refund) {
        $user_id = $refund->get_meta('_pos_user_id', true);
        $user = $user_id ? get_user_by('id', $user_id) : null;
        $previous_refunds[] = [
            'id'     => $refund->get_id(),
            'amount' => abs($refund->get_amount()),
            'date'   => $refund->get_date_created()->date('d M Y'),
            'reason' => $refund->get_reason(),
            'user'   => $user ? $user->display_name : 'System'
        ];
    }
    
    foreach ($order->get_items(['line_item', 'shipping']) as $id => $item) {
        if ($item->is_type('line_item')) {
            $qty_refunded = (int) wc_get_order_item_meta($id, '_qty_refunded', true);
            $qty_refundable = $item->get_quantity() - $qty_refunded;


$item_refund_date = '';
$cashier_name = 'Unknown';

if ($qty_refunded > 0) {
    $latest_timestamp = 0;
    $latest_refund = null;

    foreach ($order->get_refunds() as $refund) {
        $refund_time = $refund->get_date_created()->getTimestamp();

        foreach ($refund->get_items() as $refund_item) {
            if ($refund_item->get_meta('_refunded_item_id') == $id) {
                if ($refund_time > $latest_timestamp) {
                    $latest_timestamp = $refund_time;
                    $latest_refund = $refund;
                }
            }
        }
    }

    if ($latest_refund) {
        $item_refund_date = $latest_refund->get_date_created()->date('d M Y');

        $user_id = $latest_refund->get_meta('_pos_user_id', true);
        if ($user_id) {
            $user = get_userdata($user_id);
            if ($user) {
                $first = trim(get_user_meta($user_id, 'first_name', true));
                $last  = trim(get_user_meta($user_id, 'last_name', true));

                if (!empty($first) || !empty($last)) {
                    $cashier_name = trim($first . ' ' . $last);
                } else {
                    $cashier_name = $user->display_name; // fallback to display_name (often username or custom)
                }
            }
        }
    }
}

            $product = $item->get_product();
            $image_url = $product 
                ? wp_get_attachment_image_url($product->get_image_id(), 'thumbnail') 
                : wc_placeholder_img_src('thumbnail');

            $items[] = [
                'original_item_id' => $id,
                'name'             => $item->get_name(),
                'quantity'         => $item->get_quantity(),
                'qty_refundable'   => $qty_refundable,
                'refund_date'      => $item_refund_date,
                'price_paid'       => wc_format_decimal($item->get_total() / $item->get_quantity()),
                // Pre-discount, pre-tax unit price (needed to mirror receipt discount exactly)
                'original_price'   => wc_format_decimal($item->get_subtotal() / $item->get_quantity()),
                'tax_paid'         => wc_format_decimal($item->get_total_tax() / $item->get_quantity()),
                'image'            => $image_url,
                'cashier_name'     => $cashier_name,
                'id'               => $item->get_product_id(),
                'variation_id'     => $item->get_variation_id()
            ];
        }
        
   elseif ($item->is_type('shipping') && $order_total_shipping > 0) {
    $shipping_total_refunded = abs($order->get_total_shipping_refunded());
    $shipping_refundable_amount = $order_total_shipping - $shipping_total_refunded;

    $is_shipping_refunded = ($shipping_refundable_amount <= 0);

    $items[] = [
        'original_item_id'    => $id,
        'name'                => 'Shipping',
        'quantity'            => 1,
        'qty_refundable'      => $is_shipping_refunded ? 0 : 1,
        'price_paid'          => $is_shipping_refunded ? $order_total_shipping : $shipping_refundable_amount,
        'is_shipping_refund'  => true,
        'tax_paid'            => $item->get_total_tax(),
        'image'               => '',
        'refund_date'         => '', 
        'status_text'         => $is_shipping_refunded ? 'Shipping Fully Refunded' : 'Refundable Shipping'
    ];
  }
 }


    // Get shipping information for refund cart display
    $shipping_items = $order->get_items('shipping');
    $shipping_method = '';
    $shipping_total = 0;
    foreach ($shipping_items as $shipping_item) {
        $shipping_method = $shipping_item->get_name();
        $shipping_total = $shipping_item->get_total();
        break; // Take first shipping method
    }

    // Get coupon information
    $coupons = $order->get_coupon_codes();
    $coupon_code = !empty($coupons) ? $coupons[0] : '';
    $coupon_discount = abs($order->get_discount_total());

    // Get cashier discount (stored in order meta)
    $cashier_discount = abs(floatval($order->get_meta('_cashier_discount_amount', true)));

    wp_send_json_success([
        'items' => $items,
        'order_summary' => [
            'total'            => $order->get_total(),
            'total_refunded'   => $order->get_total_refunded(),
            'date'             => $order->get_date_created() ? $order->get_date_created()->date('Y-m-d H:i:s') : '',
            'refund_history'   => $previous_refunds,
            'cashier'          => wp_get_current_user()->display_name,
            'payment_method'   => $order->get_payment_method_title(),
            'coupon_code'      => $coupon_code,
            'coupon_discount'  => $coupon_discount,
            'cashier_discount' => $cashier_discount,
            'shipping_method'  => $shipping_method,
            'shipping_total'   => $shipping_total
        ]
    ]);
}


function loycart_ajax_get_customer_address() {
    check_ajax_referer('loycart_pos_nonce', 'nonce');
    $cid = isset($_POST['customer_id']) ? absint(wp_unslash($_POST['customer_id'])) : 0;
    
    if ($cid <= 0) {
        wp_send_json_error(__('Invalid customer ID.', 'loycart-pos'));
        return;
    }
    
    $c = new WC_Customer($cid);
    

    wp_send_json_success([
        'shipping' => $c->get_shipping(),
        'billing_email' => $c->get_billing_email(),
        'billing_phone' => $c->get_billing_phone(),
    ]);
}

function loycart_ajax_save_cart_state() {
    check_ajax_referer('loycart_pos_nonce', 'nonce');
    if (!current_user_can('manage_woocommerce')) { wp_send_json_error(__('Denied.', 'loycart-pos'), 403); return; }
    

    
    if ( function_exists( 'WC' ) && WC()->session ) {

        remove_action( 'shutdown', array( WC()->session, 'write_session' ), 99 );
        

        remove_action( 'wp_loaded', array( WC()->cart, 'maybe_set_cart_cookies' ), 10 );
    }
    

    
    $raw = isset($_POST['cart_state']) ? sanitize_textarea_field(wp_unslash($_POST['cart_state'])) : '';
    

    if (!empty($raw) && $raw !== '[]') {
        update_user_meta(get_current_user_id(), '_loycart_pos_active_cart', $raw);
    } else {

        delete_user_meta(get_current_user_id(), '_loycart_pos_active_cart');
    }
    
    wp_send_json_success();
}

function loycart_ajax_load_cart_state() {
    check_ajax_referer('loycart_pos_nonce', 'nonce');
    if (!current_user_can('manage_woocommerce')) { wp_send_json_error(__('Denied.', 'loycart-pos'), 403); return; }
    $uid = get_current_user_id();
    $json = get_user_meta($uid, '_loycart_pos_active_cart', true);
    if (!is_string($json) || empty($json)) { wp_send_json_success(['data' => null]); return; }
    $arr = json_decode($json, true);
    if (json_last_error() !== JSON_ERROR_NONE) { delete_user_meta($uid, '_loycart_pos_active_cart'); wp_send_json_success(['data' => null]); return; }
    wp_send_json_success(['data' => $arr]);
}

function loycart_ajax_clear_cart_state() {
    check_ajax_referer('loycart_pos_nonce', 'nonce');
    if (!current_user_can('manage_woocommerce')) { wp_send_json_error(__('Denied.', 'loycart-pos'), 403); return; }
    delete_user_meta(get_current_user_id(), '_loycart_pos_active_cart');
    wp_send_json_success();
}

function loycart_ajax_create_customer() {
    check_ajax_referer('loycart_pos_nonce', 'nonce');
    

    $email = isset($_POST['email']) ? sanitize_email(wp_unslash($_POST['email'])) : '';
    $fname = isset($_POST['first_name']) ? sanitize_text_field(wp_unslash($_POST['first_name'])) : '';
    $lname = isset($_POST['last_name']) ? sanitize_text_field(wp_unslash($_POST['last_name'])) : '';
    
    if(email_exists($email)) { wp_send_json_error(__('Exists', 'loycart-pos')); return; }
    

    $uid = wp_create_user($email, wp_generate_password(), $email);

    $display_name = !empty($fname) ? sanitize_text_field($fname) . ' POS-' . $uid : 'Customer-' . $uid;
    wp_update_user([
        'ID'           => $uid, 
        'first_name'   => $fname, 
        'last_name'    => $lname, 
        'nickname'     => $display_name, 
        'display_name' => $display_name,
        'role'         => 'customer'
    ]);
    

    $address_keys = [
        'first_name', 'last_name', 'email', 'phone', 
        'address_1', 'address_2', 'city', 'postcode', 
        'country', 'state'
    ];
    
    foreach ($address_keys as $key) {
        if (isset($_POST[$key])) {
            $value = sanitize_text_field(wp_unslash($_POST[$key]));
            

            update_user_meta($uid, 'billing_' . $key, $value);


            update_user_meta($uid, 'shipping_' . $key, $value);
        }
    }


    wp_send_json_success(['id'=>$uid, 'name'=>$display_name]);
}


function loycart_ajax_validate_cart() {
    check_ajax_referer('loycart_pos_nonce', 'nonce');
    
    $cart_raw = isset($_POST['cart']) ? sanitize_textarea_field(wp_unslash($_POST['cart'])) : '[]';
    $stale_cart = json_decode($cart_raw, true);


    $validation_results = loycart_pos_validate_stale_cart_data($stale_cart);
    
    wp_send_json_success($validation_results);
}




function loycart_pos_validate_stale_cart_data($stale_cart) {
    if (!is_array($stale_cart) || empty($stale_cart)) {
        return ['cart' => [], 'warnings' => []];
    }

    $fresh_cart = [];
    $warnings = [];
    $precision = wc_get_price_decimals();

    foreach ($stale_cart as $item) {
        $product_id = isset($item['variation_id']) && $item['variation_id'] ? $item['variation_id'] : $item['id'];
        $product = wc_get_product($product_id);

        if (!$product || !in_array($product->get_status(), array('publish', 'private'))) {
            /* translators: %s: Item name */
            $warnings[] = sprintf(__('Item "%s" is no longer available and has been removed.', 'loycart-pos'), $item['name']);
            continue;
        }

        $item['name'] = $product->get_name(); 

        $saved_quantity = absint($item['quantity']);
        if ($product->managing_stock()) {
            $max_stock = $product->get_stock_quantity();
            if ($saved_quantity > $max_stock) {
                if ($max_stock <= 0) {
                    /* translators: %s: Product name */
                    $warnings[] = sprintf(__('"%s" is now out of stock and has been removed.', 'loycart-pos'), $product->get_name());
                    $item['quantity'] = 0;
                } else {
                    /* translators: 1: Item name, 2: New quantity, 3: Old quantity */
                    $warnings[] = sprintf(__('Stock for "%1$s" is low. Quantity reduced from %3$s to %2$s.', 'loycart-pos'), $product->get_name(), $max_stock, $saved_quantity);
                    $item['quantity'] = $max_stock;
                }
            }
        }
        
        if($item['quantity'] <= 0) {
            continue;
        }

        $fresh_price = wc_get_price_to_display($product);
        $stale_price = floatval($item['price']);
        
        if (abs($stale_price - $fresh_price) > (1 / pow(10, $precision))) {
            // translators: 1: Product name, 2: Old price, 3: New price.
            $warnings[] = sprintf(__('Price for "%1$s" updated from %2$s to %3$s.', 'loycart-pos'), $product->get_name(), wc_price($stale_price), wc_price($fresh_price));
            $item['price'] = $fresh_price;
        }

        $item['regular_price'] = wc_get_price_to_display($product, ['price' => $product->get_regular_price()]);
        $item['sale_price'] = wc_get_price_to_display($product, ['price' => $product->get_sale_price()]);
        $item['on_sale'] = $product->is_on_sale();
        
        $fresh_cart[] = $item;
    }
    
    return ['cart' => $fresh_cart, 'warnings' => $warnings];
}



function loycart_ajax_get_label_data() {
    check_ajax_referer('loycart_pos_nonce', 'nonce');
    
    if (!current_user_can('manage_woocommerce')) {
        wp_send_json_error(__('Permission denied.', 'loycart-pos'), 403);
        return;
    }

    $order_id = isset($_POST['order_id']) ? absint(wp_unslash($_POST['order_id'])) : 0;
    $order = wc_get_order($order_id);

    if (!$order) {
        wp_send_json_error(__('Order not found.', 'loycart-pos'));
        return;
    }


    $data = [
        'order_number'   => $order->get_order_number(),
        'customer_name'  => $order->get_shipping_first_name() . ' ' . $order->get_shipping_last_name(),
        'order_id'       => $order_id,
        'label_type'     => 'shipping', // Hardcoded as shipping
        'date'           => $order->get_date_created()->format('Y-m-d'),
        'customer_id'    => $order->get_customer_id(),
        'customer_phone' => $order->get_billing_phone(),
        'label_address'  => $order->get_formatted_shipping_address(),
        'label_title'    => __('Ship To Address', 'loycart-pos'),
        'barcode_id'     => $order->get_order_number(),
    ];

    $package_details = loycart_pos_calculate_order_package_details($order);
    $data['weight']     = $package_details['weight']; 
    $data['dimensions'] = $package_details['dimensions'];

    wp_send_json_success($data);
}


function loycart_ajax_open_cash_drawer() {
    check_ajax_referer('loycart_pos_nonce', 'nonce');
    if (!current_user_can('manage_woocommerce')) {
        wp_send_json_error(__('Permission denied.', 'loycart-pos'), 403);
        return;
    }
    
    wp_send_json_success([
        'message' => __('Cash drawer signal sent.', 'loycart-pos'),
        'command' => 'kick_drawer'
    ]);
}
add_action('wp_ajax_loycart_ajax_open_cash_drawer', 'loycart_ajax_open_cash_drawer');


function loycart_ajax_get_available_coupons() {
    check_ajax_referer('loycart_pos_nonce', 'nonce');
    
    if (!current_user_can('manage_woocommerce')) {
        wp_send_json_error(__('Unauthorized', 'loycart-pos'), 403);
        return;
    }

    $customer_id = isset($_POST['customer_id']) ? absint(wp_unslash($_POST['customer_id'])) : 0;

    $args = [
        'post_type'      => 'shop_coupon',
        'post_status'    => 'publish',
        'posts_per_page' => -1,
        'orderby'        => 'title',
        'order'          => 'ASC'
    ];

    $coupon_posts = get_posts($args);
    $data = [];


    foreach ($coupon_posts as $post) {
        $coupon = new WC_Coupon($post->ID);
        $expiry_date = $coupon->get_date_expires();
        if ($expiry_date && current_time('timestamp') > $expiry_date->getTimestamp()) {
            continue;
        }

        // Filter store credit coupons: show only for the selected customer, otherwise hide them entirely
        $is_store_credit = get_post_meta($post->ID, '_loycart_store_credit', true) === 'yes';
        if ($is_store_credit) {
            if ($customer_id <= 0) {
                // No customer selected: do not list any store credit coupons
                continue;
            }
            $owner_id = absint(get_post_meta($post->ID, '_loycart_customer_id', true));
            if ($owner_id !== $customer_id) {
                // Different customer's credit: hide from list
                continue;
            }
            // Hide if value is 0.00
            $amount = floatval($coupon->get_amount());
            if ($amount <= 0.0) {
                continue;
            }
        }

        $type_label = str_replace('_', ' ', $coupon->get_discount_type());

        $data[] = [
            'id'          => $coupon->get_id(),
            'code'        => strtoupper($coupon->get_code()),
            'description' => $coupon->get_description() ?: __('No description provided.', 'loycart-pos'),
            'type'        => $coupon->get_discount_type(),
            'type_label'  => ucfirst($type_label),
            'amount'      => $coupon->get_amount(),
        ];
    }

    wp_send_json_success($data);
}
add_action('wp_ajax_loycart_ajax_get_available_coupons', 'loycart_ajax_get_available_coupons');


// Store Credit: return customer coupon code and balance
function loycart_ajax_get_store_credit() {
    check_ajax_referer('loycart_pos_nonce', 'nonce');

    if (!current_user_can('manage_woocommerce')) {
        wp_send_json_error(__('Unauthorized', 'loycart-pos'), 403);
        return;
    }

    $cid = isset($_POST['customer_id']) ? absint(wp_unslash($_POST['customer_id'])) : 0;
    if ($cid <= 0) {
        wp_send_json_error(__('Invalid customer ID.', 'loycart-pos'));
        return;
    }

    // Ensure store credit module is available
    if (!function_exists('loycart_pos_get_store_credit_settings')) {
        wp_send_json_error(__('Store credit module not available.', 'loycart-pos'));
        return;
    }

    $settings = loycart_pos_get_store_credit_settings();
    if (!$settings['enabled']) {
        // Still return deterministic code with zero amount
        $code = function_exists('loycart_pos_credit_coupon_code') ? loycart_pos_credit_coupon_code($cid) : '';
        wp_send_json_success(['code' => $code, 'amount' => 0]);
        return;
    }

    // Create or fetch the coupon and return its current amount
    $coupon_id = loycart_pos_get_or_create_credit_coupon($cid);
    if (!$coupon_id) {
        wp_send_json_error(__('Unable to read or create credit coupon.', 'loycart-pos'));
        return;
    }

    $code = get_the_title($coupon_id);
    $amount = floatval(get_post_meta($coupon_id, 'coupon_amount', true));
    wp_send_json_success(['code' => strtoupper($code), 'amount' => $amount]);
}
add_action('wp_ajax_loycart_pos_get_store_credit', 'loycart_ajax_get_store_credit');




add_action('wp_ajax_loycart_get_sale_data', 'loycart_get_sale_data');

// =========================
// POS Settings (AJAX): Get & Save
// Register globally so the actions are always available
// =========================
function loycart_pos_get_settings_ajax() {
    check_ajax_referer('loycart_pos_nonce', 'nonce');
    if (!current_user_can('manage_woocommerce')) {
        wp_send_json_error(__('Unauthorized', 'loycart-pos'), 403);
        return;
    }
    if (!function_exists('loycart_pos_get_store_credit_settings')) {
        wp_send_json_error(__('Settings module unavailable.', 'loycart-pos'));
        return;
    }
    // Explicitly load every field from the database, do not rely on helper defaults
    $settings = [
        'store_credit' => [
            'enabled' => get_option('loycart_pos_credit_enabled', 'no') === 'yes',
            'earn_percent' => (float) get_option('loycart_pos_credit_earn_percent', '0'),
            'apply_online' => get_option('loycart_pos_credit_apply_online', 'no') === 'yes',
            'exclude_sale_items_earning' => get_option('loycart_pos_credit_exclude_sale_items_earning', 'no') === 'yes',
            'exclude_sale_items_redeeming' => get_option('loycart_pos_credit_exclude_sale_items_redeeming', 'no') === 'yes',
        ],
        'receipt' => [
            'store_name'     => get_option('woocommerce_pos_store_name', get_bloginfo('name')),
            'store_phone'    => get_option('woocommerce_pos_store_phone', ''),
            'footer_message' => get_option('woocommerce_pos_receipt_footer', __('Thank you for your purchase!', 'loycart-pos')),
            'paper_width'    => get_option('loycart_pos_receipt_paper_width', '80mm'),
            'usb_direct_print' => get_option('loycart_pos_enable_usb_direct_print', 'no') === 'yes',
        ],
        'ui' => [
            'default_theme'   => get_option('loycart_pos_default_theme', 'light'),
            'show_theme_toggle'   => get_option('loycart_pos_show_theme_toggle', 'yes') === 'yes',
            'show_notifications'  => get_option('loycart_pos_show_notifications', 'no') === 'yes',
            'products_per_page' => intval(get_option('loycart_pos_products_per_page', 24)),
            'default_sort' => get_option('loycart_pos_default_sort', 'popular'),
            'auto_redirect_pos' => get_option('loycart_pos_auto_redirect_pos', 'no') === 'yes',
        ],
        'visibility' => [
            'allow_private_detail_lookups' => get_option('loycart_pos_allow_private_detail_lookups', 'yes') === 'yes',
        ],
    ];
    // Add explicit logging for debugging
    if (defined('WP_DEBUG') && WP_DEBUG) {
        error_log('LoyCart POS settings loaded: ' . print_r($settings, true));
    }
    wp_send_json_success($settings);
}
add_action('wp_ajax_loycart_pos_get_settings', 'loycart_pos_get_settings_ajax');

function loycart_pos_save_settings_ajax() {
    check_ajax_referer('loycart_pos_nonce', 'nonce');
    if (!current_user_can('manage_woocommerce')) {
        wp_send_json_error(__('Unauthorized', 'loycart-pos'), 403);
        return;
    }
    // Accept posted values under store_credit, receipt, ui, visibility keys
    // Deep sanitize incoming arrays from POST
    $sanitize_deep = function($value) use (&$sanitize_deep) {
        if (is_array($value)) {
            $out = [];
            foreach ($value as $k => $v) {
                $out[is_string($k) ? sanitize_key($k) : $k] = $sanitize_deep($v);
            }
            return $out;
        }
        if (is_string($value)) {
            return sanitize_text_field($value);
        }
        return $value;
    };

    $raw_sc = filter_input(INPUT_POST, 'store_credit', FILTER_DEFAULT, FILTER_REQUIRE_ARRAY);
    $raw_rc = filter_input(INPUT_POST, 'receipt', FILTER_DEFAULT, FILTER_REQUIRE_ARRAY);
    $raw_ui = filter_input(INPUT_POST, 'ui', FILTER_DEFAULT, FILTER_REQUIRE_ARRAY);
    $raw_vs = filter_input(INPUT_POST, 'visibility', FILTER_DEFAULT, FILTER_REQUIRE_ARRAY);

    $sc = is_array($raw_sc) ? $sanitize_deep( wp_unslash( $raw_sc ) ) : [];
    $rc = is_array($raw_rc) ? $sanitize_deep( wp_unslash( $raw_rc ) ) : [];
    $ui = is_array($raw_ui) ? $sanitize_deep( wp_unslash( $raw_ui ) ) : [];
    $vs = is_array($raw_vs) ? $sanitize_deep( wp_unslash( $raw_vs ) ) : [];

    // Always set all options, even if missing from payload (default to false/empty)
    // Store Credit
    update_option('loycart_pos_credit_enabled', isset($sc['enabled']) && $sc['enabled'] ? 'yes' : 'no');
    update_option('loycart_pos_credit_apply_online', isset($sc['apply_online']) && $sc['apply_online'] ? 'yes' : 'no');
    update_option('loycart_pos_credit_exclude_sale_items_earning', isset($sc['exclude_sale_items_earning']) && $sc['exclude_sale_items_earning'] ? 'yes' : 'no');
    update_option('loycart_pos_credit_exclude_sale_items_redeeming', isset($sc['exclude_sale_items_redeeming']) && $sc['exclude_sale_items_redeeming'] ? 'yes' : 'no');
    update_option('loycart_pos_credit_earn_percent', isset($sc['earn_percent']) ? (string)max(0, floatval($sc['earn_percent'])) : '0');

    // Receipt
    update_option('woocommerce_pos_store_name', isset($rc['store_name']) ? sanitize_text_field($rc['store_name']) : '');
    update_option('woocommerce_pos_store_phone', isset($rc['store_phone']) ? sanitize_text_field($rc['store_phone']) : '');
    update_option('woocommerce_pos_receipt_footer', isset($rc['footer_message']) ? sanitize_textarea_field($rc['footer_message']) : '');
    update_option('loycart_pos_receipt_paper_width', isset($rc['paper_width']) ? (in_array($rc['paper_width'], ['80mm','58mm','A4','Letter'], true) ? $rc['paper_width'] : '80mm') : '80mm');
    update_option('loycart_pos_enable_usb_direct_print', isset($rc['usb_direct_print']) && $rc['usb_direct_print'] ? 'yes' : 'no');

    // UI
    update_option('loycart_pos_default_theme', isset($ui['default_theme']) && $ui['default_theme'] === 'dark' ? 'dark' : 'light');
    update_option('loycart_pos_show_theme_toggle', isset($ui['show_theme_toggle']) && $ui['show_theme_toggle'] ? 'yes' : 'no');
    update_option('loycart_pos_show_notifications', isset($ui['show_notifications']) && $ui['show_notifications'] ? 'yes' : 'no');
    update_option('loycart_pos_auto_redirect_pos', isset($ui['auto_redirect_pos']) && $ui['auto_redirect_pos'] ? 'yes' : 'no');
    update_option('loycart_pos_products_per_page', isset($ui['products_per_page']) ? max(1, min(200, intval($ui['products_per_page']))) : 24);
    update_option('loycart_pos_default_sort', isset($ui['default_sort']) && in_array($ui['default_sort'], ['popular','name_asc','name_desc','price_asc','price_desc','sale','newest'], true) ? $ui['default_sort'] : 'popular');

    // Visibility
    update_option('loycart_pos_allow_private_detail_lookups', isset($vs['allow_private_detail_lookups']) && $vs['allow_private_detail_lookups'] ? 'yes' : 'no');

    // Coupon meta update logic (unchanged)
    $get_bool = function($key) use ($sc) {
        if (!isset($sc[$key])) return 'no';
        $val = $sc[$key];
        if (is_string($val)) { $val = strtolower($val); }
        return in_array($val, ['1','true','yes','on'], true) ? 'yes' : 'no';
    };
    $enabled     = $get_bool('enabled');
    $applyOnline = $get_bool('apply_online');
    $excEarn     = $get_bool('exclude_sale_items_earning');
    $excRedeem   = $get_bool('exclude_sale_items_redeeming');
    $earnPercent = isset($sc['earn_percent']) ? sanitize_text_field($sc['earn_percent']) : '0';
    $earnPercent = (string) max(0, floatval($earnPercent));

    // Persist Store Credit
    update_option('loycart_pos_credit_enabled', $enabled);
    update_option('loycart_pos_credit_apply_online', $applyOnline);
    update_option('loycart_pos_credit_exclude_sale_items_earning', $excEarn);
    update_option('loycart_pos_credit_exclude_sale_items_redeeming', $excRedeem);
    update_option('loycart_pos_credit_earn_percent', $earnPercent);

        // If store credit is disabled, unpublish all store credit coupons
        if ($enabled === 'no' && function_exists('loycart_pos_unpublish_store_credit_coupons')) {
            loycart_pos_unpublish_store_credit_coupons();
        }
        // If store credit is enabled, publish all store credit coupons
        if ($enabled === 'yes' && function_exists('loycart_pos_publish_store_credit_coupons')) {
            loycart_pos_publish_store_credit_coupons();
        }
    // If redeeming policy changed, align existing coupons meta
    if (function_exists('get_posts')) {
        $coupon_posts = get_posts([
            'post_type'      => 'shop_coupon',
            'post_status'    => 'publish',
            'posts_per_page' => -1,
            // phpcs:disable WordPress.DB.SlowDBQuery.slow_db_query_meta_key,WordPress.DB.SlowDBQuery.slow_db_query_meta_value
            'meta_key'       => '_loycart_store_credit',
            'meta_value'     => 'yes',
            // phpcs:enable WordPress.DB.SlowDBQuery.slow_db_query_meta_key,WordPress.DB.SlowDBQuery.slow_db_query_meta_value
            'fields'         => 'ids'
        ]);
        $flag = ($excRedeem === 'yes') ? 'yes' : 'no';
        foreach ($coupon_posts as $cid) {
            update_post_meta($cid, 'exclude_sale_items', $flag);
        }
    }

    // Persist Receipt options
    if (!empty($rc)) {
        if (isset($rc['store_name']))     update_option('woocommerce_pos_store_name', sanitize_text_field($rc['store_name']));
        if (isset($rc['store_phone']))    update_option('woocommerce_pos_store_phone', sanitize_text_field($rc['store_phone']));
        if (isset($rc['footer_message'])) update_option('woocommerce_pos_receipt_footer', sanitize_textarea_field($rc['footer_message']));
        if (isset($rc['paper_width']))    update_option('loycart_pos_receipt_paper_width', in_array($rc['paper_width'], ['80mm','58mm','A4','Letter'], true) ? $rc['paper_width'] : '80mm');
        if (isset($rc['usb_direct_print'])) {
            $usb = in_array(strtolower((string)$rc['usb_direct_print']), ['1','true','yes','on'], true) ? 'yes' : 'no';
            update_option('loycart_pos_enable_usb_direct_print', $usb);
        }
    }

    // Persist UI defaults/toggles
    if (!empty($ui)) {
        $theme   = isset($ui['default_theme']) ? sanitize_text_field($ui['default_theme']) : 'light';
        $showTheme   = isset($ui['show_theme_toggle']) && in_array(strtolower((string)$ui['show_theme_toggle']), ['1','true','yes','on'], true) ? 'yes' : 'no';
        $showNotifications = isset($ui['show_notifications']) && in_array(strtolower((string)$ui['show_notifications']), ['1','true','yes','on'], true) ? 'yes' : 'no';
            $autoRedirect = isset($ui['auto_redirect_pos']) && in_array(strtolower((string)$ui['auto_redirect_pos']), ['1','true','yes','on'], true) ? 'yes' : 'no';
        update_option('loycart_pos_default_theme', $theme === 'dark' ? 'dark' : 'light');
        update_option('loycart_pos_show_theme_toggle', $showTheme);
        update_option('loycart_pos_show_notifications', $showNotifications);
            update_option('loycart_pos_auto_redirect_pos', $autoRedirect);
        // Save products_per_page if present
        if (isset($ui['products_per_page'])) {
            $ppp = max(1, min(200, intval($ui['products_per_page'])));
            update_option('loycart_pos_products_per_page', $ppp);
        }
        // Save default_sort if present
        if (isset($ui['default_sort'])) {
            $allowed_sorts = ['popular','name_asc','name_desc','price_asc','price_desc','sale','newest'];
            $sort = in_array($ui['default_sort'], $allowed_sorts, true) ? $ui['default_sort'] : 'popular';
            update_option('loycart_pos_default_sort', $sort);
        }
    }

    // Persist visibility controls
    if (!empty($vs)) {
        $allow = isset($vs['allow_private_detail_lookups']) && in_array(strtolower((string)$vs['allow_private_detail_lookups']), ['1','true','yes','on'], true) ? 'yes' : 'no';
        update_option('loycart_pos_allow_private_detail_lookups', $allow);
    }

    wp_send_json_success(['message' => __('Settings saved.', 'loycart-pos')]);
}
add_action('wp_ajax_loycart_pos_save_settings', 'loycart_pos_save_settings_ajax');

function loycart_get_sale_data() {
    check_ajax_referer('loycart_pos_nonce', 'nonce');

    $order_id = isset($_POST['order_id']) ? absint($_POST['order_id']) : 0;
    $for_refund = isset($_POST['for_refund']) && in_array(strtolower((string)$_POST['for_refund']), ['true', '1', 'yes', 'on']);

    $order = wc_get_order($order_id);
    
    if (!$order) {
        wp_send_json_error(['message' => __('Order not found.', 'loycart-pos')]);
        return;
    }

    // =============================================
    // 1. Basic order information (same for both)
    // =============================================
    $response = [
        'order_id'             => $order->get_id(),
        'order_number'         => $order->get_order_number(),
        'date_created'         => $order->get_date_created()->date('d/m/Y, H:i:s'),
        'shipping_address'     => $order->get_formatted_shipping_address(),
        'payment_method_title' => $order->get_payment_method_title(),
        'is_refund'            => ($order->get_type() === 'shop_order_refund'),
    ];

    // =============================================
    // 2. Coupon information
    // =============================================
    $applied_coupons = [];
    foreach ($order->get_coupon_codes() as $code) {
        $applied_coupons[] = [
            'code'   => strtoupper($code),
            'amount' => (float) $order->get_total_discount(),
        ];
    }
    $response['coupons'] = $applied_coupons;

    // =============================================
    // 3. Prepare totals - different logic for refund vs normal
    // =============================================
    if ($for_refund) {
    $refund_subtotal    = 0;
    $refund_tax         = 0;
    $refund_shipping    = 0;
    $original_subtotal  = 0;

    // Items
    foreach ($order->get_items() as $item) {
        $qty = (float) $item->get_quantity();
        if ($qty <= 0) continue;

        $line_subtotal_original = (float) $item->get_subtotal();
        $line_total_discounted  = (float) $item->get_total();

        $original_subtotal += $line_subtotal_original;
        $refund_subtotal += $line_total_discounted;

        // Tax on items (already in order, use discounted portion)
        $refund_tax += (float) $item->get_total_tax();
    }

    // Shipping
    $refund_shipping = (float) $order->get_shipping_total();

    // Shipping tax - get from order shipping items or fallback to order total tax minus item tax
    $shipping_tax = 0;
    foreach ($order->get_items('shipping') as $shipping_item) {
        $shipping_tax += (float) $shipping_item->get_total_tax();
    }

    if ($shipping_tax == 0) {
        // Fallback: if no shipping items, use total tax minus items tax
        $total_tax_in_order = (float) $order->get_total_tax();
        $items_tax = 0;
        foreach ($order->get_items() as $item) {
            $items_tax += (float) $item->get_total_tax();
        }
        $shipping_tax = $total_tax_in_order - $items_tax;
    }

    $refund_tax += $shipping_tax;

    // Build totals (subtotal = items only, shipping separate, VAT on both)
    $formatted_totals = [
        ['label' => __('Subtotal', 'loycart-pos'), 'value' => wc_price($refund_subtotal), 'key' => 'subtotal'],
        ['label' => __('Shipping', 'loycart-pos'), 'value' => wc_price($refund_shipping), 'key' => 'shipping_total'],
        ['label' => __('VAT', 'loycart-pos'), 'value' => wc_price($refund_tax), 'key' => 'total_tax'],
        ['label' => '<strong>' . __('Total Refund', 'loycart-pos') . '</strong>',
         'value' => wc_price($refund_subtotal + $refund_shipping + $refund_tax),
         'key' => 'order_total']
    ];

    $response['totals']          = $formatted_totals;
    $response['subtotal']        = $refund_subtotal;        // items only (matches regular receipts)
    $response['total_tax']       = $refund_tax;             // items + shipping tax
    $response['shipping_total']  = $refund_shipping;
    $response['gross_subtotal']  = $original_subtotal;
    $response['total']           = -($refund_subtotal + $refund_shipping + $refund_tax);
}// negative for refund

    else {
        // ── NORMAL ORDER / ORIGINAL RECEIPT ──
        // Build custom totals array to show cashier discounts separately
        $formatted_totals = [];
        
        // Calculate gross subtotal (before any discounts) and cashier discounts
        $gross_subtotal = 0;
        $cashier_discount_total = 0;
        $items_total_after_discounts = 0;
        
        foreach ($order->get_items() as $item) {
            $item_subtotal = (float) $item->get_subtotal();
            $item_total = (float) $item->get_total();
            $item_qty = abs((float) $item->get_quantity());
            
            $gross_subtotal += $item_subtotal;
            $items_total_after_discounts += $item_total;
            
            // Calculate cashier discount from subtotal vs total difference
            // This includes any line item discounts applied (excluding coupon which is order-level)
            $line_discount = $item_subtotal - $item_total;
            
            // Check if this is a cashier discount vs sale price
            $discount_amount = (float) $item->get_meta('_cashier_discount_amount', true);
            $discount_percent = (float) $item->get_meta('_cashier_discount_percent', true);
            
            // If we have cashier discount meta, use that to identify cashier discounts
            if ($discount_amount > 0) {
                // Direct discount amount per unit
                $cashier_discount_total += $discount_amount * $item_qty;
            } elseif ($discount_percent > 0) {
                // Percentage discount - calculate from original price
                $original_price = (float) $item->get_meta('_original_price', true);
                if (!$original_price) {
                    // Fallback to regular price from product
                    $product = $item->get_product();
                    if ($product) {
                        $original_price = (float) $product->get_regular_price();
                    }
                }
                if ($original_price > 0) {
                    $discount_per_unit = $original_price * ($discount_percent / 100);
                    $cashier_discount_total += $discount_per_unit * $item_qty;
                }
            }
        }
        
        // If we still don't have cashier discounts from meta, calculate from line discounts
        // This is a fallback for cases where meta wasn't stored properly
        if ($cashier_discount_total < 0.01) {
            $total_line_discounts = $gross_subtotal - $items_total_after_discounts;
            $coupon_discount = (float) $order->get_total_discount();
            
            // If there are line discounts beyond the coupon, those are cashier discounts
            if ($total_line_discounts > $coupon_discount + 0.01) {
                $cashier_discount_total = $total_line_discounts - $coupon_discount;
            }
        }
        
        // Build totals array
        $formatted_totals[] = [
            'label' => __('Subtotal', 'loycart-pos'),
            'value' => wc_price($gross_subtotal),
            'key'   => 'cart_subtotal'
        ];
        
        // Add cashier discount line if present
        if ($cashier_discount_total > 0.01) {
            $formatted_totals[] = [
                'label' => __('Cashier Discounts', 'loycart-pos'),
                'value' => '-' . wc_price($cashier_discount_total),
                'key'   => 'cashier_discount'
            ];
        }
        
        // Add coupon line if present - format as "Coupon (CODE):"
        if ($order->get_total_discount() > 0) {
            $coupon_discount = $order->get_total_discount();
            $coupon_label = __('Coupon', 'loycart-pos');
            
            // Get coupon code and format as "Coupon (CODE):"
            $coupons = $order->get_coupon_codes();
            if (!empty($coupons)) {
                $coupon_code = strtoupper($coupons[0]); // Use first coupon, uppercase
                // translators: %s is the coupon code.
                $coupon_label = sprintf(__('Coupon (%s)', 'loycart-pos'), $coupon_code);
            }
            
            $formatted_totals[] = [
                'label' => $coupon_label,
                'value' => '-' . wc_price($coupon_discount),
                'key'   => 'order_discount'
            ];
        }
        
        // Add shipping line if present
        if ($order->get_shipping_total() > 0) {
            $shipping_method = '';
            foreach ($order->get_items('shipping') as $shipping_item) {
                $shipping_method = $shipping_item->get_name();
                break;
            }
            
            $formatted_totals[] = [
                'label' => __('Shipping', 'loycart-pos') . ($shipping_method ? ': ' . $shipping_method : ''),
                'value' => wc_price($order->get_shipping_total()),
                'key'   => 'shipping'
            ];
        }
        
        // Add tax line
        $tax_label = WC()->countries->tax_or_vat();
        $formatted_totals[] = [
            'label' => $tax_label,
            'value' => wc_price($order->get_total_tax()),
            'key'   => 'order_tax'
        ];
        
        // Add total line
        $formatted_totals[] = [
            'label' => '<strong>' . __('Total', 'loycart-pos') . '</strong>',
            'value' => '<strong>' . wc_price($order->get_total()) . '</strong>',
            'key'   => 'order_total'
        ];

        $response['totals']         = $formatted_totals;
        $response['gross_subtotal'] = $gross_subtotal;
        $response['subtotal']       = (float) $order->get_subtotal();
        $response['total_tax']      = (float) $order->get_total_tax();
        $response['shipping_total'] = (float) $order->get_shipping_total();
        $response['total']          = (float) $order->get_total();
        $response['cashier_discount_total'] = $cashier_discount_total;
    }

    wp_send_json_success($response);
    }


    // Store Credit Reports: return summary HTML for POS modal
    function loycart_ajax_get_credit_reports() {
        check_ajax_referer('loycart_pos_nonce', 'nonce');

        // Restrict to WooCommerce managers to view aggregated reports
        if (!current_user_can('manage_woocommerce')) {
            wp_send_json_error(__('Unauthorized', 'loycart-pos'), 403);
            return;
        }

        // Validate and sanitize date inputs (YYYY-MM-DD)
        $from_raw = isset($_POST['from']) ? sanitize_text_field( wp_unslash($_POST['from']) ) : null;
        $to_raw   = isset($_POST['to'])   ? sanitize_text_field( wp_unslash($_POST['to']) )   : null;
        // If arrays are provided (e.g., from[]), coerce to first scalar string value
        if (is_array($from_raw)) { $from_raw = count($from_raw) ? reset($from_raw) : null; }
        if (is_array($to_raw))   { $to_raw   = count($to_raw)   ? reset($to_raw)   : null; }
        $from = is_string($from_raw) ? $from_raw : gmdate('Y-m-d', time() - 30 * DAY_IN_SECONDS);
        $to   = is_string($to_raw)   ? $to_raw   : gmdate('Y-m-d');

        $is_valid_date = function($d) { return is_string($d) && preg_match('/^\d{4}-\d{2}-\d{2}$/', $d); };
        if (!$is_valid_date($from)) { $from = ''; }
        if (!$is_valid_date($to))   { $to   = ''; }

        $date_args = [];
        if ($from) { $date_args['after']  = $from . ' 00:00:00'; }
        if ($to)   { $date_args['before'] = $to   . ' 23:59:59'; }
        if (isset($date_args['after']) || isset($date_args['before'])) { $date_args['inclusive'] = true; }

        $order_query_args = [
            'limit'    => -1,
            'orderby'  => 'date',
            'order'    => 'ASC',
            'return'   => 'objects',
            'status'   => ['completed', 'processing', 'on-hold', 'refunded'],
        ];
        if (!empty($date_args)) {
            $order_query_args['date_created'] = $date_args;
        }

        try {
            $orders = wc_get_orders($order_query_args);
        } catch (\Throwable $t) {
            $fallback_args = $order_query_args;
            unset($fallback_args['date_created']);
            $orders = wc_get_orders($fallback_args);
        }

        $total_earned = 0.0;
        $total_redeemed = 0.0;
        foreach ($orders as $order) {
            $total_earned   += floatval($order->get_meta('_loycart_credit_earned'));
            $total_redeemed += floatval($order->get_meta('_loycart_credit_redeemed'));
        }

        // Top customers by current balance
        $coupon_posts = get_posts([
            'post_type'      => 'shop_coupon',
            'post_status'    => 'publish',
            'posts_per_page' => -1,
            // phpcs:disable WordPress.DB.SlowDBQuery.slow_db_query_meta_key,WordPress.DB.SlowDBQuery.slow_db_query_meta_value
            'meta_key'       => '_loycart_store_credit',
            'meta_value'     => 'yes',
            // phpcs:enable WordPress.DB.SlowDBQuery.slow_db_query_meta_key,WordPress.DB.SlowDBQuery.slow_db_query_meta_value
            'fields'         => 'ids'
        ]);

        $balances = [];
        foreach ($coupon_posts as $cid) {
            $amount = floatval(get_post_meta($cid, 'coupon_amount', true));
            $uid    = intval(get_post_meta($cid, '_loycart_customer_id', true));
            if ($amount > 0 && $uid > 0) {
                $balances[] = [ 'user_id' => $uid, 'amount' => $amount ];
            }
        }
        usort($balances, function($a, $b) { return $b['amount'] <=> $a['amount']; });
        $top_balances = array_slice($balances, 0, 5);

        ob_start();
        echo '<div class="credit-reports-summary">';
        echo '<div class="summary-cards" style="display:flex; gap:16px; flex-wrap:wrap;">';
        echo '<div class="summary-card" style="flex:1 1 220px; padding:12px; border:1px solid var(--border); border-radius:8px;">'
            . '<div style="font-weight:600; margin-bottom:6px;">' . esc_html__('Total Earned', 'loycart-pos') . '</div>'
            . '<div style="font-size:1.2em;">' . wp_kses_post(wc_price($total_earned)) . '</div>'
            . '</div>';
        echo '<div class="summary-card" style="flex:1 1 220px; padding:12px; border:1px solid var(--border); border-radius:8px;">'
            . '<div style="font-weight:600; margin-bottom:6px;">' . esc_html__('Total Redeemed', 'loycart-pos') . '</div>'
            . '<div style="font-size:1.2em;">' . wp_kses_post(wc_price($total_redeemed)) . '</div>'
            . '</div>';
        echo '</div>';

        echo '<h4 style="margin-top:16px;">' . esc_html__('Top Customers by Balance', 'loycart-pos') . '</h4>';
        if (!empty($top_balances)) {
            echo '<table class="widefat" style="width:100%; max-width:700px;">';
            echo '<thead><tr><th style="text-align:left;">' . esc_html__('Customer', 'loycart-pos') . '</th><th style="text-align:right;">' . esc_html__('Balance', 'loycart-pos') . '</th></tr></thead><tbody>';
            foreach ($top_balances as $entry) {
                $user = get_userdata($entry['user_id']);
                $name = $user ? $user->display_name : ('User #' . $entry['user_id']);
                echo '<tr><td>' . esc_html($name) . '</td><td style="text-align:right;">' . wp_kses_post(wc_price($entry['amount'])) . '</td></tr>';
            }
            echo '</tbody></table>';
        } else {
            echo '<p>' . esc_html__('No customers with a positive balance.', 'loycart-pos') . '</p>';
        }
        echo '</div>';
        $html = ob_get_clean();

        wp_send_json_success([ 'html' => $html ]);
    }
    add_action('wp_ajax_loycart_pos_get_credit_reports', 'loycart_ajax_get_credit_reports');

    ?>