
var lastFocusedElement = null; 
var isPrinting = false; 


function debounce(func, wait, immediate) {
    var timeout;
    return function() {
        var context = this, args = arguments;
        var later = function() {
            timeout = null;
            if (!immediate) func.apply(context, args);
        };
        var callNow = immediate && !timeout;
        clearTimeout(timeout);
        timeout = setTimeout(later, wait);
        if (callNow) func.apply(context, args);
    };
};


window.isMobileApp = function() {
    const ua = navigator.userAgent || navigator.vendor || window.opera;
    const isIOSWebview = /(iPhone|iPod|iPad).*AppleWebKit(?!.*Safari)/i.test(ua);
    return isIOSWebview || window.webkit && window.webkit.messageHandlers;
};


// Logger module - centralizes logging for the POS app
if (typeof loycart === 'undefined') { window.loycart = {}; }
loycart.logger = {
    debug: false,
    log: function() { if (this.debug) { console.log.apply(console, arguments); } },
    error: function() { console.error.apply(console, arguments); }
};

// Check for sold individually limit
loycart.checkSoldIndividuallyLimit = function(productId) {
    const currentCartQuantity = loycart.cart.reduce((acc, item) => {
        if (item.id === productId && !item.is_custom) {
            return acc + Math.abs(item.quantity);
        }
        return acc;
    }, 0);
    return currentCartQuantity;
};

(function($) { 

// Force Decimal.js to use ROUND_HALF_UP globally for all calculations
if (typeof Decimal !== 'undefined') {
    Decimal.set({ rounding: 4 });
}

 window.triggerIframePrint = function(htmlContent, windowTitle, paperSize = '80mm') {
    
    const ua = navigator.userAgent || navigator.vendor || window.opera;
    const isIOSWebview = /(iPhone|iPod|iPad).*AppleWebKit(?!.*Safari)/i.test(ua);
    const isMobileApp = isIOSWebview || (window.webkit && window.webkit.messageHandlers);

    if (isMobileApp) {
        const printWindow = window.open('', '_blank');
        
        if (printWindow) {
            
            printWindow.document.open();
            printWindow.document.write(`
    <!DOCTYPE html>
    <html>
    <head>
        <meta charset="UTF-8">
        <meta name="viewport" content="width=device-width, initial-scale=1.0">
        <title>${windowTitle}</title>
        <style>
            body { margin: 0; padding: 20px; font-family: -apple-system, sans-serif; text-align: center; }
            .mobile-print-nav { 
                background: #f1f1f1; padding: 15px; margin-bottom: 20px; 
                border-radius: 8px; display: block;
            }
            .print-btn { 
                background: #007cba; color: white; border: none; 
                padding: 10px 20px; border-radius: 4px; font-weight: bold; 
            }
            img, svg { max-width: 100%; height: auto; }
            @media print { .mobile-print-nav { display: none; } }
        </style>
    </head>
    <body>
        <div class="mobile-print-nav">
            <button class="print-btn" onclick="window.print()">${window.loycartPOS?.strings?.tapHereToPrint || 'Tap Here to Print'}</button>
            <p style="font-size: 12px; margin-top: 10px;">${window.loycartPOS?.strings?.iosSharePrintHint || 'If the button does nothing, use the iOS Share icon to Print.'}</p>
        </div>
        ${htmlContent}
    </body>
    </html>
`);
            printWindow.document.close();

            window.showNotification(window.loycartPOS?.strings?.printContentOpenedShare || 'Print content opened. Tap the Share icon to Print/Save.', 'info');
        } else {
            window.showNotification(window.loycartPOS?.strings?.popupBlockedEnable || 'Pop-up blocked. Please enable pop-ups in your app settings.', 'error');
        }
        return;
    }
    


    isPrinting = true; 

    jQuery('#loycartPrintFrame').remove(); 
    
    const $iframe = jQuery('<iframe>', { 
        name: 'loycartPrintFrame',
        id: 'loycartPrintFrame',
        style: 'position:absolute; top:-1000px; left:-1000px;'
    }).appendTo('body'); 
    
    const iframeWin = $iframe[0].contentWindow;
    const iframeDoc = iframeWin.document;

    iframeDoc.open();
    iframeDoc.write('<!DOCTYPE html><html><head><title>' + windowTitle + '</title>');
    
    if (paperSize === '4x6') {
        iframeDoc.write('<style>@page { size: 4in 6in; margin: 0; } body { margin: 0; padding: 0; }</style>');
    }
    
    iframeDoc.write('</head><body>');
    iframeDoc.write(htmlContent); 
    iframeDoc.write('</body></html>');
    iframeDoc.close();

    setTimeout(() => {
        const $body = jQuery('body'); 
        $body.addClass('loycart-printing'); 
        
        try {
            iframeWin.focus();
            iframeWin.print();

            jQuery(window).one('focus', function() {
                setTimeout(() => {
                     isPrinting = false;
                }, 100); 
                $iframe.remove();
                $body.removeClass('loycart-printing');
            });
            
        } catch (e) {
            console.error("Print error:", e);
            window.showNotification(window.loycartPOS?.strings?.errorTryingToPrint || 'Error trying to print.', 'error');
            isPrinting = false;
        }
    }, 250);
};

})(jQuery);



window.formatPrice = function(price) {
    price = parseFloat(price);
    if (isNaN(price)) {
        price = 0;
    }

    const posConfig = (typeof loycartPOS !== 'undefined') ? loycartPOS : {};
    const symbol = posConfig.currency_symbol || '£';
    const decimals = parseInt(posConfig.price_decimals) || 2;
    const decimalSep = posConfig.price_decimal_sep || '.';
    const thousandSep = posConfig.price_thousand_sep || ',';
    const position = posConfig.currency_pos || 'left';

    if (typeof Decimal !== 'undefined') {
        price = new Decimal(price).toDecimalPlaces(decimals, Decimal.ROUND_HALF_UP).toNumber();
    }

    const numberFixed = price.toFixed(decimals);
    const parts = numberFixed.split('.');
    let integerPart = parts[0];
    const decimalPart = parts.length > 1 ? parts[1] : '';
    integerPart = integerPart.replace(/\B(?=(\d{3})+(?!\d))/g, thousandSep);
    let formattedNumber = integerPart;
    if (decimals > 0) {
        formattedNumber += decimalSep + decimalPart;
    }


    
    switch (position) {
        case 'left':
            return symbol + formattedNumber;
        case 'right':
            return formattedNumber + symbol;
        case 'left_space':
            return symbol + ' ' + formattedNumber;
        case 'right_space':
            return formattedNumber + ' ' + symbol;
        default:
            return symbol + formattedNumber;
    }
}


function formatAttributeName(key) {
    if (typeof key !== 'string') return '';
    let name = key.replace(/^attribute_/, '').replace(/^pa_/, '').replace(/-/g, ' ');
    return name.replace(/\b\w/g, char => char.toUpperCase());
}


function generatePriceHTML(productObject) {
    if (!productObject) return '';

    if (typeof Decimal === 'undefined') {
        const price = parseFloat(productObject.price) || 0;
        return `<span class="current-price">${formatPrice(price)}</span>`;
    }

    let priceHtml = '';

    const price = new Decimal(productObject.price || 0);
    const regular_price = new Decimal(productObject.regular_price || 0);
    const max_price = new Decimal(productObject.max_price || 0);
    const zero = new Decimal(0);

    if (productObject.type === 'variable') {
        const minPrice = price.isFinite() ? price : zero;
        const maxPrice = max_price.isFinite() && max_price.gt(minPrice) ? max_price : minPrice;

        if (maxPrice.gt(minPrice)) {
            priceHtml = `<span class="current-price"> ${formatPrice(minPrice.toNumber())} - ${formatPrice(maxPrice.toNumber())}</span>`;
        } else {
            priceHtml = `<span class="current-price">${formatPrice(minPrice.toNumber())}</span>`;
        }

    } else if (productObject.discount_percent && productObject.discount_percent > 0) {

        const originalSinglePrice = regular_price.gt(price) ? regular_price : price;
        const discountPercent = new Decimal(productObject.discount_percent);

        const multiplier = new Decimal(1).minus(discountPercent.div(100));
        const discountedPrice = originalSinglePrice.times(multiplier);
        
        const quantity = new Decimal(productObject.quantity || 1);
        const discountValue = originalSinglePrice.times(quantity).times(discountPercent).div(100);

        priceHtml = `<div>
                        <span class="original-price">${formatPrice(originalSinglePrice.toNumber())}</span>
                        <span class="current-price">${formatPrice(discountedPrice.toNumber())}</span>
                     </div>
                     <div class="discount-details">
                        Cart Discount ${productObject.discount_percent}% Saving ${formatPrice(discountValue.toNumber())}
                     </div>`;
                     
    } else if (productObject.on_sale && regular_price.gt(zero) && regular_price.gt(price)) {
        const saleSaving = regular_price.minus(price);
        const salePercent = regular_price.gt(zero) ? saleSaving.div(regular_price).times(100).toDecimalPlaces(0).toNumber() : 0;

        priceHtml = `<div>
                        <span class="original-price">${formatPrice(regular_price.toNumber())}</span>
                        <span class="current-price">${formatPrice(price.toNumber())}</span>
                     </div>
                     <div class="discount-details">`;
    } else {
        const displayPrice = price.gt(zero) ? price : zero;
        priceHtml = `<span class="current-price">${formatPrice(displayPrice.toNumber())}</span>`;
    }

    return priceHtml;
}




function generateReceiptHTML(response, cartItems, paperWidth) {
    const storeDetails = {
    name:          loycartPOS.store_name     || '',
    address:       loycartPOS.store_address  || '',
    phone:         loycartPOS.store_phone    || '', 
    logo:          loycartPOS.store_logo_url || '',
    footer_message: loycartPOS.footer_message || 'Thank you for your purchase!',
    paper_width:   paperWidth                || '80mm'
};

    const isFullPage = (paperWidth === 'A4' || paperWidth === 'Letter');
    const isRTL = loycartPOS.is_rtl === '1' || loycartPOS.is_rtl === true;
    const isRefund = response.is_refund || false;

    let styleBlock = isFullPage ? `
        body { font-family: Arial, sans-serif; font-size: 11pt; margin: 20mm; direction: ${isRTL ? 'rtl' : 'ltr'}; color: #000; }
        .center { text-align: center; }
        .rit { width: 100%; border-collapse: collapse; margin-top: 5mm; }
        .rit td { padding: 10px 0; text-align: left; vertical-align: top; border-bottom: 1px dashed #333; }
        .rt { width: 50%; float: right; margin-top: 5mm; }
        .rtr { display: flex; justify-content: space-between; padding: 5px 0; }
        .rtl { font-weight: bold; border-top: 2px solid #000; margin-top: 2mm; padding-top: 2mm; font-size: 1.25em; }
        .refund-header { color: #d63638; font-weight: bold; font-size: 1.5em; margin: 2mm 0; border: 2px solid #d63638; padding: 2mm; }
    ` : `
        body { font-family: monospace; font-size: 10pt; width: ${storeDetails.paper_width}; padding: 3mm; direction: ${isRTL ? 'rtl' : 'ltr'}; color: #000; line-height: 1.3; }
        .center { text-align: center; }
        .rit { width: 100%; border-collapse: collapse; margin: 2mm 0; }
        .rit th { padding: 4px 0; text-align: left; border-bottom: 1px solid #000; font-weight: bold; }
        .rit td { padding: 8px 0; text-align: left; vertical-align: top; border-bottom: 1px dashed #000; }
        .rtr { display: flex; justify-content: space-between; padding: 2px 0; }
        .rtl { font-weight: bold; border-top: 1px dashed #000; margin-top: 2mm; padding-top: 2mm; font-size: 1.15em; }
        .sep { border-top: 1px dashed #000; margin: 3mm 0; }
        .saving-label { font-size: 0.85em; font-style: italic; color: #444; display: block; margin-top: 2px; }
        .refund-header { font-weight: bold; font-size: 1.2em; border: 1px solid #000; padding: 1mm; margin-bottom: 2mm; }
    `;

    let receipt = `<!DOCTYPE html><html><head><meta charset="UTF-8"><style>${styleBlock}</style></head><body><div class="r">`;

    // Header (unchanged)
    receipt += `<div class="center">`;
    if (storeDetails.logo) receipt += `<img src="${storeDetails.logo}" style="max-height:28mm; margin-bottom:3mm;"><br>`;
    receipt += `<h2 style="margin:0 0 1mm 0; font-size: 1.4em;">${storeDetails.name}</h2>${storeDetails.address}<br>Tel: ${storeDetails.phone}</div><div class="sep"></div>`;

    // Refund title
    if (isRefund) {
        receipt += `<div class="center"><div class="refund-header">*** REFUND RECEIPT ***</div></div>`;
    }

    receipt += `<div class="center"><b>ORDER: #${response.order_id}</b><br>DATE: ${response.date_created}</div><div class="sep"></div>`;

    // Items table (looks good – keep it)
    receipt += `<table class="rit"><thead><tr><th>Item Details</th><th style="text-align:right">Qty / Total</th></tr></thead><tbody>`;

    cartItems.forEach(item => {
        const originalPrice = parseFloat(item.regular_price) || parseFloat(item.price) || 0;
        const qty = Math.abs(parseInt(item.quantity) || 1);
        const cashierDiscountAmount = Math.abs(parseFloat(item.discount_amount) || 0);
        const itemDiscountPercent = parseFloat(item.discount_percent) || 0;
        
        let discountLabel = '', totalSaving = 0, netPrice = originalPrice;

        if (cashierDiscountAmount > 0) {
            netPrice = originalPrice - cashierDiscountAmount;
            totalSaving = cashierDiscountAmount * qty;
            const percent = Math.round((cashierDiscountAmount / originalPrice) * 100);
            discountLabel = `Cashier Discount ${percent}% Saving £${formatPrice(totalSaving).replace('£', '')}`;
        } 
        else if (itemDiscountPercent > 0) {
            const unitSaving = originalPrice * (itemDiscountPercent / 100);
            totalSaving = unitSaving * qty;
            netPrice = originalPrice - unitSaving;
            discountLabel = `Cashier Discount ${itemDiscountPercent}% Saving ${formatPrice(totalSaving)}`;
        } 
        else if (parseFloat(item.price) < originalPrice) {
            netPrice = parseFloat(item.price);
            totalSaving = (originalPrice - netPrice) * qty;
            const percent = Math.round(((originalPrice - netPrice) / originalPrice) * 100);
            discountLabel = `Sale Discount ${percent}% Saving ${formatPrice(totalSaving)}`;
        }

        receipt += `<tr><td style="padding-right: 5px;"><b>${item.name}</b>`;
        
        if (item.attributes && Object.keys(item.attributes).length > 0) {
            let varDetails = Object.keys(item.attributes).map(key => {
                let label = key.replace('attribute_pa_', '').replace('attribute_', '').replace(/-/g, ' ');
                label = label.charAt(0).toUpperCase() + label.slice(1);
                return `${label}: ${item.attributes[key]}`;
            }).join(', ');
            if (varDetails) receipt += `<br><small style="color:#555;">${varDetails}</small>`;
        }

        receipt += `<br>`;
        if (totalSaving > 0) receipt += `<span style="text-decoration:line-through; color:#777; font-size:0.9em;">${formatPrice(originalPrice)}</span> `;
        receipt += `<span>${formatPrice(netPrice)}</span>`;
        if (discountLabel) receipt += `<br><span class="saving-label">${discountLabel}</span>`;
        receipt += `</td><td style="text-align:right; white-space: nowrap;">`;
        receipt += `<small>x${Math.abs(qty)}</small>  <b>${formatPrice(Math.abs(netPrice * qty))}</b></td></tr>`;
    });
    receipt += `</tbody></table>`;

    // ── TOTALS SECTION ──  ← This is the fixed part
    receipt += `<div class="rt" style="margin-left: auto; width: 100%;">`;

    if (isRefund) {
    // ── PRIORITY: Use values from server (after PHP fix) ──
    let subtotal = 0;
    let cashierDiscount = 0;
    let couponDiscount = 0;
    let tax      = 0;
    let shipping = 0;
    let grandTotal = 0;

    // Get values from server response
    subtotal = Math.abs(parseFloat(response.subtotal) || 0);
    cashierDiscount = Math.abs(parseFloat(response.cashier_discount) || 0);
    couponDiscount = Math.abs(parseFloat(response.coupon_discount) || 0);
    tax = Math.abs(parseFloat(response.total_tax) || 0);
    shipping = Math.abs(parseFloat(response.shipping_total) || 0);
    const shippingTax = Math.abs(parseFloat(response.shipping_tax) || 0);
    
    // Add shipping tax to total tax for combined VAT line
    tax += shippingTax;

    // Grand total – prefer server value
    grandTotal = Math.abs(parseFloat(response.total) || 0);

    // ── Now build the receipt lines (matching cart display) ──
    receipt += `<div class="rtr"><span>Subtotal:</span><span>${formatPrice(subtotal)}</span></div>`;

    if (cashierDiscount > 0) {
        receipt += `<div class="rtr"><span>${window.loycartPOS?.strings?.receiptCashierDiscountsLabel || 'Cashier Discounts:'}</span><span>-${formatPrice(cashierDiscount)}</span></div>`;
    }

    if (couponDiscount > 0) {
        const couponCode = response.coupon_code || 'Applied';
        const couponLabel = (window.loycartPOS?.strings?.receiptCouponLabel || 'Coupon (%s):').replace('%s', couponCode);
        receipt += `<div class="rtr"><span>${couponLabel}</span><span>-${formatPrice(couponDiscount)}</span></div>`;
    }

    if (shipping > 0) {
        receipt += `<div class="rtr"><span>${window.loycartPOS?.strings?.receiptShippingLabel || 'Shipping:'}</span><span>${formatPrice(shipping)}</span></div>`;
    }

    receipt += `<div class="rtr"><span>VAT:</span><span>${formatPrice(tax)}</span></div>`;

    receipt += `<div class="rtr rtl"><span><strong>${window.loycartPOS?.strings?.receiptTotalRefundLabel || 'Total Refund:'}</strong></span>` +
               `<span><strong>${formatPrice(grandTotal)}</strong></span></div>`;
}
    else {
        // Normal sale — keep original logic (server-provided totals array)
        if (response.totals && response.totals.length > 0) {
            response.totals.forEach(total => {
                const isGrandTotal = (total.key === 'order_total');
                let label = total.label.replace(/:+$/, '');
                receipt += `<div class="rtr ${isGrandTotal ? 'rtl' : ''}">`;
                receipt += `<span>${label}:</span><span>${total.value}</span></div>`;
            });
        }
    }

    receipt += `</div>`;

    // Shipping address & footer (unchanged)
    if (response.shipping_address && response.shipping_address.trim() !== "") {
        receipt += `<div class="sep"></div><div style="font-size: 0.9em;"><b>${window.loycartPOS?.strings?.receiptShippingToLabel || 'SHIPPING TO:'}</b><br>${response.shipping_address.replace(/\n/g, '<br>')}</div>`;
    }

    receipt += `<div class="sep"></div><div class="center" style="font-size: 1.1em; font-weight: bold;">${storeDetails.footer_message}</div>`;
    const copyLabel = isRefund ? (window.loycartPOS?.strings?.receiptCopyLabelRefund || 'Refund Copy') : (window.loycartPOS?.strings?.receiptCopyLabelCustomer || 'Customer Copy');
    receipt += `<div style="font-size: 0.8em; margin-top: 1mm; text-align:center;">*** ${copyLabel} ***</div>`;

    // Barcode (improved centering)
    if (response.order_id) {
        const orderNum = String(response.order_id);
        receipt += `
        <div class="center" style="margin-top:4mm;">
            <svg id="barcode"></svg>
        </div>
        <div class="center" style="font-size:1.1em; margin-top:2mm; letter-spacing:1px;">
            #${orderNum}
        </div>

        <script src="${loycartPOS.plugin_url}assets/js/JsBarcode.all.min.js"></script>
        <script>
            try {
                JsBarcode("#barcode", "${orderNum}", {
                    format: "CODE128",
                    lineColor: "#000",
                    width: 2,
                    height: 60,
                    fontSize: 16,
                    margin: 8,
                    displayValue: false
                });
            } catch(e) {
                console.error("Barcode error:", e);
            }
        </script>`;
    }

    receipt += `</div></body></html>`;
    return receipt;
}




// Helper: Compute display refund total with server-first fallback.
// Returns a positive number suitable for UI display (Return to Customer, etc.)
window.computeRefundDisplayTotal = function(resp) {
    try {
        if (!resp || typeof resp !== 'object') return Math.abs(window.splitPaymentTotalDue || 0);

        const serverTotal = Math.abs(parseFloat(resp.total) || 0);
        if (serverTotal > 0) return serverTotal;

        // Fallback: recompute from server-provided fields (match receipt logic)
        const subtotal       = Math.abs(parseFloat(resp.subtotal) || 0);
        const couponDiscount = Math.abs(parseFloat(resp.coupon_discount) || 0);
        const tax            = Math.abs(parseFloat(resp.total_tax) || 0);
        const shipping       = Math.abs(parseFloat(resp.shipping_total) || 0);
        const shippingTax    = Math.abs(parseFloat(resp.shipping_tax) || 0);

        // Combine shipping tax into VAT for display; do not subtract cashier discount here
        const vatCombined = tax + shippingTax;
        const total = Math.abs((subtotal - couponDiscount) + vatCombined + shipping);
        return total;
    } catch (e) {
        console.warn('computeRefundDisplayTotal error:', e);
        return Math.abs(window.splitPaymentTotalDue || 0);
    }
};


function generateLabelHTML(labelData) {
    const style = `
        <style>
            @media print {
                @page {
                    size: 89mm 28mm; /* Standard Dymo 30252 size */
                    margin: 0;
                }
            }

            html, body {
                font-family: Arial, sans-serif;
                margin: 0;
                padding: 0;
                width: 89mm;
                height: 28mm;
                overflow: hidden;
                line-height: 1.1; /* Tighter line height */
                page-break-inside: avoid;
            }
            .label-container {
                width: 100%;
                height: 100%;
                display: flex;
                flex-direction: column;
                justify-content: center; /* Center vertically */
                align-items: center;   /* Center horizontally */
                text-align: center;
                page-break-inside: avoid;
                box-sizing: border-box; /* Include padding in size */
                padding: 2mm 3mm; /* Reduced padding */
            }
            
            .label-content-top {
                 line-height: 1.15;
            }

            .label-content-bottom {
                margin-top: 1mm; /* Reduced gap */
                /* This ensures the top part stays centered even if this is empty */
                min-height: 9mm; 
            }
            

            .price { font-size: 12pt; font-weight: bold; margin: 0; }
            .name { font-size: 8pt; font-weight: bold; margin: 0.5mm 0 0 0; word-wrap: break-word; }
            .barcode-container { padding-top: 0.5mm; }
            #barcode {
                max-height: 6.5mm;
                width: 100%;
                display: block;
            }
            .barcode-text { font-size: 6pt; margin: 0; letter-spacing: 0.5px; }
            .sku { font-size: 6pt; margin: 0.5mm 0 0 0; color: #333; }
        </style>
    `;


    let displayName = labelData.name;
    if (labelData.variation) {
        displayName += ` - ${labelData.variation}`;
    }


    let barcodeHtml = '';
    if (labelData.barcode && labelData.barcode.trim() !== '') {
        barcodeHtml = `
            <div class="barcode-container">
                <svg id="barcode"></svg>
            </div>
            <p class="barcode-text">${labelData.barcode}</p>
        `;
    }
    

    let skuHtml = '';
    if (labelData.sku && labelData.sku.trim() !== '') {
        skuHtml = `<p class="sku">SKU: ${labelData.sku}</p>`;
    }
    
    const body = `
        <div class="label-container">
            <div class="label-content-top">
                <p class="price">${labelData.price}</p>
                <p class="name">${displayName}</p>
            </div>
            <div class="label-content-bottom">
                ${barcodeHtml}
                
            </div>
        </div>
    `;

    return `<!DOCTYPE html><html><head><title>${window.loycartPOS?.strings?.printLabelTitle || 'Print Label'}</title>${style}</head><body>${body}</body></html>`;
}



(function($) { 


    function showNotification(message, type = 'success') {
        // Check if notifications are enabled in settings (default: hidden)
        try {
            var showNotifications = true; // fallback to true if missing
            if (window.loycartPOS && window.loycartPOS.settings && typeof window.loycartPOS.settings.ui !== 'undefined') {
                showNotifications = !!window.loycartPOS.settings.ui.show_notifications;
            }
            // If not enabled, do not show notification
            if (!showNotifications) return;
        } catch (e) { loycart.logger.error('[POS] showNotification error:', e); }
        // Do not hide store name or date/time when notification is shown
        // Deduplicate identical notifications within a short window to avoid spam
        window._posNotifyLast = window._posNotifyLast || { msg: '', ts: 0 };
        const now = Date.now();
        if (window._posNotifyLast.msg === message && (now - window._posNotifyLast.ts) < 1200) {
            return; // Skip duplicate
        }
        window._posNotifyLast = { msg: message, ts: now };


        // Prefer admin bar notification area if present
        let $container = jQuery('#loycart-pos-adminbar-notifications');
        if ($container.length) {
            $container.find('.pos-notification').remove();
        } else {
            // Fallback to legacy body container
            $container = jQuery('#pos-notification-container');
            if (!$container.length) {
                $container = jQuery('<div id="pos-notification-container"></div>');
                jQuery('body').append($container);
            }
        }

        const $notification = jQuery(`<div class="pos-notification ${type}" role="alert" aria-live="${type === 'error' || type === 'warning' ? 'assertive' : 'polite'}">${message}</div>`);
        $container.append($notification);
        setTimeout(() => {
            $notification.addClass('show');
        }, 10);
        setTimeout(() => {
            $notification.removeClass('show');
            setTimeout(() => {
                $notification.remove();
                // No need to restore store name or date/time visibility
            }, 300);
        }, 3000);
    }


    window.openModal = function($modal) {
        if (!$modal || !$modal.length) {
             console.error("openModal called with invalid modal element.");
             return;
        }
        
        if (!isPrinting) {
            lastFocusedElement = document.activeElement; 
        }
        $modal.addClass('show').attr('aria-hidden', 'false');
        setTimeout(() => {
            
            const $focusable = $modal.find('input:not([type="hidden"]):not(:disabled), select:not(:disabled), textarea:not(:disabled), button:not(:disabled), [tabindex]:not([tabindex="-1"])').filter(':visible').first();
            try {
             
                if ($focusable.length && $focusable[0]) {
                    $focusable[0].focus();
                }
            } catch (e) {
                console.warn("Could not focus element in modal:", e);
            
                try {

                    $modal.find('.modal-content').attr('tabindex', '-1')[0].focus();
                } catch (fe) {
                    console.error("Could not focus modal content as fallback:", fe);
                }
            }
        }, 100);
    }



    window.closeModal = function($modal) {
        if (!$modal || !$modal.length) {
             console.error("closeModal called with invalid modal element.");
             return;
        }

        $modal.find('.modal-content[tabindex="-1"]').removeAttr('tabindex');

        if (document.activeElement && $modal.find(document.activeElement).length > 0) {
            try { document.activeElement.blur(); } catch(e) {}
        }
        
 
        if (!isPrinting && lastFocusedElement) { 
            try {
  
                const $lastElement = jQuery(lastFocusedElement);
                if ($lastElement.length && $lastElement.is(':visible') && $lastElement[0]) {

                    $lastElement[0].focus();
                } else {

                     jQuery('#product-search')[0].focus();
                }
            } catch (e) {
                 jQuery('#product-search')[0].focus(); 
            }
        } else if (!isPrinting) {

             jQuery('#product-search')[0].focus();
        }
        
        lastFocusedElement = null; 
        $modal.removeClass('show').attr('aria-hidden', 'true');
    }


    function printReceiptInternal(response, cartItems, debug = false, paperWidth = '80mm') {
        const receiptHTML = generateReceiptHTML(response, cartItems, paperWidth); 

        jQuery('#loycartPrintFrame').remove();

        const $iframe = jQuery('<iframe>', {
            name: 'loycartPrintFrame',
            id: 'loycartPrintFrame',
            style: 'position:absolute; top:-1000px; left:-1000px;'
        }).appendTo('body');
        
        const iframeWin = $iframe[0].contentWindow;
        const iframeDoc = iframeWin.document;

        iframeDoc.open();
        iframeDoc.write(receiptHTML);
        iframeDoc.close();
        
        setTimeout(() => {
            const $body = jQuery('body');
            
            $body.addClass('loycart-printing');
            
            try {
                iframeWin.focus();
                iframeWin.print();
            } catch (e) {
                console.error("Error triggering print dialog:", e);
                showNotification(window.loycartPOS?.strings?.errorTryingToPrint || 'Error trying to print.', 'error');
            }
            
            try {
                window.focus();
                if (lastFocusedElement) { jQuery(lastFocusedElement).focus(); }
                else { jQuery('#product-search').focus(); }
            } catch(e) {}

            if (!debug) {
                setTimeout(() => {
                    $iframe.remove();
                    $body.removeClass('loycart-printing');
                }, 2000); 
            }
            
        }, 250);
    }

    window.showNotification = showNotification;
    window.printReceiptInternal = printReceiptInternal;





window.generate58mmReceiptString = function() {
    const saleData = window.lastSaleResponse || {};
    const serverItems = saleData.line_items || [];
    const cartData = (typeof loycart !== 'undefined' && loycart.cart) ? loycart.cart : {};
    const posConfig = (typeof loycartPOS !== 'undefined') ? loycartPOS : {};
    const currencySym = posConfig.currency_symbol || '£';
    
    const decodeHTML = (html) => {
        if (!html) return "";
        let txt = document.createElement("textarea");
        txt.innerHTML = html.replace(/<br\s*\/?>/gi, "\n");
        // Forcibly remove @ symbol from branding
        return txt.value.replace(/@/g, '').replace(/\u00A0/g, ' ').trim();
    };

    const hardwareSym = (currencySym === '£') ? "\x9C" : currencySym;
    const targetWidth = 30; 

    // ESC/POS Commands
    const init       = [0x1B, 0x40];      
    const singleByte = [0x1C, 0x2E];      
    const charTable  = [0x1B, 0x74, 0x02]; 
    const center     = [0x1B, 0x61, 0x01];  
    const left       = [0x1B, 0x61, 0x00];  
    const doubleOn   = [0x1B, 0x21, 0x10];  
    const doubleOff  = [0x1B, 0x21, 0x00];  
    const boldOn     = [0x1B, 0x45, 0x01];  
    const boldOff    = [0x1B, 0x45, 0x00];  
    const cut        = [0x1D, 0x56, 0x41]; 

    let receipt = String.fromCharCode(...init, ...singleByte, ...charTable, ...center);

    // HEADER - Clean Store Name + Blank Line
    receipt += String.fromCharCode(...doubleOn, ...boldOn) + decodeHTML(posConfig.store_name || "LOYCART POS") + String.fromCharCode(...doubleOff) + "\n\n";
    receipt += decodeHTML(posConfig.store_address || "") + "\n";
    receipt += "Tel: " + decodeHTML(posConfig.store_phone || "") + "\n------------------------------\n";

    // TRANSACTION INFO
    receipt += String.fromCharCode(...boldOn) + "ORDER: #" + (saleData.order_id || "SALE") + String.fromCharCode(...boldOff) + "\n";
    receipt += "DATE: " + (saleData.date_created || new Date().toLocaleString('en-GB')) + "\n------------------------------\n" + String.fromCharCode(...left);

    // ITEM LISTING - Supporting both Sale and Cashier discounts
    const itemsToProcess = serverItems.length > 0 ? serverItems : Object.values(cartData);
    
    itemsToProcess.forEach(item => {
        receipt += String.fromCharCode(...boldOn) + decodeHTML(item.name).substring(0, targetWidth) + String.fromCharCode(...boldOff) + "\n";
        
        const qty = parseInt(item.quantity);
        const currentPrice = parseFloat(item.price);
        const regPrice = parseFloat(item.regular_price || item.price);
        const cashierDiscount = parseFloat(item.discount_amount || 0); // Manual keypad discount
        
        // Final item price after manual discount
        const netPrice = cashierDiscount > 0 ? (regPrice - cashierDiscount) : currentPrice;
        const itemTotal = netPrice * qty;

        // Line 2: Pricing Logic with "Was/Now"
        let priceRow = " ";
        if (regPrice > netPrice + 0.01) {
            priceRow += "Was:" + hardwareSym + regPrice.toFixed(2) + " Now:" + hardwareSym + netPrice.toFixed(2);
        } else {
            priceRow += hardwareSym + netPrice.toFixed(2);
        }

        // Quantity positioned before the total
        const totalPart = qty + "x " + hardwareSym + itemTotal.toFixed(2);
        const spaces = targetWidth - priceRow.length - totalPart.length;
        receipt += priceRow + " ".repeat(Math.max(1, spaces)) + totalPart + "\n";

        // Line 3: Discount Label (Cashier vs Sale)
        let savingLabel = "";
        const diff = regPrice - netPrice;
        if (diff > 0.01) {
            const percent = Math.round((diff / regPrice) * 100);
            const totalSaved = (diff * qty).toFixed(2);
            // If item has a discount_amount, it's a Cashier Discount
            const type = (cashierDiscount > 0) ? "Cashier Discount" : "Sale Discount";
            savingLabel = `${type} ${percent}% Saving ${hardwareSym}${totalSaved}`;
        }

        if (savingLabel) {
            receipt += savingLabel.substring(0, targetWidth) + "\n";
        }

        receipt += "------------------------------\n";
    });

    // ──────────────────────────────────────────────────────────────
    // TOTALS - Fixed version: respects discounted values for refunds
    // ──────────────────────────────────────────────────────────────
    const isRefund = !!saleData.is_refund;

    let printSubtotal   = 0;
    let printCashierDisc = 0;
    let printCouponDisc = 0;
    let printTax        = 0;
    let printShipping   = 0;
    let printGrandTotal = 0;

    if (isRefund) {
        // Refund mode: use discounted values sent by server
        printSubtotal   = Math.abs(parseFloat(saleData.subtotal || 0));
        printCashierDisc = Math.abs(parseFloat(saleData.cashier_discount || 0));
        printCouponDisc = Math.abs(parseFloat(saleData.coupon_discount || 0));
        printTax        = Math.abs(parseFloat(saleData.total_tax || 0));
        printShipping   = Math.abs(parseFloat(saleData.shipping_total || 0));
        
        // Add shipping tax to total tax for combined VAT line
        const shippingTaxAmount = Math.abs(parseFloat(saleData.shipping_tax || 0));
        printTax += shippingTaxAmount;
        
        printGrandTotal = Math.abs(parseFloat(saleData.total || 0));

        // Strong fallback: recalculate from items if server value missing/wrong
        if (printSubtotal < 0.01 && itemsToProcess.length > 0) {
            itemsToProcess.forEach(item => {
                const qty = Math.abs(parseInt(item.quantity || 1));
                const netPrice = parseFloat(item.price || 0); // already discounted
                printSubtotal += netPrice * qty;
            });
        }

        // Final safety net for grand total
        if (printGrandTotal < 0.01) {
            printGrandTotal = printSubtotal - printCashierDisc - printCouponDisc + printShipping + printTax;
        }

        // Print refund totals with separate discount lines
        receipt += "Subtotal:".padEnd(30) + 
                   hardwareSym + printSubtotal.toFixed(2).padStart(8) + "\n";

        if (printCashierDisc > 0) {
            receipt += "Cashier Discounts:".padEnd(30) + 
                       "-" + hardwareSym + printCashierDisc.toFixed(2).padStart(7) + "\n";
        }

        if (printCouponDisc > 0) {
            const couponCode = saleData.coupon_code || 'Applied';
            const couponLabel = "Coupon (" + couponCode + "):";
            receipt += couponLabel.padEnd(30) + 
                       "-" + hardwareSym + printCouponDisc.toFixed(2).padStart(7) + "\n";
        }

        if (printShipping > 0) {
            receipt += "Shipping:".padEnd(30) + 
                       hardwareSym + printShipping.toFixed(2).padStart(8) + "\n";
        }        
        const printShippingTax = Math.abs(parseFloat(saleData.shipping_tax || 0));
        if (printShippingTax > 0) {
            receipt += "Shipping Tax:".padEnd(30) + 
                       hardwareSym + printShippingTax.toFixed(2).padStart(8) + "\n";
        }
        receipt += "VAT:".padEnd(30) + 
                   hardwareSym + printTax.toFixed(2).padStart(8) + "\n";

        receipt += String.fromCharCode(...boldOn) +
                   "Total Refund:".padEnd(30) +
                   hardwareSym + printGrandTotal.toFixed(2).padStart(8) +
                   String.fromCharCode(...boldOff) + "\n";
    }
    else {
        // Normal sale → original logic unchanged
        if (saleData.totals && Array.isArray(saleData.totals)) {
            saleData.totals.forEach(total => {
                if (total.key === 'payment_method') return;
                const isGrandTotal = (total.key === 'order_total');
                const isShipping = (total.key === 'shipping_total');
                
                let label = isShipping ? "Shipping" : decodeHTML(total.label).replace(/:+$/, '');
                let valText = decodeHTML(total.value).replace(new RegExp(currencySym, 'g'), hardwareSym);

                if (isShipping) {
                    const priceMatch = valText.match(/[\x9C$€]?\d+\.\d{2}/);
                    valText = priceMatch ? priceMatch[0] : valText;
                }

                if (isGrandTotal) receipt += String.fromCharCode(...boldOn);
                const spaces = targetWidth - label.length - valText.length;
                receipt += label + " ".repeat(Math.max(1, spaces)) + valText + "\n";
                if (isGrandTotal) receipt += String.fromCharCode(...boldOff);
            });
        }
    }

    // PAYMENT LINE
    if (saleData.payment_method_title) {
        let pMethod = decodeHTML(saleData.payment_method_title).replace('POS - ', '').trim();
        const pLabel = "Payment";
        const displayMethod = pMethod.substring(0, 15);
        const pSpace = targetWidth - pLabel.length - displayMethod.length;
        receipt += pLabel + " ".repeat(Math.max(1, pSpace)) + displayMethod + "\n";
    }

    // SHIPPING TO & FOOTER
    if (saleData.shipping_address && saleData.shipping_address.trim() !== "") {
        receipt += "------------------------------\n" + String.fromCharCode(...left, ...boldOn) + "SHIPPING TO:" + String.fromCharCode(...boldOff) + "\n";
        receipt += decodeHTML(saleData.shipping_address) + "\n";
    }

    receipt += "------------------------------\n" + String.fromCharCode(...center) + decodeHTML(posConfig.footer_message || "Thank You!") + "\n";
    
    // NATIVE BARCODE
    if (saleData.order_id) {
        const bData = String(saleData.order_id);
        const barcodeCmd = [0x1D, 0x68, 0x40, 0x1D, 0x77, 0x02, 0x1D, 0x48, 0x02, 0x1D, 0x6B, 0x04];
        receipt += "\n" + String.fromCharCode(...barcodeCmd) + bData + "\x00\n";
    }

    receipt += "\n\n\n\n" + String.fromCharCode(...cut);
    return receipt;
};




})(jQuery); 