// --- SCANNER MODULE ---
loycart.scanner = {
    detectionHandler: null,
    lastResult: null,
    matchCount: 0,
    isRunning: false,
    nativeGetContext: null,
    keydownHandler: null,
    context: 'add-product', // 'add-product' or 'header'

    applyCanvasOverride() {
        if (!this.nativeGetContext) {
            this.nativeGetContext = HTMLCanvasElement.prototype.getContext;
        }
        const self = this;
        HTMLCanvasElement.prototype.getContext = function(type, attributes) {
            if (type === '2d') {
                attributes = attributes || {};
                attributes.willReadFrequently = true;
            }
            return self.nativeGetContext.call(this, type, attributes);
        };
    },

    restoreCanvasOverride() {
        if (this.nativeGetContext) {
            HTMLCanvasElement.prototype.getContext = this.nativeGetContext;
        }
    },

    resetFeedback(message = 'Requesting camera access...') {
        const $feedback = jQuery('#scanner-feedback');
        $feedback.text(message);
    },

    attachEscHandler() {
        this.keydownHandler = (e) => {
            if (e.key === 'Escape' && jQuery('#barcode-scanner-modal').is(':visible')) {
                e.preventDefault();
                this.closeScanner(true);
            }
        };
        jQuery(document).on('keydown.scanner', this.keydownHandler);
    },

    detachEscHandler() {
        jQuery(document).off('keydown.scanner');
        this.keydownHandler = null;
    },

    openScanner: function(context = 'add-product') {
        if (this.isRunning) return;

        this.context = context;
        const $modal = jQuery('#barcode-scanner-modal');
        const $feedback = jQuery('#scanner-feedback');
        const $scannerArea = jQuery('#scanner-area');
        
        this.lastResult = null;
        this.matchCount = 0;
        this.resetFeedback();

        // Clear focus to prevent soft keyboards interfering
        if (jQuery(':focus').length) {
            jQuery(':focus').trigger('blur');
        }

        $modal.css({
            'z-index': '999999',
            'display': 'flex'
        }).attr('aria-hidden', 'false');
        
        if (typeof openModal === 'function') {
            openModal($modal);
        }

        if (typeof Quagga === 'undefined') {
            $feedback.text(window.loycartPOS?.strings?.scannerQuaggaNotLoaded || 'Error: QuaggaJS library not loaded.');
            return;
        }

        this.applyCanvasOverride();
        this.isRunning = true;
        this.attachEscHandler();

        this.detectionHandler = (result) => {
            const code = result?.codeResult?.code;
            if (code) {
                if (code === this.lastResult) {
                    this.matchCount++;
                } else {
                    this.lastResult = code;
                    this.matchCount = 0;
                }

                if (this.matchCount >= 2) { // 3 consecutive detections (0,1,2)
                    this.onScanSuccess(code);
                }
            }
        };

        Quagga.init({
            inputStream: {
                name: "Live",
                type: "LiveStream",
                target: $scannerArea[0],
                constraints: {
                    width: { ideal: 1280 },
                    height: { ideal: 720 },
                    facingMode: "environment"
                }
            },
            locator: {
                patchSize: "medium",
                halfSample: false
            },
            numOfWorkers: navigator.hardwareConcurrency || 2,
            frequency: 10,
            decoder: {
                readers: [
                    "ean_reader",
                    "ean_8_reader",
                    "code_128_reader",
                    "upc_reader",
                    "upc_e_reader"
                ]
            }
        }, (err) => {
            if (err) {
                console.error("Quagga init failed:", err);
                this.isRunning = false;
                this.restoreCanvasOverride();
                $feedback.text('Camera access denied or unavailable.');
                return;
            }

            Quagga.start();
            Quagga.onDetected(this.detectionHandler);
            $feedback.text('Ready - Point camera at barcode');
        });
    },



    onScanSuccess: function(barcode) {
        // Stop the scanner and hide the modal
        this.closeScanner(true); 
        
        // Route scan result based on context
        if (this.context === 'add-product' && jQuery('#add-new-product-modal').is(':visible')) {
            // Scanning from add-product modal
            jQuery('#add-new-product-barcode').val(barcode).trigger('input').trigger('change');
            jQuery('#add-new-product-stock').trigger('focus');
        } else {
            // Scanning from header or other contexts - use product manager
            loycart.productManager.handleBarcodeScan(barcode);
        }
        
        const scannedMsg = (window.loycartPOS?.strings?.scannedPrefix || 'Scanned: %s').replace('%s', barcode);
        showNotification(scannedMsg, 'success');
    },


    closeScanner: function(shouldCloseModal = true) {
        const $modal = jQuery('#barcode-scanner-modal');
        
        if (typeof Quagga !== 'undefined') {
            try { Quagga.offDetected(this.detectionHandler); } catch(e) {}
            try { Quagga.stop(); } catch(e) {}
        }

        const $video = jQuery('#scanner-video')[0];
        if ($video && $video.srcObject) {
            $video.srcObject.getTracks().forEach(track => track.stop());
            $video.srcObject = null;
        }

        jQuery('#scanner-area').empty();

        this.restoreCanvasOverride();
        this.detachEscHandler();
        this.isRunning = false;

        if (shouldCloseModal) {
            $modal.hide().attr('aria-hidden', 'true').css('z-index', '');
            if (typeof closeModal === 'function') {
                closeModal($modal);
            }
        } else {
            // keep modal open, reset feedback for retry
            this.resetFeedback('Requesting camera access...');
        }
    }



};