/**
 * LoyCart POS - Refund Manager Module
 * 
 * Handles refund/return mode operations:
 * - Switches UI to refund mode
 * - Searches and loads customer orders
 * - Displays refundable items from selected order
 * - Calculates refund totals (discount-aware)
 * - Processes refunds and generates refund receipts
 * 
 * @namespace loycart.refundManager
 */
loycart.refundManager = {

    orderHistoryPage: 1,
    orderHistorySearch: '',
    orderHistoryStatus: 'not_refunded', 
    orderHistoryDate: '',

    /**
     * Toggle between normal POS mode and refund mode
     * @param {boolean} isReturn - True to enter refund mode
     */
    setUIMode: window.setUIMode = function (isReturn) {
    window.isReturnMode = isReturn;
    jQuery('body').toggleClass('return-mode-active', isReturn);

    // Reference to the original product browsing header
    const $originalHeader = jQuery('.cart-header');

    // Create or reuse the refund mode header
    let $refundHeader = jQuery('#refund-mode-header');
    if ($refundHeader.length === 0 && isReturn) {
        $refundHeader = jQuery(`
            <header id="refund-mode-header" class="refund-mode-header">
                <div class="refund-mode-title">
                    <span class="dashicons dashicons-arrow-left-alt2"></span>
                    ${loycartPOS.strings && loycartPOS.strings.refundModeTitle ? loycartPOS.strings.refundModeTitle : 'Refund / Return Mode'}
                </div>
                <div class="refund-mode-subtitle">
                    ${loycartPOS.strings && loycartPOS.strings.refundModeSubtitle ? loycartPOS.strings.refundModeSubtitle : "Select items from the customer's previous order to refund"}
                </div>
            </header>
        `);
        
        // Insert it where it makes sense (after original header or at top of products area)
        $originalHeader.after($refundHeader);
    }

    if (isReturn) {
        // Hide normal browsing controls
        $originalHeader.hide();
        jQuery('#refund-mode-header').show();

        // Clear products and show placeholder
        if (window.$productsList) {
            const placeholder = (loycartPOS.strings && loycartPOS.strings.refundSelectOrderPlaceholder) ? loycartPOS.strings.refundSelectOrderPlaceholder : 'Please select an order to see refundable items.';
            window.$productsList.html(
                `<div class="no-products-found">${placeholder}</div>`
            );
        }
        jQuery('.pagination-controls').hide();
    } else {
        // Restore normal mode
        $originalHeader.show();
        jQuery('#refund-mode-header').hide();

        window.originalOrderId = null;
        window.refundableOrderItems = [];

        jQuery('#original-order-info').hide();
        jQuery('#refund-summary-section').empty();
        jQuery('.pagination-controls').show();
        jQuery('#cart-checkout-actions').show();
        jQuery('#refund-actions-group').hide();

        if (window.$productsList) {
            window.$productsList.empty();
        }

        // Reload normal products
        loycart.productManager.loadProducts(1);
    }
},
    
    displayOrderContext: function(order) {
        if (!order || !order.id) {
            jQuery('#original-order-info').hide();
            return;
        }

        let rawDate = order.date_created || order.date || order.order_date || order.date_completed || '';
        
        if (!rawDate) {
            for (let key in order) {
                if (typeof order[key] === 'string' && order[key].includes('-')) {
                    rawDate = order[key];
                    break;
                }
            }
        }

        let displayDate = rawDate; 
        if (rawDate && rawDate.length > 5) {
            const isoDate = rawDate.toString().replace(' ', 'T');
            const dateObj = new Date(isoDate);
            if (!isNaN(dateObj.getTime())) {
                displayDate = dateObj.toLocaleDateString(undefined, {
                    year: 'numeric', month: 'short', day: 'numeric'
                });
            }
        }

        jQuery('#refund-order-id').text('#' + order.id);
        jQuery('#refund-order-total').text(formatPrice(order.total));
        
        if (displayDate) {
            jQuery('#refund-order-date').text(displayDate).show();
            jQuery('.context-divider-date').show();
        } else {
            jQuery('#refund-order-date').text('Date N/A').show();
            jQuery('.context-divider-date').show();
        }
        
        jQuery('#original-order-info').fadeIn(200);
    },     
        
    loadCustomerOrders: function (page = 1) {
        const customerId = $customerId.val();
        if (!customerId || customerId === '0') return;

        this.orderHistoryPage = page;

        const loadingTxt = (loycartPOS.strings && loycartPOS.strings.refundLoadingOrders) ? loycartPOS.strings.refundLoadingOrders : 'Loading orders...';
        const $customerOrdersList = jQuery('#customer-order-history-list').html(`<p class="empty-state-message">${loadingTxt}</p>`);
        const $paginationControls = jQuery('#refund-order-pagination');
        $paginationControls.hide();
        
        jQuery('#refund-order-lookup-modal .spinner').addClass('is-active');

        api_get_customer_orders(customerId, this.orderHistorySearch, this.orderHistoryStatus, this.orderHistoryDate, page)
            .done(function (response) {
                $customerOrdersList.empty();
                jQuery('#refund-order-lookup-modal .spinner').removeClass('is-active');

                if (response.success && response.data && response.data.orders && response.data.orders.length > 0) {
                    const orders = response.data.orders;
                    const pagination = response.data.pagination;

                    orders.forEach(function (order) {
                        const totalRaw = parseFloat(order.total_raw);
                        const totalRefunded = parseFloat(order.total_refunded);
                        
                        let isFullyRefunded = totalRefunded >= totalRaw;
                        let isPartiallyRefunded = (totalRefunded > 0 && totalRefunded < totalRaw);

                        let statusClass = isFullyRefunded ? 'status-fully-refunded' : (isPartiallyRefunded ? 'status-partially-refunded' : 'status-eligible');
                        let statusText = isFullyRefunded
                            ? (loycartPOS.strings && loycartPOS.strings.refundStatusFully ? loycartPOS.strings.refundStatusFully : 'Fully Refunded')
                            : (isPartiallyRefunded
                                ? (loycartPOS.strings && loycartPOS.strings.refundStatusPartial ? loycartPOS.strings.refundStatusPartial : 'Partial Refund')
                                : (loycartPOS.strings && loycartPOS.strings.refundStatusEligible ? loycartPOS.strings.refundStatusEligible : 'Eligible for Refund'));

                        const $orderItem = jQuery(`
                            <div class="customer-order-item ${statusClass}" role="button" tabindex="0" data-order-id="${order.id}">
                                <div class="customer-order-info">
                                    <div class="order-id-meta"><strong>Order #${order.id}</strong> (${order.item_count} items)</div>
                                    <div class="order-total-meta">
                                        <span>Date: ${order.date_created}</span>
                                        <span>Total: <strong>${order.total}</strong></span>
                                        <span class="order-refund-status">${statusText}</span>
                                    </div>
                                </div>
                                <div class="customer-actions">
                                    <button class="button button-secondary select-order-for-refund" data-order-id="${order.id}">Select</button>
                                </div>
                            </div>
                        `);
                        $customerOrdersList.append($orderItem);
                    });

                    jQuery('#refund-current-page').text(pagination.current_page);
                    jQuery('#refund-total-pages').text(pagination.total_pages);
                    jQuery('#refund-prev-page').prop('disabled', pagination.current_page === 1);
                    jQuery('#refund-next-page').prop('disabled', pagination.current_page >= pagination.total_pages);
                    $paginationControls.show();

                } else {
                    $customerOrdersList.html('<p class="empty-state-message">No orders found.</p>');
                }
            })
            .fail(function () {
                jQuery('#refund-order-lookup-modal .spinner').removeClass('is-active');
                $customerOrdersList.html('<p class="empty-state-message">Failed to load orders.</p>');
            });
    },        

enterReturnModeForOrder: window.enterReturnModeForOrder = function (orderId) {
    jQuery('#refund-summary-section').empty();
    jQuery('#original-order-info').hide();
    jQuery('#cart-checkout-actions, #refund-actions-group').hide();

    // CRITICAL: Clear cart and shipping before loading order
    loycart.cart = [];
    window.selectedShipping = { id: null, cost: 0, label: '' };
    
    // Force cart UI update to clear everything
    if (loycart.cartManager && loycart.cartManager.updateCart) {
        loycart.cartManager.updateCart();
    }
    
    originalOrderId = orderId;
    {
        const tmpl = (loycartPOS.strings && loycartPOS.strings.verifyingOrder) ? loycartPOS.strings.verifyingOrder : 'Verifying Order #%s...';
        showNotification(tmpl.replace('%s', orderId), 'info');
    }

    api_getRefundableOrderItems(orderId)
        .done(function (response) {
            if (!response.success || !response.data || !response.data.items) {
                const errorMsg = (response.data && typeof response.data === 'string')
                    ? response.data
                    : 'Order not found or invalid response.';
                showNotification('Error: ' + errorMsg + ' Please try again.', 'error');
                // Don't reload - let user try again
                return;
            }

            window.refundableOrderItems = response.data.items;
            const summary = response.data.order_summary;

            // Set coupon code if present
            window.appliedCoupon = summary.coupon_code || '';

            // Clear shipping again to ensure it's not set
            window.selectedShipping = { id: null, cost: 0, label: '' };

            loycart.refundManager.displayOrderContext({
                id: orderId,
                total: summary.total,
                date_created: summary.date,
                refund_history: summary.refund_history || [],
                total_refunded: summary.total_refunded || 0
            });

            loycart.refundManager.displayRefundableItems();

            const isFullyRefundedOrder = window.refundableOrderItems.length > 0 &&
                window.refundableOrderItems.every(item => 
                    item.qty_refundable <= 0 && !item.is_shipping_refund
                ) &&
                !window.refundableOrderItems.some(item => 
                    item.is_shipping_refund && item.qty_refundable > 0
                );

            if (isFullyRefundedOrder) {
                // Fully refunded state
                jQuery('#process-refund-button').hide();
                jQuery('#footer-print-refund-receipt').show();  // ← new unique ID

                jQuery('#cancel-return-button')
                    .text('Exit')
                    .off('click')
                    .on('click', function () {
                        loycart.refundManager.setUIMode(false);
                    });

                // Footer print button: triggers direct print (using your existing print system)
                jQuery('#footer-print-refund-receipt')
                    .off('click.footer-print')
                    .on('click.footer-print', function (e) {
                        e.preventDefault();

                        // Prepare refund data
                        const refundData = {
                            is_refund: true,
                            order_id: orderId,
                            date_created: summary.date || new Date().toISOString().split('T')[0],
                            subtotal: Math.abs(summary.total || 0) * -1,
                            total: Math.abs(summary.total || 0) * -1,
                            total_tax: 0, // add real values if available
                            shipping_total: 0
                        };

                        const refundCartItems = window.refundableOrderItems
                            .filter(item => item.qty_refundable <= 0 && !item.is_shipping_refund)
                            .map(item => ({
                                name: item.name,
                                quantity: item.quantity,
                                price: parseFloat(item.price_paid) + parseFloat(item.tax_paid),
                                regular_price: parseFloat(item.price_paid) + parseFloat(item.tax_paid),
                                discount_amount: 0,
                                discount_percent: 0,
                                attributes: {}
                            }));

                        const receiptHTML = generateReceiptHTML(refundData, refundCartItems, '80mm');

                        // Use your proven print function
                        printReceiptInternal(refundData, refundCartItems, false, '80mm');

                        const printTxt = (loycartPOS.strings && loycartPOS.strings.printingRefundReceipt) ? loycartPOS.strings.printingRefundReceipt : 'Refund receipt printing...';
                        showNotification(printTxt, 'success');
                    });
            } else {
                jQuery('#process-refund-button').show();
                jQuery('#footer-print-refund-receipt').hide();

                jQuery('#cancel-return-button')
                    .text('Cancel Return')
                    .off('click')
                    .on('click', function () {
                        loycart.refundManager.setUIMode(false);
                        // Smooth reset instead of page reload
                        setTimeout(() => {
                            loycart.cartManager.clearCartAndCustomer();
                        }, 300);
                    });
            }

            {
                const tmplLoaded = (loycartPOS.strings && loycartPOS.strings.orderLoaded) ? loycartPOS.strings.orderLoaded : 'Order #%s loaded.';
                showNotification(tmplLoaded.replace('%s', orderId), 'success');
            }

            if (window.$refundOrderLookupModal) {
                closeModal(window.$refundOrderLookupModal);
            }
        })
        .fail(function () {
            const connErr = (loycartPOS.strings && loycartPOS.strings.serverConnectionError) ? loycartPOS.strings.serverConnectionError : 'Server connection error. Please check your connection and try again.';
            showNotification(connErr, 'error');
            // Don't reload - let user try again
        });
},

displayRefundableItems: function () {
    $productsList.empty();

    if (!refundableOrderItems || refundableOrderItems.length === 0) {
        const noneTxt = (loycartPOS.strings && loycartPOS.strings.noItemsFound) ? loycartPOS.strings.noItemsFound : 'No items found.';
        $productsList.html(`<div class="no-products-found">${noneTxt}</div>`);
        return;
    }

    refundableOrderItems.forEach(item => {
        const pricePaidGross = (parseFloat(item.price_paid) || 0) + (parseFloat(item.tax_paid) || 0);
        const isFullyRefunded = item.qty_refundable <= 0;
        const isPartiallyRefunded = !isFullyRefunded && item.qty_refundable < item.quantity;
        const qtyRefunded = item.quantity - item.qty_refundable;

        const dateValue = item.refund_date || "";
        let refundDateHtml = '';
        if ((isFullyRefunded || isPartiallyRefunded) && dateValue) {
            refundDateHtml = `
                <div class="refund-date-label" style="font-size: 11px; color: #d63638; margin-top: 4px; font-weight: bold; display: flex; align-items: center; gap: 4px;">
                    <span class="dashicons dashicons-calendar-alt" style="font-size: 13px; width: 13px; height: 13px;"></span>
                    <span>Refunded: ${dateValue}</span>
                </div>`;
        }

        let extraRefundInfo = '';
        if (qtyRefunded > 0) {
            extraRefundInfo = `
                <div class="refund-extra-info" style="font-size: 11px; color: #555; margin-top: 3px; line-height: 1.3;">
                    ${qtyRefunded} of ${item.quantity} refunded<br>
                    Cashier: ${item.cashier_name || 'System'}
                </div>`;
        }

        let statusText = isFullyRefunded ?
            (loycartPOS.strings && loycartPOS.strings.refundStatusFully ? loycartPOS.strings.refundStatusFully : 'Fully Refunded') :
            (isPartiallyRefunded ?
                `${loycartPOS.strings && loycartPOS.strings.refundStatusPartial ? loycartPOS.strings.refundStatusPartial : 'Partial Refund'} (${item.qty_refundable} left)` :
                `${(loycartPOS.strings && loycartPOS.strings.refundableQty ? loycartPOS.strings.refundableQty : 'Refundable: %s').replace('%s', item.qty_refundable)}`);

        let statusClass = isFullyRefunded ? 'refunded' : (isPartiallyRefunded ? 'partial-refunded' : '');

        const imageSrc = (item.image && item.image !== 'false') ? item.image : loycartPOS.placeholder_url;

        // ── SHIPPING ITEM ─────────────────────────────────────────────────────
        if (item.is_shipping_refund) {
            statusText = isFullyRefunded
                ? (loycartPOS.strings && loycartPOS.strings.shippingFullyRefunded ? loycartPOS.strings.shippingFullyRefunded : 'Shipping Fully Refunded')
                : (loycartPOS.strings && loycartPOS.strings.refundableShipping ? loycartPOS.strings.refundableShipping : 'Refundable Shipping');
            statusClass = isFullyRefunded ? 'refunded' : '';

            const $product_el = jQuery(`
                <div class="product shipping-item ${isFullyRefunded ? 'disabled' : ''}"
                     data-id="${item.original_item_id}"
                     style="min-height: 260px; display: flex; flex-direction: column; ${isFullyRefunded ? 'padding-bottom: 12px;' : ''}">
                     <div class="product-image" style="flex: 0 0 auto; display: flex; justify-content: center; align-items: center; padding: 20px 10px; background: #f8f9fa; border-bottom: 1px solid #dee2e6; min-height: 120px;">
                     <span class="dashicons dashicons-airplane" style="font-size: 80px; width: 80px; height: 80px; color: ${isFullyRefunded ? '#6c757d' : 'var(--accent, #007cba)'}; opacity: ${isFullyRefunded ? '0.65' : '1'};"></span>
                     </div>
                    <div class="product-note-wrapper" style="flex: 0 0 auto; padding: 8px; text-align: center;">
                        <span class="stock-status ${statusClass}" style="font-size: 13px; padding: 4px 10px;">${statusText}</span>
                        ${refundDateHtml}
                    </div>
                    <div class="product-info" style="flex: 1; display: flex; flex-direction: column; justify-content: space-between; min-height: 80px; padding: 10px;">
                        <div class="product-name" style="margin-bottom: 8px; font-weight: 600; text-align: center;">${item.name}</div>
                        <div class="product-price-area">
                            <div class="product-price" style="text-align: center;">
                                <span style="font-size: 1.3em; font-weight: bold;">${formatPrice(pricePaidGross)}</span>
                            </div>
                            ${extraRefundInfo}
                        </div>
                    </div>
                </div>`);

            $product_el.data('product-data', item);
            $productsList.append($product_el);
            return; // Skip normal product rendering
        }

        // ── NORMAL PRODUCT CARD ──────────────────────────────────────────────
        const $product_el = jQuery(`
            <div class="product ${isFullyRefunded ? 'disabled' : ''} ${isPartiallyRefunded ? 'partial-refunded' : ''}"
                 data-id="${item.original_item_id}"
                 style="min-height: 260px; display: flex; flex-direction: column; ${isFullyRefunded || isPartiallyRefunded ? 'padding-bottom: 12px;' : ''}">
                <div class="product-image" style="flex: 0 0 auto;">
                    <img src="${imageSrc}" alt="${item.name}" onerror="this.onerror=null;this.src='${loycartPOS.placeholder_url}';">
                </div>
                <div class="product-note-wrapper" style="flex: 0 0 auto;">
                    <span class="stock-status ${statusClass}">${statusText}</span>
                    ${refundDateHtml}
                </div>
                <div class="product-info" style="flex: 1; display: flex; flex-direction: column; justify-content: space-between; min-height: 80px;">
                    <div class="product-name" style="margin-bottom: 6px;">${item.name}</div>
                    <div class="product-price-area">
                        <div class="product-price">
                            <span>${formatPrice(pricePaidGross)}</span>
                        </div>
                        ${extraRefundInfo}
                    </div>
                </div>
            </div>`);

        $product_el.data('product-data', item);
        $productsList.append($product_el);
    });
},

    updateRefundableItemState: function () {
        if (!isReturnMode) return;
        $productsList.find('.product').each(function () {
            const $card = jQuery(this);
            const productData = $card.data('product-data');
            if (!productData) return;

            // Handle shipping card separately
            if (productData.is_shipping_refund) {
                const shippingSelected = window.selectedShipping && window.selectedShipping.id;
                if (shippingSelected) {
                    const addedTxt = (loycartPOS.strings && loycartPOS.strings.refundableShipping) ? loycartPOS.strings.refundableShipping : 'Refundable Shipping';
                    $card.addClass('disabled').find('.stock-status').text(addedTxt).addClass('refunded');
                } else {
                    const refShipTxt = (loycartPOS.strings && loycartPOS.strings.refundableShipping) ? loycartPOS.strings.refundableShipping : 'Refundable Shipping';
                    $card.removeClass('disabled').find('.stock-status').text(refShipTxt).removeClass('refunded');
                }
                return;
            }

            // Handle regular items
            const cartItem = loycart.cart.find(item => item.original_item_id === productData.original_item_id);
            const qtyInCart = cartItem ? Math.abs(cartItem.quantity) : 0;
            const qtyRefundable = parseInt(productData.qty_refundable);

            if (qtyInCart >= qtyRefundable) {
                const itemRefTxt = (loycartPOS.strings && loycartPOS.strings.itemRefunded) ? loycartPOS.strings.itemRefunded : 'Item Refunded';
                $card.addClass('disabled').find('.stock-status').text(itemRefTxt).addClass('refunded');
            } else {
                const refQtyTmpl = (loycartPOS.strings && loycartPOS.strings.refundableQty) ? loycartPOS.strings.refundableQty : 'Refundable: %s';
                $card.removeClass('disabled').find('.stock-status').text(refQtyTmpl.replace('%s', qtyRefundable)).removeClass('refunded');
            }
        });
    }
};