// --- PAYMENT MANAGER MODULE ---
// Contains all logic for handling cash, card, and split payments
// ----------------------------

// Global selectors
const $cartTotal = window.$cartTotal;
const $cartSubtotal = window.$cartSubtotal;
const $cartDiscount = window.$cartDiscount;
const $cartDiscountRow = window.$cartDiscountRow;
const $cartTax = window.$cartTax;
const $cartTaxRow = window.$cartTaxRow;
const $modalFinalTotal = window.$modalFinalTotal;
const $modalChange = window.$modalChange;
const $amountTendered = window.$amountTendered;

loycart.paymentManager = {

    updateChange: function () {
        const amount = parseFloat($amountTendered.text()) || 0;
        const finalTotal = parseFloat($modalFinalTotal.text().replace(/[^0-9.-]+/g, "")) || 0;
        const change = amount - finalTotal;
        $modalChange.text(formatPrice(change >= 0 ? change : 0));
    },

    initializeSplitPaymentModal: function () {
        window.pendingPayments = [];
        window.splitPaymentTotalDue = window.splitPaymentTotalDue || 0;

if (window.splitPaymentTotalDue <= 0.001) {
    showNotification('Cannot process payment with zero or invalid total.', 'error');
    closeModal(jQuery('#split-payment-modal')); // Close if somehow opened
    return;
}
window.splitPaymentAmountRemaining = window.splitPaymentTotalDue;

        jQuery('#modal-original-subtotal').text($cartSubtotal.text());
        const discountText = $cartDiscount.text();
        if ($cartDiscountRow.is(':visible')) {
            jQuery('#modal-original-discount').text(discountText);
            jQuery('#modal-original-discount-row').show();
        } else {
            jQuery('#modal-original-discount-row').hide();
        }
        const shippingText = jQuery('#cart-shipping-total').text();
        if (jQuery('#cart-shipping-row').is(':visible')) {
            jQuery('#modal-original-shipping').text(shippingText);
            jQuery('#modal-original-shipping-row').show();
        } else {
            jQuery('#modal-original-shipping-row').hide();
        }
        const taxText = $cartTax.text();
        if ($cartTaxRow.is(':visible')) {
            jQuery('#modal-original-tax').text(taxText);
            jQuery('#modal-original-tax-row').show();
        } else {
            jQuery('#modal-original-tax-row').hide();
        }
        jQuery('#modal-original-total').text(formatPrice(window.splitPaymentTotalDue));

        splitPaymentInput = window.splitPaymentTotalDue.toFixed(2);
        splitInputHasDecimal = true;
        splitInputIsNew = false;
        jQuery('#split-amount-label').text('Amount Tendered:');
        

        this.updateSplitPaymentModalUI();
        openModal($splitPaymentModal);
    },

    updateSplitPaymentModalUI: function () {
        jQuery('#split-total-due').text(formatPrice(window.splitPaymentTotalDue));
        jQuery('#split-amount-remaining').text(formatPrice(window.splitPaymentAmountRemaining));

        const $splitAmountInput = jQuery('#split-amount-input');
        if ($splitAmountInput.length) {
            if ($splitAmountInput.val() !== splitPaymentInput) {
                $splitAmountInput.val(splitPaymentInput);
            }
        } else {
            loycart.logger.error("#split-amount-input not found!");
        }

        const currentInputAmountCheck = parseFloat(splitPaymentInput);
        const labelText = (!isNaN(currentInputAmountCheck) && currentInputAmountCheck > window.splitPaymentAmountRemaining + 0.001) ?
            'Amount Tendered:' :
            'Amount Tendered:';
        jQuery('#split-amount-label').text(labelText);

        const $list = jQuery('#split-payments-applied-list').empty();
        if (window.pendingPayments.length > 0) {
            window.pendingPayments.forEach((payment, index) => {


                let removeButtonHtml = '';

                if (payment.method === 'cash' || payment.method === 'cheque') {
                    removeButtonHtml = `<button class="split-payment-item-remove" aria-label="Remove this payment">&times;</button>`;
                } else {

                    removeButtonHtml = `<span class="split-payment-item-locked dashicons dashicons-lock" title="This payment cannot be removed"></span>`;
                }


                const $item = jQuery(`
                    <div class="split-payment-item" data-index="${index}">
                        <span class="split-payment-item-details">
                            <span class="dashicons dashicons-yes-alt"></span>
                            ${payment.title}: ${formatPrice(payment.amount)}
                        </span>
                        ${removeButtonHtml} </div>
                `);
                $list.append($item);
            });
        }


        const totalTendered = window.pendingPayments.reduce((sum, p) => sum + p.amount, 0);
        let changeDue = 0;

        if (totalTendered > window.splitPaymentTotalDue) {
            changeDue = totalTendered - window.splitPaymentTotalDue;
        }

        const $splitChangeDue = jQuery('#split-change-due');
        const $splitChangeRow = jQuery('#split-change-row');
        if ($splitChangeDue.length && $splitChangeRow.length) {
            $splitChangeDue.text(formatPrice(changeDue));
            $splitChangeRow.show();
        } else {
            loycart.logger.error("Change display elements not found!");
        }

        const isComplete = window.splitPaymentAmountRemaining < 0.001;

        const $completeButton = jQuery('#split-complete-sale');
        $completeButton.prop('disabled', !isComplete);

        if (isComplete && changeDue > 0.001) {
            $completeButton.text(`Complete & Give ${formatPrice(changeDue)} Change`);
        } else {
            $completeButton.text('Complete Sale');
        }

        jQuery('#split-payment-buttons .modal-payment-btn').prop('disabled', isComplete);
        jQuery('#split-keypad button, .split-action-button').prop('disabled', isComplete);
        jQuery('#split-quick-amounts button').prop('disabled', isComplete);
        if ($splitAmountInput.length) {
            $splitAmountInput.prop('readonly', isComplete);
            $splitAmountInput.toggleClass('disabled-input', isComplete);
        }
    },

    addPendingPayment: function (payment) {
        window.pendingPayments.push(payment);
        window.splitPaymentAmountRemaining = window.splitPaymentAmountRemaining - payment.amount;
        if (window.splitPaymentAmountRemaining < 0.001) {
            window.splitPaymentAmountRemaining = 0;
        }

        const nextInput = window.splitPaymentAmountRemaining > 0 ? window.splitPaymentAmountRemaining.toFixed(2) : '0.00';
        splitPaymentInput = nextInput;
        splitInputHasDecimal = nextInput.includes('.');
        splitInputIsNew = true;
        this.updateSplitPaymentModalUI();
    },


showReceiptModal: function (saleResponse, cartItemsCopy) {

window.lastSaleResponse = saleResponse;

        const savedPaperWidth = (window.loycartPOS.settings && window.loycartPOS.settings.receipt && window.loycartPOS.settings.receipt.paper_width) || '80mm';
        const $printReceipt = jQuery('#print-receipt');
        const $labelButton = jQuery('#print-order-label-button');
        const $dismissReceipt = jQuery('#dismiss-receipt');

        const $receiptPreviewWrapper = jQuery('#receipt-preview-wrapper').length ? jQuery('#receipt-preview-wrapper') : $receiptModal.find('.modal-content');

        const applyPreviewWidth = (width) => {
            $receiptPreviewWrapper.removeClass('width-80mm width-58mm width-A4 width-Letter');
            if (width === '80mm') {
                $receiptPreviewWrapper.addClass('width-80mm');
            } else if (width === '58mm') {
                $receiptPreviewWrapper.addClass('width-58mm');
            } else if (width === 'A4') {
                $receiptPreviewWrapper.addClass('width-A4');
            } else if (width === 'Letter') {
                $receiptPreviewWrapper.addClass('width-Letter');
            }
        };

        const activeWidth = savedPaperWidth || '80mm';
        applyPreviewWidth(activeWidth);
        jQuery('#receipt-paper-width-indicator span').text(activeWidth);
        openModal($receiptModal);


        const onReceiptPrinted = () => {
            // Close modal immediately for better UX
            closeModal($receiptModal);
            window.isProcessing = false;
            
            // Clear cart asynchronously
            loycart.cartManager.clearCartAndCustomer()
                .done(function() {
                    // Cart cleared successfully
                })
                .fail(function() {
                    showNotification('Failed to clear cart on server.', 'error');
                });
        };

        // AUTHORITATIVE RECEIPT PRINTING: Uses saleResponse to show both discounts
        $printReceipt.off('click').on('click', function () {
            const paperWidth = $paperWidthSelect.val();
            // No localStorage: update global settings for preview only
            window.loycartPOS.settings.receipt.paper_width = paperWidth;

            // This pulls the cashier_discount and coupon_discount from the server response
            const receiptHTML = generateReceiptHTML(saleResponse, cartItemsCopy, paperWidth);

            window.triggerIframePrint(
                receiptHTML,
                'Sales Receipt - #' + (saleResponse.order_id || 'N/A'),
                paperWidth
            );
        });

        $labelButton.off('click').on('click', function() {
            loycart.paymentManager.printLabelInternal(saleResponse);
        });

        const commonResetLogic = () => {
            loycart.cartManager.clearCartAndCustomer()
                .done(function() {
                    closeModal($receiptModal);
                    window.isProcessing = false;
                    // Cart cleared successfully
                })
                .fail(function() {
                    showNotification('Failed to clear cart, please try again.', 'error');
                    closeModal($receiptModal);
                    window.isProcessing = false;
                });
        };
    },

    printLabelInternal: function(saleResponse) {
        const dataToSend = {
            action: 'loycart_ajax_get_label_data',
            nonce: loycartPOS.nonce,
            order_id: saleResponse.order_id,
            label_type: 'shipping',
            shipping_method_id: saleResponse.shipping_method_id
        };

        showNotification('Fetching shipping label data...', 'info');

        jQuery.post(loycartPOS.ajax_url, dataToSend)
            .done(function(response) {
                if (response.success && response.data.label_address) {
                    response.data.label_type = 'shipping';
                    const labelHTML = loycart.labelManager.buildLabelHtml(response.data);
                    window.triggerIframePrint(labelHTML, `Shipping Label - #${response.data.order_id}`, '4x6');
                    showNotification('Shipping Label print initiated.', 'success');
                } else {
                    showNotification(response.data.error || 'Failed to fetch shipping data.', 'error');
                }
            })
            .fail(function() {
                showNotification('Server error while fetching label.', 'error');
            });
    },

    handleDiscountKeypad: function ($button) {
    const value = String($button.data('value'));
    let currentDisplay = $discountValueDisplay.text();
    
    // Handle the visual input display
    if (value === 'clear') {
        currentDisplay = '0';
    } else if (value === '.') {
        if (!currentDisplay.includes('.')) currentDisplay += '.';
    } else {
        currentDisplay = (currentDisplay === '0' && value !== '.') ? value : currentDisplay + value;
    }

    let numericValue = parseFloat(currentDisplay);
    if (isNaN(numericValue)) {
        numericValue = 0;
        currentDisplay = '0';
    }

    // Logic for Fixed Amount validation
    if (window.selectedProductId !== null && loycart.cart[window.selectedProductId]) {
        const productInCart = loycart.cart[window.selectedProductId];
        const originalPrice = parseFloat(productInCart.price); // £39.99 for Blouse

        // Ensure the discount doesn't exceed the product price
        if (numericValue > originalPrice) {
            numericValue = originalPrice;
            currentDisplay = String(originalPrice);
            showNotification('Discount cannot exceed the item price.', 'warning');
        }

        $discountValueDisplay.text(currentDisplay);

        // Update the preview to show the "New Price" instead of "Savings"
        // This makes the UI clearer for the cashier
        const newPrice = originalPrice - numericValue;
        $discountPreviewAmount.text(formatPrice(newPrice)); 
    }
},

applyDiscount: function () {
    if (window.selectedProductId !== null && loycart.cart[window.selectedProductId]) {
        const discountVal = parseFloat($discountValueDisplay.text());
        
        if (isNaN(discountVal) || discountVal < 0) {
            showNotification('Please enter a valid discount amount.', 'error');
            return;
        }

        // SWAP: We now use discount_amount instead of discount_percent
        loycart.cart[window.selectedProductId].discount_amount = discountVal;
        loycart.cart[window.selectedProductId].discount_percent = 0; // Clear old logic for sync

        loycart.cartManager.invalidateSelectedShipping();
        loycart.cartManager.updateCart();
        closeModal($discountModal);
        showNotification('Fixed discount applied.', 'success');
        window.selectedProductId = null;
    }
},

removeDiscount: function () {
    if (window.selectedProductId !== null && loycart.cart[window.selectedProductId]) {
        // Clear both values to ensure a clean reset
        loycart.cart[window.selectedProductId].discount_amount = 0;
        loycart.cart[window.selectedProductId].discount_percent = 0;
        
        loycart.cartManager.invalidateSelectedShipping();
        loycart.cartManager.updateCart();
        closeModal($discountModal);
        showNotification(strings.discountRemoved, 'success');
        window.selectedProductId = null;
    }
},

    handleSplitKeypad: function ($button) {
        const $inputField = jQuery('#split-amount-input');
        if (!$inputField.length || $inputField.prop('readonly')) return;

        const value = $button.data('value');
        let currentVal = $inputField.val();

        if (value === 'clear') {
            currentVal = '0.00';
            splitInputHasDecimal = false;
            splitInputIsNew = true;
            jQuery('#split-amount-label').text('Amount to Pay:');
        } else if (value === '.') {
            if (splitInputIsNew) {
                currentVal = '0.';
                splitInputIsNew = false;
            } else if (!splitInputHasDecimal) {
                if (currentVal === '') currentVal = '0';
                currentVal += '.';
            }
            splitInputHasDecimal = true;
        } else {
            const parts = currentVal.split('.');
            if (splitInputIsNew) {
                currentVal = value;
                splitInputIsNew = false;
                splitInputHasDecimal = false;
            } else if (splitInputHasDecimal && parts.length > 1 && parts[1].length >= 2) {
                // Ignore
            } else if (currentVal === '0' || currentVal === '0.00') {
                if (splitInputHasDecimal && currentVal.includes('.')) {
                    currentVal += value;
                } else if (value !== '0') {
                    currentVal = value;
                }
            } else if (currentVal === '0.') {
                currentVal += value;
            } else {
                currentVal += value;
            }
        }

        $inputField.val(currentVal);
        splitPaymentInput = currentVal;
        this.updateSplitPaymentModalUI();
    },

    handleSplitQuickTender: function ($button) {
        const $inputField = jQuery('#split-amount-input');
        if (!$inputField.length || $inputField.prop('readonly')) return;

        let newAmount = 0;
        const amountType = $button.data('amount-type');
        const amountValue = $button.data('amount-value');
        const currentAmount = parseFloat($inputField.val()) || 0;

        if (amountType === 'exact') {
            newAmount = (window.splitPaymentAmountRemaining > 0.001) ? window.splitPaymentAmountRemaining : 0;
        } else if (amountValue !== undefined) {
            newAmount = currentAmount + parseFloat(amountValue);
        }

        if (!isNaN(newAmount) && newAmount >= 0) {
            splitPaymentInput = newAmount.toFixed(2);
            splitInputHasDecimal = splitPaymentInput.includes('.');
            splitInputIsNew = true;
            $inputField.val(splitPaymentInput);
            this.updateSplitPaymentModalUI();
        }
    },

    handlePostSaleActions: function(saleResponse) {
        const $completeButton = jQuery('#split-complete-sale');
        $completeButton.hide();
        $splitPaymentModal.find('.modal-header h3').text(`Sale Completed! (#${saleResponse.order_id})`);
        $splitPaymentModal.find('.modal-body :input, .modal-body button').prop('disabled', true);
        jQuery('#payment-post-sale-actions').css('display', 'flex');

        const shippingMethodId = saleResponse.shipping_method_id || null;
        if (shippingMethodId !== null && shippingMethodId !== '') {
            $inlinePrintLabel.show();
            jQuery('#shipping-button-text').text(loycartPOS.strings.printShippingLabel);
        } else {
            $inlinePrintLabel.hide();
        }

        $newSaleButton.prop('disabled', false);
        const totalTendered = window.pendingPayments.reduce((sum, p) => sum + p.amount, 0);
        const change = Math.max(0, totalTendered - saleResponse.total);
        $splitPaymentModal.find('#split-amount-label').text(change > 0.001 ? 'Change Given:' : 'Total Paid:');
        $splitPaymentModal.find('#split-amount-input').val(change > 0.001 ? change.toFixed(2) : saleResponse.total.toFixed(2));
        $splitPaymentModal.find('.modal-body').scrollTop(0);
    }
}; // Closing the paymentManager object