/**
 * LoyCart POS - Coupon Manager Module
 * 
 * Manages coupon selection and application:
 * - Opens coupon picker modal
 * - Fetches available coupons from server
 * - Applies selected coupon to cart
 * - Triggers cart total recalculation
 * 
 * @namespace loycart.couponManager
 */
loycart.couponManager = {

    init: function() {
        const self = this;

        // 1. Open Modal Trigger
        jQuery('body').on('click', '#apply-coupon-btn', function(e) {
            e.preventDefault();
            e.stopPropagation();
            self.openCouponModal();
        });

        // 2. Remove Coupon Trigger (Main UI)
        jQuery('body').on('click', '#remove-applied-coupon', function(e) {
            e.preventDefault();
            window.appliedCoupon = '';
            showNotification(window.loycartPOS?.strings?.couponRemoved || 'Coupon removed.', 'info');

            const currentCustomerId = jQuery('#customer-id').val() || '0';
            jQuery('#coupon-no-customer-notice').toggle(currentCustomerId === '0');

            if (loycart.cartManager) {
                loycart.cartManager.getCartTotalsFromServer();
                loycart.cartManager.debouncedSaveCartToServer();
            }
        });

        // 3. Close Modal (X and Cancel buttons)
        jQuery('body').on('click', '#coupon-picker-modal .modal-close', function(e) {
            e.preventDefault();
            closeModal(jQuery('#coupon-picker-modal'));
        });

        // 4. Clear Selection Button logic
        jQuery('body').on('click', '#clear-coupon-selection', function(e) {
            e.preventDefault();
            window.appliedCoupon = '';

            if (loycart.cartManager && typeof loycart.cartManager.getCartTotalsFromServer === 'function') {
                loycart.cartManager.getCartTotalsFromServer();
                loycart.cartManager.debouncedSaveCartToServer();
            }

            showNotification(window.loycartPOS?.strings?.couponSelectionCleared || 'Coupon selection cleared.', 'info');
            closeModal(jQuery('#coupon-picker-modal'));
        });

        // 5. Select Coupon Trigger (Inside Modal Grid)
        jQuery('body').on('click touchend', '.select-coupon-btn', function(e) {
            e.preventDefault();
            e.stopPropagation();
            const couponCode = jQuery(this).data('code');
            self.applyCouponToCart(couponCode);
        });

        // 6. Escape Key Support
        jQuery(document).on('keydown', function(e) {
            if (e.key === 'Escape' && jQuery('#coupon-picker-modal').is(':visible')) {
                closeModal(jQuery('#coupon-picker-modal'));
            }
        });

        // Initial prompt update on load
    },


    openCouponModal: function() {
        const $modal = jQuery('#coupon-picker-modal');

        if (!$modal.length) {
            console.error('ERROR: #coupon-picker-modal not found in the HTML.');
            showNotification(window.loycartPOS?.strings?.uiErrorCouponModalMissing || 'UI Error: Coupon Modal missing.', 'error');
            return;
        }

        const $list = jQuery('#available-coupons-list');
        const $spinner = $modal.find('.spinner');
        openModal($modal);
        $list.html(`<p class="empty-state">${window.loycartPOS?.strings?.fetchingAvailableCoupons || 'Fetching available coupons...'}</p>`);

        const currentCustomerId = jQuery('#customer-id').val() || '0';
        jQuery('#coupon-no-customer-notice').toggle(currentCustomerId === '0');

        window.api_getAvailableCoupons(currentCustomerId)
            .done(function(response) {
                $spinner.removeClass('is-active');
                $list.empty();

                if (response.success && Array.isArray(response.data) && response.data.length > 0) {
                    response.data.forEach(function(coupon) {
                        const amountDisplay = (coupon.type && coupon.type.indexOf('percent') !== -1)
                            ? coupon.amount + '%'
                            : formatPrice(coupon.amount);

                        const html = `
                            <div class="coupon-item-card">
                                <div class="coupon-details">
                                    <span class="coupon-code">${coupon.code}</span>
                                    <span class="coupon-type">${coupon.type_label || ''}</span>
                                    <p class="coupon-desc">${coupon.description || ''}</p>
                                </div>
                                <div class="coupon-action">
                                    <div class="coupon-value">${amountDisplay}</div>
                                    <button class="button button-primary select-coupon-btn" data-code="${coupon.code}">Apply</button>
                                </div>
                            </div>`;
                        $list.append(html);
                    });
                } else {
                    $list.html(`<p class="empty-state">${window.loycartPOS?.strings?.noActiveCouponsFound || 'No active coupons found.'}</p>`);
                }
            })
            .fail(function() {
                $spinner.removeClass('is-active');
                showNotification(window.loycartPOS?.strings?.failedLoadCoupons || 'Failed to load coupons.', 'error');
            });
    },

    applyCouponToCart: function(code) {
        window.appliedCoupon = code;

        // Visual confirmation
        showNotification((window.loycartPOS?.strings?.couponSelected || 'Coupon %s selected.').replace('%s', code), 'success');

        // FORCE the cart to recalculate
        if (loycart.cartManager && typeof loycart.cartManager.getCartTotalsFromServer === 'function') {
            loycart.cartManager.getCartTotalsFromServer();
        } else {
            console.error('Cart Manager not found or getCartTotalsFromServer missing');
        }

        // Save cart state with the applied coupon
        if (loycart.cartManager && typeof loycart.cartManager.debouncedSaveCartToServer === 'function') {
            loycart.cartManager.debouncedSaveCartToServer();
        }

        closeModal(jQuery('#coupon-picker-modal'));
    }
};

// Start the module
loycart.couponManager.init();