/**
 * LoyCart POS - Cart Manager Module
 * 
 * Handles all cart operations including:
 * - Adding/removing items from cart
 * - Calculating cart totals (local & server-side)
 * - Displaying cart UI with proper formatting
 * - Managing discounts and refund calculations
 * - Syncing with server for accurate pricing
 * 
 * @namespace loycart.cartManager
 */
(function($) {
    // Ensure the namespace exists before defining the module
    window.loycart = window.loycart || {};

    loycart.cartManager = {
        // Helper: compare variation attributes robustly
        _areAttributesEqual: function(a, b) {
            const isEmpty = (x) => (
                x == null ||
                (Array.isArray(x) && x.length === 0) ||
                (typeof x === 'object' && !Array.isArray(x) && Object.keys(x).length === 0)
            );

            if (isEmpty(a) && isEmpty(b)) return true;

            const toObject = (x) => {
                if (x == null) return {};
                if (Array.isArray(x)) {
                    // Convert array to keyed object for stable comparison
                    return x.reduce((acc, val, idx) => { acc[String(idx)] = String(val); return acc; }, {});
                }
                // Ensure string values for stable compare
                return Object.fromEntries(Object.entries(x).map(([k, v]) => [String(k), String(v)]));
            };

            const objA = toObject(a);
            const objB = toObject(b);
            const keysA = Object.keys(objA).sort();
            const keysB = Object.keys(objB).sort();
            if (keysA.length !== keysB.length) return false;
            for (let i = 0; i < keysA.length; i++) {
                if (keysA[i] !== keysB[i]) return false;
                if (objA[keysA[i]] !== objB[keysB[i]]) return false;
            }
            return true;
        },

        /**
         * Clear cart, customer selection, coupons and reset UI smoothly without reload
         */
        clearCartAndCustomer: function() {
    // ...existing code...
    const self = this;

    // Re-enable products that were in the cart before clearing
    const cartBeforeClearing = [...loycart.cart]; // Copy the cart before clearing
    if (loycart.productManager && cartBeforeClearing.length > 0) {
        // ...existing code...
        loycart.productManager.updateProductStock(cartBeforeClearing);
    }

    loycart.cart = [];
    // ...existing code...
    window.appliedCoupon = ''; 
    loycart.cashierDiscount = 0;

    return window.api_clearCartOnServer()
        .done(function() {
            // ...existing code...
            showNotification('Cart cleared.', 'info');

            // Reset customer UI
            window.$customerId.val('0');
            window.$selectedCustomerName.text('');
            window.$selectedCustomerIndicator.hide();
            window.$guestCustomerIndicator.show();
            jQuery('#customer-cart-indicator').addClass('is-guest');

            // Reset shipping
            window.selectedShipping = {
                id: null,
                cost: 0,
                label: ''
            };
            loycart.shippingManager.updateShippingCallToAction();

            // Reset order note
            window.orderNote = '';
            jQuery('#add-note-button').removeClass('note-added').html('<span class="dashicons dashicons-edit-page"></span> Add Note');

            // Update cart UI
            self.updateCart();
            // Refresh product grid to update disabled/enabled state of add-to-cart buttons
            if (typeof loycart.productManager !== 'undefined' && typeof loycart.productManager.loadProducts === 'function') {
                loycart.productManager.loadProducts();
            }
            self.resetTotalsUI();

            // Refresh held sales count and update hold/retrieve button state
            loycart.heldSalesManager.refreshHeldSales();

            // Reload product grid to ensure fresh state
            loycart.productManager.loadProducts();

            // ...existing code...
        })
        .fail(function() {
            // ...existing code...
            showNotification('Failed to clear cart on server.', 'error');
            // Still reset locally
            self.updateCart();
            self.resetTotalsUI();
            // Refresh held sales count and update button state even on failure
            loycart.heldSalesManager.refreshHeldSales();
        });
},

        resetTotalsUI: function () {
    const zeroPrice = formatPrice(0);
    
    jQuery('#cart-subtotal').text(zeroPrice);
    jQuery('#cart-total-amount').text(zeroPrice).removeClass('calculating');
    jQuery('#header-cart-total .total-amount').text(zeroPrice);
    

    // Ensure discount and coupon rows are hidden when cart is empty
    jQuery('#wc-coupon-row').hide();
    jQuery('#cashier-discount-row').hide();

    if (window.$cartDiscountRow) window.$cartDiscountRow.hide();
    if (window.$cartTaxRow) window.$cartTaxRow.hide();
    jQuery('#cart-tax-row').hide();
    jQuery('#cart-shipping-row').hide();
    jQuery('#cart-shipping-tax-row').hide();
    
    jQuery('#checkout-button')
        .text(loycartPOS.strings.checkoutDefaultText)
        .prop('disabled', true)
        .attr('aria-disabled', 'true');

    window.splitPaymentTotalDue = 0;
},

        updateCart: function (isInitialLoad = false) {
            // ...existing code...
            if (!window.$cartItems) {
                // ...existing code...
                return;
            }

            // Collect editing states before re-rendering
            const editingIndices = [];
            window.$cartItems.find('.cart-item.is-editing').each(function() {
                editingIndices.push(jQuery(this).data('index'));
            });

            // Clear cart items container before rendering
            window.$cartItems.empty();

            clearTimeout(window.cartTotalsTimer);

            let clientSubtotal = 0;
            let clientDiscountTotal = 0;

            if (!loycart.cart || !Array.isArray(loycart.cart) || loycart.cart.length === 0) {
                // ...existing code...
                window.isCartLoading = false;
    this.resetTotalsUI();
    
    
    window.$cartItems.append(`
        <div class="empty-cart">
            <span class="dashicons dashicons-cart"></span>
            <p>${window.strings.cartIsEmpty || 'Your cart is empty!'}</p>
            <small>Start by scanning or searching for a product.</small>
        </div>
    `);
    jQuery('#checkout-button').prop('disabled', true).attr('aria-disabled', 'true');
    jQuery('#add-shipping-button').prop('disabled', true).attr('aria-disabled', 'false');
    window.selectedShipping = { id: null, cost: 0, label: '' };
} else {

    window.isCartLoading = false;


    jQuery('#add-shipping-button').prop('disabled', false).attr('aria-disabled', 'false');

    loycart.cart.forEach((item, index) => {
                    const itemQuantity = parseInt(item.quantity) || 0;
                    const itemDiscountPercent = parseFloat(item.discount_percent) || 0;
                    const isShipping = item.is_shipping_refund || false;


                    if (isShipping) {
                        return;
                    }

                    let priceHtml = '';
                    let itemTotal;
                    let imageHtml = '';

                    imageHtml = `<img src="${item.image}" alt="${item.name}" class="cart-item-thumbnail" onerror="this.onerror=null;this.src='${loycartPOS.placeholder_image_url}';">`;

                   
if (window.isReturnMode) {
    // In refund mode, show NET price (without tax) - tax is shown separately in totals
    const pricePaidNet = parseFloat(item.price) || 0;
    const taxPaidPerUnit = parseFloat(item.tax_paid) || 0;
    itemTotal = pricePaidNet * itemQuantity; // Display NET total (tax shown separately)
    priceHtml = `<span class="current-price">${formatPrice(pricePaidNet)}</span>`; // Display NET price only
} else {
   
    const itemPrice = parseFloat(item.price) || 0;
    const itemQuantity = parseInt(item.quantity) || 0;

    const itemDiscountPercent = parseFloat(item.discount_percent) || 0;
    const fixedDiscountAmount = parseFloat(item.discount_amount || 0); // New flat discount field

    let netUnitPrice = itemPrice;
    let totalDiscountValue = 0;

    if (fixedDiscountAmount > 0) {
        netUnitPrice = itemPrice - fixedDiscountAmount;
        totalDiscountValue = fixedDiscountAmount * itemQuantity;
    } else if (itemDiscountPercent > 0) {

        netUnitPrice = itemPrice * (1 - itemDiscountPercent / 100);
        totalDiscountValue = (itemPrice * itemQuantity * itemDiscountPercent) / 100;
    }

    itemTotal = netUnitPrice * itemQuantity;
    clientSubtotal += (itemPrice * itemQuantity);
    clientDiscountTotal += totalDiscountValue;

    if (fixedDiscountAmount > 0) {
    const cashierPercent = Math.round((fixedDiscountAmount / itemPrice) * 100);
    priceHtml = `
        <div>
            <span class="original-price">${formatPrice(itemPrice)}</span> 
            <span class="current-price" style="font-weight: bold; color: #000;">${formatPrice(netUnitPrice)}</span>
        </div>
        <div class="discount-details" style="color: #28a745;">Cashier Discount ${cashierPercent}% Saving ${formatPrice(totalDiscountValue)}</div>
    `;
     } else if (itemDiscountPercent > 0) {
        priceHtml = `
            <div>
                <span class="original-price">${formatPrice(itemPrice)}</span> 
                <span class="current-price">${formatPrice(netUnitPrice)}</span>
            </div>
            <div class="discount-details">Cashier Discount ${itemDiscountPercent}% Saving ${formatPrice(totalDiscountValue)}</div>
        `;
    } else if (item.on_sale && item.regular_price > item.sale_price) {
        const saleSaving = item.regular_price - item.sale_price;
        const salePercent = (item.regular_price > 0) ? Math.round((saleSaving / item.regular_price) * 100) : 0;
        priceHtml = `
            <div><span class="original-price">${formatPrice(item.regular_price)}</span> <span class="current-price">${formatPrice(item.sale_price)}</span></div>
            <div class="discount-details">Sale Discount ${salePercent}% Saving ${formatPrice(saleSaving)}</div>
        `;
    } else {
        priceHtml = `<span class="current-price">${formatPrice(itemPrice)}</span>`;
    }
}


  let variationDetails = '';
if (item.variation_id && item.attributes && Object.keys(item.attributes).length > 0) {
    variationDetails = Object.keys(item.attributes)
        .map(key => {
            const value = item.attributes[key];
            // Skip empty values or values that match the product name
            if (!value || value === item.name) return '';
            
            let label = key.replace('attribute_pa_', '').replace('attribute_', '');
            
            label = label.split(/[-_]/).map(word => word.charAt(0).toUpperCase() + word.slice(1)).join(' ');
            
            return `<span class="cart-item-attribute">${label}: ${value}</span>`;
        })
        .filter(Boolean)
        .join(', ');
    if (variationDetails) variationDetails = `<div class="cart-item-variation-details">${variationDetails}</div>`;
}

                   
const isItemOnSale = item.on_sale === true || (item.regular_price > item.price);


const discountButtonHtml = !isItemOnSale ? `
    <button class="cart-discount-btn" title="Apply Discount" aria-label="Apply discount to this item">
        <span class="dashicons dashicons-tag"></span>
    </button>` : ''; // Hide if on sale

const actionButtonHtml = window.isReturnMode ? 
    `<button class="cart-remove-btn direct-remove" title="Remove Item" aria-label="Remove this item from cart" style="color: #d63638;"><span class="dashicons dashicons-trash"></span></button>` : 
    `<button class="cart-edit-toggle" title="Edit Item" aria-label="Edit this item"><span class="dashicons dashicons-edit"></span></button>`;


const $item_el = jQuery(`
    <div class="cart-item ${window.isReturnMode ? 'is-refund-item' : ''}" data-index="${index}" data-id="${item.id}" ${item.variation_id ? `data-variation-id="${item.variation_id}"` : ''}>
        <div class="cart-item-main">
            <div class="cart-item-thumbnail-wrapper">${imageHtml}</div>
            <div class="cart-item-details">
                <div class="cart-item-name">${item.is_custom ? item.custom_name : item.name}</div>
                ${variationDetails}
                <div class="cart-item-price-wrapper">${priceHtml}</div>
            </div>
            <div class="cart-item-subtotal-wrapper">
                <div class="cart-item-subtotal">
                    <div class="final-gross-total">
                        <small style="font-weight: normal; margin-right: 5px; color: #666;">x${Math.abs(itemQuantity)}</small> 
                        ${formatPrice(Math.abs(itemTotal))}
                    </div>
                </div>
                ${actionButtonHtml}
            </div>
        </div>
        
        ${!window.isReturnMode ? `
        <div class="cart-controls-wrapper is-hidden">
            <div class="cart-quantity-controls">
                ${!item.sold_individually ? `
                    <button class="cart-quantity-btn minus" data-action="decrease">-</button>
                    <span class="cart-item-quantity-display">${Math.abs(item.quantity)}</span>
                    <button class="cart-quantity-btn plus" data-action="increase">+</button>
                ` : `<span class="sold-individually-note">Limited to one per customer</span>`}
            </div>
            <div class="cart-action-controls">
                ${discountButtonHtml}
                <button class="cart-remove-btn" title="Remove Item" aria-label="Remove this item from cart">
                    <span class="dashicons dashicons-trash"></span>
                </button>
            </div>
        </div>` : ''}
    </div>
`);
                    // Animate slide-in for new cart items
                    $item_el.css({opacity: 0, transform: 'translateY(30px)'});
                    window.$cartItems.append($item_el);
                    setTimeout(() => {
                        $item_el.css({transition: 'opacity 0.3s, transform 0.3s', opacity: 1, transform: 'translateY(0)'});
                    }, 10);
                });
                jQuery('#checkout-button').prop('disabled', false).attr('aria-disabled', 'false');
            }

            // Re-apply editing states after re-rendering
            editingIndices.forEach(idx => {
                const $item = window.$cartItems.find(`.cart-item[data-index="${idx}"]`);
                if ($item.length) {
                    $item.addClass('is-editing');
                    $item.find('.cart-controls-wrapper').slideDown(200).css('display', 'flex');
                }
            });

            const totalItemsCount = loycart.cart.reduce((sum, item) => sum + Math.abs(parseInt(item.quantity) || 0), 0);
            jQuery('.header-item-count').text(totalItemsCount);

            
            const clientShipping = window.selectedShipping.cost || 0;
            const currentTotal = window.isReturnMode ? 
                Math.abs(loycart.cart.reduce((sum, i) => sum + ((parseFloat(i.price) + (parseFloat(i.tax_paid) || 0)) * i.quantity), 0)) : 
                (clientSubtotal - clientDiscountTotal + clientShipping);

            jQuery('#header-cart-total .total-amount').text(formatPrice(currentTotal));

            
            if (isInitialLoad) {
                this.getCartTotalsFromServer();
            } else {
                window.cartTotalsTimer = setTimeout(() => { this.getCartTotalsFromServer(); }, 250);
            }
            
            this.debouncedSaveCartToServer();

            if (window.isReturnMode && loycart.refundManager) {
                loycart.refundManager.updateRefundableItemState();
            }

            // Update hold/retrieve button state whenever cart changes
            loycart.heldSalesManager.updateHoldRetrieveButton();
        },

        invalidateSelectedShipping: function () {
            if (window.selectedShipping.id !== null) {
                window.selectedShipping = { id: null, cost: 0, label: '' };
                showNotification('Shipping was reset due to a cart change. Please re-calculate.', 'warning');
            }
        },

        checkStock: function (product, quantity, variation = null) {
            const productToCheck = variation || product;
            if (productToCheck.manage_stock === false) {
                return true;
            }
            const currentCartQuantity = loycart.cart.reduce((acc, item) => {
                if ((variation && item.variation_id === variation.id) || (!variation && item.id === product.id)) {
                    return acc + item.quantity;
                }
                return acc;
            }, 0);

            if (productToCheck.stock_quantity !== null && (currentCartQuantity + quantity) > productToCheck.stock_quantity) {
                showNotification(window.strings.notEnoughStock.replace('%s', productToCheck.stock_quantity), 'error');
                return false;
            }
            return true;
        },

        addToCart: function (product, quantity = 1, variationId = null, attributes = {}) {

            if (window.isReturnMode) {
                // Don't add shipping as a cart item - it's handled in selectedShipping
                if (product.is_shipping_refund) {
                    return;
                }
                
                quantity = -Math.abs(quantity);

                const originalItemId = product.original_item_id;
                const pricePaid = parseFloat(product.price_paid);
                const qtyRefundable = parseInt(product.qty_refundable);
                const itemAttributes = product.attributes;

                if (!originalItemId) {
                    showNotification('This is not a valid refundable item.', 'error');
                    return;
                }

                const existingCartItem = loycart.cart.find(item => item.original_item_id === originalItemId);
                const qtyInCart = existingCartItem ? Math.abs(existingCartItem.quantity) : 0;

                if (qtyInCart + Math.abs(quantity) > qtyRefundable) {
                    showNotification(`Cannot return more items than were purchased. ${qtyRefundable} available.`, 'error');
                    return;
                }

                if (existingCartItem) {
                    existingCartItem.quantity += quantity;
                } else {
                    loycart.cart.push({
                        id: product.id,
                        variation_id: product.variation_id,
                        original_item_id: originalItemId,
                        name: product.name,
                        image: product.image,
                        price: pricePaid,
                        quantity: quantity,
                        attributes: itemAttributes,
                        discount_percent: product.discount_percent || 0,
                        is_shipping_refund: false,
                        original_price: product.original_price ? parseFloat(product.original_price) : pricePaid,
                        tax_paid: product.tax_paid || 0,
                    });
                }
                this.updateCart();
                showNotification('Item added to return.', 'success');
                if (loycart.refundManager) loycart.refundManager.updateRefundableItemState();
                return;
            }

            if (window.isProcessing) {
                return;
            }

            const soldIndividually = product.sold_individually || false;
            if (soldIndividually) {
                if (loycart.cart.some(item => (item.id === product.id || (variationId && item.variation_id === variationId)))) {
                    showNotification('This product is limited to one per order.', 'error');
                    return;
                }
                quantity = 1;
            }

            const variationData = variationId ? product.variations?.find(v => v.id === variationId) : null;

            if (!this.checkStock(product, quantity, variationData)) {
                return;
            }

            let itemToAdd = {
                id: product.id,
                name: product.name,
                custom_name: product.name,
                is_custom: product.is_custom || false,
                price: parseFloat(product.price),
                quantity: parseInt(quantity),
                discount_percent: 0,
                image: product.image || '',
                sold_individually: soldIndividually,
                on_sale: product.on_sale,
                regular_price: (product.regular_price && product.regular_price !== '') ? parseFloat(product.regular_price) : parseFloat(product.price),
                sale_price: (product.sale_price && product.sale_price !== '') ? parseFloat(product.sale_price) : parseFloat(product.price),
                is_virtual: product.is_virtual || false,
                variation_id: variationId,
                attributes: attributes
            };

            if (variationData) {
                itemToAdd.price = parseFloat(variationData.price);
                itemToAdd.image = variationData.image || product.image;
                itemToAdd.sold_individually = variationData.sold_individually || soldIndividually;
                itemToAdd.on_sale = variationData.on_sale;
                itemToAdd.regular_price = (variationData.regular_price && variationData.regular_price !== '') ? parseFloat(variationData.regular_price) : parseFloat(variationData.price);
                itemToAdd.sale_price = (variationData.sale_price && variationData.sale_price !== '') ? parseFloat(variationData.sale_price) : parseFloat(variationData.price);
                itemToAdd.is_virtual = variationData.is_virtual || false;
            }

    const existingItemIndex = soldIndividually ? -1 :
    loycart.cart.findIndex(item =>
        item.id === itemToAdd.id &&
        (Number(item.variation_id) || 0) === (Number(itemToAdd.variation_id) || 0) &&
        // Robust compare: treat empty arrays/objects equally and ignore key order
        loycart.cartManager._areAttributesEqual(item.attributes, itemToAdd.attributes) &&
        // Merge only if discount amounts/percents match exactly
        (Number(item.discount_amount) || 0) === (Number(itemToAdd.discount_amount) || 0) &&
        (Number(item.discount_percent) || 0) === (Number(itemToAdd.discount_percent) || 0)
    );

            if (existingItemIndex > -1) {
                loycart.cart[existingItemIndex].quantity += itemToAdd.quantity;
                
            } else {
                loycart.cart.push(itemToAdd);
                
            }

            this.updateCart();
            // Restore: reload product grid after cart change
            if (typeof loycart.productManager !== 'undefined' && typeof loycart.productManager.loadProducts === 'function') {
                loycart.productManager.loadProducts();
            }
            showNotification(window.strings.productAddedToCart.replace('%s', itemToAdd.name), 'success');

            if (soldIndividually) {
                const targetId = variationId || product.id;
                if (window.$productsList) {
                    const $productCard = window.$productsList.find(`.product[data-id="${targetId}"]`);
                    if ($productCard.length) {
                        $productCard.addClass('disabled').attr('aria-disabled', 'true');
                    }
                }
            }
        },

        handleQuantityChange: function ($button) {
            const index = $button.closest('.cart-item').data('index');
            const action = $button.data('action');
            if (loycart.cart[index]) {
                if (action === 'increase') {
                    const product = loycart.cart[index];
                    const mainProductData = window.$productsList ? window.$productsList.find(`.product[data-id="${product.id}"]`).data('product-data') || product : product;
                    const variationData = product.variation_id ? mainProductData.variations?.find(v => v.id === product.variation_id) : null;

                    if (!this.checkStock(mainProductData, 1, variationData)) {
                        return;
                    }
                    loycart.cart[index].quantity++;

                } else if (action === 'decrease' && loycart.cart[index].quantity > 1) {
                    loycart.cart[index].quantity--;
                } else {
                    return;
                }
                this.invalidateSelectedShipping();
                this.updateCart();
            }
        },

        handleRemoveItem: function ($button) {
            const $itemToRemove = $button.closest('.cart-item');
            const index = $itemToRemove.data('index');
            const itemData = loycart.cart[index];

            if (!itemData) return;

            $itemToRemove.addClass('removing');

            setTimeout(() => {
                if (itemData.sold_individually && window.$productsList) {
                    const productId = itemData.variation_id || itemData.id;
                    const $productCard = window.$productsList.find(`.product[data-id="${productId}"]`);
                    if ($productCard.length) {
                        $productCard.removeClass('disabled').attr('aria-disabled', 'false');
                        $productCard.find('.add-to-cart-btn').prop('disabled', false);
                    }
                }
                this.invalidateSelectedShipping();
                loycart.cart.splice(index, 1);
                this.updateCart();
                // Restore: reload product grid after cart change
                if (typeof loycart.productManager !== 'undefined' && typeof loycart.productManager.loadProducts === 'function') {
                    loycart.productManager.loadProducts();
                }
                showNotification(window.strings.itemRemoved, 'success');
            }, 300);
        },

        handleDiscountClick: function ($button) {
            window.selectedProductId = $button.closest('.cart-item').data('index');
            const productInCart = loycart.cart[window.selectedProductId];
            if (!productInCart) return;

            if (window.$discountValueDisplay) window.$discountValueDisplay.text(productInCart.discount_percent || 0);
            const savings = (parseFloat(productInCart.price) * productInCart.quantity * (parseFloat(productInCart.discount_percent) || 0)) / 100;
            if (window.$discountPreviewAmount) window.$discountPreviewAmount.text(formatPrice(savings));
            jQuery('#discount-product-image').attr('src', productInCart.image);
            jQuery('#discount-product-name').text(productInCart.name);
            jQuery('#discount-product-price').text(formatPrice(productInCart.price)).data('original-price', parseFloat(productInCart.price));
            openModal(window.$discountModal);
        },

getCartTotalsFromServer: function (retryCount = 0) {
    const deferred = jQuery.Deferred();
    
  
    if (loycart.cart.length === 0 && !window.selectedShipping.id) {
        this.resetTotalsUI();
        return deferred.resolve().promise();
    }

    jQuery('#cart-total-amount').addClass('calculating');

    const MAX_RETRIES = 2;
    const self = this;

   
    if (this.totalsTimeout) clearTimeout(this.totalsTimeout);
    this.totalsTimeout = setTimeout(() => {
        api_getCartTotals(
            JSON.stringify(loycart.cart), 
            window.$customerId.val() || '0', 
            window.selectedShipping, 
            window.originalOrderId,
            window.appliedCoupon 
        )
        .done(function (response) {
            if (response.success) {
                const t = response.data;
                const taxTotal = parseFloat(t.total_tax) || 0;
                const finalTotal = parseFloat(t.total) || 0;
                const displayTotal = window.isReturnMode ? Math.abs(finalTotal) : Math.abs(finalTotal);
                const formattedTotal = formatPrice(displayTotal);

                
                window.splitPaymentTotalDue = displayTotal;

                
                if (window.$cartTax && window.$cartTaxRow) {
                    window.$cartTax.text(formatPrice(Math.abs(taxTotal)));
                    if (Math.abs(taxTotal) > 0.001) {
                        window.$cartTaxRow.show();
                    } else {
                        window.$cartTaxRow.hide();
                    }
                }
                
                
                if (window.$cartSubtotal) {
                    // For refunds, mirror the receipt: show subtotal BEFORE discount
                    let finalSubtotalDisplay = window.isReturnMode
                        ? Math.abs(parseFloat(t.gross_subtotal_before_discount || t.subtotal) || 0)
                        : Math.abs(parseFloat(t.subtotal) || 0);
                    
                    const taxLabel = loycartPOS.tax_label || 'Tax';
                    if (loycartPOS.tax_display_mode === 'incl') {
                        jQuery('#tax-label-text').text(`${taxLabel} Included:`);
                    } else {
                        jQuery('#tax-label-text').text(`${taxLabel}:`);
                    }
                    window.$cartSubtotal.text(formatPrice(finalSubtotalDisplay));
                }

                
                if (Math.abs(parseFloat(t.shipping_total) || 0) > 0.001) {
                    const shippingLabel = window.selectedShipping && window.selectedShipping.label ? window.selectedShipping.label : 'Shipping';
                    const shippingCost = Math.abs(parseFloat(t.shipping_total) || 0);
                    const shippingTax = Math.abs(parseFloat(t.shipping_tax) || 0);
                    const removeBtnHtml = window.isReturnMode ? '' : '<button id="remove-cart-shipping" class="remove-total-item">×</button>';
                    jQuery('#cart-shipping-row').html(`
                        <div class="total-label">${shippingLabel}:</div>
                        <div class="total-value">
                            <span id="cart-shipping-total">${formatPrice(shippingCost)}</span>
                            ${removeBtnHtml}
                        </div>
                    `).show();
                    
                    // Show shipping tax as a separate line only in normal sale mode.
                    // In refund mode, shipping tax is combined into the VAT line (t.total_tax)
                    if (!window.isReturnMode && shippingTax > 0.001) {
                        jQuery('#cart-shipping-tax-row').html(`
                            <div class="total-label">Shipping Tax:</div>
                            <div class="total-value">
                                <span id="cart-shipping-tax-total">${formatPrice(shippingTax)}</span>
                            </div>
                        `).show();
                    } else {
                        jQuery('#cart-shipping-tax-row').hide();
                    }
                } else {
                    jQuery('#cart-shipping-row').hide();
                    jQuery('#cart-shipping-tax-row').hide();
                }

                // Display cashier and coupon discounts (both in normal and refund mode)
                const cashierDisc = parseFloat(t.cashier_discount) || 0;
                const couponDisc = parseFloat(t.coupon_discount) || 0;

                // Always show cashier discount if present
                if (Math.abs(cashierDisc) > 0.001) {
                    jQuery('#cashier-discount-val').text(`-${formatPrice(Math.abs(cashierDisc))}`);
                    jQuery('#cashier-discount-row').show();
                } else {
                    jQuery('#cashier-discount-row').hide();
                }

                // Always show coupon discount if present (refund or normal mode)
                if (Math.abs(couponDisc) > 0.001) {
                    const activeCode = window.appliedCoupon || "Applied"; 
                    const $couponRow = jQuery('#wc-coupon-row');
                    $couponRow.children('span').first().text(`Coupon (${activeCode}):`);
                    jQuery('#wc-coupon-val').text(`-${formatPrice(Math.abs(couponDisc))}`);
                    $couponRow.show();
                } else {
                    jQuery('#wc-coupon-row').hide();
                }

                
                const $mainTotal = jQuery('#cart-total-amount');
                if ($mainTotal.length) {
                    $mainTotal.text(formattedTotal).removeClass('calculating');
                }
                
                const btnPrefix = window.isReturnMode ? 'Process Refund ' : 'Charge ';
                const $checkoutBtn = jQuery('#checkout-button');
                $checkoutBtn.text(btnPrefix + formattedTotal);
                
                if (displayTotal > 0) {
                    $checkoutBtn.prop('disabled', false).attr('aria-disabled', 'false');
                } else {
                    $checkoutBtn.prop('disabled', true).attr('aria-disabled', 'true');
                }
                
                jQuery('#header-cart-total .total-amount').text(formattedTotal);
                deferred.resolve(response);

            } else if (retryCount < MAX_RETRIES) {
                setTimeout(() => self.getCartTotalsFromServer(retryCount + 1), 2000);
            } else {
                showNotification('Error calculating totals.', 'error');
                deferred.reject(response);
            }
        })
        .fail(function (jqXHR) {
            if (retryCount < MAX_RETRIES) {
                setTimeout(() => self.getCartTotalsFromServer(retryCount + 1), 2000);
            } else {
                deferred.reject(jqXHR);
            }
        });
    }, 100);

    return deferred.promise();
},

resetPOSStateAfterSale: function (soldItems) {
    const clearPromise = api_clearCartOnServer();
    loycart.cart = [];
    window.selectedShipping = { id: null, cost: 0, label: '' };
    window.guestShippingAddress = null;


    window.appliedCoupon = ''; 
    const $couponRow = jQuery('#wc-coupon-row');
    if ($couponRow.length) {
        $couponRow.children('span').first().text('Coupon:'); 
        $couponRow.hide();
    }
    

    this.resetTotalsUI();
    this.updateCart();

    window.orderNote = '';
    jQuery('#add-note-button').removeClass('note-added').html('<span class="dashicons dashicons-edit-page"></span> Add Note');

    jQuery('#remove-customer-from-cart').trigger('click');
    if (loycart.productManager) loycart.productManager.updateProductStock(soldItems);

    if (window.$productsList) {
        window.$productsList.find('.product.disabled').each(function () {
            const $productEl = jQuery(this);
            const productData = $productEl.data('product-data');
            if (productData && productData.sold_individually && productData.stock_status !== 'outofstock') {
                $productEl.removeClass('disabled').attr('aria-disabled', 'false').attr('tabindex', '0');
            }
        });
    }

    return clearPromise;
},
        
        printReceipt: function(response, paperWidth) {
    const self = this;
   
    const receiptHtml = generateReceiptHTML(response, loycart.cart, paperWidth);

    
    const $tempDiv = jQuery('<div style="display:none;">').append(receiptHtml);
    jQuery('body').append($tempDiv);

    
    const barcodeSvg = $tempDiv.find('#barcode')[0];
    if (barcodeSvg && typeof JsBarcode !== 'undefined') {
        JsBarcode(barcodeSvg, response.order_id.toString(), {
            format: "CODE128",
            width: 2,
            height: 40,
            displayValue: false
        });
    }

    const finalReceiptHtml = $tempDiv.html();
    $tempDiv.remove();

    
    return finalReceiptHtml;
},

        debouncedSaveCartToServer: debounce(function () {
            const customerId = window.$customerId ? window.$customerId.val() || '0' : '0';
            if (loycart.cart.length === 0 && customerId === '0' && window.selectedShipping.id === null) {
                api_clearCartOnServer();
                return;
            }

            // Saving indicator handled via notifications; no inline status badge

            const stateToSave = {
                cart: loycart.cart,
                customerId: customerId,
                customerName: window.$selectedCustomerName ? window.$selectedCustomerName.text() || '' : '',
                selectedShipping: window.selectedShipping,
                guestShippingAddress: window.guestShippingAddress,
                appliedCoupon: window.appliedCoupon || '',
                cartSessionId: window.cartSessionId || '',
                lastModified: Date.now()
            };

            api_saveCartToServer(stateToSave)
                .done(function() {
                    showNotification('Cart saved.', 'success');
                })
                .fail(function () {
                    showNotification('Cart not saved. Check connection.', 'error');
                });
        }, 1000),

        loadCartFromServer: function () {
            const self = this;
            return api_loadCartFromServer()
                .then(function (loadResponse) {
                    if (loadResponse.success) {
                        const savedState = loadResponse.data.data;
                        const serverLastModified = loadResponse.data.lastModified || 0;
                        
                        // Conflict detection: warn if server state is newer
                        if (serverLastModified > 0 && window.lastLoadedTimestamp && serverLastModified > window.lastLoadedTimestamp + 5000) {
                            showNotification('Stale cart state detected; server has newer version.', 'warning');
                            
                        }
                        
                        if (savedState && Array.isArray(savedState.cart) && savedState.cart.length > 0) {
                            return api_validate_cart(savedState.cart).then(function (validationResponse) {
                                return { savedState: savedState, validation: validationResponse, serverLastModified: serverLastModified };
                            });
                        } else {
                            const emptyState = { customerId: '0', customerName: '', cart: [], selectedShipping: { id: null, cost: 0, label: '' }, guestShippingAddress: null, appliedCoupon: '', cartSessionId: window.cartSessionId || '', lastModified: Date.now() };
                            return { savedState: emptyState, validation: null, serverLastModified: serverLastModified };
                        }
                    } else {
                        return jQuery.Deferred().reject(loadResponse.data || "Failed to load cart state.");
                    }
                })
                .done(function (responses) {
                    const savedState = responses.savedState;
                    const validation = responses.validation;
                    window.lastLoadedTimestamp = responses.serverLastModified || Date.now();

                    if (validation && validation.success) {
                        loycart.cart = validation.data.cart;
                    } else {
                        loycart.cart = savedState.cart;
                    }

                    const savedCustomerId = savedState.customerId || '0';
                    const customerName = savedState.customerName || '';

                    if (savedCustomerId !== '0' && customerName !== '') {
                        if (loycart.customerManager) loycart.customerManager.addSelectedCustomerToCart(savedCustomerId, customerName);
                    } else {
                        jQuery('#remove-customer-from-cart').trigger('click');
                    }

                    window.selectedShipping = savedState.selectedShipping || { id: null, cost: 0, label: '' };
                    window.guestShippingAddress = savedState.guestShippingAddress || null;
                    window.appliedCoupon = savedState.appliedCoupon || '';
                    window.cartSessionId = savedState.cartSessionId || window.cartSessionId || '';
                    
                    // FIX: Indicate initial load to bypass debounce on refresh
                    self.updateCart(true); 
                })
                .fail(function (jqXHR) {
                    let errorMsg = typeof jqXHR === 'string' ? jqXHR : "Could not connect to load cart state.";
                    showNotification(errorMsg, "error");
                    loycart.cart = [];
                    window.selectedShipping = { id: null, cost: 0, label: '' };
                    window.appliedCoupon = '';
                    jQuery('#remove-customer-from-cart').trigger('click');
                    self.updateCart();
                });
        }
    };
})(jQuery);
// --- END CART MANAGER MODULE ---