// --- SETTINGS PERSISTENCE LOGIC (like cart) ---
const debouncedSaveSettingsToServer = debounce(function () {
    // Gather settings from modal fields (if present), else from in-memory object
    // Always include every setting, even if unchecked/empty
    const payload = {
        store_credit: {
            enabled: jQuery('#pos-setting-credit-enabled').is(':checked') ? true : false,
            earn_percent: jQuery('#pos-setting-earn-percent').val() || '',
            apply_online: jQuery('#pos-setting-apply-online').is(':checked') ? true : false,
            exclude_sale_items_earning: jQuery('#pos-setting-exc-earn').is(':checked') ? true : false,
            exclude_sale_items_redeeming: jQuery('#pos-setting-exc-redeem').is(':checked') ? true : false
        },
        receipt: {
            store_name: jQuery('#pos-setting-store-name').val() || '',
            store_phone: jQuery('#pos-setting-store-phone').val() || '',
            footer_message: jQuery('#pos-setting-receipt-footer').val() || '',
            paper_width: jQuery('#pos-setting-paper-width').val() || '80mm',
            usb_direct_print: jQuery('#pos-setting-usb-direct-print').is(':checked') ? true : false
        },
        ui: {
            products_per_page: parseInt(jQuery('#pos-setting-products-per-page').val(), 10) || 24,
            default_sort: jQuery('#pos-setting-default-sort').val() || 'popular',
            default_theme: jQuery('#pos-setting-default-theme').val() || 'light',
            show_theme_toggle: jQuery('#pos-setting-show-theme-toggle').is(':checked') ? true : false,
            show_notifications: jQuery('#pos-setting-show-notifications').is(':checked') ? true : false,
            auto_redirect_pos: jQuery('#pos-setting-auto-redirect-pos').is(':checked') ? true : false
        },
        visibility: {
            allow_private_detail_lookups: jQuery('#pos-setting-allow-private').is(':checked') ? true : false
        }
    };
    console.log('[POS] Saving settings to server:', JSON.parse(JSON.stringify(payload)));
    if (window.api_savePosSettings) {
        window.api_savePosSettings(payload)
            .done(function(resp){
                console.log('[POS] Save settings response:', resp);
                if (resp && resp.success) {
                    showNotification('Settings saved.', 'success');
                } else {
                    showNotification('Failed to save settings.', 'error');
                }
            })
            .fail(function(xhr, status, error){
                console.error('[POS] Error saving settings:', status, error, xhr && xhr.responseText);
                showNotification('Error saving settings.', 'error');
            });
    }
}, 1000);
jQuery(document).ready(function ($) {
    // Load POS settings on startup for persistent UI options, then load products only after settings are ready
    window.api_getPosSettings()
        .done(function(resp){
            if (resp && resp.success && resp.data) {
                if (!window.loycartPOS) window.loycartPOS = {};
                window.loycartPOS.settings = resp.data;
                // --- Apply UI defaults from server settings to main UI ---
                const ui = resp.data.ui || {};
                if (typeof ui.default_sort !== 'undefined') {
                    window.currentSort = ui.default_sort;
                    if (window.$sortSelect) {
                        window.$sortSelect.val(ui.default_sort).trigger('change');
                    }
                }
                if (typeof ui.products_per_page !== 'undefined') {
                    window.itemsPerPage = parseInt(ui.products_per_page, 10) || 24;
                }
                // Add more UI syncs here as needed
            }
            // Now that settings are loaded and itemsPerPage is set, load products
            if (window.loycart && window.loycart.productManager && typeof window.loycart.productManager.loadProducts === 'function') {
                window.loycart.productManager.loadProducts(1, false);
            }
        })
        .fail(function(xhr, status, error){
            console.error('[POS] Error loading settings:', status, error, xhr && xhr.responseText);
        });
    // --- POS HEADER DATE/TIME UPDATER ---
    function updatePosHeaderDateTime() {
    const dateSpan = document.getElementById('pos-current-date');
    const timeSpan = document.getElementById('pos-current-time');
    if (!dateSpan || !timeSpan) return;

    const now = new Date();

    // Day name (Mon, Tue, etc.)
    const weekday = now.toLocaleString('en-US', { weekday: 'short' });

    // Day of month with ordinal suffix (1st, 2nd, 3rd, 4th, ...)
    const day = now.getDate();
    const dayWithOrdinal = day + getOrdinalSuffix(day);

    // Month (short: Jan, Feb, Mar, ...)
    const month = now.toLocaleString('en-US', { month: 'short' });

    const year = now.getFullYear();

    // Final date line: "Tue 24th Feb 2026"
    dateSpan.textContent = `${weekday} ${dayWithOrdinal} ${month} ${year}`;

    // Time: HH:MM:SS
    const hours = now.getHours().toString().padStart(2, '0');
    const mins  = now.getMinutes().toString().padStart(2, '0');
    const secs  = now.getSeconds().toString().padStart(2, '0');
    timeSpan.textContent = `${hours}:${mins}:${secs}`;
}

// Helper function for st, nd, rd, th
function getOrdinalSuffix(n) {
    const lastDigit = n % 10;
    const lastTwo   = n % 100;

    if (lastTwo >= 11 && lastTwo <= 13) return 'th';
    switch (lastDigit) {
        case 1:  return 'st';
        case 2:  return 'nd';
        case 3:  return 'rd';
        default: return 'th';
    }
}

// Start updating
setInterval(updatePosHeaderDateTime, 1000);
updatePosHeaderDateTime();   // initial call
    
    
    
    // Scroll cart panel into view when header-cart-total is clicked
    $('#header-cart-total').on('click', function() {
        var cartPanel = document.querySelector('.cart-panel');
        if (cartPanel) {
            cartPanel.scrollIntoView({ behavior: 'smooth', block: 'end' });
        }
    });

    // --- ALL JQUERY ELEMENT VARIABLES (ATTACHED TO WINDOW FOR GLOBAL MODULE ACCESS) ---
    window.$createCustomerModal = jQuery('#create-customer-modal');
    window.$clearCartModal = jQuery('#clear-cart-modal');
    window.$refundOrderLookupModal = jQuery('#refund-order-lookup-modal');
    window.$refundCompleteModal = jQuery('#refund-complete-modal');
    window.$productsList = jQuery('#products-list');
    window.$cartItems = jQuery('#cart-items');
    window.$cartSubtotal = jQuery('#cart-subtotal');
    window.$cartDiscount = jQuery('#cart-discount');
    window.$cartDiscountRow = jQuery('#cart-discount-row');
    window.$cartTotal = jQuery('#cart-total');
    window.$productSearch = jQuery('#product-search');
    window.$categorySelect = jQuery('#category-select');
    window.$sortSelect = jQuery('#sort-select');
    window.$prevPage = jQuery('#prev-page');
    window.$nextPage = jQuery('#next-page');
    window.$currentPage = jQuery('#current-page');
    window.$itemsFrom = jQuery('#items-from');
    window.$itemsTo = jQuery('#items-to');
    window.$totalItems = jQuery('#total-items');
    window.$customerSearch = jQuery('#modal-customer-search');
    window.$customerResults = jQuery('#modal-customer-results');
    window.$customerModal = jQuery('#customer-modal');
    window.$guestCustomerIndicator = jQuery('#guest-customer-indicator');
    window.$selectedCustomerIndicator = jQuery('#selected-customer-indicator');
    window.$selectedCustomerName = jQuery('#selected-customer-name');
    window.$customerId = jQuery('#customer-id');
    window.$discountModal = jQuery('#discount-modal');
    window.$cashModal = jQuery('#cash-modal');
    window.$discountValueDisplay = jQuery('#discount-value');
    window.$discountPreviewAmount = jQuery('#discount-preview-amount');
    window.$amountTendered = jQuery('#amount-tendered');
    window.$modalCartTotal = jQuery('#modal-cart-total');
    window.$modalDiscount = jQuery('#modal-discount');
    window.$modalDiscountRow = jQuery('#modal-discount-row');
    window.$modalFinalTotal = jQuery('#modal-final-total');
    window.$modalChange = jQuery('#modal-change');
    window.$receiptModal = jQuery('#receipt-modal');
    window.$printReceipt = jQuery('#print-receipt');
    window.$dismissReceipt = jQuery('#dismiss-receipt');
    window.$cardConfirmModal = jQuery('#card-confirm-modal');
    window.$cardModalFinalTotal = jQuery('#card-modal-final-total');
    window.$cardModalConfirm = jQuery('#card-modal-confirm');
    window.$cardModalCancel = jQuery('#card-modal-cancel');
    window.$payCashButton = jQuery('#pay-cash');
    window.$payCardButton = jQuery('#pay-card');
    window.$payChequeButton = jQuery('#pay-cheque');
    window.$payBankButton = jQuery('#pay-bank');
    window.$holdSaleButton = jQuery('#hold-sale-button');
    window.$retrieveSaleButton = jQuery('#retrieve-sale-button');
    window.$heldSalesModal = jQuery('#held-sales-modal');
    window.$holdSaleModal = jQuery('#hold-sale-modal');
    window.$heldCartsList = jQuery('#held-carts-list');
    window.$cartTaxRow = jQuery('#cart-tax-row');
    window.$cartTax = jQuery('#cart-tax');
    window.$addNewProductModal = jQuery('#add-new-product-modal');
    window.$addNewProductValue = jQuery('#add-new-product-value');
    window.$shippingModal = jQuery('#shipping-modal');
    window.$productInfoModal = jQuery('#product-info-modal');
    window.$advancedPendingPaymentModal = jQuery('#advanced-pending-payment-modal');

    // --- Set initial sort-select value from settings/localStorage and trigger change ---
    if (window.$sortSelect && window.currentSort) {
        window.$sortSelect.val(window.currentSort).trigger('change');
    }
    // --- Force sort-select value and trigger at end of ready ---
    setTimeout(function() {
        if (window.$sortSelect && window.currentSort) {
            window.$sortSelect.val(window.currentSort).trigger('change');
        }
    }, 0);
    // --- DARK MODE TOGGLE ---
    window.toggleDarkMode = function() {
        const isDark = jQuery('body').hasClass('dark-mode');
        if (isDark) {
            jQuery('body').removeClass('dark-mode');
            // localStorage.setItem('loycart_dark_mode', 'off');
        } else {
            jQuery('body').addClass('dark-mode');
            // localStorage.setItem('loycart_dark_mode', 'on');
        }
        updateAdminBarButton();
    };
    
    function updateAdminBarButton() {
        const isDark = jQuery('body').hasClass('dark-mode');
        const $btn = jQuery('#wp-admin-bar-loycart-dark-mode a');
        if ($btn.length) {
            if (isDark) {
                $btn.html('☀️ Light Mode');
            } else {
                $btn.html('🌙 Dark Mode');
            }
        }
    }
    
    jQuery(document).ready(function() {
        // Restore dark mode preference on load
        // Always use server setting for dark mode
        const defaultDark = (window.loycartPOS.settings && window.loycartPOS.settings.ui && window.loycartPOS.settings.ui.default_theme === 'dark');
        jQuery('body').toggleClass('dark-mode', defaultDark);

        updateAdminBarButton();

        // Apply UI defaults from saved POS settings (only if localStorage not set)

        
        jQuery('#toggle-theme-button, #topbar-theme-toggle').show();

        if (typeof window.api_getPosSettings === 'function') {
            window.api_getPosSettings()
                .done(function(resp){
                    if (resp && resp.success && resp.data) {
                        const ui = resp.data.ui || {};
                        const rc = resp.data.receipt || {};
                        // Theme default: always use server
                        const defaultDark = (ui.default_theme === 'dark');
                        jQuery('body').toggleClass('dark-mode', defaultDark);
                        updateAdminBarButton();

                        // Show/hide admin bar theme toggle based on setting
                        jQuery('#wp-admin-bar-loycart-dark-mode').toggle(!!ui.show_theme_toggle);
                     
                        jQuery('#toggle-theme-button, #topbar-theme-toggle').show();

                        // Apply receipt-related settings (from server only)
                        if (typeof rc.paper_width !== 'undefined') {
                            const validWidths = ['80mm','58mm','A4','Letter'];
                            const width = validWidths.indexOf(rc.paper_width) !== -1 ? rc.paper_width : '80mm';
                            jQuery('#receipt-paper-width-indicator span').text(width);
                        }
                        const usbEnabled = !!rc.usb_direct_print;
                        jQuery('#printer-control-area').toggle(usbEnabled);
                    }
                });
        }
    });
    // --- END DARK MODE ---

    // --- ACCESSIBILITY IMPROVEMENTS ---
    jQuery('#product-search-clear').attr('aria-label', 'Clear search input').hide();
    jQuery('#sort-select').attr('aria-label', 'Sort products');
    jQuery('#category-select').attr('aria-label', 'Filter products by category');
    jQuery('#checkout-button').attr('aria-label', 'Proceed to checkout');
    jQuery('#hold-sale-button').attr('aria-label', 'Hold current sale');
    jQuery('#retrieve-sale-button').attr('aria-label', 'Retrieve held sale');
    jQuery('#new-sale-button').attr('aria-label', 'Start new sale');
    jQuery('#add-shipping-button').attr('aria-label', 'Add shipping');
    jQuery('#add-note-button').attr('aria-label', 'Add order note');
    jQuery('#refund-button').attr('aria-label', 'Enter refund mode');
    jQuery('#clear-cart-button').attr('aria-label', 'Clear cart');
    jQuery('#toggle-theme-button').attr('aria-label', 'Toggle dark mode');
    jQuery('#topbar-theme-toggle').attr('aria-label', 'Toggle dark mode');
    jQuery('#filters-toggle').attr('aria-label', 'Open filters modal');
    // --- END ACCESSIBILITY IMPROVEMENTS ---

    window.$shippingConfirmModal = jQuery('#shipping-confirm-modal');
    window.$splitPaymentModal = jQuery('#payment-method-modal');
    window.$tenderedDisplay = jQuery('#tendered-display');
    window.$inlinePrintReceipt = jQuery('#print-receipt-inline');
    window.$inlinePrintLabel = jQuery('#print-order-label-inline');
    window.$newSaleButton = jQuery('#new-sale-button');
    window.$orderNoteModal = jQuery('#order-note-modal');
    // --- END JQUERY ELEMENT VARIABLES ---

// --- ALL GLOBAL STATE VARIABLES (ATTACHED TO WINDOW FOR MODULE ACCESS) ---
    const ajaxUrl = loycartPOS.ajax_url;
    window.nonce = loycartPOS.nonce; 
    const currencySymbol = loycartPOS.currency_symbol || '£';
    
    // Use server settings only
    window.itemsPerPage = (window.loycartPOS.settings && window.loycartPOS.settings.ui && window.loycartPOS.settings.ui.products_per_page)
        ? parseInt(window.loycartPOS.settings.ui.products_per_page, 10)
        : 24;
    
    window.maxDiscount = 100;
    const isRTL = loycartPOS.is_rtl === '1' || loycartPOS.is_rtl === true;
    window.strings = loycartPOS.strings || {};
    window.cartTotalsTimer = null; 
    loycart.cart = [];
    window.currentPage = 1; 
    window.currentCategory = 'all'; 
    if (window.loycartPOS.settings && window.loycartPOS.settings.ui && window.loycartPOS.settings.ui.default_sort) {
        window.currentSort = window.loycartPOS.settings.ui.default_sort;
    } else {
        window.currentSort = 'popular';
    }
    window.currentSearch = ''; 
    window.isProcessing = false; 
    window.isScanning = false; 
    window.isReturnMode = false; 
    window.originalOrderId = null; 
    window.refundableOrderItems = [];
    window.orderNote = '';
    window.customerToSelect = null; 
    window.shippingModalContext = ''; 
    window.searchInputBarcodeTimeout = null;
    window.heldSalesCount = 0;
    window.globalBarcodeBuffer = '';
    window.globalBarcodeTimer = null;
    const BARCODE_INTERCHAR_TIMEOUT_MS = 250;
    const BARCODE_MIN_SCAN_LENGTH = 6;
    
    // Authoritative data storage for direct USB printing
    window.lastSaleResponse = null; 
    window.lastReceiptText = '';

    window.selectedShipping = {
        id: null,
        cost: 0,
        label: ''
    };
    window.pendingPayments = [];
    window.splitPaymentTotalDue = 0;
    window.splitPaymentAmountRemaining = 0;
    window.splitPaymentInput = '0.00'; 
    window.splitInputIsNew = true; 

    // These are shared between modules
    window.selectedProductId = null;
    window.selectedProductForOptions = null; 
    window.selectedVariation = null; 
    window.guestShippingAddress = null;

// --- KEYBOARD SHORTCUTS MODULE ---
    /**
     * Ctrl+S = Complete Sale
     * Ctrl+R = Enter Refund Mode
     * Ctrl+H = Hold Sale
     * Ctrl+D = Apply Coupon/Discount
     * Ctrl+? = Show help
     */
    jQuery(document).on('keydown', function(e) {
        // Don't trigger if user is typing in an input field (except search which has special handling)
        const $activeEl = jQuery(document.activeElement);
        const isInputActive = $activeEl.is('input:not(#product-search), textarea, select');
        
        if (isInputActive) {
            return;
        }

        // Ctrl key combinations
        if (e.ctrlKey || e.metaKey) {
            const key = e.key.toLowerCase();

            // Ctrl+S: Checkout/Complete Sale
            if (key === 's') {
                e.preventDefault();
                const $checkoutBtn = jQuery('#checkout-button');
                if ($checkoutBtn.length && !$checkoutBtn.prop('disabled')) {
                    $checkoutBtn.click();
                    showNotification('Checkout initiated (Ctrl+S)', 'info');
                    return;
                }
            }

            // Ctrl+R: Enter Refund Mode
            if (key === 'r') {
                e.preventDefault();
                const $returnBtn = jQuery('#return-mode-button');
                if ($returnBtn.length) {
                    $returnBtn.click();
                    showNotification(window.loycartPOS?.strings?.shortcutRefundModeNotification || 'Refund Mode activated (Ctrl+R)', 'info');
                    return;
                }
            }

            // Ctrl+H: Hold Sale
            if (key === 'h') {
                e.preventDefault();
                const $holdBtn = jQuery('#hold-sale-button');
                if ($holdBtn.length && !$holdBtn.prop('disabled')) {
                    $holdBtn.click();
                    showNotification(window.loycartPOS?.strings?.shortcutHoldSaleNotification || 'Hold Sale (Ctrl+H)', 'info');
                    return;
                }
            }

            // Ctrl+D: Apply Discount/Coupon
            if (key === 'd') {
                e.preventDefault();
                const $couponBtn = jQuery('#apply-coupon-btn');
                if ($couponBtn.length) {
                    $couponBtn.click();
                    showNotification(window.loycartPOS?.strings?.shortcutApplyCouponNotification || 'Apply Coupon (Ctrl+D)', 'info');
                    return;
                }
            }

            // Ctrl+?: Show keyboard shortcuts help
            if (key === '?' || key === '/') {
                e.preventDefault();
                showKeyboardShortcutsHelp();
                return;
            }

            
        }
    });

    /**
     * Display keyboard shortcuts help modal
     */
    function showKeyboardShortcutsHelp() {
        const helpContent = `
            <div style="padding: 20px; max-width: 500px; font-family: monospace; line-height: 1.8;">
                <h3 style="margin-top: 0;">${window.loycartPOS?.strings?.keyboardShortcutsTitle || 'LoyCart POS - Keyboard Shortcuts'}</h3>
                <table style="width: 100%; border-collapse: collapse;">
                    <tr style="border-bottom: 1px solid #ddd; padding: 8px 0;">
                        <td style="padding: 8px; font-weight: bold; width: 40%;">Ctrl + S</td>
                        <td style="padding: 8px;">${window.loycartPOS?.strings?.shortcutCompleteSale || 'Complete Sale / Checkout'}</td>
                    </tr>
                    <tr style="border-bottom: 1px solid #ddd;">
                        <td style="padding: 8px; font-weight: bold;">Ctrl + R</td>
                        <td style="padding: 8px;">${window.loycartPOS?.strings?.shortcutEnterRefundMode || 'Enter Refund Mode'}</td>
                    </tr>
                    <tr style="border-bottom: 1px solid #ddd;">
                        <td style="padding: 8px; font-weight: bold;">Ctrl + H</td>
                        <td style="padding: 8px;">${window.loycartPOS?.strings?.shortcutHoldSale || 'Hold Sale'}</td>
                    </tr>
                    <tr style="border-bottom: 1px solid #ddd;">
                        <td style="padding: 8px; font-weight: bold;">Ctrl + D</td>
                        <td style="padding: 8px;">${window.loycartPOS?.strings?.shortcutApplyCouponDiscount || 'Apply Coupon/Discount'}</td>
                    </tr>
                    <tr style="border-bottom: 1px solid #ddd;">
                        <td style="padding: 8px; font-weight: bold;">Ctrl/Cmd + Alt + D</td>
                        <td style="padding: 8px;">${window.loycartPOS?.strings?.shortcutToggleCompactLayout || 'Toggle Compact Layout'}</td>
                    </tr>
                    <tr>
                        <td style="padding: 8px; font-weight: bold;">Ctrl + ?</td>
                        <td style="padding: 8px;">${window.loycartPOS?.strings?.shortcutShowHelp || 'Show this help'}</td>
                    </tr>
                </table>
                <p style="font-size: 12px; color: #666; margin-top: 20px;">
                    <em>${window.loycartPOS?.strings?.shortcutHintWorksEverywhere || 'Shortcuts work from anywhere except while typing in input fields.'}</em>
                </p>
            </div>
        `;

        const $helpModal = jQuery(`
            <div id="keyboard-shortcuts-modal" class="modal" aria-hidden="false" role="dialog" aria-labelledby="shortcuts-modal-title">
                <div class="modal-content" style="width: 600px;">
                    <span class="modal-close" aria-label="Close">&times;</span>
                    <div id="shortcuts-modal-title" style="display: none;">Keyboard Shortcuts</div>
                    ${helpContent}
                </div>
            </div>
        `);

        jQuery('body').append($helpModal);
        openModal($helpModal);

        $helpModal.on('click', '.modal-close', function() {
            closeModal($helpModal);
            $helpModal.remove();
        });

        jQuery(document).on('keydown.shortcutsModal', function(e) {
            if (e.key === 'Escape') {
                closeModal($helpModal);
                $helpModal.remove();
                jQuery(document).off('keydown.shortcutsModal');
            }
        });
    }
// --- END KEYBOARD SHORTCUTS ---

// --- SALE COMPLETION LOGIC ---
    $splitPaymentModal.on('click', '#split-complete-sale', function () {
        if (window.isProcessing || jQuery(this).prop('disabled')) {
            return;
        }
        window.isProcessing = true;
        const $btn = jQuery(this);
        const originalBtnText = $btn.text();
        $btn.text('Processing...');

        const saleData = {
            cart: JSON.stringify(loycart.cart),
            payments: JSON.stringify(window.pendingPayments || []),
            customer_id: jQuery('#customer-id').val() || '0',
            order_note: window.orderNote || '',
            shipping_method_id: (window.selectedShipping && window.selectedShipping.id) ? window.selectedShipping.id : '',
            shipping_cost: (window.selectedShipping && window.selectedShipping.cost) ? window.selectedShipping.cost : 0,
            shipping_label: (window.selectedShipping && window.selectedShipping.label) ? window.selectedShipping.label : '',
        };

        if (window.guestShippingAddress) {
            saleData.guest_address = window.guestShippingAddress;
        }

        api_submitSale(saleData)
    .done(function (response) {
        if (response.success) {
            const orderId = response.data.order_id;
            const cartItemsCopy = JSON.parse(JSON.stringify(loycart.cart));

            // Use the nonce specifically for the get_sale_data action
            jQuery.post(loycartPOS.ajax_url, {
                action: 'loycart_get_sale_data',
                order_id: orderId,
                nonce: loycartPOS.nonce // Ensure this nonce is passed to avoid 403
            }, function(res) {
                if(res.success) {
                    const richSaleData = res.data;
                    window.lastSaleResponse = richSaleData; 

                    if (typeof window.generate58mmReceiptString === 'function') {
                        window.lastReceiptText = window.generate58mmReceiptString();
                    }

                    $receiptModal.data('saleResponse', richSaleData);
                    $receiptModal.data('cartCopy', cartItemsCopy);
                    
                    const hasShippingValue = richSaleData.shipping_total && parseFloat(richSaleData.shipping_total) !== 0;
                    const hasShippingAddress = richSaleData.shipping_address && richSaleData.shipping_address.trim() !== "";

                    if (hasShippingValue || hasShippingAddress) {
                        jQuery('#print-order-label-button').css('display', 'flex');
                    } else {
                        jQuery('#print-order-label-button').hide();
                    }

                    closeModal($splitPaymentModal);
                    openModal($receiptModal);
                    window.isProcessing = false;
                }
            }).fail(function(xhr) {
                console.error("Secondary Data Fetch Failed:", xhr.status);
                showNotification('Order completed, but failed to load receipt data. Please check Order History.', 'warning');
                window.isProcessing = false;
            });
        } else {
            showNotification(response.data || 'Failed to complete sale.', 'error');
            window.isProcessing = false;
        }
    })
    .fail(function(xhr, status, error) {
        console.error("Sale submission failed:", xhr.status, error);
        showNotification('Failed to complete sale. Please try again.', 'error');
        window.isProcessing = false;
    });
    });
    let usbDevice; // Connection persists across sales

jQuery(document).ready(function($) {
    // 1. Authorization check: Show connection area only for WooCommerce managers
    if (typeof loycartHardware !== 'undefined' && loycartHardware.isAuthorized) {
        $('#printer-control-area').show();
    }

    // 2. Hardware Connection Handshake
    $('#connect-usb-printer').on('click', async function() {
        try {
            // Browsers require a user gesture (click) to request a device
            usbDevice = await navigator.usb.requestDevice({ filters: [] });
            
            await usbDevice.open();
            await usbDevice.selectConfiguration(1);
            await usbDevice.claimInterface(0);
            
            $(this).css('background', '#28a745').text('Printer Ready');
            showNotification("USB Printer Linked: " + usbDevice.productName, 'success');
        } catch (err) {
            console.error("USB Connection failed:", err);
            showNotification("Connection failed. Ensure HTTPS is active.", "error");
        }
    });

 

$('#print-usb-direct').on('click', async function() {
    if (!usbDevice) return showNotification("Printer not connected", "error");

    try {
        const receiptText = window.lastReceiptText || window.generate58mmReceiptString(); 
        
        // --- NEW RAW BYTE CONVERSION ---
        // Converts string characters directly to 8-bit bytes (0-255)
        // This prevents the browser from adding extra 'ghost' characters
        const data = new Uint8Array(receiptText.length);
        for (let i = 0; i < receiptText.length; i++) {
            data[i] = receiptText.charCodeAt(i) & 0xFF;
        }

        const endpoint = usbDevice.configuration.interfaces[0].alternate.endpoints.find(e => e.direction === 'out');
        await usbDevice.transferOut(endpoint.endpointNumber, data);
        
        showNotification("Professional Receipt Printed", "success");
    } catch (err) {
        console.error("USB Print Error:", err);
    }
});

});



window.closeMoreActions = function() {
    const $button = jQuery('#more-actions-toggle');
    const $secondaryActions = jQuery('#cart-secondary-actions');
    if ($secondaryActions.is(':visible')) {
        $secondaryActions.slideUp(200);
        $button.html('<span class="dashicons dashicons-screenoptions"></span> More Actions');
        $button.removeClass('is-open');
    }
}





    jQuery('body').on('click', '#remove-cart-shipping', function () {
        

        window.selectedShipping = {
            id: null,
            cost: 0,
            label: ''
        };
        loycart.cartManager.getCartTotalsFromServer();
        
        if (window.isReturnMode) {
            showNotification('Shipping removed from refund.', 'info');
            if (loycart.refundManager) loycart.refundManager.updateRefundableItemState();
        } else {
            loycart.shippingManager.updateShippingCallToAction();
            loycart.cartManager.debouncedSaveCartToServer();
            setTimeout(() => {
                loycart.shippingManager.initiateShippingProcess();
            }, 150);
            showNotification('Shipping removed. Please select new option.', 'info');
        }
    });


    if (loycartPOS.is_bacs_enabled) {
        jQuery('#pay-bank-button').show();
    } else {
        jQuery('#pay-bank-button').hide();
    }
    if (loycartPOS.is_cheque_enabled) {
        jQuery('#pay-cheque-button').show();
    } else {
        jQuery('#pay-cheque-button').hide();
    }

    jQuery('#customer-cart-indicator').addClass('is-guest');

    // Product grid filters
    $categorySelect.on('change', function () {
        currentCategory = jQuery(this).val();
        currentPage = 1;
        loycart.productManager.loadProducts();
    });
    $sortSelect.on('change', function () {
        currentSort = jQuery(this).val();
        currentPage = 1;
        loycart.productManager.loadProducts();
        // Sync Sale chip state
        const isSale = currentSort === 'sale';
        jQuery('#sale-chip').toggleClass('active', isSale).attr('aria-pressed', isSale ? 'true' : 'false');
    });
    $prevPage.on('click', function () {
        if (!$prevPage.prop('disabled') && currentPage > 1) {
            currentPage--;
            loycart.productManager.loadProducts(currentPage);
        }
    });
    $nextPage.on('click', function () {
        if (!$nextPage.prop('disabled')) {
            currentPage++;
            loycart.productManager.loadProducts(currentPage);
        }
    });
    
    jQuery('#page-select').on('change', function () {
    const newPage = parseInt(jQuery(this).val());
    if (newPage !== currentPage) {
        currentPage = newPage;
        loycart.productManager.loadProducts(currentPage);
    }
});

    // Clear product search quickly
    jQuery(document).on('click', '#product-search-clear', function () {
        const $input = jQuery('#product-search');
        $input.val('');
        currentSearch = '';
        loycart.productManager.loadProducts(1);
        jQuery('#product-search-clear').hide();
        // Do not refocus after clearing search
    });

    // Topbar quick actions
    jQuery(document).on('click', '#topbar-theme-toggle', function () {
        // Delegate to existing toggle button
        const $btn = jQuery('#toggle-theme-button');
        if ($btn.length) { $btn.trigger('click'); }
        else { window.toggleDarkMode && window.toggleDarkMode(); }
    });
    

    // Sale quick chip toggle
    jQuery(document).on('click', '#sale-chip', function () {
        const $chip = jQuery(this);
        const isActive = $chip.hasClass('active');
        if (isActive) {
            const prev = $chip.data('prev-sort') || 'popular';
            window.$sortSelect.val(prev).trigger('change');
            $chip.removeClass('active').attr('aria-pressed', 'false');
        } else {
            $chip.data('prev-sort', window.$sortSelect.val());
            window.$sortSelect.val('sale').trigger('change');
            $chip.addClass('active').attr('aria-pressed', 'true');
        }
    });

    // Filters modal
    jQuery(document).on('click', '#filters-toggle', function () {
        // Sync modal controls with current state
        jQuery('#filters-sort-select').val(window.$sortSelect.val());
        jQuery('#filters-category-select').val(window.$categorySelect.val());
        openModal(jQuery('#filters-modal'));
    });
    jQuery(document).on('click', '#filters-cancel, #filters-modal .modal-close', function () {
        closeModal(jQuery('#filters-modal'));
    });
    jQuery(document).on('click', '#filters-apply', function () {
        const selSort = jQuery('#filters-sort-select').val();
        const selCat = jQuery('#filters-category-select').val();
        window.$sortSelect.val(selSort).trigger('change');
        window.$categorySelect.val(selCat).trigger('change');
        closeModal(jQuery('#filters-modal'));
    });

    // Store Credit Reports modal
    function loadCreditReports() {
        const from = jQuery('#credit-reports-from').val();
        const to = jQuery('#credit-reports-to').val();
        const $content = jQuery('#credit-reports-content');
        $content.html('<p>' + (strings.loading || 'Loading...') + '</p>');
        window.api_getStoreCreditReports(from, to)
            .done(function(resp) {
                if (resp && resp.success && resp.data && resp.data.html) {
                    $content.html(resp.data.html);
                } else {
                    $content.html('<p>' + (strings.noData || 'No data available for the selected range.') + '</p>');
                }
            })
            .fail(function(jqXHR) {
                loycart.logger.error('Error loading credit reports:', jqXHR.responseText);
                showNotification('Error loading reports.', 'error');
                $content.html('<p class="error">' + (strings.error || 'An error occurred while loading reports.') + '</p>');
            });
    }

    jQuery(document).on('click', '#store-credit-reports-button', function () {
        openModal(jQuery('#credit-reports-modal'));
        loadCreditReports();
    });
    jQuery(document).on('click', '#credit-reports-refresh', function () {
        loadCreditReports();
    });
    jQuery(document).on('click', '#credit-reports-modal .modal-close', function () {
        closeModal(jQuery('#credit-reports-modal'));
    });

    // POS Settings modal
    function populatePosSettingsForm(data) {
        try {
            const sc = data && data.store_credit ? data.store_credit : {};
            jQuery('#pos-setting-credit-enabled').prop('checked', !!sc.enabled);
            jQuery('#pos-setting-earn-percent').val(typeof sc.earn_percent !== 'undefined' ? sc.earn_percent : '2');
            jQuery('#pos-setting-apply-online').prop('checked', !!sc.apply_online);
            jQuery('#pos-setting-exc-earn').prop('checked', !!sc.exclude_sale_items_earning);
            jQuery('#pos-setting-exc-redeem').prop('checked', !!sc.exclude_sale_items_redeeming);

            const rc = data && data.receipt ? data.receipt : {};
            if (typeof rc.store_name !== 'undefined') jQuery('#pos-setting-store-name').val(rc.store_name);
            if (typeof rc.store_phone !== 'undefined') jQuery('#pos-setting-store-phone').val(rc.store_phone);
            if (typeof rc.footer_message !== 'undefined') jQuery('#pos-setting-receipt-footer').val(rc.footer_message);
            if (typeof rc.paper_width !== 'undefined') jQuery('#pos-setting-paper-width').val(rc.paper_width);
            jQuery('#pos-setting-usb-direct-print').prop('checked', !!rc.usb_direct_print);

            const ui = data && data.ui ? data.ui : {};
            if (typeof ui.products_per_page !== 'undefined') jQuery('#pos-setting-products-per-page').val(ui.products_per_page);
            if (typeof ui.default_sort !== 'undefined') jQuery('#pos-setting-default-sort').val(ui.default_sort);
            if (typeof ui.default_theme !== 'undefined') jQuery('#pos-setting-default-theme').val(ui.default_theme);
            jQuery('#pos-setting-show-theme-toggle').prop('checked', !!ui.show_theme_toggle);
            jQuery('#pos-setting-show-notifications').prop('checked', !!ui.show_notifications);
            jQuery('#pos-setting-auto-redirect-pos').prop('checked', !!ui.auto_redirect_pos);

            const vis = data && data.visibility ? data.visibility : {};
            jQuery('#pos-setting-allow-private').prop('checked', !!vis.allow_private_detail_lookups);
        } catch(e) {
            loycart.logger.error('Error populating settings form:', e);
        }
    }

    jQuery(document).on('click', '#pos-settings-button', function () {
        openModal(jQuery('#pos-settings-modal'));
        // Load settings
        window.api_getPosSettings()
            .done(function(resp){
                if (resp && resp.success && resp.data) {
                    // Always assign settings to window.loycartPOS.settings
                    if (!window.loycartPOS) window.loycartPOS = {};
                    window.loycartPOS.settings = resp.data;
                    populatePosSettingsForm(resp.data);
                } else {
                    showNotification('Failed to load settings.', 'error');
                }
            })
            .fail(function(){ showNotification('Error loading settings.', 'error'); });
    });

    jQuery(document).on('click', '#pos-settings-save', function () {
        const payload = {
            store_credit: {
                enabled: jQuery('#pos-setting-credit-enabled').is(':checked'),
                earn_percent: jQuery('#pos-setting-earn-percent').val(),
                apply_online: jQuery('#pos-setting-apply-online').is(':checked'),
                exclude_sale_items_earning: jQuery('#pos-setting-exc-earn').is(':checked'),
                exclude_sale_items_redeeming: jQuery('#pos-setting-exc-redeem').is(':checked')
            },
            receipt: {
                store_name: jQuery('#pos-setting-store-name').val(),
                store_phone: jQuery('#pos-setting-store-phone').val(),
                footer_message: jQuery('#pos-setting-receipt-footer').val(),
                paper_width: jQuery('#pos-setting-paper-width').val(),
                usb_direct_print: jQuery('#pos-setting-usb-direct-print').is(':checked')
            },
            ui: {
                products_per_page: parseInt(jQuery('#pos-setting-products-per-page').val(), 10) || 24,
                default_sort: jQuery('#pos-setting-default-sort').val(),
                default_theme: jQuery('#pos-setting-default-theme').val(),
                show_theme_toggle: jQuery('#pos-setting-show-theme-toggle').is(':checked'),
                     show_notifications: jQuery('#pos-setting-show-notifications').is(':checked'),
                     auto_redirect_pos: jQuery('#pos-setting-auto-redirect-pos').is(':checked')
            },
            visibility: {
                allow_private_detail_lookups: jQuery('#pos-setting-allow-private').is(':checked')
            }
        };
        window.api_savePosSettings(payload)
            .done(function(resp){
                if (resp && resp.success) {
                    showNotification('Settings saved.', 'success');
                    // Reflect USB direct print toggle immediately in the receipt modal
                    const usbEnabled = jQuery('#pos-setting-usb-direct-print').is(':checked');
                    jQuery('#printer-control-area').toggle(usbEnabled);
                    // Update paper width indicator
                    const pw = jQuery('#pos-setting-paper-width').val() || '80mm';
                    jQuery('#receipt-paper-width-indicator span').text(pw);

                    // Update global variables from modal fields
                    const perPage = parseInt(jQuery('#pos-setting-products-per-page').val(), 10) || 24;
                    const sortBy = jQuery('#pos-setting-default-sort').val() || 'popular';
                    window.itemsPerPage = perPage;
                    window.currentSort = sortBy;

                    // Refresh product grid and header controls
                    loycart.productManager.loadProducts();

                    // Update header controls to reflect new settings
                    if (window.$sortSelect) {
                        window.$sortSelect.val(sortBy);
                    }

                    // Update admin bar theme toggle visibility
                    const showThemeToggle = jQuery('#pos-setting-show-theme-toggle').is(':checked');
                    jQuery('#wp-admin-bar-loycart-dark-mode').toggle(showThemeToggle);

                    // Update notifications setting in global JS
                    if (window.loycartPOS && window.loycartPOS.settings && window.loycartPOS.settings.ui) {
                        window.loycartPOS.settings.ui.show_notifications = jQuery('#pos-setting-show-notifications').is(':checked');
                        if (window.loycartPOS.settings.ui.show_notifications) {
                            setTimeout(function(){
                                showNotification('Settings Saved.', 'success');
                            }, 400);
                        }
                    }
                    // Stay on settings page instead of closing modal
                    // closeModal(jQuery('#pos-settings-modal'));
                } else {
                    showNotification('Failed to save settings.', 'error');
                }
            })
            .fail(function(){ showNotification('Error saving settings.', 'error'); });
    });
    jQuery(document).on('click', '#pos-settings-modal .modal-close', function(){
        closeModal(jQuery('#pos-settings-modal'));
    });

    // Restrict add-to-cart to the dedicated button for better touch UX
    $productsList.on('click keypress', '.add-to-cart-btn', function (e) {
        if (e.type === 'keypress' && (e.key !== 'Enter' && e.key !== ' ')) return;
        e.preventDefault();
        e.stopPropagation();
        const $btn = jQuery(this);
        // Prevent all action/animation if disabled (for out-of-stock or sold-individually-in-cart)
        if ($btn.prop('disabled') || $btn.hasClass('disabled')) return;
        // Visual tap + pulse feedback
        $btn.addClass('tapped pulse cart-flash-check');
        setTimeout(() => { $btn.removeClass('tapped'); }, 160);
        setTimeout(() => { $btn.removeClass('pulse'); }, 460);
        setTimeout(() => { $btn.removeClass('cart-flash-check'); }, 420);
        const $productCard = jQuery(this).closest('.product');
        loycart.productManager.handleProductClick($productCard);
    });

    // Make product image act like Add to Cart in normal browsing
    $productsList.on('click', '.product:not(.disabled) .product-image', function (e) {
        e.preventDefault();
        e.stopPropagation();
        // Visual tap feedback on image
        const $imgWrap = jQuery(this);
        $imgWrap.addClass('tapped');
        setTimeout(() => { $imgWrap.removeClass('tapped'); }, 150);
        const $productCard = jQuery(this).closest('.product');
        loycart.productManager.handleProductClick($productCard);
    });

    // Prevent product card clicks from triggering add-to-cart or info modal implicitly
    $productsList.on('click', '.product:not(.disabled)', function (e) {
        // If click originates from action icons, handlers will manage
        if (jQuery(e.target).closest('.product-info-icon, .product-print-icon, .add-to-cart-btn').length > 0) {
            return;
        }
        // In refund mode, clicking the card should add the item to the refund cart
        if (window.isReturnMode) {
            const $productCard = jQuery(this);
            loycart.productManager.handleProductClick($productCard);
            return;
        }
        // Otherwise, do nothing (disable info modal on card click)
    });

    // Disable right-click and long-press context menu on product cards
    $productsList.on('contextmenu', '.product', function (e) {
        e.preventDefault();
    });


    $productSearch.on('keydown', function (e) { 

        if (e.key === 'Enter') {
            e.preventDefault(); 
            if (searchInputBarcodeTimeout) clearTimeout(searchInputBarcodeTimeout);

            const input = jQuery(this).val();
            currentSearch = input.trim();

            // Always perform a text search on Enter
            $productSearch.blur(); // Remove cursor from search after manual search
            loycart.productManager.loadProducts();
            return;
        }
       
        // Do not run searches while typing; only act on Enter
    });

    $productSearch.on('input', function () {
        const hasValue = jQuery(this).val().trim() !== '';
        jQuery('#product-search-clear').toggle(hasValue);
    });

    jQuery(document).on('keypress', function (e) {
        const $target = jQuery(e.target);
        if (jQuery('.modal.show').length > 0 || $target.is('input, textarea, select')) {
            globalBarcodeBuffer = '';
            return;
        }

        const charCode = e.which || e.keyCode;

        if (charCode === 13) { 

            if (globalBarcodeBuffer.length >= BARCODE_MIN_SCAN_LENGTH) {
                $productSearch.blur();
                e.preventDefault();
                e.stopPropagation();


                loycart.productManager.handleBarcodeScan(globalBarcodeBuffer.trim());
            }
            globalBarcodeBuffer = '';
            return;
        }

        const char = String.fromCharCode(charCode);
        if (charCode >= 32 && charCode <= 126) {
            globalBarcodeBuffer += char;
        }

        if (globalBarcodeTimer) clearTimeout(globalBarcodeTimer);
        globalBarcodeTimer = setTimeout(() => {
            globalBarcodeBuffer = '';
        }, 500);
    });

    
    $cartItems.on('click', '.cart-quantity-btn', function () {
        loycart.cartManager.handleQuantityChange(jQuery(this));
    });

    $cartItems.on('click', '.cart-remove-btn', function () {
        loycart.cartManager.handleRemoveItem(jQuery(this));
    });

    $cartItems.on('click', '.cart-discount-btn', function () {
        loycart.cartManager.handleDiscountClick(jQuery(this));
    });

    $cartItems.on('click', '.cart-edit-toggle', function () {
        const $item = jQuery(this).closest('.cart-item');
        const $controls = $item.find('.cart-controls-wrapper');

        
        $item.toggleClass('is-editing');

        
        if ($controls.is(':visible')) {
            
            $controls.slideUp(200);
        } else {
            
            $controls.slideDown(200).css('display', 'flex');
        }
    });

    
    $discountModal.on('click', '.modal-close', function () {
        closeModal($discountModal);
        selectedProductId = null;
    });

    $discountModal.on('click', '.discount-keypad button', function () {
        loycart.paymentManager.handleDiscountKeypad(jQuery(this));
    });

    $discountModal.on('click', '#apply-discount', function () {
        loycart.paymentManager.applyDiscount();
    });

    $discountModal.on('click', '#remove-discount', function () {
        loycart.paymentManager.removeDiscount();
    });

    
    $cashModal.on('click', '.modal-close, #modal-cancel', function () {
        closeModal($cashModal);
        loycart.paymentManager.resetChangeModal();
        if (lastSaleResponse) {
            loycart.paymentManager.showReceiptModal(lastSaleResponse, JSON.parse(JSON.stringify(cart)));
            lastSaleResponse = null;
        }
    });
    

    $cardConfirmModal.on('click', '.modal-close, #card-modal-cancel', function () {
        $cardConfirmModal.removeData('payment'); // Clean up
        closeModal($cardConfirmModal);
    });
    $cardConfirmModal.on('click', '#card-modal-confirm', function () {
        const payment = $cardConfirmModal.data('payment');
        if (payment) {
            loycart.paymentManager.addPendingPayment(payment);
            closeModal($cardConfirmModal);
            $cardConfirmModal.removeData('payment');
        } else {
            
            const finalTotal = parseFloat($cardModalFinalTotal.text().replace(/[^0-9.-]+/g, ""));
            loycart.paymentManager.submitSale('card', finalTotal);
        }
    });



    $heldSalesModal.on('click', '.modal-close', function () {
        closeModal($heldSalesModal);
    });
    jQuery('#held-modal-close').on('click', function () {
        closeModal($heldSalesModal);
    });

    $heldCartsList.on('click keypress', '.resume-held-cart', function (event) {
        if (event.type === 'keypress' && (event.key !== 'Enter' && event.key !== ' ')) return;
        event.preventDefault();
        event.stopPropagation();

        if (isProcessing) {
            //showNotification(strings.posIsBusy, 'warning');
            return;
        }
        if (loycart.cart.length > 0) {
            showNotification(strings.completeOrHoldCurrentSale, 'warning');
            return;
        }

        isProcessing = true;
        const heldCartId = jQuery(this).closest('.held-cart-item').data('id');

    api_resumeHeldCart(heldCartId)
    .done(function (response) {
        if (response.success) {
            api_clearCartOnServer();
            
            if (response.data && Array.isArray(response.data.cart)) {
                
                loycart.cart = response.data.cart;
                
                const resumedCustomerId = parseInt(response.data.customer_id) || 0;

                if (resumedCustomerId > 0) {
                    api_searchCustomers(resumedCustomerId)
                        .done(function (custRes) {
                            if (custRes.success && custRes.data && custRes.data.length > 0) {
                                loycart.customerManager.addSelectedCustomerToCart(custRes.data[0].id, custRes.data[0].name);
                            }
                        });
                } else {
                    
                    window.$customerId.val('0');
                    window.$selectedCustomerName.text('');
                    window.$selectedCustomerIndicator.hide();
                    window.$guestCustomerIndicator.show();
                    jQuery('#customer-cart-indicator').addClass('is-guest');
                }
            } else {
                loycart.cart = [];
                window.$customerId.val('0');
                showNotification(strings.invalidHeldCartData, 'error');
            }

            
            loycart.cartManager.updateCart(); 
            
            showNotification(strings.saleResumedSuccess, 'success');
            closeModal($heldSalesModal);
            loycart.heldSalesManager.refreshHeldSales();
        } else {
            const errorMessage = (response.data && (response.data.message || response.data)) || strings.failedResumeSale;
            showNotification(errorMessage, 'error');
        }
    })
    });

    $heldCartsList.on('click keypress', '.delete-held-cart', function (event) {
        if (event.type === 'keypress' && (event.key !== 'Enter' && event.key !== ' ')) return;
        event.preventDefault();
        event.stopPropagation();

        if (isProcessing) {
            //showNotification(strings.posIsBusy, 'warning');
            return;
        }

        const heldCartId = jQuery(this).closest('.held-cart-item').data('id');
        const $itemToRemove = jQuery(this).closest('.held-cart-item');

        isProcessing = true;

        api_deleteHeldCart(heldCartId)
            .done(function (response) {
                if (response.success) {
                    $itemToRemove.fadeOut(300, function () {
                        jQuery(this).remove();
                        loycart.heldSalesManager.refreshHeldSales();
                    });
                    showNotification(strings.heldSaleDeletedSuccess, 'success');

                } else {
                    const errorMessage = (response.data && (response.data.message || response.data)) || strings.failedDeleteHeldSale;
                    showNotification(errorMessage, 'error');
                }
            })
            .fail(function () {
                showNotification(strings.errorDeletingHeldSale, 'error');
            })
            .always(function () {
                isProcessing = false;
            });
    });

    // --- INFO MODAL HANDLERS ---

    // Attach info button handler with proper event delegation
    jQuery(document).on('click', '.product-info-icon', function (e) {
        e.stopPropagation();
        e.preventDefault();
        // Visual tap feedback
        const $icon = jQuery(this);
        $icon.addClass('tapped');
        setTimeout(() => { $icon.removeClass('tapped'); }, 150);
        const $productCard = jQuery(this).closest('.product');
        const productData = $productCard.data('product-data');

        if (productData) {
            const needsFullFetch = !productData.gallery_images || productData.gallery_images.length === 0;

            if (needsFullFetch) {
                loycart.productManager.fetchAndShowProductInfo(String(productData.id));
            } else {
                loycart.modals.showProductInfoModal(productData);
            }
        }
    });
    
    // Print label directly from product card
    jQuery(document).on('click', '.product-print-icon', function (e) {
        e.stopPropagation();
        e.preventDefault();
        // Visual tap feedback
        const $icon = jQuery(this);
        $icon.addClass('tapped');
        setTimeout(() => { $icon.removeClass('tapped'); }, 150);
        const $productCard = jQuery(this).closest('.product');
        const productData = $productCard.data('product-data');

        if (!productData) return;

        // Variable products: open modal so cashier can select options; label printing remains in-modal
        if (productData.type === 'variable') {
            const needsFullFetch = !productData.gallery_images || productData.gallery_images.length === 0;
            if (needsFullFetch) {
                loycart.productManager.fetchAndShowProductInfo(String(productData.id));
            } else {
                loycart.modals.showProductInfoModal(productData);
            }
            return;
        }

        // Non-variable: print label immediately
        const labelData = {
            name: productData.name,
            variation: '',
            price: formatPrice(productData.price),
            barcode: productData.barcode,
            sku: productData.sku || ''
        };

        let labelHtml = generateLabelHTML(labelData);

        if (labelData.barcode && String(labelData.barcode).trim() !== '') {
            const barcodeScript = `
                <script src="${loycartPOS.plugin_url}/assets/js/JsBarcode.all.min.js"></script>
                <script>
                    window.onload = function() {
                        JsBarcode("#barcode", "${labelData.barcode}", {
                            format: "CODE128",
                            displayValue: false,
                            fontSize: 14,
                            margin: 0,
                            height: 40
                        });
                    };
                </script>
            `;
            labelHtml = labelHtml.replace('</body>', barcodeScript + '</body>');
        }

        window.triggerIframePrint(labelHtml, 'Product Label - ' + labelData.name, '4x6');
    });
    

    jQuery('#product-info-modal').on('click', '.modal-close', function () {
        closeModal($productInfoModal);
    });

    jQuery('#product-info-modal').on('change', '.info-modal-variation-select', loycart.modals.updateInfoModalVariationDisplay);

    jQuery('#info-modal-add-to-cart').on('click', function () {
    const $modal = jQuery('#product-info-modal');

    const product = $modal.data('product-data');
    const selectedVariation = $modal.data('selected-variation'); 
    const quantity = parseInt(jQuery('#info-modal-quantity').val()) || 1;

    if (!product) return;

    const $modalImage = $modal.find('.product-info-main-image');

    if (product.type === 'variable') {
        if (selectedVariation) {

            loycart.cartManager.addToCart(product, quantity, selectedVariation.id, selectedVariation.attributes);
            loycart.productManager.animateToCart($modalImage, {id: product.id, variation_id: selectedVariation.id});
            closeModal($modal);
        } else {
            showNotification('Please select all product options.', 'error');
        }
    } else {

        loycart.cartManager.addToCart(product, quantity);
        loycart.productManager.animateToCart($modalImage, {id: product.id});
        closeModal($modal);
    }
});


jQuery('#product-info-modal').on('click', '#print-label-button', function () {
    const $modal = jQuery('#product-info-modal');
    const product = $modal.data('product-data');
    const selectedVariation = $modal.data('selected-variation');
    let labelData = {};

    if (product.type === 'variable') {
        if (!selectedVariation) {
            showNotification('Please select options.', 'warning');
            return;
        }
        labelData = {
            name: product.name,
            variation: Object.values(selectedVariation.attributes).join(' / '),
            price: formatPrice(selectedVariation.price),
            barcode: selectedVariation.barcode,
            sku: selectedVariation.sku || ''
        };
    } else {
        labelData = {
            name: product.name,
            variation: '',
            price: formatPrice(product.price),
            barcode: product.barcode,
            sku: product.sku || ''
        };
    }

    let labelHtml = generateLabelHTML(labelData);

    if (labelData.barcode && labelData.barcode.trim() !== '') {
        const barcodeScript = `
            <script src="${loycartPOS.plugin_url}/assets/js/JsBarcode.all.min.js"></script>
            <script>
                window.onload = function() {
                    JsBarcode("#barcode", "${labelData.barcode}", {
                        format: "CODE128",
                        displayValue: false,
                        fontSize: 14,
                        margin: 0,
                        height: 40
                    });
                };
            </script>
        `;
        
        labelHtml = labelHtml.replace('</body>', barcodeScript + '</body>');
    }

    window.triggerIframePrint(labelHtml, 'Product Label - ' + labelData.name, '4x6');
});




jQuery('body').on('click', '#print-shipping-label', function () {
    if (!lastSaleResponse) {
        showNotification('Error: Last sale data not found.', 'error');
        return;
    }

    const labelData = {
        order_number: lastSaleResponse.order_id,
        customer_name: lastSaleResponse.customer_name || 'Guest',
        label_address: lastSaleResponse.shipping_address,
        label_type: 'shipping',
        label_title: 'Shipping Label',
        barcode_id: String(lastSaleResponse.order_id).replace(/\D/g, '') 
    };

    let labelHtml = loycart.labelManager.buildLabelHtml(labelData);

    window.triggerIframePrint(labelHtml, 'Shipping Label - Order #' + labelData.order_number, '4x6');
});


    jQuery('body').on('click', '#edit-selected-customer', function () {
        openModal($customerModal);
        $customerSearch.focus();
    });

    jQuery('#customer-modal-button, #add-guest-customer-button').on('click', function () {
        openModal($customerModal);
        $customerSearch.focus();
    });

    $customerModal.on('click', '.modal-close, #customer-modal-close', function () {
        closeModal($customerModal);
    });

    $customerSearch.on('keyup', function (e) {
        const query = jQuery(this).val().trim();
        if (e.key === 'Escape') {
            $customerResults.empty().html(`<p class="empty-state-message">${strings.searchForCustomer}</p>`);
            return;
        }
        if (query.length < 2) {
            $customerResults.empty().html(`<p class="empty-state-message">${strings.searchForCustomer}</p>`);
            return;
        }

        if (this.searchTimeout) clearTimeout(this.searchTimeout);

        this.searchTimeout = setTimeout(() => {
            $customerResults.html(`<p class="empty-state-message">${strings.searching}</p>`);

            api_searchCustomers(query)
                .done(function (response) {
                    $customerResults.empty();
                    if (response.success && response.data && response.data.length > 0) {
                        response.data.forEach(customer => {
                            const $result = jQuery(`
                                <div class="customer-result-item">
                                    <div class="customer-info">
                                        <div class="customer-name">${customer.name}</div>
                                        <div class="customer-email">${customer.email||''}</div>
                                        ${customer.phone ? `<div class="customer-phone">${customer.phone}</div>` : ''}
                                        <div class="customer-stats">Orders: ${customer.total_orders} | Spent: ${customer.total_spent}</div>
                                    </div>
                                    <div class="customer-actions">
                                        <button class="button select-customer-button" data-id="${customer.id}" data-name="${customer.name}">${strings.select}</button>
                                    </div>
                                </div>`);
                            $customerResults.append($result);
                        });
                    } else {
                        $customerResults.html(`<p class="empty-state-message">${strings.noCustomersFound}</p>`);
                    }
                })
                .fail(function () {
                    $customerResults.html(`<p class="empty-state-message">Error searching for customers.</p>`);
                });

        }, 300);
    });

    $customerResults.on('click', '.select-customer-button', function () {
        const customerId = jQuery(this).data('id');
        const customerName = jQuery(this).data('name');

        closeModal($customerModal);

        loycart.customerManager.addSelectedCustomerToCart(customerId, customerName);

        const orderRequiresShipping = loycart.cart.some(item => !item.is_virtual);
        if (orderRequiresShipping) {
            setTimeout(function () {
                loycart.shippingManager.initiateShippingProcess();
            }, 100);
        }
    });

    jQuery('#remove-customer-from-cart').off('click').on('click', function () {
    loycart.customerManager.removeCustomerFromCartState();
    });

    jQuery('#create-new-customer-button').on('click', function (e) {
        e.preventDefault();
        e.stopPropagation();
        closeModal($customerModal);
        jQuery('#create-customer-modal .pos-input').val('');
        openModal($createCustomerModal);
    });

    $createCustomerModal.on('click', '#save-new-customer', function () {
        const $thisButton = jQuery(this);
        const customerData = {
            first_name: jQuery('#new_customer_first_name').val().trim(),
            last_name: jQuery('#new_customer_last_name').val().trim(),
            email: jQuery('#new_customer_email').val().trim(),
            phone: jQuery('#new_customer_phone').val().trim(),
            address_1: jQuery('#new_customer_address_1').val().trim(),
            address_2: jQuery('#new_customer_address_2').val().trim(),
            city: jQuery('#new_customer_city').val().trim(),
            postcode: jQuery('#new_customer_postcode').val().trim(),
            country: jQuery('#new_customer_country').val(),
            state: jQuery('#new_customer_state').val().trim()
        };

        for (const key in customerData) {
            if (key === 'address_2') {
                continue;
            }
            if (!customerData[key]) {
                const fieldName = key.replace(/_/g, ' ').replace(/\b\w/g, l => l.toUpperCase());
                showNotification(`'${fieldName}' is a required field.`, 'error');
                return;
            }
        }
        const emailRegex = /^[^\s@]+@[^\s@]+\.[^\s@]+$/;
        if (!emailRegex.test(customerData.email)) {
            showNotification('Please enter a valid email address.', 'error');
            return;
        }

        $thisButton.text('Creating...').prop('disabled', true);

        api_createCustomer(customerData)
            .done(function (response) {
                if (response.success) {
                    const newCustomer = response.data;
                    closeModal($createCustomerModal);

                    const orderRequiresShipping = loycart.cart.some(item => !item.is_virtual);

                    if (orderRequiresShipping) {
                        customerToSelect = {
                            id: newCustomer.id,
                            name: newCustomer.name
                        };
                        const $shippingModal = jQuery('#shipping-modal');
                        const $guestForm = jQuery('#guest-shipping-form');

                        jQuery('#shipping_first_name').val(customerData.first_name);
                        jQuery('#shipping_last_name').val(customerData.last_name);
                        jQuery('#shipping_email').val(customerData.email);
                        jQuery('#shipping_phone').val(customerData.phone);
                        jQuery('#shipping_address_1').val(customerData.address_1);
                        jQuery('#shipping_address_2').val(customerData.address_2);
                        jQuery('#shipping_city').val(customerData.city);
                        jQuery('#shipping_postcode').val(customerData.postcode);
                        jQuery('#shipping_country').val(customerData.country);
                        jQuery('#shipping_state').val(customerData.state);

                        $guestForm.show();
                        $guestForm.find('#create_account_checkbox').closest('.modal-form-group').hide();
                        $shippingModal.find('.guest-form-intro').text('Confirm or edit the shipping address for this order.');
                        jQuery('#shipping-options-content').html('<p class="empty-state-message">Click "Calculate Shipping" to see available options.</p>');

                        openModal($shippingModal);
                    } else {
                        loycart.customerManager.addSelectedCustomerToCart(newCustomer.id, newCustomer.name);
                        showNotification(`Customer '${newCustomer.name}' created and selected.`, 'success');
                    }
                    if (typeof window.unlockPOS === 'function') {
                        window.unlockPOS();
                    }
                } else {
                    showNotification(response.data, 'error');
                }
            })
            .fail(function () {
                showNotification('An error occurred. Please try again.', 'error');
            })
            .always(function () {
                $thisButton.text('Create & Select Customer').prop('disabled', false);
            });
    });

    $createCustomerModal.on('click', '.modal-close', function () {
        closeModal($createCustomerModal);
        if (typeof window.lockPOS === 'function') {
            window.lockPOS();
        }
    });

   
    jQuery('#add-new-product-button').on('click', function() {
        jQuery('#add-new-product-modal .pos-input').val('');
        jQuery('.add-new-product-schedule-fields').hide();
        

        jQuery('#new_product_visibility').val('pos_only');

   
        jQuery('.add-new-product-options-collapsible').hide(); 
        jQuery('#toggle-new-product-options').html('<span class="dashicons dashicons-plus" style="vertical-align: text-top;"></span> More Options');
        
 
        jQuery('#add-new-product-image-id').val('');
        jQuery('#add-new-product-image-uploader .image-preview').attr('src', loycartPOS.placeholder_image_url);
        jQuery('#add-new-product-remove-image').hide();
        
  
        const $taxControls = jQuery('#add-new-product-tax-controls');
        if (loycartPOS.tax_enabled === 'true') {
  
            const $taxStatusSelect = jQuery('#add-new-product-tax-status').empty();
            const $taxClassSelect = jQuery('#add-new-product-tax-class').empty();
            
            
            $taxStatusSelect.append(jQuery('<option>', { value: '' }).text('Select Status...'));
            $.each(loycartPOS.tax_statuses, (key, value) => $taxStatusSelect.append(jQuery('<option>', { value: key }).text(value)));

            
            $taxClassSelect.append(jQuery('<option>', { value: '' }).text('Standard Rates'));
            $.each(loycartPOS.tax_classes, (key, value) => {
                if (key !== '') { 
                    $taxClassSelect.append(jQuery('<option>', { value: key }).text(value));
                }
            });

            $taxStatusSelect.val('taxable');

            $taxClassSelect.val('');

            $taxControls.show();
        } else {
            $taxControls.hide();
        }
        openModal($addNewProductModal);
    });

    
    jQuery('body').on('click', '#toggle-new-product-options', function() {
        const $collapsible = jQuery('.add-new-product-options-collapsible');
        const $button = jQuery(this);
        
        $collapsible.slideToggle(200, function() {
         
            if ($collapsible.is(':visible')) {
                // If it's now VISIBLE (open)
                $button.html('<span class="dashicons dashicons-minus" style="vertical-align: text-top;"></span> Less Options');
            } else {
                // If it's now HIDDEN (closed)
                $button.html('<span class="dashicons dashicons-plus" style="vertical-align: text-top;"></span> More Options');
            }
        });
    });

    
    jQuery('body').on('click', '#barcode-scan-button', function(e) {
        e.preventDefault();
        loycart.scanner.openScanner('add-product');
    });

    jQuery('body').on('click', '#header-barcode-scan-button', function(e) {
        e.preventDefault();
        loycart.scanner.openScanner('header');
    });

    $addNewProductModal.on('click', '.modal-close', function () {
        closeModal($addNewProductModal);
    });
    $addNewProductModal.on('click', '.sale-price-schedule-link', function (e) {
        e.preventDefault();
        jQuery('.add-new-product-schedule-fields').slideToggle(200);
    });
    $addNewProductModal.on('keydown', '#add-new-product-barcode', function (e) {
        if (e.key === 'Enter' || e.keyCode === 13) {
            e.preventDefault();
        }
    });
    
    
    

    /**
     * Unified function to handle both 'Save & Add to Cart' and 'Save & Add Another'
     * @param {boolean} addToCart - If true, adds to cart; if false, just closes modal and resets.
     */
function processNewProductSubmission(addToCart) {
    const name = jQuery('#add-new-product-name').val().trim();
    const regular_price = jQuery('#add-new-product-regular-price').val().trim();
    const sale_price = jQuery('#add-new-product-sale-price').val().trim();
    const sale_from = jQuery('#add-new-product-date-from').val().trim();
    const sale_to = jQuery('#add-new-product-date-to').val().trim();
    const sku = jQuery('#add-new-product-sku').val().trim();
    const barcode = jQuery('#add-new-product-barcode').val().trim();
    const stock = jQuery('#add-new-product-stock').val();
    const description = jQuery('#add-new-product-description').val().trim();
    const tax_status = jQuery('#add-new-product-tax-status').val();
    const tax_class = jQuery('#add-new-product-tax-class').val();
    
    // Fetch selected visibility from dropdown
    const visibility = jQuery('#new_product_visibility').val();
    
    if (name === '') { showNotification('Item Name is required.', 'error'); return; }
    if (regular_price === '' || isNaN(parseFloat(regular_price)) || parseFloat(regular_price) <= 0) { 
        showNotification('A valid positive Price is required.', 'error'); 
        return; 
    }
    if (sale_price !== '' && parseFloat(sale_price) >= parseFloat(regular_price)) { 
        showNotification('Sale Price must be less than Regular Price.', 'error'); 
        return; 
    }

    const $modalButtons = $addNewProductModal.find('.modal-footer button');
    $modalButtons.prop('disabled', true);
    const originalText = jQuery('#add-add-new-product-to-cart').text();
    jQuery('#add-add-new-product-to-cart').text('Creating...');

    const productData = {
        name: name,
        regular_price: regular_price,
        sale_price: sale_price,
        sale_from: sale_from,
        sale_to: sale_to,
        image_id: jQuery('#add-new-product-image-id').val(),
        category_id: jQuery('#add-new-product-category').val(),
        sku: sku,
        barcode: barcode,
        tax_status: tax_status,
        tax_class: tax_class,
        stock: stock,
        description: description,
        visibility: visibility // Passes 'pos_only', 'public', or 'hidden' to the API
    };

    api_createCustomProduct(productData)
        .done(function(response) {
            if (response.success) {
                const newProduct = response.data;
                newProduct.is_custom = true;
                
                if (addToCart) {
                    loycart.cartManager.addToCart(newProduct);
                    closeModal($addNewProductModal);
                    // Refresh product grid to show newly created product
                    loycart.productManager.loadProducts();
                } else {
                    showNotification(`Product '${newProduct.name}' created successfully!`, 'success');
                    // Refresh product grid to show newly created product
                    loycart.productManager.loadProducts();
                    // Reset modal fields
                    jQuery('#add-new-product-name').val('');
                    jQuery('#add-new-product-regular-price').val('');
                    jQuery('#add-new-product-sale-price').val('');
                    jQuery('#add-new-product-stock').val('');
                    jQuery('#add-new-product-barcode').val('');
                    jQuery('#add-new-product-description').val('');
                    // Reset visibility to default
                    jQuery('#new_product_visibility').val('pos_only');
                    jQuery('#add-new-product-name').focus();
                }
            } else {
                showNotification('Error: ' + (response.data || 'Could not create product.'), 'error');
            }
        })
        .fail(function() {
            showNotification('Could not connect to the server.', 'error');
        })
        .always(function() {
            $modalButtons.prop('disabled', false);
            jQuery('#add-add-new-product-to-cart').text(originalText);
        });
}


    jQuery('#add-add-new-product-to-cart').on('click', function() {
        processNewProductSubmission(true);
    });

    // ðŸ›‘ NEW HANDLER: Save & Add Another
    jQuery('#save-and-add-another').on('click', function() {
        processNewProductSubmission(false);
    });

    // Add new product image uploader
    let mediaFrame;
    jQuery('#add-new-product-upload-image').on('click', function (e) {
        e.preventDefault();
        if (mediaFrame) {
            mediaFrame.open();
            return;
        }
        mediaFrame = wp.media({
            title: 'Select an Image for the Custom Product',
            button: {
                text: 'Use this image'
            },
            multiple: false
        });
        mediaFrame.on('select', function () {
            const attachment = mediaFrame.state().get('selection').first().toJSON();
            jQuery('#add-new-product-image-id').val(attachment.id);
            jQuery('#add-new-product-image-uploader .image-preview').attr('src', attachment.sizes.thumbnail ? attachment.sizes.thumbnail.url : attachment.url);
            jQuery('#add-new-product-remove-image').show();
        });
        mediaFrame.open();
    });

    jQuery('#add-new-product-remove-image').on('click', function (e) {
        e.preventDefault();
        jQuery('#add-new-product-image-id').val('');
        jQuery('#add-new-product-image-uploader .image-preview').attr('src', loycartPOS.placeholder_image_url);
        jQuery(this).hide();
    });
    
    

    // Clear Cart Confirmation Modal Logic
    jQuery('#clear-cart-button').on('click', function(e) {
        e.preventDefault();
        openModal(window.$clearCartModal);
    });

    // Cancel button or close (X) closes modal
    jQuery('#clear-cart-cancel-btn, #clear-cart-modal .modal-close').on('click', function(e) {
        e.preventDefault();
        closeModal(window.$clearCartModal);
    });

    // Confirm button clears cart and closes modal
    jQuery('#clear-cart-confirm-btn').on('click', function(e) {
        e.preventDefault();
        closeModal(window.$clearCartModal);
        loycart.cartManager.clearCartAndCustomer();
    });

    // Shipping modal handlers
    jQuery('#add-shipping-button').on('click', function () {
        loycart.shippingManager.initiateShippingProcess();
    });

    jQuery('#calculate-guest-shipping').on('click', function () {
        const address = {
            first_name: jQuery('#shipping_first_name').val().trim(),
            last_name: jQuery('#shipping_last_name').val().trim(),
            email: jQuery('#shipping_email').val().trim(),
            phone: jQuery('#shipping_phone').val().trim(),
            address_1: jQuery('#shipping_address_1').val().trim(),
            address_2: jQuery('#shipping_address_2').val().trim(),
            city: jQuery('#shipping_city').val().trim(),
            postcode: jQuery('#shipping_postcode').val().trim(),
            country: jQuery('#shipping_country').val(),
            state: jQuery('#shipping_state').val().trim(),
            create_account: jQuery('#create_account_checkbox').is(':checked')
        };

        for (const key in address) {
            if (key === 'address_2' || key === 'create_account') {
                continue;
            }
            if (!address[key]) {
                const fieldName = key.replace(/_/g, ' ').replace(/\b\w/g, l => l.toUpperCase());
                showNotification(`'${fieldName}' is a required field.`, 'error');
                return;
            }
        }

        const emailRegex = /^[^\s@]+@[^\s@]+\.[^\s@]+$/;
        if (!emailRegex.test(address.email)) {
            showNotification('Please enter a valid email address.', 'error');
            return;
        }

        window.guestShippingAddress = address;
        loycart.shippingManager.fetchShippingOptions(null, window.guestShippingAddress);
    });

    jQuery('#shipping-options-content').on('click', '.shipping-method-item', function () {
        const $this = jQuery(this);
        jQuery('.shipping-method-item').removeClass('selected');
        $this.addClass('selected');

        window.selectedShipping = {
            id: $this.data('id'),
            cost: parseFloat($this.data('cost')),
            label: $this.data('label')
        };

        jQuery('#apply-shipping-button').prop('disabled', false);
    });

jQuery('#apply-shipping-button').on('click', function () {
    if (selectedShipping.id) {
        let notificationMessage = 'Shipping method applied to cart.';

        if (customerToSelect) {

            loycart.customerManager.addSelectedCustomerToCart(customerToSelect.id, customerToSelect.name);
            customerToSelect = null;
            notificationMessage = 'Customer and shipping method applied.';
        }


        closeModal(jQuery('#shipping-modal'));


        loycart.cartManager.getCartTotalsFromServer()
            .done(function() {

                if (loycart.paymentManager && loycart.paymentManager.initializeSplitPaymentModal) {
                    loycart.paymentManager.initializeSplitPaymentModal();
                    showNotification(notificationMessage, 'success');
                } else {
                    showNotification('Payment module not ready. Please click Charge.', 'warning');
                }
            })
            .fail(function() {
                showNotification('Failed to update cart totals. Please try again.', 'error');
            });
        

        loycart.shippingManager.updateShippingCallToAction();
        loycart.cartManager.debouncedSaveCartToServer(); // Non-blocking async save

    } else {
        showNotification('Please select a shipping method before applying.', 'warning');
    }
});

    $shippingModal.on('click', '.modal-close', function () {
        closeModal($shippingModal);
    });

    // Payment / Checkout Handlers
jQuery('#checkout-button').on('click', function () {
    if (jQuery(this).prop('disabled')) return;

    const $btn = jQuery(this);
    const orderRequiresShipping = loycart.cart.some(item => !item.is_virtual);

    if (orderRequiresShipping && window.selectedShipping.id === null) {
        // Shipping needed but not selected → open confirm modal
        window.shippingModalContext = 'checkout';
        openModal(window.$shippingConfirmModal);
        return;
    }

    // Temporarily disable button to prevent double clicks
    $btn.prop('disabled', true).text('Calculating...');

    // Force fresh server totals before opening payment modal
    loycart.cartManager.getCartTotalsFromServer()
        .done(function () {
            // Server totals are now up to date
            // window.splitPaymentTotalDue has been correctly set in getCartTotalsFromServer()

            if (window.splitPaymentTotalDue <= 0.001) {
                showNotification('Cannot process payment with zero or invalid total.', 'error');
                $btn.prop('disabled', false).text(loycartPOS.strings.checkoutDefaultText);
                return;
            }

            // Safe to open payment modal
            openModal(window.$splitPaymentModal || jQuery('#split-payment-modal'));
            loycart.paymentManager.initializeSplitPaymentModal();

            // Restore button text (it will update again via server totals anyway)
            $btn.prop('disabled', false);
        })
        .fail(function () {
            showNotification('Failed to calculate totals. Please try again.', 'error');
            $btn.prop('disabled', false).text(loycartPOS.strings.checkoutDefaultText);
        });
});

    jQuery('body').on('click', '#shipping-confirm-modal .modal-close', function () {
        closeModal($shippingConfirmModal);
    });

    jQuery('#confirm-add-shipping').on('click', function () {
        closeModal($shippingConfirmModal);
        setTimeout(function () {
            loycart.shippingManager.initiateShippingProcess();
        }, 100);
    });

    jQuery('#confirm-no-shipping').on('click', function () {
        closeModal($shippingConfirmModal);

        if (shippingModalContext === 'checkout') {
            loycart.paymentManager.initializeSplitPaymentModal();
        } else {
            if (customerToSelect) {
                loycart.customerManager.addSelectedCustomerToCart(customerToSelect.id, customerToSelect.name);
                customerToSelect = null;
            }
        }
    });
    


    $splitPaymentModal.on('click', '.modal-close', function () {
        closeModal($splitPaymentModal);
    });

    $splitPaymentModal.on('click', '#split-keypad button, .split-action-button', function () {
        loycart.paymentManager.handleSplitKeypad(jQuery(this));
    });

    $splitPaymentModal.on('click', '#split-quick-amounts .quick-cash-btn:not(:disabled)', function () {
        loycart.paymentManager.handleSplitQuickTender(jQuery(this));
    });

    $splitPaymentModal.on('input', '#split-amount-input', function () {
        let text = jQuery(this).val();
        text = text.replace(/[^0-9.]/g, '');
        const parts = text.split('.');
        if (parts.length > 2) {
            text = parts[0] + '.' + parts.slice(1).join('');
        }
        if (parts.length > 1 && parts[1].length > 2) {
            text = parts[0] + '.' + parts[1].substring(0, 2);
        }

        if (text !== jQuery(this).val()) {
            jQuery(this).val(text);
        }

        splitPaymentInput = text || '0.00';
        splitInputHasDecimal = splitPaymentInput.includes('.');
        loycart.paymentManager.updateSplitPaymentModalUI();
    });

    $splitPaymentModal.on('click', '.modal-payment-btn:not(:disabled)', function () {
        let amount = parseFloat(splitPaymentInput);
        if (isNaN(amount) || amount <= 0.001) {
            showNotification('Amount must be greater than zero.', 'error');
            return;
        }

        const paymentMethod = jQuery(this).data('payment-method');

        const amountToAdd = parseFloat(amount.toFixed(2));
        if (amountToAdd <= 0) return;

        const payment = {
            method: jQuery(this).data('payment-method'),
            title: jQuery(this).data('title') || jQuery(this).text().trim(),
            amount: amountToAdd
        };

        if (payment.method === 'cheque' || payment.method === 'bank_transfer') {
            const $advModal = jQuery('#advanced-pending-payment-modal');
            $advModal.find('#advanced-pending-message').text(`This will add a ${payment.title} payment of ${formatPrice(payment.amount)} to the order.`);
            const $instructionsDiv = $advModal.find('#pending-payment-instructions');
            if (payment.method === 'cheque') {
                $instructionsDiv.html(loycartPOS.cheque_instructions || '<p>No instructions available.</p>');
            } else {
                $instructionsDiv.html(loycartPOS.bacs_instructions || '<p>No instructions available.</p>');
            }

            $advModal.find('.pos-button[data-action="complete"]').hide();
            $advModal.find('.pos-button[data-action="hold-in-pos"]').hide();
            $advModal.find('.pos-button[data-action="on-hold"]')
                .text('Add Payment')
                .data('payment', payment)
                .show();

            openModal($advModal);
        } else if (payment.method === 'card') {
            $cardModalFinalTotal.text(formatPrice(amountToAdd));
            $cardConfirmModal.data('payment', payment);
            openModal($cardConfirmModal);
        } else {
            loycart.paymentManager.addPendingPayment(payment);
        }
    });

    $splitPaymentModal.on('click', '.split-payment-item-remove', function () {
        const index = jQuery(this).closest('.split-payment-item').data('index');
        if (index !== undefined && pendingPayments[index]) {
            const payment = pendingPayments[index];

            pendingPayments.splice(index, 1);
            splitPaymentAmountRemaining = splitPaymentAmountRemaining + payment.amount;

            if (pendingPayments.length === 0) {
                splitPaymentAmountRemaining = splitPaymentTotalDue;
            }

            const nextInput = splitPaymentAmountRemaining.toFixed(2);
            splitPaymentInput = nextInput;
            splitInputHasDecimal = nextInput.includes('.');

            loycart.paymentManager.updateSplitPaymentModalUI();
            loycart.cartManager.debouncedSaveCartToServer();
        }
    });




    $advancedPendingPaymentModal.on('click', '.modal-close', function () {
        $advancedPendingPaymentModal.find('.pos-button[data-action="on-hold"]').text('Place Order On Hold').removeData('payment');
        closeModal($advancedPendingPaymentModal);
    });

    $advancedPendingPaymentModal.on('click', '.pos-button[data-action="on-hold"]', function () {
        const payment = jQuery(this).data('payment');
        if (payment) {
            loycart.paymentManager.addPendingPayment(payment);
            closeModal($advancedPendingPaymentModal);
            jQuery(this).text('Place Order On Hold').removeData('payment');
        }
    });


    jQuery('#return-mode-button').on('click', function () {
        if (loycart.cart.length > 0) {
            showNotification('Please complete or empty the current sale before starting a return.', 'warning');
            return;
        }

        const customerId = $customerId.val();


        const $customerOrdersListContainer = $refundOrderLookupModal.find('#customer-order-history-container');
        const $customerOrdersList = $refundOrderLookupModal.find('#customer-order-history-list');
        const $manualLookup = $refundOrderLookupModal.find('#manual-order-id-input').closest('.modal-form-group');
        const $paginationControls = $refundOrderLookupModal.find('#refund-order-pagination');
        // --- END UI elements ---
        if (customerId && customerId !== '0') {

            $customerOrdersListContainer.show();
            $manualLookup.hide();
            $customerOrdersList.html('<p class="empty-state-message">Loading orders...</p>');


    api_get_customer_orders(customerId)
    .done(function (response) {
    const $customerOrdersList = $refundOrderLookupModal.find('#customer-order-history-list');
    const $paginationControls = $refundOrderLookupModal.find('#refund-order-pagination');
    $customerOrdersList.empty();
    
    if (response.success && response.data && response.data.orders && response.data.orders.length > 0) {
        const orders = response.data.orders;
        const pagination = response.data.pagination;
        
        orders.forEach(function (order) {

            const totalRaw = parseFloat(order.total_raw);
            const totalRefunded = parseFloat(order.total_refunded);
            

            const isPartiallyRefunded = (totalRefunded > 0 && totalRefunded < totalRaw);
            const isFullyRefunded = totalRefunded >= totalRaw;
            let statusClass = '';
            let statusText = 'Not Refunded';
            
            if (isFullyRefunded) {
                statusClass = 'status-fully-refunded';
                statusText = 'Fully Refunded';
            } else if (isPartiallyRefunded) {
                statusClass = 'status-partially-refunded';
                statusText = 'Partial Refund';
            } else {
                statusText = 'Eligible for Refund';

                statusClass = 'status-eligible'; 
            }

            const $orderItem = jQuery(`
                <div class="customer-order-item ${statusClass}" role="button" tabindex="0" data-order-id="${order.id}">
                    <div class="customer-order-info">
                        <div class="order-id-meta">
                            <strong>Order #${order.id}</strong> (${order.item_count} items)
                        </div>
                        <div class="order-total-meta">
                            <span>Date: ${order.date_created}</span>
                            <span>Total: <strong>${order.total}</strong></span>
                            <span class="order-refund-status">${statusText}</span>
                        </div>
                    </div>
                    <div class="customer-actions">
                        <button class="button button-secondary select-order-for-refund" data-order-id="${order.id}">Select</button>
                    </div>
                </div>
            `);
            $customerOrdersList.append($orderItem);
        });

        // Update Pagination UI
        jQuery('#refund-current-page').text(pagination.current_page);
        jQuery('#refund-total-pages').text(pagination.total_pages);
        jQuery('#refund-prev-page').prop('disabled', pagination.current_page === 1);
        jQuery('#refund-next-page').prop('disabled', pagination.current_page >= pagination.total_pages);
        $paginationControls.show();

    } else {
        $customerOrdersList.html('<p class="empty-state-message">No recent orders found for this customer.</p>');
        $paginationControls.hide();
    }
})
                .fail(function () {
                    $customerOrdersList.html('<p class="empty-state-message">Failed to load orders. Please try again.</p>');
                    $paginationControls.hide();
                });

        } else {
            $customerOrdersListContainer.hide();
            $manualLookup.show();
            jQuery('#manual-order-id-input').val('');
            $paginationControls.hide();
        }

        loycart.refundManager.setUIMode(true);
        const activatedTxt = (loycartPOS.strings && loycartPOS.strings.refundModeActivated) ? loycartPOS.strings.refundModeActivated : 'Return Mode Activated.';
        showNotification(activatedTxt, 'warning');
        openModal($refundOrderLookupModal);
    });

    $refundOrderLookupModal.on('click', '.select-order-for-refund', function () {
        const orderId = jQuery(this).data('order-id');
        loycart.refundManager.enterReturnModeForOrder(orderId);
    });

    $refundOrderLookupModal.on('click', '#process-manual-order-id', function () {
        const orderId = jQuery('#manual-order-id-input').val();
        if (orderId) {
            loycart.refundManager.enterReturnModeForOrder(orderId);
        }
    });

    $refundOrderLookupModal.on('click', '.modal-close', function () {
        loycart.refundManager.setUIMode(false);
        closeModal($refundOrderLookupModal);
        const canceledTxt = (loycartPOS.strings && loycartPOS.strings.refundModeCanceled) ? loycartPOS.strings.refundModeCanceled : 'Return Mode Canceled';
        showNotification(canceledTxt, 'info');
    });

// Step 1: Open reason modal when clicking main Process Refund button
jQuery('#process-refund-button').on('click', function () {
    // Validation: check if cart has items OR shipping is selected
    const hasCartItems = loycart.cart && loycart.cart.length > 0;
    const hasShipping = window.selectedShipping && window.selectedShipping.id;
    
    if (!hasCartItems && !hasShipping) {
        showNotification('No items selected for refund', 'warning');
        return;
    }

    // Open the reason modal (this is the missing step!)
    openModal(jQuery('#refund-reason-modal'));
    
    // Optional: Clear previous reason
    jQuery('#refund-reason-textarea').val('');
});

// Add close handlers for refund reason modal
jQuery('#refund-reason-modal').on('click', '.modal-close', function () {
    closeModal(jQuery('#refund-reason-modal'));
});


jQuery('#submit-refund-with-reason').on('click', function () {
        const reason = jQuery('#refund-reason-textarea').val().trim();

        if (reason === '') {
            showNotification('Please provide a reason for the refund.', 'error');
            return;
        }

        const $thisButton = jQuery(this);
        $thisButton.text('Processing...').prop('disabled', true);

        // FIX 1: Use loycart.cart instead of 'cart'
        const cartItemsCopy = JSON.parse(JSON.stringify(loycart.cart));

        // FIX 2: Use window.originalOrderId and loycart.cart for the API call
        api_processRefund(window.originalOrderId, JSON.stringify(loycart.cart), reason)
            .done(function (response) {
                if (response.success) {
                    closeModal(jQuery('#refund-reason-modal'));

                    // Prefer centralized helper (server-first, safe fallback)
                    const refundLiability = (typeof window.computeRefundDisplayTotal === 'function')
                        ? window.computeRefundDisplayTotal(response.data)
                        : (response.data && response.data.total ? Math.abs(parseFloat(response.data.total) || 0) : Math.abs(window.splitPaymentTotalDue || 0));

                    const receiptData = response.data;
                    receiptData.total = refundLiability;
                    
                    // Use the cart items from the API response, not the processed cart
                    // This ensures we have the correct refund data
                    const refundCartItems = response.data.cart_items || cartItemsCopy;
                    
                    $refundCompleteModal.data('receiptData', receiptData);
                    $refundCompleteModal.data('cartItems', refundCartItems);

                    jQuery('#refund-amount-display').text(formatPrice(refundLiability)); 

                    // Debug-only invariant check: compare server total vs helper
                    try {
                        if (loycartPOS && loycartPOS.debug_refunds) {
                            const serverAbs = response.data && response.data.total ? Math.abs(parseFloat(response.data.total) || 0) : 0;
                            const diff = Math.abs(serverAbs - refundLiability);
                            if (diff > 0.001) {
                                console.warn('Refund total mismatch:', { serverAbs, refundLiability, response: response.data });
                                showNotification('Debug: refund total mismatch detected. Using safe display total.', 'warning');
                            }
                        }
                    } catch (e) {
                        /* noop */
                    }
                    openModal($refundCompleteModal);

                } else {
                    showNotification(response.data.message || response.data, 'error');
                }
            })
            .fail(function () {
                showNotification('An error occurred while processing the refund.', 'error');
            })
            .always(function () {
                $thisButton.text('Process Refund').prop('disabled', false);
            });
    });



jQuery('#refund-date-filter').on('change', function() {
    loycart.refundManager.orderHistoryDate = jQuery(this).val();
    loycart.refundManager.loadCustomerOrders(1); 
}); 
    

jQuery('#refund-modal-done').on('click', function () {
    closeModal($refundCompleteModal);
    jQuery('#refund-summary-section').empty();
    loycart.cart = []; // Clear cart
    window.selectedShipping = { id: null, cost: 0, label: '' }; // Clear shipping
    loycart.cartManager.updateCart();
    loycart.refundManager.setUIMode(false);
    jQuery('#remove-customer-from-cart').trigger('click');
    jQuery('.pagination-controls').show(); 

    $refundCompleteModal.removeData('receiptData');
    window.$cartItems.empty(); // Clear cart display
    
    // Reset UI smoothly instead of reloading
    loycart.cartManager.resetTotalsUI();
    loycart.productManager.loadProducts();
    showNotification('Refund completed. POS reset.', 'success');
});




    jQuery('#refund-print-receipt').on('click', function () {
        const receiptData = $refundCompleteModal.data('receiptData');
        const cartItems = $refundCompleteModal.data('cartItems');

        const paperWidth = (window.loycartPOS.settings && window.loycartPOS.settings.receipt && window.loycartPOS.settings.receipt.paper_width) || '80mm';

        if (receiptData && cartItems) {
            printReceiptInternal(receiptData, cartItems, false, paperWidth);
        } else {
            showNotification('Error: Receipt data not found.', 'error');
        }
    });



    jQuery('#cancel-return-button, #exit-refund-view-button').on('click', function () {
    jQuery('#refund-summary-section').empty();
    loycart.cart = []; // FIX: Ensure namespace is used
    loycart.cartManager.updateCart();
    loycart.refundManager.setUIMode(false);
    jQuery('#remove-customer-from-cart').trigger('click');
});


    jQuery('#add-note-button').on('click', function () {
        jQuery('#order-note-textarea').val(window.orderNote);
        openModal($orderNoteModal);
    });
    
    
    

jQuery('#refund-order-lookup-modal').on('change', '#refund-status-select', function() {
    loycart.refundManager.orderHistoryStatus = jQuery(this).val();
    loycart.refundManager.loadCustomerOrders(1); 
});


const debouncedRefundSearch = debounce(function(query) {
    loycart.refundManager.orderHistorySearch = query;
    loycart.refundManager.loadCustomerOrders(1);
}, 500);

jQuery('#refund-order-search').on('input', function() {
    debouncedRefundSearch(jQuery(this).val().trim());
});


jQuery('#refund-order-pagination').on('click', '#refund-prev-page:not(:disabled)', function() {
    loycart.refundManager.loadCustomerOrders(loycart.refundManager.orderHistoryPage - 1);
});

jQuery('#refund-order-pagination').on('click', '#refund-next-page:not(:disabled)', function() {
    loycart.refundManager.loadCustomerOrders(loycart.refundManager.orderHistoryPage + 1);
});

  

    $orderNoteModal.on('click', '#save-order-note', function () {
        window.orderNote = jQuery('#order-note-textarea').val().trim();
        if (window.orderNote) {
            jQuery('#add-note-button').addClass('note-added').html('<span class="dashicons dashicons-yes-alt"></span> Note Added');
            showNotification('Order note saved.', 'success');
        } else {
            jQuery('#add-note-button').removeClass('note-added').html('<span class="dashicons dashicons-edit-page"></span> Add Note');
        }
        closeModal($orderNoteModal);
    });

    $orderNoteModal.on('click', '.modal-close', function () {
        closeModal($orderNoteModal);
    });


    jQuery('#product-info-modal').on('click', '.product-info-thumbnail', function () {
        const newSrc = jQuery(this).data('full-src');
        jQuery('#product-info-modal .product-info-main-image').attr('src', newSrc);
        jQuery('#product-info-modal .product-info-thumbnail').removeClass('active');
        jQuery(this).addClass('active');
    });


    // Clear the product info modal when it is hidden
jQuery(document).on('click', '#product-info-modal .modal-close', function() {
    jQuery('#footer-price-display').html('');
    jQuery('#footer-stock-display').html('').attr('class', '');
    jQuery('#product-info-modal').removeData('selected-variation');
    jQuery('#product-info-modal').removeData('product-data');
});
    
    
    jQuery(document).on('click', '#barcode-scanner-modal .modal-close', function(e) {
    e.preventDefault();
    loycart.scanner.closeScanner(true);
    });

    // Close scanner on backdrop click (outside modal-content)
    jQuery(document).on('click', '#barcode-scanner-modal', function(e) {
        if (e.target === this) {
            loycart.scanner.closeScanner(true);
        }
    });



    // Hold Sale Modal logic
    jQuery('body').on('click', '#hold-sale-button', function () {
        if (jQuery(this).prop('disabled')) return;
        const action = jQuery(this).data('action');

        if (action === 'retrieve') {
            openModal($heldSalesModal);
            loycart.heldSalesManager.refreshHeldSales();
        } else {
            // Open the hold sale modal for note entry using openModal for consistency
            console.log('[POS] Hold Sale button clicked, opening modal');
            window.$holdSaleModal.find('#hold-sale-note').val('');
            window.$holdSaleModal.addClass('show');
            // openModal(window.$holdSaleModal); // Not needed for this modal
            setTimeout(function() {
                window.$holdSaleModal.find('#hold-sale-note').triggerHandler('focus');
            }, 200);
        }
    });

    // Modal close logic
    jQuery('body').on('click', '#hold-sale-modal-close, #hold-sale-cancel-btn', function () {
        window.$holdSaleModal.removeClass('show');
        // closeModal(window.$holdSaleModal); // Not needed for this modal
    });

    // Confirm hold sale with note
    jQuery('body').on('click', '#hold-sale-confirm-btn', function () {
        // Get note value
        var note = window.$holdSaleModal.find('#hold-sale-note').val().trim();
        var currentCustomerName = window.$selectedCustomerName ? window.$selectedCustomerName.text() : '';
        var strings = window.loycartPOS ? window.loycartPOS.strings : {};
        var heldName = currentCustomerName ?
            (strings.defaultHeldOrderFor ? strings.defaultHeldOrderFor.replace('%s', currentCustomerName) : 'Held for ' + currentCustomerName) :
            (strings.defaultHeldAtTime ? strings.defaultHeldAtTime.replace('%s', new Date().toLocaleTimeString([], { hour: '2-digit', minute: '2-digit' })) : 'Held at ' + new Date().toLocaleTimeString([], { hour: '2-digit', minute: '2-digit' }));
        if (note) {
            heldName += ' - ' + note;
        }
        loycart.heldSalesManager.holdSaleAjax(heldName);
        closeModal(window.$holdSaleModal);
    });

    // Optional: Close modal on outside click
    // Remove outside click close for hold-sale-modal (handled by openModal/closeModal)




$receiptModal.on('click', '#print-receipt', function() {
    const saleData = $receiptModal.data('saleResponse');
    const cartCopy = $receiptModal.data('cartCopy');
    const paperWidth = (window.loycartPOS.settings && window.loycartPOS.settings.receipt && window.loycartPOS.settings.receipt.paper_width) || '80mm';

    if (saleData && cartCopy) {
        
        const finalHtml = loycart.cartManager.printReceipt(saleData, paperWidth);
        window.triggerIframePrint(finalHtml, 'Receipt #' + saleData.order_id, paperWidth);
    } else {
        showNotification('Error: Receipt data missing.', 'error');
    }
});


$receiptModal.on('click', '#dismiss-receipt', function() {
    console.log('[New Sale] Button clicked');
    closeModal($receiptModal);
    console.log('[New Sale] Modal closed, calling clearCartAndCustomer');
    loycart.cartManager.clearCartAndCustomer();
});


$receiptModal.on('click', '#print-order-label-button', function(e) {
    e.preventDefault();
    const saleData = $receiptModal.data('saleResponse');

    if (saleData) {
        const labelData = {
            order_number: saleData.order_id,
            customer_name: saleData.customer_name || 'Guest',
            label_address: saleData.shipping_address || '',
            label_type: 'shipping',
            label_title: 'Shipping Label',
            barcode_id: String(saleData.order_id).replace(/\D/g, '') 
        };

        if (loycart.labelManager && typeof loycart.labelManager.buildLabelHtml === 'function') {
            
            let labelHtml = loycart.labelManager.buildLabelHtml(labelData);
           
            const $tempDiv = jQuery('<div style="display:none;">').append(labelHtml);
            jQuery('body').append($tempDiv);

            const barcodeSvg = $tempDiv.find('#barcode')[0];
            if (barcodeSvg && typeof JsBarcode !== 'undefined') {
                JsBarcode(barcodeSvg, labelData.barcode_id, {
                    format: "CODE128",
                    width: 2,
                    height: 40,
                    displayValue: true
                });
            }

            const finalLabelHtml = $tempDiv.html();
            $tempDiv.remove();

            window.triggerIframePrint(finalLabelHtml, 'Order Label - #' + labelData.order_number, '4x6');
        }
    }
});




jQuery('body').on('click', '#more-actions-toggle', function () {
    const $button = jQuery(this);
    const $secondaryActions = jQuery('#cart-secondary-actions');

    if ($secondaryActions.is(':visible')) {

        window.closeMoreActions(); 
    } else {

        $secondaryActions.slideDown(200);
        $button.html('<span class="dashicons dashicons-no-alt"></span> More Actions');
        $button.addClass('is-open');

        

        // Sync theme toggle button label with current state
        const isDark = jQuery('body').hasClass('dark-mode');
        const $themeBtn = jQuery('#toggle-theme-button');
        if ($themeBtn.length) {
            const label = isDark ? 'Light Mode' : 'Dark Mode';
            const icon = isDark ? 'dashicons-lightbulb' : 'dashicons-moon';
            $themeBtn.html('<span class="dashicons ' + icon + '"></span> ' + label);
        }
    }
});

// Ensure secondary actions are hidden by default on page load
jQuery(function() {
    jQuery('#cart-secondary-actions').hide();
});

// Define closeMoreActions globally if not already defined
if (typeof window.closeMoreActions !== 'function') {
    window.closeMoreActions = function() {
        var $secondaryActions = jQuery('#cart-secondary-actions');
        var $button = jQuery('#more-actions-toggle');
        $secondaryActions.slideUp(200);
        $button.html('<span class="dashicons dashicons-screenoptions"></span> More Actions');
        $button.removeClass('is-open');
    };
}


jQuery('#cart-secondary-actions').on('click', 'button', function() {
    window.closeMoreActions();
});
    


// Theme toggle button handler
jQuery('body').on('click', '#toggle-theme-button', function (e) {
    e.preventDefault();
    const $body = jQuery('body');
    const isDark = $body.hasClass('dark-mode');
    const newDark = !isDark;
    $body.toggleClass('dark-mode', newDark);
    window.loycartPOS.settings.ui.default_theme = newDark ? 'dark' : 'light';
    updateAdminBarButton();
    showNotification(newDark ? 'Dark Mode enabled' : 'Light Mode enabled', 'info');

    // Update button label to next state
    const label = newDark ? 'Light Mode' : 'Dark Mode';
    const icon = newDark ? 'dashicons-lightbulb' : 'dashicons-moon';
    jQuery(this).html('<span class="dashicons ' + icon + '"></span> ' + label);

    // Persist theme change to server (debounced)
    debouncedSaveSettingsToServer();
// Persist all settings changes (debounced, like cart)
jQuery(document).on('change', '#pos-setting-paper-width, #pos-setting-products-per-page, #pos-setting-default-sort, #pos-setting-default-theme, #pos-setting-show-theme-toggle, #pos-setting-show-notifications, #pos-setting-auto-redirect-pos, #pos-setting-credit-enabled, #pos-setting-earn-percent, #pos-setting-apply-online, #pos-setting-exc-earn, #pos-setting-exc-redeem, #pos-setting-store-name, #pos-setting-store-phone, #pos-setting-receipt-footer, #pos-setting-usb-direct-print, #pos-setting-allow-private', function() {
    debouncedSaveSettingsToServer();
});

    // Persist theme change to server (debounced)
    debouncedSaveSettingsToServer();

    // Immediately persist theme change to server
    if (window.api_savePosSettings && window.loycartPOS && window.loycartPOS.settings) {
        window.api_savePosSettings(window.loycartPOS.settings);
    }

// Persist all settings changes (debounced, like cart)
jQuery(document).on('change', '#pos-setting-paper-width, #pos-setting-products-per-page, #pos-setting-default-sort, #pos-setting-default-theme, #pos-setting-show-theme-toggle, #pos-setting-show-notifications, #pos-setting-auto-redirect-pos, #pos-setting-credit-enabled, #pos-setting-earn-percent, #pos-setting-apply-online, #pos-setting-exc-earn, #pos-setting-exc-redeem, #pos-setting-store-name, #pos-setting-store-phone, #pos-setting-receipt-footer, #pos-setting-usb-direct-print, #pos-setting-allow-private', function() {
    debouncedSaveSettingsToServer();
});
});
    

jQuery('body').on('click', '#open-drawer-button', function() {

    window.closeMoreActions(); 
    

    const $button = jQuery(this).prop('disabled', true);
    const originalText = $button.text();
    $button.text('Opening...');


    api_openCashDrawer()
        .done(function(response) {
            if (response.success) {
                showNotification('Cash drawer opened.', 'success');
            } else {
                showNotification('Error: Could not send cash drawer command.', 'error');
            }
        })
        .fail(function() {
            showNotification('Connection error: Failed to reach server for cash drawer command.', 'error');
        })
        .always(function() {

            $button.text(originalText).prop('disabled', false);
        });
});




window.isCartLoading = true;
    loycart.cartManager.loadCartFromServer().always(function () {       
        window.isCartLoading = false;
        loycart.productManager.loadProducts();
        loycart.heldSalesManager.refreshHeldSales();
    });
});

// --- 5. GLOBAL CLEANUP (OUTSIDE READY BLOCK) ---
window.loycartFinalizeSaleCleanup = function() {
    window.isProcessing = false; 
    if (loycart.cartManager && typeof loycart.cartManager.clearCartAndCustomer === 'function') {
        loycart.cartManager.clearCartAndCustomer();
    }
    if (loycart.cartManager && typeof loycart.cartManager.debouncedSaveCartToServer === 'function') {
        loycart.cartManager.debouncedSaveCartToServer(); 
    }
};