<?php
/**
 * Manage points earning rule template class
 *
 * @package WooCommerce Points and Rewards
 * @version 1.0.0
 */

namespace DDWCPointsAndRewards\Templates\Admin\Points_Rules\Points_Earning;

use DevDiggers\Framework\Includes\DDFW_Layout;
use DDWCPointsAndRewards\Helper\Rule\DDWCPR_Rule_Helper;

defined( 'ABSPATH' ) || exit;

if ( ! class_exists( 'DDWCPR_Manage_Points_Earning_Rule_Template' ) ) {
	/**
	 * Manage points earning rule template class
	 */
	class DDWCPR_Manage_Points_Earning_Rule_Template {
		/**
		 * Configuration Variable
		 *
		 * @var array
		 */
		protected $ddwcpr_configuration;

		/**
		 * Rule Helper Variable
		 *
		 * @var object
		 */
		protected $rule_helper;

		/**
		 * Construct
		 * 
		 * @param array $ddwcpr_configuration
		 */
		public function __construct( $ddwcpr_configuration ) {
			$this->ddwcpr_configuration = $ddwcpr_configuration;
			$this->rule_helper          = new DDWCPR_Rule_Helper( $ddwcpr_configuration );

			$this->ddwcpr_perform_save_rule();
			$this->ddwcpr_get_manage_earning_rule_template();
		}

		/**
		 * Perform save rule function
		 *
		 * @return void
		 */
		public function ddwcpr_perform_save_rule() {
			if ( ! empty( $_POST['ddwcpr_save_rule_submit'] ) && ! empty( $_POST['ddwcpr_save_rule_submit_nonce'] ) && wp_verify_nonce( sanitize_text_field( wp_unslash( $_POST['ddwcpr_save_rule_submit_nonce'] ) ), 'ddwcpr_save_rule_submit_nonce_action' ) ) {
				$this->rule_helper->ddwcpr_save_earning_rule();
			}
		}

		/**
		 * Get Manage Earning Rule Template function
		 *
		 * @return void
		 */
		public function ddwcpr_get_manage_earning_rule_template() {
			$earning_rule = [
				'name'            => '',
				'description'     => '',
				'priority'        => 1,
				'type'            => '',
				'applicable_to'   => 'all_users',
				'start_date'      => '',
				'icon'            => '',
				'end_date'        => '',
				'show'            => '',
				'status'          => 'active',
				'additional_data' => [
					'minimum_rating'           => '',
					'points_type'              => '',
					'minimum_order_value'      => '',
					'apply_to_products'        => 'all_products',
					'applied_products'         => [],
					'applied_categories'       => [],
					'points_value'             => '',
					'points_per_amount'        => '',
					'level_points'             => [],
					'maximum_points_per_order' => '',
					'applicable_days'          => [],
					'specific_user_roles'      => [],
					'specific_levels'          => [],
				],
			];

			if ( 'edit' === $_GET['action'] && ! empty( $_GET['ddwcpr-id'] ) ) {
				$id           = intval( wp_unslash( $_GET['ddwcpr-id'] ) );              // WPCS: input var ok.
				$earning_rule = $this->rule_helper->ddwcpr_get_earning_rule_by_id( $id );

				$earning_rule['additional_data'] = maybe_unserialize( $earning_rule['additional_data'] );
				$earning_rule['start_date']      = $earning_rule['start_date'] ? date( 'Y-m-d', strtotime( $earning_rule['start_date'] ) ) : $earning_rule['start_date'];
				$earning_rule['end_date']        = $earning_rule['end_date'] ? date( 'Y-m-d', strtotime( $earning_rule['end_date'] ) ) : $earning_rule['end_date'];
			}

			extract( $earning_rule );
			extract( $earning_rule['additional_data'] );

			if ( 'add' === $_GET['action'] ) {
				$heading = esc_html__( 'Add Rule', 'loyaltyx-points-and-rewards-for-woocommerce' );
			} else {
				$heading = esc_html__( 'Edit Rule', 'loyaltyx-points-and-rewards-for-woocommerce' );
			}

			$page = ! empty( $_GET['page'] ) ? sanitize_text_field( wp_unslash( $_GET['page'] ) ) : '';
			$menu = ! empty( $_GET['menu'] ) ? sanitize_text_field( wp_unslash( $_GET['menu'] ) ) : '';

			if ( ! empty( $_GET['tab'] ) ) {
				$tab             = sanitize_text_field( wp_unslash( $_GET['tab'] ) );
				$back_button_url = admin_url( "admin.php?page={$page}&menu={$menu}&tab={$tab}" );
			} else {
				$back_button_url = admin_url( "admin.php?page={$page}&menu={$menu}" );
			}

			$rule_types = ddwcpr_get_earning_rule_types();
			ob_start();
			?>
			<div class="ddwcpr-rules-types-container">
				<div class="ddwcpr-rules-types-heading">
					<h3>
						<?php
						if ( 'edit' === $_GET['action'] ) {
							esc_html_e( 'Rule Type', 'loyaltyx-points-and-rewards-for-woocommerce' );
						} else {
							esc_html_e( 'Choose Rule Type', 'loyaltyx-points-and-rewards-for-woocommerce' );
						}
						?>
					</h3>
					<span class="button ddwcpr-change-rule-type-button <?php echo esc_attr( empty( $type ) || 'edit' === $_GET['action'] ? 'ddfw-hide' : '' ); ?>"><?php esc_html_e( 'Change', 'loyaltyx-points-and-rewards-for-woocommerce' ); ?></span>
				</div>
				<div class="ddwcpr-rules-types <?php echo esc_attr( empty( $type ) ? 'ddwcpr-grid3' : '' ); ?>">
					<?php
					foreach ( $rule_types as $rule_type => $rule ) {
						?>
						<div class="ddwcpr-rule-type-card <?php echo esc_attr( ! empty( $type ) && $rule_type !== $type ? 'ddfw-hide' : '' ); ?> <?php echo esc_attr( ! empty( $rule['pro_version'] ) && $rule['pro_version'] ? 'ddfw-upgrade-to-pro-tag-wrapper' : '' ); ?>" data-rule-type="<?php echo esc_attr( $rule_type ); ?>" <?php echo esc_attr( ! empty( $rule['pro_version'] ) && $rule['pro_version'] ? 'disabled' : '' ); ?>>
							<span class="ddwcpr-rule-type-card-icon-wrapper">
								<img src="<?php echo esc_attr( ddwcpr_get_rules_icon( $rule_type ) ); ?>" />
							</span>
							<h4><?php echo esc_html( $rule['heading'] ); ?></h4>
							<p><?php echo wp_kses_post( $rule['description'] ); ?></p>
							<?php
							if ( ! empty( $rule['pro_version'] ) && $rule['pro_version'] ) {
								ddfw_pro_tag();
							}
							?>
						</div>
						<?php
					}
					?>
				</div>
			</div>
			<?php
			$rules_types_html = ob_get_clean();

			$args = [
				[
					'header' => [
						'heading'             => $heading,
						'back_button_enabled' => true,
						'back_button_url'     => $back_button_url,
						'description'         => esc_html__( 'Configure the conditions under which customers can earn points on your store.', 'loyaltyx-points-and-rewards-for-woocommerce' ),
					],
					'after_header_html' => $rules_types_html,
				],
				[
					'header' => [
						'heading'             => esc_html__( 'Rule Details', 'loyaltyx-points-and-rewards-for-woocommerce' ),
						'description'         => esc_html__( 'Define the basic information for this earning rule, including its name, description, and priority.', 'loyaltyx-points-and-rewards-for-woocommerce' ),
					],
					'class' => 'ddwcpr-rules-form-container' . esc_attr( 'add' === $_GET['action'] ? ' ddfw-hide' : '' ),
					'fields' => [
						[
							'type'        => 'text',
							'label'       => esc_html__( 'Rule Name', 'loyaltyx-points-and-rewards-for-woocommerce' ),
							'description' => esc_html__( 'Enter a meaningful name for the earning rule (e.g., "Points for Product Purchase").', 'loyaltyx-points-and-rewards-for-woocommerce' ),
							'value'       => $name,
							'id'          => 'ddwcpr-rule-name',
							'required'    => true,
						],
						[
							'type'        => 'textarea',
							'label'       => esc_html__( 'Rule Description', 'loyaltyx-points-and-rewards-for-woocommerce' ),
							'description' => esc_html__( 'Enter a meaningful description for the earning rule.', 'loyaltyx-points-and-rewards-for-woocommerce' ),
							'value'       => $description,
							'id'          => 'ddwcpr-rule-description',
						],
						[
							'type'              => 'number',
							'label'             => esc_html__( 'Priority', 'loyaltyx-points-and-rewards-for-woocommerce' ),
							'description'       => esc_html__( 'Assign a priority to this rule. Lower numbers (e.g., "1") mean higher priority and take effect first.', 'loyaltyx-points-and-rewards-for-woocommerce' ),
							'value'             => $priority,
							'id'                => 'ddwcpr-rule-priority',
							'required'          => true,
							'custom_attributes' => [
								'min' => 1,
							],
						],
						[
							'type'        => 'select',
							'label'       => esc_html__( 'Type', 'loyaltyx-points-and-rewards-for-woocommerce' ),
							'description' => esc_html__( "Choose the event or action that will award points to customers.", 'loyaltyx-points-and-rewards-for-woocommerce' ),
							'id'          => 'ddwcpr-rule-type',
							'value'       => $type,
							'field_class' => [ 'ddfw-hide' ],
							'required'    => true,
							'options'     => array_reduce(
								array_keys( $rule_types ),
								function( $carry, $rule_type ) use ( $rule_types ) {
									$carry[ $rule_type ] = esc_html( $rule_types[ $rule_type ]['heading'] );
									return $carry;
								},
								[ '' => esc_html__( 'Choose an option', 'loyaltyx-points-and-rewards-for-woocommerce' ) ]
							),
							'show_fields' => [
								'purchase_amount'   => [ 'ddwcpr-rule-minimum-order-value', 'ddwcpr-rule-maximum-points-per-order', 'ddwcpr-rule-points-type', 'ddwcpr-rule-points-value' ],
							],
						],
						[
							'type'        => 'select',
							'label'       => esc_html__( 'Points Type', 'loyaltyx-points-and-rewards-for-woocommerce' ),
							'description' => esc_html__( 'Choose the type of points to award for this rule.', 'loyaltyx-points-and-rewards-for-woocommerce' ),
							'id'          => 'ddwcpr-rule-points-type',
							'value'       => $points_type,
							'required'    => true,
							'options'     => [
								'fixed_points_based_on_amount' => esc_html__( 'Fixed Points Based on Amount', 'loyaltyx-points-and-rewards-for-woocommerce' ),
								'fixed_points'                 => esc_html__( 'Fixed Points', 'loyaltyx-points-and-rewards-for-woocommerce' ),
								'percentage_points'            => esc_html__( 'Percentage Points', 'loyaltyx-points-and-rewards-for-woocommerce' ),
							],
						],
						[
							'type'        => 'two_input_columns',
							'class'       => [ 'ddwcpr-rule-points-value-columns' ],
							'label'       => esc_html__( 'Points Value', 'loyaltyx-points-and-rewards-for-woocommerce' ),
							'description' => esc_html__( 'Enter the number of points to be awarded.', 'loyaltyx-points-and-rewards-for-woocommerce' ),
							'columns'     => [
								[
									'type'    => 'number',
									'id'      => 'ddwcpr-rule-points-per-amount',
									'name'    => '_ddwcpr_rule_points_per_amount',
									'value'   => $points_per_amount,
									'min'     => 0,
									'prefix'  => esc_html__( 'For each', 'loyaltyx-points-and-rewards-for-woocommerce' ) . '&nbsp;',
									'suffix'  => '&nbsp;' . get_woocommerce_currency_symbol() . '(' . get_woocommerce_currency() . ')' . '&nbsp;',
									'wrapper' => 'span',
								],
								[
									'type'    => 'number',
									'id'      => 'ddwcpr-rule-points-value',
									'name'    => '_ddwcpr_rule_points_value',
									'value'   => $points_value,
									'min'     => 0,
									'suffix'  => '&nbsp;<span id="ddwcpr-rule-points-value-points">' . esc_html__( 'Points', 'loyaltyx-points-and-rewards-for-woocommerce' ) . '</span><span id="ddwcpr-rule-points-value-percentage">%</span>',
									'wrapper' => 'span',
								],
							],
						],
						[
							'type'          => 'image',
							'label'         => esc_html__( 'Icon', 'loyaltyx-points-and-rewards-for-woocommerce' ),
							'description'   => esc_html__( 'Choose an icon (100x100px) to represent rule.', 'loyaltyx-points-and-rewards-for-woocommerce' ),
							'id'            => 'ddwcpr-rule-icon',
							'value'         => $icon,
							'default_image' => ddwcpr_get_rules_icon( $type ),
						],
						[
							'type'        => 'select',
							'label'       => esc_html__( 'Show to Customer', 'loyaltyx-points-and-rewards-for-woocommerce' ),
							'description' => esc_html__( "Choose whether to display this rule in the customer’s dashboard so they are informed about it.", 'loyaltyx-points-and-rewards-for-woocommerce' ),
							'id'          => 'ddwcpr-rule-show',
							'value'       => $show,
							'required'    => true,
							'options'     => [
								'show'   => esc_html__( 'Show', 'loyaltyx-points-and-rewards-for-woocommerce' ),
								'hide'   => esc_html__( 'Hide', 'loyaltyx-points-and-rewards-for-woocommerce' ),
							],
						],
						[
							'type'        => 'select',
							'label'       => esc_html__( 'Status', 'loyaltyx-points-and-rewards-for-woocommerce' ),
							'description' => esc_html__( "Set the status to 'active' if you want users earn points from this rule.", 'loyaltyx-points-and-rewards-for-woocommerce' ),
							'id'          => 'ddwcpr-rule-status',
							'value'       => $status,
							'required'    => true,
							'options'     => [
								'active'   => esc_html__( 'Active', 'loyaltyx-points-and-rewards-for-woocommerce' ),
								'inactive' => esc_html__( 'Inactive', 'loyaltyx-points-and-rewards-for-woocommerce' ),
							],
						],
					],
				],
				[
					'header' => [
						'heading'     => esc_html__( 'Conditions', 'loyaltyx-points-and-rewards-for-woocommerce' ),
						'description' => esc_html__( 'These settings help use the reward to particular use-cases or marketing goals.', 'loyaltyx-points-and-rewards-for-woocommerce' ),
					],
					'class' => 'ddfw-upgrade-to-pro-tag-wrapper ddwcpr-rules-form-container' . esc_attr( 'add' === $_GET['action'] ? ' ddfw-hide' : '' ),
					'fields' => [
						[
							'type'              => 'number',
							'label'             => esc_html__( 'Minimum Order Value', 'loyaltyx-points-and-rewards-for-woocommerce' ),
							'description'       => esc_html__( 'Set the minimum order value required to earn points.', 'loyaltyx-points-and-rewards-for-woocommerce' ),
							'value'             => $minimum_order_value,
							'id'                => 'ddwcpr-rule-minimum-order-value',
							'custom_attributes' => [
								'disabled'    => 'disabled',
							],
						],
						[
							'type'              => 'number',
							'label'             => esc_html__( 'Maximum Points Per Order', 'loyaltyx-points-and-rewards-for-woocommerce' ),
							'description'       => esc_html__( 'Enter the number of points earned, regardless of order size.', 'loyaltyx-points-and-rewards-for-woocommerce' ),
							'value'             => $maximum_points_per_order,
							'id'                => 'ddwcpr-rule-maximum-points-per-order',
							'custom_attributes' => [
								'disabled'    => 'disabled',
							],
						],
						[
							'type'        => 'select',
							'label'       => esc_html__( 'Applicable on Days', 'loyaltyx-points-and-rewards-for-woocommerce' ),
							'placeholder' => esc_html__( 'All Days', 'loyaltyx-points-and-rewards-for-woocommerce' ),
							'description' => esc_html__( 'Choose the days of the week when this rule can be applied. Leave empty to allow all days.', 'loyaltyx-points-and-rewards-for-woocommerce' ),
							'id'          => 'ddwcpr-rule-applicable-days',
							'name'        => '_ddwcpr_rule_applicable_days[]',
							'value'       => $applicable_days,
							'options'     => [
								'monday'    => esc_html__( 'Monday', 'loyaltyx-points-and-rewards-for-woocommerce' ),
								'tuesday'   => esc_html__( 'Tuesday', 'loyaltyx-points-and-rewards-for-woocommerce' ),
								'wednesday' => esc_html__( 'Wednesday', 'loyaltyx-points-and-rewards-for-woocommerce' ),
								'thursday'  => esc_html__( 'Thursday', 'loyaltyx-points-and-rewards-for-woocommerce' ),
								'friday'    => esc_html__( 'Friday', 'loyaltyx-points-and-rewards-for-woocommerce' ),
								'saturday'  => esc_html__( 'Saturday', 'loyaltyx-points-and-rewards-for-woocommerce' ),
								'sunday'    => esc_html__( 'Sunday', 'loyaltyx-points-and-rewards-for-woocommerce' ),
							],
							'custom_attributes' => [
								'multiple' => true,
								'disabled' => 'disabled',
							]
						],
						[
							'type'        => 'radio',
							'label'       => esc_html__( 'Applicable to', 'loyaltyx-points-and-rewards-for-woocommerce' ),
							'description' => esc_html__( "Choose which users this rule should apply to.", 'loyaltyx-points-and-rewards-for-woocommerce' ),
							'id'          => 'ddwcpr-rule-applicable-to',
							'value'       => $applicable_to,
							'required'    => true,
							'options'     => [
								'all_users'           => esc_html__( 'All Users', 'loyaltyx-points-and-rewards-for-woocommerce' ),
								'specific_user_roles' => esc_html__( 'Specific User Roles', 'loyaltyx-points-and-rewards-for-woocommerce' ),
								'specific_levels'     => esc_html__( 'Specific Levels', 'loyaltyx-points-and-rewards-for-woocommerce' ),
							],
							'show_fields' => [
								'specific_user_roles' => [ 'ddwcpr-rule-specific-user-roles' ],
							],
							'custom_attributes' => [
								'disabled'    => 'disabled',
							],
						],
						[
							'type'        => 'user_roles',
							'label'       => esc_html__( 'Choose User Roles', 'loyaltyx-points-and-rewards-for-woocommerce' ),
							'description' => esc_html__( 'Restrict earning rule to specific user roles.', 'loyaltyx-points-and-rewards-for-woocommerce' ),
							'id'          => 'ddwcpr-rule-specific-user-roles',
							'name'        => '_ddwcpr_rule_specific_user_roles[]',
							'value'       => $specific_user_roles,
							'custom_attributes' => [
								'multiple'    => true,
								'disabled'    => 'disabled',
							]
						],
					],
				],
				[
					'header' => [
						'heading'     => esc_html__( 'Schedule', 'loyaltyx-points-and-rewards-for-woocommerce' ),
						'description' => esc_html__( 'Set a time window during which this earning rule will be active and visible to customers.', 'loyaltyx-points-and-rewards-for-woocommerce' ),
					],
					'class' => 'ddwcpr-rules-form-container' . esc_attr( 'add' === $_GET['action'] ? ' ddfw-hide' : '' ),
					'fields' => [
						[
							'type'        => 'two_input_columns',
							'class'       => [ 'ddwcpr-two-input-columns' ],
							'td_class'    => [ 'ddfw-padding-0' ],
							'label'       => '',
							'description' => esc_html__( 'Leave empty to start immediately.', 'loyaltyx-points-and-rewards-for-woocommerce' ),
							'columns'     => [
								[
									'type'        => 'date',
									'label'       => esc_html__( 'Start Date', 'loyaltyx-points-and-rewards-for-woocommerce' ),
									'id'          => 'ddwcpr-rule-start-date',
									'name'        => '_ddwcpr_rule_start_date',
									'value'       => $start_date,
									'class'       => [ 'regular-text' ],
									'wrapper'     => 'label',
								],
								[
									'type'        => 'date',
									'label'       => esc_html__( 'End Date', 'loyaltyx-points-and-rewards-for-woocommerce' ),
									'id'          => 'ddwcpr-rule-end-date',
									'name'        => '_ddwcpr_rule_end_date',
									'value'       => $end_date,
									'class'       => [ 'regular-text' ],
									'wrapper'     => 'label',
								],
							],
						],
					],
				],
			];

			$layout = new DDFW_Layout();
			$layout->get_form_section_layout(
				$args,
				'',
				[
					'name'                => 'ddwcpr_save_rule_submit',
					'value'               => esc_html__( 'Save Rule', 'loyaltyx-points-and-rewards-for-woocommerce' ),
					'button_parent_class' => 'ddwcpr-rules-form-container' . esc_attr( 'add' === $_GET['action'] ? ' ddfw-hide' : '' ),
				]
			);
		}
	}
}
