<?php
/**
 * Manual adjustment template class
 *
 * @package WooCommerce Points and Rewards
 * @version 1.0.0
 */

namespace DDWCPointsAndRewards\Templates\Admin\Manual_Adjustment;

use DDWCPointsAndRewards\Helper\Point\DDWCPR_Point_Helper;
use DevDiggers\Framework\Includes\DDFW_SVG;

defined( 'ABSPATH' ) || exit;

if ( ! class_exists( 'DDWCPR_Manual_Adjustment_Template' ) ) {
	/**
	 * Manual adjustment template class
	 */
	class DDWCPR_Manual_Adjustment_Template {
		/**
		 * Configuration Variable
		 *
		 * @var array
		 */
		protected $ddwcpr_configuration;

		/**
		 * Construct
		 * 
		 * @param array $ddwcpr_configuration
		 */
		public function __construct( $ddwcpr_configuration ) {
			$this->ddwcpr_configuration = $ddwcpr_configuration;

			$this->ddwcpr_enqueue_scripts();
			$this->ddwcpr_get_manual_adjustment_template();
		}

		/**
		 * Enqueue scripts and styles
		 *
		 * @return void
		 */
		public function ddwcpr_enqueue_scripts() {
		wp_enqueue_style( 'ddwcpr-manual-adjustment-style' );
		wp_enqueue_script( 'ddwcpr-manual-adjustment-script' );
		
		// Localize script with AJAX data
		wp_localize_script(
			'ddwcpr-manual-adjustment-script',
			'ddwcprManualAdjustmentObject',
			[
				'ajax' => [
					'ajaxUrl'   => admin_url( 'admin-ajax.php' ),
					'ajaxNonce' => wp_create_nonce( 'ddwcpr-nonce' ),
				],
				'i18n' => [
					'enterValidPoints' => esc_html__( 'Please enter a valid points amount.', 'loyaltyx-points-and-rewards-for-woocommerce' ),
					'noUsersFound'     => esc_html__( 'No users found for processing.', 'loyaltyx-points-and-rewards-for-woocommerce' ),
					'selectUsers'       => esc_html__( 'Please select at least one user.', 'loyaltyx-points-and-rewards-for-woocommerce' ),
					'processingError'   => esc_html__( 'An error occurred while processing. Please try again.', 'loyaltyx-points-and-rewards-for-woocommerce' ),
					'processingAdjustment' => esc_html__( 'Processing points adjustment...', 'loyaltyx-points-and-rewards-for-woocommerce' ),
				],
			]
		);
		}

		/**
		 * Get Manual Adjustment Template function
		 *
		 * @return void
		 */
		public function ddwcpr_get_manual_adjustment_template() {
			?>
			<div class="ddwcpr-wizard">
				<!-- Generic Progress Bar Template -->
				<script type="text/html" id="tmpl-ddwcpr-progress-bar">
					<div class="ddwcpr-progress-wrap">
						<div class="ddwcpr-progress-bar-bg">
							<div class="ddwcpr-progress-bar" style="width:{{ data.percent }}%" data-percent="{{ data.percent }}"></div>
						</div>
						<div class="ddwcpr-progress-label">
							<span>{{ data.message || 'Processing...' }}</span>
							<span class="ddwcpr-progress-text">{{ data.processed }}/{{ data.total }}</span>
						</div>
					</div>
				</script>

				<!-- Error State Template -->
				<script type="text/html" id="tmpl-ddwcpr-wizard-error">
					<div class="ddwcpr-wizard-error">
						<span class="ddwcpr-wizard-error-icon">⚠</span>
						<h3><?php esc_html_e( 'Processing Error', 'loyaltyx-points-and-rewards-for-woocommerce' ); ?></h3>
						<p>{{ data.message }}</p>
						<button class="button button-secondary ddwcpr-wizard-retry"><?php esc_html_e( 'Try Again', 'loyaltyx-points-and-rewards-for-woocommerce' ); ?></button>
					</div>
				</script>

				<!-- Success Summary Template -->
				<script type="text/html" id="tmpl-ddwcpr-manual-adjustment-summary">
					<div class="ddwcpr-wizard-summary-card">
						<div class="ddwcpr-wizard-summary-header">
							<span class="ddwcpr-wizard-summary-success">
								<svg width="24" height="24" fill="none"><circle cx="12" cy="12" r="12" fill="#22C55E"/><path d="M7 13l3 3 7-7" stroke="#fff" stroke-width="2" stroke-linecap="round" stroke-linejoin="round"/></svg>
							</span>
							<span class="ddwcpr-wizard-summary-title"><?php esc_html_e( 'Manual Adjustment Complete', 'loyaltyx-points-and-rewards-for-woocommerce' ); ?></span>
						</div>
						<div class="ddwcpr-wizard-summary-details">
							<div class="ddwcpr-wizard-summary-stats">
								<div class="ddwcpr-wizard-summary-row">
									<span class="ddwcpr-wizard-summary-label ddwcpr-success">✔ <?php esc_html_e( 'Total users processed', 'loyaltyx-points-and-rewards-for-woocommerce' ); ?></span>
									<span class="ddwcpr-wizard-summary-value">{{ data.totalUsers }}</span>
								</div>
								<div class="ddwcpr-wizard-summary-row">
									<span class="ddwcpr-wizard-summary-label ddwcpr-success">✔ <?php esc_html_e( 'Successfully processed', 'loyaltyx-points-and-rewards-for-woocommerce' ); ?></span>
									<span class="ddwcpr-wizard-summary-value">{{ data.successCount }}</span>
								</div>
								<div class="ddwcpr-wizard-summary-row">
									<span class="ddwcpr-wizard-summary-label ddwcpr-warning">⚠ <?php esc_html_e( 'Failed to process', 'loyaltyx-points-and-rewards-for-woocommerce' ); ?></span>
									<span class="ddwcpr-wizard-summary-value">{{ data.errorCount }}</span>
								</div>
							</div>
							<# if ( data.failedResults && data.failedResults.length > 0 ) { #>
							<div class="ddwcpr-wizard-summary-details">
								<details class="ddwcpr-details">
									<summary class="ddwcpr-details-summary">
										<span class="ddwcpr-details-icon">📋</span>
										<?php esc_html_e( 'View Failed Process Details', 'loyaltyx-points-and-rewards-for-woocommerce' ); ?>
										<span class="ddwcpr-details-count">({{ data.failedResults.length }})</span>
									</summary>
									<div class="ddwcpr-details-content">
										<div class="ddwcpr-details-list">
											<# _.each( data.failedResults, function( result ) { #>
											<div class="ddwcpr-details-item">
												<div class="ddwcpr-details-item-header">
													<span class="ddwcpr-details-user">{{ result.display_name }} ({{ result.user_login }})</span>
													<span class="ddwcpr-details-status ddwcpr-error"><?php esc_html_e( 'Failed', 'loyaltyx-points-and-rewards-for-woocommerce' ); ?></span>
												</div>
												<div class="ddwcpr-details-item-content">
													<p class="ddwcpr-details-message">{{ result.message }}</p>
													<div class="ddwcpr-details-info">
														<span class="ddwcpr-details-label"><?php esc_html_e( 'Action:', 'loyaltyx-points-and-rewards-for-woocommerce' ); ?></span>
														<span class="ddwcpr-details-value">{{ result.action_type }}</span>
													</div>
													<div class="ddwcpr-details-info">
														<span class="ddwcpr-details-label"><?php esc_html_e( 'Points Requested:', 'loyaltyx-points-and-rewards-for-woocommerce' ); ?></span>
														<span class="ddwcpr-details-value">{{ result.points_requested }}</span>
													</div>
													<div class="ddwcpr-details-info">
														<span class="ddwcpr-details-label"><?php esc_html_e( 'Current Balance:', 'loyaltyx-points-and-rewards-for-woocommerce' ); ?></span>
														<span class="ddwcpr-details-value">{{ result.points_balance }}</span>
													</div>
												</div>
											</div>
											<# }); #>
										</div>
									</div>
								</details>
							</div>
							<# } #>
							<div class="ddwcpr-step-actions-final ddwcpr-step-actions">
								<button type="button" class="button button-secondary ddwcpr-start-again"><?php esc_html_e( 'Start Again', 'loyaltyx-points-and-rewards-for-woocommerce' ); ?></button>
								<a href="<?php echo esc_url( admin_url( 'admin.php?page=' . sanitize_text_field( wp_unslash( $_GET['page'] ) ) . '&menu=' . sanitize_text_field( wp_unslash( $_GET['menu'] ) ) ) ); ?>" class="button button-primary"><?php esc_html_e( 'Done', 'loyaltyx-points-and-rewards-for-woocommerce' ); ?></a>
							</div>
						</div>
					</div>
				</script>

				<div class="ddwcpr-steps">
					<div class="ddwcpr-step active" data-step="1">
						<span class="ddwcpr-step-index">1</span>
						<span class="ddwcpr-step-label"><?php esc_html_e( 'Select Users & Points', 'loyaltyx-points-and-rewards-for-woocommerce' ); ?></span>
					</div>
					<div class="ddwcpr-step" data-step="2">
						<span class="ddwcpr-step-index">2</span>
						<span class="ddwcpr-step-label"><?php esc_html_e( 'Process Adjustment', 'loyaltyx-points-and-rewards-for-woocommerce' ); ?></span>
					</div>
					<div class="ddwcpr-step" data-step="3">
						<span class="ddwcpr-step-index">3</span>
						<span class="ddwcpr-step-label"><?php esc_html_e( 'Complete', 'loyaltyx-points-and-rewards-for-woocommerce' ); ?></span>
					</div>
				</div>

				<form id="ddwcpr-manual-adjustment-form" method="post">
					<?php wp_nonce_field( 'ddwcpr_manual_adjustment_submit_nonce_action', 'ddwcpr_manual_adjustment_submit_nonce' ); ?>

					<div class="ddwcpr-step-content active" data-step="1">
						<div class="ddwcpr-step-header">
							<h2><?php esc_html_e( 'Manual Add or Deduct Points', 'loyaltyx-points-and-rewards-for-woocommerce' ); ?></h2>
							<p class="ddwcpr-step-description"><?php esc_html_e( 'You can add or deduct points for single or multiple users from here.', 'loyaltyx-points-and-rewards-for-woocommerce' ); ?></p>
						</div>

						<div class="ddwcpr-manual-adjustment-fields">
							<!-- User Selection Section -->
							<div class="ddwcpr-field-section">
								<h3><?php esc_html_e( 'User Selection', 'loyaltyx-points-and-rewards-for-woocommerce' ); ?></h3>

								<div class="ddwcpr-field-option">
									<label class="ddwcpr-field-checkbox">
										<input type="checkbox" id="ddwcpr-select-all-users" name="ddwcpr_select_all_users">
										<div class="ddwcpr-card-content">
											<span class="ddwcpr-card-label"><?php esc_html_e( 'Select All Users', 'loyaltyx-points-and-rewards-for-woocommerce' ); ?></span>
											<span class="ddwcpr-card-description"><?php esc_html_e( 'Apply points adjustment to all users in the system', 'loyaltyx-points-and-rewards-for-woocommerce' ); ?></span>
										</div>
										<div class="ddwcpr-card-icon">
											<?php
											DDFW_SVG::get_svg_icon(
												'checkmark-circle',
												false,
												[
													'size'         => 30,
													'stroke_width' => 1.5,
												]
											);
											?>
										</div>
									</label>
								</div>

								<div class="ddwcpr-divider">
									<span class="ddwcpr-divider-text"><?php esc_html_e( 'OR', 'loyaltyx-points-and-rewards-for-woocommerce' ); ?></span>
								</div>

								<div class="ddwcpr-field-group" id="ddwcpr-individual-users-section">
									<label for="ddwcpr-users" class="ddwcpr-field-label">
										<?php esc_html_e( 'Select Users', 'loyaltyx-points-and-rewards-for-woocommerce' ); ?>
										<abbr title="<?php esc_html_e( 'Required', 'loyaltyx-points-and-rewards-for-woocommerce' ); ?>" class="required">*</abbr>
									</label>
									<select name="_ddwcpr_users[]" id="ddwcpr-users" class="regular-text ddfw-users" multiple data-placeholder="<?php esc_attr_e( 'Search Users...', 'loyaltyx-points-and-rewards-for-woocommerce' ); ?>"></select>
									<p class="description"><?php esc_html_e( 'Select single or multiple users for the operation.', 'loyaltyx-points-and-rewards-for-woocommerce' ); ?></p>
								</div>
							</div>

							<!-- Points Configuration Section -->
							<div class="ddwcpr-field-section">
								<h3><?php esc_html_e( 'Points Configuration', 'loyaltyx-points-and-rewards-for-woocommerce' ); ?></h3>

								<div class="ddwcpr-points-config">
									<div class="ddwcpr-field-grid">
										<div class="ddwcpr-field-group">
											<label for="ddwcpr-points" class="ddwcpr-field-label">
												<?php esc_html_e( 'Points Amount', 'loyaltyx-points-and-rewards-for-woocommerce' ); ?>
												<abbr title="<?php esc_html_e( 'Required', 'loyaltyx-points-and-rewards-for-woocommerce' ); ?>" class="required">*</abbr>
											</label>
											<div class="ddwcpr-input-group">
												<input type="number"
													id="ddwcpr-points"
													name="_ddwcpr_points"
													class="ddwcpr-points"
													min="1"
													step="1"
													placeholder="0"
													required>
												<span class="ddwcpr-input-suffix">pts</span>
											</div>
											<p class="description"><?php esc_html_e( 'Enter the number of points to add or deduct.', 'loyaltyx-points-and-rewards-for-woocommerce' ); ?></p>
										</div>

										<div class="ddwcpr-field-group">
											<label for="ddwcpr-action-type" class="ddwcpr-field-label">
												<?php esc_html_e( 'Action Type', 'loyaltyx-points-and-rewards-for-woocommerce' ); ?>
												<abbr title="<?php esc_html_e( 'Required', 'loyaltyx-points-and-rewards-for-woocommerce' ); ?>" class="required">*</abbr>
											</label>
											<select id="ddwcpr-action-type" name="_ddwcpr_action_type" class="ddwcpr-action-type regular-text" required>
												<option value="add" selected><?php esc_html_e( 'Add Points', 'loyaltyx-points-and-rewards-for-woocommerce' ); ?></option>
												<option value="deduct"><?php esc_html_e( 'Deduct Points', 'loyaltyx-points-and-rewards-for-woocommerce' ); ?></option>
											</select>
											<p class="description"><?php esc_html_e( 'Choose whether to add or deduct points.', 'loyaltyx-points-and-rewards-for-woocommerce' ); ?></p>
										</div>
									</div>
								</div>

								<div class="ddwcpr-description-config">
									<div class="ddwcpr-field-group">
										<label for="ddwcpr-description" class="ddwcpr-field-label">
											<?php esc_html_e( 'Description', 'loyaltyx-points-and-rewards-for-woocommerce' ); ?>
											<span class="ddwcpr-optional-badge"><?php esc_html_e( 'Optional', 'loyaltyx-points-and-rewards-for-woocommerce' ); ?></span>
										</label>
										<textarea id="ddwcpr-description"
											name="_ddwcpr_description"
											class="ddwcpr-description regular-text ddfw-full-width"
											rows="3"
											placeholder="<?php esc_attr_e( 'e.g., Manual adjustment for customer service', 'loyaltyx-points-and-rewards-for-woocommerce' ); ?>"></textarea>
										<p class="description"><?php esc_html_e( 'Add a description that will be logged and visible to users.', 'loyaltyx-points-and-rewards-for-woocommerce' ); ?></p>
									</div>
								</div>
							</div>
						</div>

						<div class="ddwcpr-step-actions">
							<button type="submit" class="button button-primary" id="ddwcpr-start-adjustment">
								<?php esc_html_e( 'Start Adjustment', 'loyaltyx-points-and-rewards-for-woocommerce' ); ?>
							</button>
						</div>
					</div>

					<div class="ddwcpr-step-content ddfw-hide" data-step="2">
						<h2><?php esc_html_e( 'Processing Points Adjustment', 'loyaltyx-points-and-rewards-for-woocommerce' ); ?></h2>
						<p class="ddwcpr-step-description"><?php esc_html_e( 'Please wait while we process your points adjustment...', 'loyaltyx-points-and-rewards-for-woocommerce' ); ?></p>
						<div class="ddwcpr-wizard-summary-wrap">
							<div class="ddwcpr-wizard-summary">
								<!-- Progress bar and results will be rendered using templates -->
							</div>
						</div>
					</div>

					<div class="ddwcpr-step-content ddfw-hide" data-step="3">
						<h2><?php esc_html_e( 'Adjustment Complete', 'loyaltyx-points-and-rewards-for-woocommerce' ); ?></h2>
						<p class="ddwcpr-step-description"><?php esc_html_e( 'Your points adjustment has been processed successfully', 'loyaltyx-points-and-rewards-for-woocommerce' ); ?></p>
						<div class="ddwcpr-wizard-summary-wrap">
							<div class="ddwcpr-wizard-summary">
								<!-- Summary will be rendered using templates -->
							</div>
						</div>
					</div>
				</form>
			</div>
			<?php
		}
	}
}
