<?php
/**
 * Global Functions
 *
 * @package WooCommerce Points and Rewards
 * @version 1.0.0
 */

defined( 'ABSPATH' ) || exit();

use DevDiggers\Framework\Includes\DDFW_SVG;

if ( ! function_exists( 'ddwcpr_get_translation' ) ) {
	/**
	 * Return translations for static words function
	 *
	 * @param string $static_word
	 * @return string
	 */
	function ddwcpr_get_translation( $static_word ) {
		$translations = [
			'manual_adjustment'   => esc_html__( 'Manual Adjustment by Admin', 'loyaltyx-points-and-rewards-for-woocommerce' ),
			'order_completion'    => esc_html__( 'Order Completion', 'loyaltyx-points-and-rewards-for-woocommerce' ),
			'order_refunded'      => esc_html__( 'Order Refunded', 'loyaltyx-points-and-rewards-for-woocommerce' ),
			'order_cancelled'     => esc_html__( 'Order Cancelled', 'loyaltyx-points-and-rewards-for-woocommerce' ),
			'order_failed'        => esc_html__( 'Order Failed', 'loyaltyx-points-and-rewards-for-woocommerce' ),
			'points_expired'      => esc_html__( 'Points Expired', 'loyaltyx-points-and-rewards-for-woocommerce' ),
			'active'              => esc_html__( 'Active', 'loyaltyx-points-and-rewards-for-woocommerce' ),
			'inactive'            => esc_html__( 'Inactive', 'loyaltyx-points-and-rewards-for-woocommerce' ),
			'purchase_amount'     => esc_html__( 'Purchase Amount', 'loyaltyx-points-and-rewards-for-woocommerce' ),
			'product_purchase'    => esc_html__( 'Product Purchase', 'loyaltyx-points-and-rewards-for-woocommerce' ),
			'referral'            => esc_html__( 'Referral', 'loyaltyx-points-and-rewards-for-woocommerce' ),
			'signup'              => esc_html__( 'Signup', 'loyaltyx-points-and-rewards-for-woocommerce' ),
			'review_submission'   => esc_html__( 'Review Submission', 'loyaltyx-points-and-rewards-for-woocommerce' ),
			'daily_login'         => esc_html__( 'Daily Login', 'loyaltyx-points-and-rewards-for-woocommerce' ),
			'birthday'            => esc_html__( 'Birthday', 'loyaltyx-points-and-rewards-for-woocommerce' ),
			'level_up_points'     => esc_html__( 'Level-Up Points', 'loyaltyx-points-and-rewards-for-woocommerce' ),
			'level_up'            => esc_html__( 'Level-Up', 'loyaltyx-points-and-rewards-for-woocommerce' ),
			'points_conversion'   => esc_html__( 'Points Conversion', 'loyaltyx-points-and-rewards-for-woocommerce' ),
			'fixed_discount'      => esc_html__( 'Fixed Discount', 'loyaltyx-points-and-rewards-for-woocommerce' ),
			'percentage_discount' => esc_html__( 'Percentage Discount', 'loyaltyx-points-and-rewards-for-woocommerce' ),
			'buy_x_get_y'         => esc_html__( 'Buy X Get Y', 'loyaltyx-points-and-rewards-for-woocommerce' ),
			'bundle_reward'       => esc_html__( 'Bundle Reward', 'loyaltyx-points-and-rewards-for-woocommerce' ),
			'free_shipping'       => esc_html__( 'Free Shipping', 'loyaltyx-points-and-rewards-for-woocommerce' ),
			'free_product'        => esc_html__( 'Free Product', 'loyaltyx-points-and-rewards-for-woocommerce' ),
			'expired'             => esc_html__( 'Expired', 'loyaltyx-points-and-rewards-for-woocommerce' ),
			'redeemed'            => esc_html__( 'Redeemed', 'loyaltyx-points-and-rewards-for-woocommerce' ),
			'import'              => esc_html__( 'Imported via CSV', 'loyaltyx-points-and-rewards-for-woocommerce' ),
		];

		/**
		 * Filter to modify translation
		 *
		 * @since 1.0.0
		 */
		$translations       = apply_filters( 'ddwcpr_modify_translations', $translations );
		$result_translation = ! empty( $translations[ $static_word ] ) ? $translations[ $static_word ] : $static_word;

		/**
		 * Filter to modify result translation
		 *
		 * @since 1.0.0
		 */
		return apply_filters( 'ddwcpr_modify_result_translation', $result_translation );
	}
}

if ( ! function_exists( 'ddwcpr_get_copy_text_field' ) ) {
	/**
	 * Get copy text field function
	 *
	 * @param string $text
	 * @param int|null $icon_size
	 * @return string
	 */
	function ddwcpr_get_copy_text_field( $text, $icon_size = 20 ) {
		?>
		<div class="ddwcpr-copy-field-container">
			<input type="text" class="ddwcpr-copy-target form-control input-text" value="<?php echo esc_attr( $text ); ?>" data-copy-text="<?php echo esc_attr( $text ); ?>" readonly />
			<a href="#" class="ddwcpr-copy-trigger" data-tooltip="<?php esc_attr_e( 'Copy', 'loyaltyx-points-and-rewards-for-woocommerce' ); ?>">
				<?php
				DDFW_SVG::get_svg_icon(
					'copy',
					false,
					[
						'size'         => $icon_size,
						'stroke_width' => 1,
					]
				);
				?>
				<span class="ddwcpr-copy-tooltip"></span>
			</a>
		</div>
		<?php
	}
}

if ( ! function_exists( 'ddwcpr_get_earning_rule_types' ) ) {
	/**
	 * Get earning rule types function
	 *
	 * @return array
	 */
	function ddwcpr_get_earning_rule_types() {
		return [
			'purchase_amount'   => [
				'heading'     => esc_html__( 'Purchase Amount', 'loyaltyx-points-and-rewards-for-woocommerce' ),
				'description' => esc_html__( 'Points earned based on the order total.', 'loyaltyx-points-and-rewards-for-woocommerce' ),
				'pro_version' => false,
			],
			'product_purchase'  => [
				'heading'     => esc_html__( 'Product Purchase', 'loyaltyx-points-and-rewards-for-woocommerce' ),
				'description' => esc_html__( 'Points earned for purchasing specific products.', 'loyaltyx-points-and-rewards-for-woocommerce' ),
				'pro_version' => true,
			],
			'referral'          => [
				'heading'     => esc_html__( 'Referral', 'loyaltyx-points-and-rewards-for-woocommerce' ),
				'description' => esc_html__( 'Points awarded for referring others.', 'loyaltyx-points-and-rewards-for-woocommerce' ),
				'pro_version' => true,
			],
			'signup'            => [
				'heading'     => esc_html__( 'Signup', 'loyaltyx-points-and-rewards-for-woocommerce' ),
				'description' => esc_html__( 'Points for creating a new account.', 'loyaltyx-points-and-rewards-for-woocommerce' ),
				'pro_version' => true,
			],
			'review_submission' => [
				'heading'     => esc_html__( 'Review Submission', 'loyaltyx-points-and-rewards-for-woocommerce' ),
				'description' => esc_html__( 'Points for submitting a product review.', 'loyaltyx-points-and-rewards-for-woocommerce' ),
				'pro_version' => true,
			],
			'daily_login'       => [
				'heading'     => esc_html__( 'Daily Login', 'loyaltyx-points-and-rewards-for-woocommerce' ),
				'description' => esc_html__( 'Earn by logging in every day and claiming points.', 'loyaltyx-points-and-rewards-for-woocommerce' ),
				'pro_version' => true,
			],
			'birthday'          => [
				'heading'     => esc_html__( 'Birthday', 'loyaltyx-points-and-rewards-for-woocommerce' ),
				'description' => esc_html__( "Points awarded on customer's birthday.", 'loyaltyx-points-and-rewards-for-woocommerce' ),
				'pro_version' => true,
			],
			'level_up_points'   => [
				'heading'     => esc_html__( 'Level-Up Points', 'loyaltyx-points-and-rewards-for-woocommerce' ),
				'description' => esc_html__( 'Additional points awarded when a customer reaches a new loyalty level.', 'loyaltyx-points-and-rewards-for-woocommerce' ),
				'pro_version' => true,
			],
		];
	}
}

if ( ! function_exists( 'ddwcpr_get_redeeming_rule_types' ) ) {
	/**
	 * Get redeeming rule types function
	 *
	 * @return array
	 */
	function ddwcpr_get_redeeming_rule_types() {
		return [
			'points_conversion' => [
				'heading'     => esc_html__( 'Points Conversion', 'loyaltyx-points-and-rewards-for-woocommerce' ),
				'description' => esc_html__( 'Allow customers to convert any number of points into coupon value based on the set ratio.', 'loyaltyx-points-and-rewards-for-woocommerce' ),
				'pro_version' => false,
			],
			'fixed_discount'    => [
				'heading'     => esc_html__( 'Fixed Discount', 'loyaltyx-points-and-rewards-for-woocommerce' ),
				/* translators: %s: coupon value */
				'description' => sprintf( esc_html__( 'Preset rewards like 500 points = %s coupon.', 'loyaltyx-points-and-rewards-for-woocommerce' ), wc_price( 50 ) ),
				'pro_version' => true,
			],
			'percentage_discount'  => [
				'heading'     => esc_html__( 'Percentage Discount', 'loyaltyx-points-and-rewards-for-woocommerce' ),
				'description' => esc_html__( 'Redeem points for a percentage discount on the cart total.', 'loyaltyx-points-and-rewards-for-woocommerce' ),
				'pro_version' => true,
			],
			'buy_x_get_y' => [
				'heading'     => esc_html__( 'Buy X Get Y', 'loyaltyx-points-and-rewards-for-woocommerce' ),
				'description' => esc_html__( 'Redeem points to buy specific product(s) and get other product(s) for free or discounted.', 'loyaltyx-points-and-rewards-for-woocommerce' ),
				'pro_version' => true,
			],
			'bundle_reward' => [
				'heading'     => esc_html__( 'Bundle Reward', 'loyaltyx-points-and-rewards-for-woocommerce' ),
				'description' => esc_html__( 'Redeem points to get product bundles at a fixed points cost.', 'loyaltyx-points-and-rewards-for-woocommerce' ),
				'pro_version' => true,
			],
			'free_shipping' => [
				'heading'     => esc_html__( 'Free Shipping', 'loyaltyx-points-and-rewards-for-woocommerce' ),
				'description' => esc_html__( 'Redeem points to get free shipping on the order.', 'loyaltyx-points-and-rewards-for-woocommerce' ),
				'pro_version' => true,
			],
			'free_product' => [
				'heading'     => esc_html__( 'Free Product', 'loyaltyx-points-and-rewards-for-woocommerce' ),
				'description' => esc_html__( 'Redeem points to receive specific product for free.', 'loyaltyx-points-and-rewards-for-woocommerce' ),
				'pro_version' => true,
			],
		];
	}
}

if ( ! function_exists( 'ddwcpr_get_points_label' ) ) {
	/**
	 * Get points label function
	 *
	 * @param int $points
	 * @return string
	 */
	function ddwcpr_get_points_label( $points ) {
		global $ddwcpr_configuration;
		return $points > 0 ? $ddwcpr_configuration['points_plural_label'] : $ddwcpr_configuration['points_singlular_label'];
	}
}

if ( ! function_exists( 'ddwcpr_get_rules_icon' ) ) {
	/**
	 * Get rules icon function
	 *
	 * @param string $rule_type
	 * @param boolean $return_array
	 * @return string|array
	 */
	function ddwcpr_get_rules_icon( $rule_type = '', $return_array = false ) {
		$rule_icons = [
			'purchase_amount'     => DDWCPR_PLUGIN_URL . 'assets/images/rules/purchase-amount.svg',
			'product_purchase'    => DDWCPR_PLUGIN_URL . 'assets/images/rules/product-purchase.svg',
			'referral'            => DDWCPR_PLUGIN_URL . 'assets/images/rules/referrals.svg',
			'signup'              => DDWCPR_PLUGIN_URL . 'assets/images/rules/signup.svg',
			'review_submission'   => DDWCPR_PLUGIN_URL . 'assets/images/rules/review-submission.svg',
			'daily_login'         => DDWCPR_PLUGIN_URL . 'assets/images/rules/daily-login.svg',
			'birthday'            => DDWCPR_PLUGIN_URL . 'assets/images/rules/birthday.svg',
			'level_up_points'     => DDWCPR_PLUGIN_URL . 'assets/images/rules/level-up.svg',
			'points_conversion'   => DDWCPR_PLUGIN_URL . 'assets/images/rules/points-conversion.svg',
			'fixed_discount'      => DDWCPR_PLUGIN_URL . 'assets/images/rules/fixed-discount.svg',
			'percentage_discount' => DDWCPR_PLUGIN_URL . 'assets/images/rules/percentage-discount.svg',
			'buy_x_get_y'         => DDWCPR_PLUGIN_URL . 'assets/images/rules/buy-x-get-y.svg',
			'bundle_reward'       => DDWCPR_PLUGIN_URL . 'assets/images/rules/bundle-reward.svg',
			'free_shipping'       => DDWCPR_PLUGIN_URL . 'assets/images/rules/free-shipping.svg',
			'free_product'        => DDWCPR_PLUGIN_URL . 'assets/images/rules/free-product.svg',
		];

		if ( $return_array ) {
			return $rule_icons;
		}

		return ! empty( $rule_icons[ $rule_type ] ) ? $rule_icons[ $rule_type ] : '';
	}
}
