<?php
/**
 * This file handles all front end action ajax callbacks.
 *
 * @version 1.0.0
 * @package WooCommerce Points and Rewards
 */

namespace DDWCPointsAndRewards\Includes\Front;

use DDWCPointsAndRewards\Helper\Rule\DDWCPR_Rule_Helper;
use DDWCPointsAndRewards\Helper\Point\DDWCPR_Point_Helper;
use DDWCPointsAndRewards\Helper\Log\DDWCPR_Log_Helper;
use DDWCPointsAndRewards\Helper\Reward\DDWCPR_Reward_Helper;

defined( 'ABSPATH' ) || exit();

if ( ! class_exists( 'DDWCPR_Front_Ajax_Functions' ) ) {    
	/**
	 * Front ajax functions class
	 */
	class DDWCPR_Front_Ajax_Functions {
		/**
		 * Configuration Variable
		 *
		 * @var array
		 */
		protected $ddwcpr_configuration;

		/**
		 * Construct
		 * 
		 * @param array $ddwcpr_configuration
		 */
		public function __construct( $ddwcpr_configuration ) {
			$this->ddwcpr_configuration = $ddwcpr_configuration;
		}

		/**
		 * Get table rows function
		 *
		 * @return void
		 */
		public function ddwcpr_get_table_rows() {
			$response = [];
			if ( check_ajax_referer( 'ddwcpr-nonce', 'nonce', false ) ) {
				if ( ! empty( $_POST['table'] ) && ! empty( $_POST['perform'] ) && ! empty( $_POST['current_page'] ) ) {
					$table        = sanitize_text_field( wp_unslash( $_POST['table'] ) );
					$perform      = sanitize_text_field( wp_unslash( $_POST['perform'] ) );
					$current_page = sanitize_text_field( wp_unslash( $_POST['current_page'] ) );
					$user_id      = get_current_user_id();
					$per_page     = 10;
					$offset       = 1 === $current_page ? 0 : ( $current_page - 1 ) * $per_page;
					$html         = '';

					if ( 'logs' === $table ) {
						$log_helper = new DDWCPR_Log_Helper( $this->ddwcpr_configuration );
						$args       = [ 'user_id' => $user_id, 'per_page' => $per_page, 'offset' => $offset ];
						$logs       = $log_helper->ddwcpr_get_logs( $args );
						ob_start();
						$log_helper->ddwcpr_render_logs_table_rows( $logs );
						$html = ob_get_clean();
					} elseif ( 'rewards' === $table ) {
						$reward_helper = new DDWCPR_Reward_Helper( $this->ddwcpr_configuration );
						$args          = [ 'user_id' => $user_id, 'per_page' => $per_page, 'offset' => $offset ];
						$rewards       = $reward_helper->ddwcpr_get_coupon_rewards( $args );
						ob_start();
						$reward_helper->ddwcpr_render_rewards_table_rows( $rewards );
						$html = ob_get_clean();
					}

					$response = [
						'success' => true,
						'message' => esc_html__( 'Table rows fetched!', 'loyaltyx-points-and-rewards-for-woocommerce' ),
						'html'    => $html,
					];
				} else {
					$response = [
						'success' => false,
						'message' => esc_html__( 'Arguments are missing.', 'loyaltyx-points-and-rewards-for-woocommerce' ),
					];
				}
			} else {
				$response = [
					'success' => false,
					'message' => esc_html__( 'Security check failed!', 'loyaltyx-points-and-rewards-for-woocommerce' ),
				];
			}
			wp_send_json( $response );
		}

		/**
		 * Get variation points message function
		 *
		 * @return void
		 */
		public function ddwcpr_get_variation_points_message() {
			$response = [];
			if ( check_ajax_referer( 'ddwcpr-nonce', 'nonce', false ) ) {
				if ( ! empty( $_POST['variation_id'] ) ) {
					$variation_id = intval( wp_unslash( $_POST['variation_id'] ) );                                // wpcs: input var okay.
					$rule_helper  = new DDWCPR_Rule_Helper( $this->ddwcpr_configuration );
					$points       = $rule_helper->ddwcpr_calculate_earning_points_by_product_id( $variation_id );
					$message      = $this->ddwcpr_configuration['points_message']['product_page']['message'];

					$points_message = '';

					if ( $points > 0 ) {
						ob_start();
						$rule_helper->ddwcpr_get_display_points_earning_message( $points, $message, 'product' );
						$points_message = ob_get_clean();
					}

					$response = [
						'success' => true,
						'message' => esc_html__( 'Points earning message fetched!', 'loyaltyx-points-and-rewards-for-woocommerce' ),
						'html'    => $points_message,
					];
				} else {
					$response = [
						'success' => false,
						'message' => esc_html__( 'Variation ID not exists!', 'loyaltyx-points-and-rewards-for-woocommerce' ),
					];
				}
			} else {
				$response = [
					'success' => false,
					'message' => esc_html__( 'Security check failed!', 'loyaltyx-points-and-rewards-for-woocommerce' ),
				];
			}
			wp_send_json( $response );
		}
	}
}
