<?php
/**
 * Thie file handles all admin end action hooks.
 *
 * @package WooCommerce Points and Rewards
 * @version 1.0.0
 */

namespace DDWCPointsAndRewards\Includes;

use DevDiggers\Framework\Includes\DDFW_Plugin_Dashboard;
use DevDiggers\Framework\Includes\DDFW_Assets;
use DevDiggers\Framework\Includes\DDFW_SVG;
use DDWCPointsAndRewards\Templates\Admin;
use DDWCPointsAndRewards\Helper\Point\DDWCPR_Point_Helper;

defined( 'ABSPATH' ) || exit();

if ( ! class_exists( 'DDWCPR_Admin_Dashboard' ) ) {
	/**
	 * Admin Dashboard Class
	 */
	class DDWCPR_Admin_Dashboard {
		/**
		 * Configuration Variable
		 *
		 * @var array
		 */
		protected $ddwcpr_configuration;

		/**
		 * Dashboard Variable
		 *
		 * @var DDFW_Plugin_Dashboard
		 */
		protected $dashboard;

		/**
		 * Construct
		 * 
		 * @param array $ddwcpr_configuration
		 */
		public function __construct( $ddwcpr_configuration ) {
			$this->ddwcpr_configuration = $ddwcpr_configuration;
			$this->ddwcpr_add_dashboard_menu();
			add_action( 'admin_enqueue_scripts', [ $this, 'ddwcpr_enqueue_admin_scripts' ] );
		}

		/**
		 * Add Admin menu function
		 *
		 * @return void
		 */
		public function ddwcpr_add_dashboard_menu() {
			ob_start();
			?>

			<!-- Icon option: Gift box with star (corner) -->
			<svg width="32" height="32" viewBox="0 0 30 30" fill="none" aria-hidden="true" focusable="false" xmlns="http://www.w3.org/2000/svg">
				<circle cx="15" cy="15" r="14" fill="#EEF3FF"/>
				<rect x="7" y="12" width="16" height="11" rx="2" fill="#EEF3FF" stroke="#0256ff" stroke-width="1.5"/>
				<rect x="7" y="9" width="16" height="4" rx="1" fill="#EEF3FF" stroke="#0256ff" stroke-width="1"/>
				<rect x="14" y="9" width="2" height="14" fill="#0256ff"/>
				<path d="M12 9C10 7 10 6 11 5C12 4 13.5 5 15 7C16.5 5 18 4 19 5C20 6 20 7 18 9H12Z" fill="#0256ff"/>
				<!-- small star on top-right corner of the gift -->
				<path d="M22 8.8L22.8 10.5L24.5 10.7L23.1 11.9L23.5 13.6L22 12.8L20.5 13.6L20.9 11.9L19.5 10.7L21.2 10.5L22 8.8Z" fill="#ffffff" stroke="#0256ff" stroke-width="2"/>
			</svg>
			<?php esc_html_e( 'LoyaltyX', 'loyaltyx-points-and-rewards-for-woocommerce' ); ?>
			<?php
			$plugin_name = ob_get_clean();

			$args = [
				'page_title'              => esc_html__( 'LoyaltyX', 'loyaltyx-points-and-rewards-for-woocommerce' ),
				'menu_title'              => esc_html__( 'LoyaltyX', 'loyaltyx-points-and-rewards-for-woocommerce' ),
				'slug'                    => 'ddwcpr-dashboard',
				'plugin_name'             => $plugin_name,
				'screen_options_callback' => [ $this, 'add_screen_options' ],
				'upgrade_url'             => '//devdiggers.com/plugin/woocommerce-points-and-rewards/?utm_source=plugin_header&utm_medium=upgrade_button&utm_campaign=header_upgrade',
				'menus'                   => [
					'dashboard'         => [
						'label'    => esc_html__( 'Dashboard', 'loyaltyx-points-and-rewards-for-woocommerce' ),
						'callback' => [ $this, 'ddwcpr_get_dashboard_template' ],
						'layout'   => 'full-width',
					],
					'users'             => [
						'label'    => esc_html__( 'Users', 'loyaltyx-points-and-rewards-for-woocommerce' ),
						'callback' => [ $this, 'ddwcpr_get_users_template' ],
						'layout'   => 'full-width',
					],
					'logs'              => [
						'label'    => esc_html__( 'Logs', 'loyaltyx-points-and-rewards-for-woocommerce' ),
						'callback' => [ $this, 'ddwcpr_get_logs_template' ],
						'layout'   => 'simple',
					],
					'expiring-points'   => [
						'label'    => esc_html__( 'Expiring Points', 'loyaltyx-points-and-rewards-for-woocommerce' ),
						'callback' => [ $this, 'ddwcpr_get_expiring_points_template' ],
						'layout'   => 'simple',
					],
					'points-rules'      => [
						'label'    => esc_html__( 'Points Rules', 'loyaltyx-points-and-rewards-for-woocommerce' ),
						'callback' => [ $this, 'ddwcpr_get_points_rules_template' ],
						'layout'   => 'sidebar',
						'tabs'     => [
							'points-earning'    => [
								'label'    => esc_html__( 'Points Earning Rules', 'loyaltyx-points-and-rewards-for-woocommerce' ),
								'icon' => DDFW_SVG::get_svg_icon(
									'points-earning',
									true,
									[ 'size' => 18 ]
								),
								'callback' => [ $this, 'ddwcpr_get_points_earning_rules_template' ],
							],
							'points-redeeming'  => [
								'label'    => esc_html__( 'Points Redeeming Rules', 'loyaltyx-points-and-rewards-for-woocommerce' ),
								'icon' => DDFW_SVG::get_svg_icon(
									'points-redeeming',
									true,
									[ 'size' => 18 ]
								),
								'callback' => [ $this, 'ddwcpr_get_points_redeeming_rules_template' ],
							],
						],
					],
					'levels-and-badges' => [
						'label'    => esc_html__( 'Levels & Badges', 'loyaltyx-points-and-rewards-for-woocommerce' ),
						'callback' => [ $this, 'ddwcpr_get_levels_and_badges_template' ],
						'layout'   => 'simple',
					],
					'configuration'     => [
						'label'    => esc_html__( 'Configuration', 'loyaltyx-points-and-rewards-for-woocommerce' ),
						'layout'   => 'sidebar',
						'tabs'     => [
							'general'           => [
								'label'    => esc_html__( 'General', 'loyaltyx-points-and-rewards-for-woocommerce' ),
								'icon'     => DDFW_SVG::get_svg_icon(
									'general',
									true,
									[ 'size' => 18 ]
								),
								'callback' => [ $this, 'ddwcpr_get_general_configuration_template' ],
							],
							'display-messages'  => [
								'label'    => esc_html__( 'Display Messages', 'loyaltyx-points-and-rewards-for-woocommerce' ),
								'icon'     => DDFW_SVG::get_svg_icon(
									'display-messages',
									true,
									[ 'size' => 18 ]
								),
								'callback' => [ $this, 'ddwcpr_get_display_messages_configuration_template' ],
							],
							'points-earning'    => [
								'label'    => esc_html__( 'Points Earning', 'loyaltyx-points-and-rewards-for-woocommerce' ),
								'icon'     => DDFW_SVG::get_svg_icon(
									'points-earning',
									true,
									[ 'size' => 18 ]
								),
								'callback' => [ $this, 'ddwcpr_get_points_earning_configuration_template' ],
							],
							'points-redeeming'  => [
								'label'    => esc_html__( 'Points Redeeming', 'loyaltyx-points-and-rewards-for-woocommerce' ),
								'icon'     => DDFW_SVG::get_svg_icon(
									'points-redeeming',
									true,
									[ 'size' => 18 ]
								),
								'callback' => [ $this, 'ddwcpr_get_points_redeeming_configuration_template' ],
							],
							'points-expiration' => [
								'label'    => esc_html__( 'Points Expiration', 'loyaltyx-points-and-rewards-for-woocommerce' ),
								'icon'     => DDFW_SVG::get_svg_icon(
									'points-expiration',
									true,
									[ 'size' => 18 ]
								),
								'callback' => [ $this, 'ddwcpr_get_points_expiration_configuration_template' ],
							],
							'layout'            => [
								'label'    => esc_html__( 'Layout', 'loyaltyx-points-and-rewards-for-woocommerce' ),
								'icon'     => DDFW_SVG::get_svg_icon(
									'layout',
									true,
									[ 'size' => 18 ]
								),
								'callback' => [ $this, 'ddwcpr_get_layout_configuration_template' ],
							],
							'emails'            => [
								'label'    => esc_html__( 'Emails', 'loyaltyx-points-and-rewards-for-woocommerce' ),
								'icon'     => DDFW_SVG::get_svg_icon(
									'emails',
									true,
									[ 'size' => 18 ]
								),
								'callback' => [ $this, 'ddwcpr_get_emails_configuration_template' ],
							],
							'shortcodes'        => [
								'label'    => esc_html__( 'Shortcodes', 'loyaltyx-points-and-rewards-for-woocommerce' ),
								'icon'     => DDFW_SVG::get_svg_icon(
									'shortcodes',
									true,
									[ 'size' => 18 ]
								),
								'callback' => [ $this, 'ddwcpr_get_shortcodes_configuration_template' ],
							],
						],
					],
				],
			];

			$this->dashboard = new DDFW_Plugin_Dashboard( $args );
		}

		/**
		 * Add screen options for the admin dashboard
		 *
		 * @return void
		 */
		public function add_screen_options() {
			global $myListTable;

			$current_menu = ! empty( $_GET['menu'] ) ? sanitize_title( wp_unslash( $_GET['menu'] ) ) : 'dashboard';  // Default to the first menu if none is set.

			$args = [
				'label'    => esc_html__( 'Results Per Page', 'loyaltyx-points-and-rewards-for-woocommerce' ),
				'default'  => 20,
				'hidden'   => 'id',
				'sanitize' => 'intval',
			];

			switch ( $current_menu ) {
				case 'users':
					$args['option'] = 'ddwcpr_users_per_page';
					$myListTable      = new Admin\Users\DDWCPR_Users_List_Template( $this->ddwcpr_configuration );
					break;
				case 'logs':
					$args['option'] = 'ddwcpr_logs_per_page';
					$myListTable      = new Admin\Logs\DDWCPR_Logs_List_Template( $this->ddwcpr_configuration );
					break;
				case 'points-rules':
					$args['option'] = 'ddwcpr_points_rules_per_page';
					$myListTable      = new Admin\Points_Rules\Points_Earning\DDWCPR_Points_Earning_Rules_List_Template( $this->ddwcpr_configuration );
					break;
			}

			add_screen_option( 'per_page', $args );
		}

		/**
		 * Dashboard Template Submenu
		 *
		 * @return void
		 */
		public function ddwcpr_get_dashboard_template() {
			new Admin\Dashboard\DDWCPR_Dashboard_Template( $this->ddwcpr_configuration );
		}

		/**
		 * Users Template Submenu
		 *
		 * @return void
		 */
		public function ddwcpr_get_users_template() {
			$current_tab = ! empty( $_GET['tab'] ) ? sanitize_text_field( wp_unslash( $_GET['tab'] ) ) : '';
			if ( 'import' === $current_tab ) {
				$import_wizard = new DDWCPR_Import_Wizard();
				$import_wizard->render();
				return;
			} elseif ( 'manual-adjustment' === $current_tab ) {
				new Admin\Manual_Adjustment\DDWCPR_Manual_Adjustment_Template( $this->ddwcpr_configuration );
			} elseif ( ! empty( $_GET['action'] ) && 'view' === sanitize_text_field( wp_unslash( $_GET['action'] ) ) ) {
				ddfw_upgrade_to_pro_section(
					[
						'image_url'     => DDWCPR_PLUGIN_URL . 'assets/images/pro-pages/view-user-page.webp',
						'heading'       => esc_html__( 'Advanced User Insights', 'loyaltyx-points-and-rewards-for-woocommerce' ),
						'description'   => esc_html__( 'Get deep insights into individual user behavior, points performance, and revenue impact with detailed analytics and visualizations.', 'loyaltyx-points-and-rewards-for-woocommerce' ),
						'list_features' => [
							esc_html__( 'Detailed User Profile with Activity Timeline', 'loyaltyx-points-and-rewards-for-woocommerce' ),
							esc_html__( 'Comprehensive Points Summary Cards (Earned, Available, Redeemed, Expired)', 'loyaltyx-points-and-rewards-for-woocommerce' ),
							esc_html__( 'User Level & Rank Tracking with Progress Indicators', 'loyaltyx-points-and-rewards-for-woocommerce' ),
							esc_html__( 'Revenue Impact Analysis Charts', 'loyaltyx-points-and-rewards-for-woocommerce' ),
							esc_html__( 'Points Performance Timeline Visualization', 'loyaltyx-points-and-rewards-for-woocommerce' ),
							esc_html__( 'Earning Sources Breakdown by Activity Type', 'loyaltyx-points-and-rewards-for-woocommerce' ),
							esc_html__( 'Recent Activities and Transactions History', 'loyaltyx-points-and-rewards-for-woocommerce' ),
							esc_html__( 'Points Expiration Alerts and Tracking', 'loyaltyx-points-and-rewards-for-woocommerce' ),
							esc_html__( 'Quick Actions for User Management', 'loyaltyx-points-and-rewards-for-woocommerce' ),
							esc_html__( 'Date Range Filtering for Custom Analysis Periods', 'loyaltyx-points-and-rewards-for-woocommerce' ),
						],
						'upgrade_url' => '//devdiggers.com/plugin/woocommerce-points-and-rewards/?utm_source=plugin_dashboard&utm_medium=upgrade_notice&utm_campaign=view_users_pro_feature',
					]
				);
			} else {
				$obj = new Admin\Users\DDWCPR_Users_List_Template( $this->ddwcpr_configuration );

				$page  = isset( $_GET['page'] ) ? sanitize_text_field( wp_unslash( $_GET['page'] ) ) : '';
				$menu  = isset( $_GET['menu'] ) ? sanitize_text_field( wp_unslash( $_GET['menu'] ) ) : '';
				$paged = isset( $_GET['paged'] ) ? sanitize_text_field( wp_unslash( $_GET['paged'] ) ) : '';
				?>
				<form method="get">
					<hr class="wp-header-end" />
					<h1 class="wp-heading-inline"><?php esc_html_e( 'Users', 'loyaltyx-points-and-rewards-for-woocommerce' ); ?></h1>
					<a href="<?php echo esc_url( admin_url( 'user-new.php' ) ); ?>" class="page-title-action button-primary button">
						<?php
						DDFW_SVG::get_svg_icon(
							'plus',
							false,
							[ 'size' => 15 ]
						);
						esc_html_e( 'Add New', 'loyaltyx-points-and-rewards-for-woocommerce' );
						?>
					</a>
					<a href="<?php echo esc_url( admin_url( "admin.php?page={$page}&menu={$menu}&tab=manual-adjustment" ) ); ?>" class="page-title-action button"><?php esc_html_e( 'Manual Adjustment', 'loyaltyx-points-and-rewards-for-woocommerce' ); ?></a>
					<a href="<?php echo esc_url( admin_url( "admin.php?page={$page}&menu={$menu}&tab=import" ) ); ?>" class="page-title-action button"><?php esc_html_e( 'Import', 'loyaltyx-points-and-rewards-for-woocommerce' ); ?></a>
					<button class="page-title-action button ddfw-upgrade-to-pro-tag-wrapper" type="button">
						<?php
						esc_html_e( 'Export', 'loyaltyx-points-and-rewards-for-woocommerce' );
						ddfw_pro_tag();
						?>
					</button>
					<input type="hidden" name="page" value="<?php echo esc_attr( $page ); // WPCS: CSRF ok. // WPCS: input var ok. ?>" />
					<input type="hidden" name="menu" value="<?php echo esc_attr( $menu ); // WPCS: CSRF ok. // WPCS: input var ok. ?>" />
					<input type="hidden" name="paged" value="<?php echo esc_attr( $paged ); // WPCS: CSRF ok. // WPCS: input var ok. ?>" />
					<?php
					wp_nonce_field( 'ddwcpr_nonce_action', 'ddwcpr_nonce' );
					$obj->prepare_items();
					$obj->search_box( esc_html__( 'Search', 'loyaltyx-points-and-rewards-for-woocommerce' ), 'search-id' );
					$obj->display();
					?>
				</form>
				<?php
			}
		}

		/**
		 * Logs template Submenu
		 *
		 * @return void
		 */
		public function ddwcpr_get_logs_template() {
			$obj = new Admin\Logs\DDWCPR_Logs_List_Template( $this->ddwcpr_configuration );
			?>
			<form method="get">
				<hr class="wp-header-end" />
				<h1 class="wp-heading-inline"><?php esc_html_e( 'Logs', 'loyaltyx-points-and-rewards-for-woocommerce' ); ?></h1>

				<input type="hidden" name="page" value="<?php echo isset( $_GET['page'] ) ? esc_attr( sanitize_text_field( wp_unslash( $_GET['page'] ) ) ) : ''; // WPCS: CSRF ok. // WPCS: input var ok. ?>" />
				<input type="hidden" name="menu" value="<?php echo isset( $_GET['menu'] ) ? esc_attr( sanitize_text_field( wp_unslash( $_GET['menu'] ) ) ) : ''; // WPCS: CSRF ok. // WPCS: input var ok. ?>" />
				<input type="hidden" name="paged" value="<?php echo isset( $_GET['paged'] ) ? esc_attr( sanitize_text_field( wp_unslash( $_GET['paged'] ) ) ) : ''; // WPCS: CSRF ok. // WPCS: input var ok. ?>" />
				<?php
				wp_nonce_field( 'ddwcpr_nonce_action', 'ddwcpr_nonce' );
				$obj->prepare_items();
				$obj->search_box( esc_html__( 'Search', 'loyaltyx-points-and-rewards-for-woocommerce' ), 'search-id' );
				$obj->display();
				?>
			</form>
			<?php
		}

		/**
		 * Expiring Points template Submenu
		 *
		 * @return void
		 */
		public function ddwcpr_get_expiring_points_template() {
			ddfw_upgrade_to_pro_section(
				[
					'image_url'     => DDWCPR_PLUGIN_URL . 'assets/images/pro-pages/expiring-points.webp',
					'heading'       => esc_html__( 'Advanced Points Expiration Management', 'loyaltyx-points-and-rewards-for-woocommerce' ),
					'description'   => esc_html__( 'Proactively manage expiring points to prevent revenue loss and encourage timely redemptions.', 'loyaltyx-points-and-rewards-for-woocommerce' ),
					'list_features' => [
						esc_html__( 'Comprehensive Expiring Points Dashboard with Real-time Tracking', 'loyaltyx-points-and-rewards-for-woocommerce' ),
						esc_html__( 'Advanced Filtering by Status, Date Range, and Specific Users', 'loyaltyx-points-and-rewards-for-woocommerce' ),
						esc_html__( 'Points Expiration Alerts and Notifications System', 'loyaltyx-points-and-rewards-for-woocommerce' ),
						esc_html__( 'Detailed Points History with Earned vs Used Breakdown', 'loyaltyx-points-and-rewards-for-woocommerce' ),
						esc_html__( 'Automated Expiration Reminders via Email', 'loyaltyx-points-and-rewards-for-woocommerce' ),
					],
					'upgrade_url' => '//devdiggers.com/plugin/woocommerce-points-and-rewards/?utm_source=plugin_dashboard&utm_medium=upgrade_notice&utm_campaign=expiring_points_pro_feature',
				]
			);
		}

		/**
		 * Get points rules template function
		 *
		 * @return void
		 */
		public function ddwcpr_get_points_rules_template() {
			new Admin\Points_Rules\DDWCPR_Points_Rules_Template( $this->ddwcpr_configuration );
		}

		/**
		 * Get levels and badges template function
		 *
		 * @return void
		 */
		public function ddwcpr_get_levels_and_badges_template() {
			ddfw_upgrade_to_pro_section(
				[
					'image_url'     => DDWCPR_PLUGIN_URL . 'assets/images/pro-pages/levels-and-badges.webp',
					'heading'       => esc_html__( 'Advanced Gamification', 'loyaltyx-points-and-rewards-for-woocommerce' ),
					'description'   => esc_html__( 'Create engaging user levels and badges to gamify your loyalty program and increase customer retention.', 'loyaltyx-points-and-rewards-for-woocommerce' ),
					'list_features' => [
						esc_html__( 'Create Unlimited User Levels with Custom Point Thresholds', 'loyaltyx-points-and-rewards-for-woocommerce' ),
						esc_html__( 'Design Custom Badges for Each Level Achievement', 'loyaltyx-points-and-rewards-for-woocommerce' ),
						esc_html__( 'Track User Progress and Level Advancement', 'loyaltyx-points-and-rewards-for-woocommerce' ),
						esc_html__( 'View Users Reached Each Level with Detailed Statistics', 'loyaltyx-points-and-rewards-for-woocommerce' ),
						esc_html__( 'Automated Level Assignment Based on Points Earned', 'loyaltyx-points-and-rewards-for-woocommerce' ),
						esc_html__( 'Visual Level Progression Indicators for Users', 'loyaltyx-points-and-rewards-for-woocommerce' ),
						esc_html__( 'Bulk Level Management (Activate, Deactivate, Delete)', 'loyaltyx-points-and-rewards-for-woocommerce' ),
						esc_html__( 'Level-Based User Segmentation and Filtering', 'loyaltyx-points-and-rewards-for-woocommerce' ),
						esc_html__( 'Enhanced User Engagement Through Gamification', 'loyaltyx-points-and-rewards-for-woocommerce' ),
					],
					'upgrade_url' => '//devdiggers.com/plugin/woocommerce-points-and-rewards/?utm_source=plugin_dashboard&utm_medium=upgrade_notice&utm_campaign=levels_badges_pro_feature',
				]
			);
		}

		/**
		 * Get points earning rules template function
		 *
		 * @return void
		 */
		public function ddwcpr_get_points_earning_rules_template() {
			new Admin\Points_Rules\DDWCPR_Points_Earning_Template( $this->ddwcpr_configuration );
		}

		/**
		 * Get points redeeming rules template function
		 *
		 * @return void
		 */
		public function ddwcpr_get_points_redeeming_rules_template() {
			new Admin\Points_Rules\DDWCPR_Points_Redeeming_Template( $this->ddwcpr_configuration );
		}

		/**
		 * Get General Configuration Template
		 *
		 * @return void
		 */
		public function ddwcpr_get_general_configuration_template() {
			new Admin\Configuration\DDWCPR_General_Configuration_Template( $this->ddwcpr_configuration );
		}

		/**
		 * Get Display Messages Configuration Template
		 *
		 * @return void
		 */
		public function ddwcpr_get_display_messages_configuration_template() {
			new Admin\Configuration\DDWCPR_Display_Messages_Configuration_Template( $this->ddwcpr_configuration );
		}

		/**
		 * Get Points Earning Configuration Template
		 *
		 * @return void
		 */
		public function ddwcpr_get_points_earning_configuration_template() {
			new Admin\Configuration\DDWCPR_Points_Earning_Configuration_Template( $this->ddwcpr_configuration );
		}

		/**
		 * Get Points Redeeming Configuration Template
		 *
		 * @return void
		 */
		public function ddwcpr_get_points_redeeming_configuration_template() {
			new Admin\Configuration\DDWCPR_Points_Redeeming_Configuration_Template( $this->ddwcpr_configuration );
		}

		/**
		 * Get Points Expiration Configuration Template
		 *
		 * @return void
		 */
		public function ddwcpr_get_points_expiration_configuration_template() {
			new Admin\Configuration\DDWCPR_Points_Expiration_Configuration_Template( $this->ddwcpr_configuration );
		}

		/**
		 * Get Layout Configuration Template
		 *
		 * @return void
		 */
		public function ddwcpr_get_layout_configuration_template() {
			new Admin\Configuration\DDWCPR_Layout_Configuration_Template( $this->ddwcpr_configuration );
		}

		/**
		 * Get Emails Configuration Template
		 *
		 * @return void
		 */
		public function ddwcpr_get_emails_configuration_template() {
			new Admin\Configuration\DDWCPR_Emails_Configuration_Template( $this->ddwcpr_configuration );
		}

		/**
		 * Get Shortcodes Configuration Template
		 *
		 * @return void
		 */
		public function ddwcpr_get_shortcodes_configuration_template() {
			new Admin\Configuration\DDWCPR_Shortcodes_Configuration_Template( $this->ddwcpr_configuration );
		}

		/**
		 * Enqueue admin scripts function
		 *
		 * @return scripts
		 */
		public function ddwcpr_enqueue_admin_scripts() {
			if ( $this->dashboard->is_a_plugin_page() ) {
				wp_enqueue_media();

				wp_register_style( 'ddwcpr-import-style', DDWCPR_PLUGIN_URL . 'assets/css/import.css', [], filemtime( DDWCPR_PLUGIN_FILE . 'assets/css/import.css' ) );
				wp_register_script( 'ddwcpr-import-script', DDWCPR_PLUGIN_URL . 'assets/js/import.js', [ 'wp-util' ], filemtime( DDWCPR_PLUGIN_FILE . 'assets/js/import.js' ) );

				wp_register_style( 'ddwcpr-manual-adjustment-style', DDWCPR_PLUGIN_URL . 'assets/css/manual-adjustment.css', [], filemtime( DDWCPR_PLUGIN_FILE . 'assets/css/manual-adjustment.css' ) );
				wp_register_script( 'ddwcpr-manual-adjustment-script', DDWCPR_PLUGIN_URL . 'assets/js/manual-adjustment.js', [ 'wp-util' ], filemtime( DDWCPR_PLUGIN_FILE . 'assets/js/manual-adjustment.js' ) );

				wp_localize_script(
					'ddwcpr-import-script',
					'ddwcprImportObject',
					[
						'ajax' => [
							'ajaxUrl'   => admin_url( 'admin-ajax.php' ),
							'ajaxNonce' => wp_create_nonce( 'ddwcpr-nonce' ),
						],
						'i18n' => [
							// Import-specific strings
							'noRowsImported'    => esc_html__( 'No rows imported.', 'loyaltyx-points-and-rewards-for-woocommerce' ),
							'allRowsImported'   => esc_html__( 'All rows imported.', 'loyaltyx-points-and-rewards-for-woocommerce' ),
							'processingImport'  => esc_html__( 'Processing import...', 'loyaltyx-points-and-rewards-for-woocommerce' ),
							'importError'       => esc_html__( 'Import error occurred', 'loyaltyx-points-and-rewards-for-woocommerce' ),
							'mapUserIdentifier' => esc_html__( 'Please map at least one of ID, Username, or Email to identify users.', 'loyaltyx-points-and-rewards-for-woocommerce' ),
							'mapPointsField'    => esc_html__( 'Please map the Points field.', 'loyaltyx-points-and-rewards-for-woocommerce' ),
							'unknownError'      => esc_html__( 'Unknown error occurred', 'loyaltyx-points-and-rewards-for-woocommerce' ),
							'ajaxError'         => esc_html__( 'AJAX error:', 'loyaltyx-points-and-rewards-for-woocommerce' ),
						],
					]
				);

				wp_localize_script(
					'ddwcpr-manual-adjustment-script',
					'ddwcprManualAdjustmentObject',
					[
						'ajax' => [
							'ajaxUrl'   => admin_url( 'admin-ajax.php' ),
							'ajaxNonce' => wp_create_nonce( 'ddwcpr-nonce' ),
						],
						'i18n' => [
							// Manual adjustment specific strings
							'processingAdjustment' => esc_html__( 'Processing points adjustment...', 'loyaltyx-points-and-rewards-for-woocommerce' ),
							'adjustmentComplete'   => esc_html__( 'Points adjustment completed successfully.', 'loyaltyx-points-and-rewards-for-woocommerce' ),
							'adjustmentError'      => esc_html__( 'Adjustment error occurred', 'loyaltyx-points-and-rewards-for-woocommerce' ),
							'selectUsers'          => esc_html__( 'Please select at least one user.', 'loyaltyx-points-and-rewards-for-woocommerce' ),
							'enterValidPoints'     => esc_html__( 'Please enter valid points amount.', 'loyaltyx-points-and-rewards-for-woocommerce' ),
							'noUsersFound'         => esc_html__( 'No users found. Please try again.', 'loyaltyx-points-and-rewards-for-woocommerce' ),
						],
					]
				);

				if ( ! empty( $_GET['menu'] ) && 'dashboard' !== $_GET['menu'] && ( 'users' !== $_GET['menu'] || ( 'users' === $_GET['menu'] && ( empty( $_GET['action'] ) || 'view' !== $_GET['action'] ) ) ) ) {
					wp_enqueue_style( 'ddwcpr-admin-style', DDWCPR_PLUGIN_URL . 'assets/css/admin.css', [ DDFW_Assets::$framework_css_handle ], filemtime( DDWCPR_PLUGIN_FILE . 'assets/css/admin.css' ) );
					wp_enqueue_script( 'ddwcpr-admin-script', DDWCPR_PLUGIN_URL . 'assets/js/admin.js', [ DDFW_Assets::$framework_js_handle, 'wp-util', 'wp-mediaelement' ], filemtime( DDWCPR_PLUGIN_FILE . 'assets/js/admin.js' ) );

					wp_localize_script(
						'ddwcpr-admin-script',
						'ddwcprAdminObject',
						[
							'ajax' => [
								'ajaxUrl'   => admin_url( 'admin-ajax.php' ),
								'ajaxNonce' => wp_create_nonce( 'ddwcpr-nonce' ),
							],
							'rules_icon'           => ddwcpr_get_rules_icon( '', true ),
							'earning_rule_types'   => ddwcpr_get_earning_rule_types(),
							'redeeming_rule_types' => ddwcpr_get_redeeming_rule_types(),
						]
					);
				}

				// Enqueue Chart.js for dashboard charts
				wp_register_script( 'chart-js', DDWCPR_PLUGIN_URL . 'assets/js/chart.js', [], filemtime( DDWCPR_PLUGIN_FILE . 'assets/js/chart.js' ) );

				// Register dashboard specific assets
				wp_register_style( 'ddwcpr-dashboard-style', DDWCPR_PLUGIN_URL . 'assets/css/dashboard.css', [ DDFW_Assets::$framework_css_handle ], filemtime( DDWCPR_PLUGIN_FILE . 'assets/css/dashboard.css' ) );
				wp_register_script( 'ddwcpr-dashboard-script', DDWCPR_PLUGIN_URL . 'assets/js/dashboard.js', [ 'jquery', 'chart-js', DDFW_Assets::$framework_js_handle ], filemtime( DDWCPR_PLUGIN_FILE . 'assets/js/dashboard.js' ) );

				if ( ! empty( $_GET['page'] ) && 'ddwcpr-dashboard' === $_GET['page'] && ( empty( $_GET['menu'] ) || 'dashboard' === $_GET['menu'] ) ) {
					wp_enqueue_style( 'ddwcpr-dashboard-style' );
				}
			}
		}
	}
}
