<?php
/**
 * Point helper
 *
 * @package WooCommerce Points and Rewards
 * @version 1.0.0
 */

namespace DDWCPointsAndRewards\Helper\Point;

use DDWCPointsAndRewards\Helper\Log\DDWCPR_Log_Helper;

defined( 'ABSPATH' ) || exit();

if ( ! class_exists( 'DDWCPR_Point_Helper' ) ) {
	/**
	 * Point helper class
	 */
	class DDWCPR_Point_Helper {
		/**
		 * Database object
		 *
		 * @var object
		 */
		protected $wpdb;

		/**
		 * Configuration Variable
		 *
		 * @var array
		 */
		protected $ddwcpr_configuration;

		/**
		 * Log Helper Variable
		 *
		 * @var object
		 */
		protected $log_helper;

		/**
		 * Construct
		 */
		public function __construct( $ddwcpr_configuration ) {
			global $wpdb;
			$this->wpdb                 = $wpdb;
			$this->ddwcpr_configuration = $ddwcpr_configuration;
			$this->log_helper           = new DDWCPR_Log_Helper( $ddwcpr_configuration );
		}

		/**
		 * Add user points function
		 *
		 * @param array $data
		 * @return int
		 */
		public function ddwcpr_add_user_points( $data ) {
			$log_id = $this->log_helper->ddwcpr_save_log( $data );
			return $log_id;
		}

		/**
		 * Deduct users points function
		 *
		 * @param array $data
		 * @param int|null $order_id
		 * @return int
		 */
		public function ddwcpr_deduct_user_points( $data, $order_id = null ) {
			$log_id = $this->log_helper->ddwcpr_save_log( $data );

			return $log_id;
		}

		/**
		 * Get user available points function
		 *
		 * @param int $user_id
		 * @return int
		 */
		public function ddwcpr_get_user_available_points( $user_id ) {
			return intval( $this->wpdb->get_var( $this->wpdb->prepare( "SELECT SUM( points_log.points ) AS points FROM {$this->wpdb->ddwcpr_points_log} as points_log LEFT JOIN {$this->wpdb->users} as users ON points_log.user_id=users.ID WHERE points_log.user_id=%d", $user_id ) ) );
		}

		/**
		 * Get user earned points function
		 *
		 * @return int
		 */
		public function ddwcpr_get_user_earned_points( $user_id ) {
			if ( empty( $user_id ) ) {
				$user_id = get_current_user_id();
			}

			return intval(
				$this->wpdb->get_var(
					$this->wpdb->prepare(
						"
						SELECT SUM(
							CASE 
								WHEN points_log.points > 0 THEN points_log.points
								ELSE 0
							END
						) - SUM(
							CASE 
								WHEN points_log.points < 0 AND points_log.event IN ('order_refunded', 'order_cancelled', 'order_failed') THEN ABS(points_log.points)
								ELSE 0
							END
						)
						FROM {$this->wpdb->ddwcpr_points_log} as points_log
						LEFT JOIN {$this->wpdb->users} as users ON points_log.user_id = users.ID
						WHERE points_log.user_id = %d
						",
						$user_id
					)
				)
			);
		}

		/**
		 * Get user total referrals function
		 *
		 * @param int $user_id
		 * @return int
		 */
		public function ddwcpr_get_user_total_referrals( $user_id ) {
			return intval( $this->wpdb->get_var(
				$this->wpdb->prepare(
					"SELECT count(u.ID)
					FROM {$this->wpdb->users} AS u
					INNER JOIN {$this->wpdb->usermeta} AS um
						ON u.ID = um.user_id
					WHERE um.meta_key = '_ddwcpr_referrer_id'
					AND um.meta_value = %d",
					$user_id
				)
			) );
		}

		/**
		 * Display points earning message function
		 *
		 * @param int $points
		 * @param string $message
		 * @return void
		 */
		public function ddwcpr_display_points_earning_message( $points, $message ) {
			$message = str_replace(
				[
					'{points_value}',
					'{points_label}',
				],
				[
					$points,
					ddwcpr_get_points_label( $points ),
				],
				$message
			);

			?>
			<div
				class="ddwcpr-earning-points-message"
				style="
					background-color: <?php echo esc_attr( $this->ddwcpr_configuration['points_message_styling']['background_color'] ); ?>;
					color           : <?php echo esc_attr( $this->ddwcpr_configuration['points_message_styling']['color'] ); ?>;
					border          : solid 1px <?php echo esc_attr( $this->ddwcpr_configuration['points_message_styling']['border_color'] ); ?>;
					font-size       : <?php echo esc_attr( $this->ddwcpr_configuration['points_message_styling']['font_size'] ); ?>;
					padding         : <?php echo esc_attr( $this->ddwcpr_configuration['points_message_styling']['padding'] ); ?>;
					margin          : <?php echo esc_attr( $this->ddwcpr_configuration['points_message_styling']['margin'] ); ?>;
					border-radius   : <?php echo esc_attr( $this->ddwcpr_configuration['points_message_styling']['border_radius'] ); ?>;
					"
				>
				<p><?php echo wp_kses( $message, array_merge( wp_kses_allowed_html( 'post' ), ddfw_kses_allowed_svg_tags() ) ); ?></p>
			</div>
			<?php
		}
	}
}
