<?php
/**
 * Plugin Name: Loyalty Links
 * Description: Only shows external links from domains that have recently referred visitors to your site.
 * Version: 1.0.1
 * Author: Jose Mortellaro
 * Author URI: https://josemortellaro.com
 * Domain Path: /languages/
 * Text Domain: loyalty-links
 * License: GPL-2.0+
 * License URI: https://www.gnu.org/licenses/gpl-2.0.html
 * Requires at least: 5.0
 * Requires PHP: 7.4
 * Tested up to: 6.9
 * Stable tag: 1.0.1
 */

defined( 'ABSPATH' ) || exit; // Exit if accessed directly.

// Define plugin constants.
define( 'LOYALTY_LINKS_VERSION', '1.0.1' );
define( 'LOYALTY_LINKS_DIR', untrailingslashit( dirname( __FILE__ ) ) );
define( 'LOYALTY_LINKS_URL', untrailingslashit( plugins_url( '', __FILE__ ) ) );

// Load settings class.
require_once LOYALTY_LINKS_DIR . '/includes/class-loyalty-links-settings.php';

// Configuration: Number of days to keep referrer records (changeable).
if ( ! defined( 'LOYALTY_LINKS_RETENTION_DAYS' ) ) {
	define( 'LOYALTY_LINKS_RETENTION_DAYS', 30 ); // Default: 30 days
}

/**
 * Main plugin class.
 */
class Loyalty_Links {
	
	/**
	 * Option name for storing referrer data.
	 */
	const OPTION_NAME = 'loyalty_links_referrers';
	
	
	/**
	 * Instance of this class.
	 *
	 * @var Loyalty_Links
	 */
	private static $instance = null;
	
	/**
	 * Get instance of this class.
	 *
	 * @return Loyalty_Links
	 */
	public static function get_instance() {
		if ( null === self::$instance ) {
			self::$instance = new self();
		}
		return self::$instance;
	}
	
	/**
	 * Constructor.
	 */
	private function __construct() {
		$this->init();
	}
	
	/**
	 * Initialize plugin.
	 */
	private function init() {
		// Initialize settings page.
		Loyalty_Links_Settings::get_instance();
		
		// Clean up old referrer records periodically.
		add_action( 'wp', array( $this, 'maybe_cleanup_referrers' ) );
		
		// Enqueue scripts and styles.
		add_action( 'wp_enqueue_scripts', array( $this, 'enqueue_scripts' ) );
		
		// Add async attribute to script tag.
		add_filter( 'script_loader_tag', array( $this, 'add_async_to_script' ), 10, 2 );
		
		// Register REST API endpoint for cache-friendly approved domains.
		add_action( 'rest_api_init', array( $this, 'register_rest_routes' ) );

		// Add "Documentation" link on the Plugins page.
		add_filter( 'plugin_action_links_' . plugin_basename( __FILE__ ), array( $this, 'plugin_action_links' ) );
	}

	/**
	 * Add "Settings" and "Documentation" links to the plugin row on the Plugins page.
	 *
	 * @param array $links Existing plugin action links.
	 * @return array Modified links.
	 */
	public function plugin_action_links( $links ) {
		$settings_url = add_query_arg(
			'page',
			Loyalty_Links_Settings::PAGE_SLUG,
			admin_url( 'options-general.php' )
		);
		$settings_link = sprintf(
			'<a href="%s">%s</a>',
			esc_url( $settings_url ),
			esc_html__( 'Settings', 'loyalty-links' )
		);
		$doc_link = sprintf(
			'<a href="%s" target="_blank" rel="noopener noreferrer">%s</a>',
			esc_url( 'https://wordpress.org/support/topic/how-to-configure-the-plugin-3/' ),
			esc_html__( 'Documentation', 'loyalty-links' )
		);
		array_unshift( $links, $doc_link, $settings_link );
		return $links;
	}
	
	
	/**
	 * Add a test referrer with a custom timestamp.
	 * Used for testing purposes without waiting for actual referrals.
	 *
	 * @param string $domain   Domain name to add as referrer.
	 * @param int    $days_ago Number of days ago to set the timestamp (default: 2).
	 * @return bool True if successful, false otherwise.
	 */
	public function add_test_referrer( $domain, $days_ago = 2 ) {
		if ( empty( $domain ) ) {
			return false;
		}
		
		// Normalize domain.
		$domain = $this->normalize_domain( $domain );
		
		// Calculate timestamp (days ago).
		$days_ago = absint( $days_ago );
		$timestamp = current_time( 'timestamp' ) - ( $days_ago * DAY_IN_SECONDS );
		
		// Get existing referrers.
		$referrers = $this->get_referrers();
		
		// Mark this domain as a test domain FIRST (so it's never approved).
		// This must happen before saving referrers to ensure it's excluded from approved list.
		$this->add_test_domain( $domain );
		
		// Add or update referrer with custom timestamp (keep existing count or set to 1).
		$existing_count = isset( $referrers[ $domain ]['count'] ) ? (int) $referrers[ $domain ]['count'] : 1;
		$referrers[ $domain ] = array(
			'timestamp' => $timestamp,
			'count'     => $existing_count,
		);
		
		// Save referrers - use autoload = false to avoid loading on every request.
		// Use update_option which works for both new and existing options.
		$result = update_option( self::OPTION_NAME, $referrers, false );
		
		// Verify the save was successful by checking the saved value.
		$saved_referrers = get_option( self::OPTION_NAME, array() );
		if ( isset( $saved_referrers[ $domain ] ) ) {
			$saved_data = $saved_referrers[ $domain ];
			$saved_timestamp = is_array( $saved_data ) && isset( $saved_data['timestamp'] ) 
				? (int) $saved_data['timestamp'] 
				: (int) $saved_data;
			if ( $saved_timestamp === $timestamp ) {
				return true; // Successfully saved.
			}
		}
		
		// If update_option returned false and the value wasn't saved, try add_option.
		if ( $result === false && ! get_option( self::OPTION_NAME, false ) ) {
			$result = add_option( self::OPTION_NAME, $referrers, '', false );
			if ( $result ) {
				return true;
			}
		}
		
		return false;
	}
	
	/**
	 * Track external referrer (internal method, called by REST API).
	 * This method is now called from JavaScript via REST API for cache compatibility.
	 *
	 * @param string $referrer_url The referrer URL to track.
	 * @return bool True if tracked successfully, false otherwise.
	 */
	private function track_referrer_internal( $referrer_url ) {
		if ( empty( $referrer_url ) ) {
			return false;
		}
		
		$referrer_host = wp_parse_url( $referrer_url, PHP_URL_HOST );
		
		// Get current site domain.
		$site_url = home_url();
		$site_host = wp_parse_url( $site_url, PHP_URL_HOST );
		
		// Only track external referrers.
		if ( empty( $referrer_host ) || $referrer_host === $site_host ) {
			return false;
		}
		
		// Normalize domain (remove www. prefix for consistency).
		$referrer_host = $this->normalize_domain( $referrer_host );
		
		// Get monitored domains from settings.
		$settings = Loyalty_Links_Settings::get_instance();
		$monitored_domains = $settings->get_monitored_domains();
		
		// Only track if domain is in monitored list.
		// If list is empty, don't track anything (plugin is effectively disabled).
		if ( empty( $monitored_domains ) || ! in_array( $referrer_host, $monitored_domains, true ) ) {
			return false;
		}
		
		// Get existing referrers.
		$referrers = $this->get_referrers();
		
		// Get current timestamp.
		$current_timestamp = current_time( 'timestamp' );
		
		// Update or add referrer with current timestamp and increment visit count.
		if ( isset( $referrers[ $referrer_host ] ) && is_array( $referrers[ $referrer_host ] ) ) {
			// Increment existing count.
			$referrers[ $referrer_host ]['count'] = (int) $referrers[ $referrer_host ]['count'] + 1;
			$referrers[ $referrer_host ]['timestamp'] = $current_timestamp;
		} else {
			// New referrer - start with count of 1.
			$referrers[ $referrer_host ] = array(
				'timestamp' => $current_timestamp,
				'count'     => 1,
			);
		}
		
		// Save referrers.
		update_option( self::OPTION_NAME, $referrers, false );
		
		return true;
	}
	
	/**
	 * Normalize domain name (remove www. prefix).
	 *
	 * @param string $domain Domain name.
	 * @return string Normalized domain.
	 */
	private function normalize_domain( $domain ) {
		return preg_replace( '/^www\./i', '', strtolower( $domain ) );
	}
	
	/**
	 * Get all referrer records.
	 * Returns array in format: domain => ['timestamp' => int, 'count' => int]
	 * Handles backward compatibility with old format: domain => timestamp
	 *
	 * @return array Array of domain => ['timestamp' => int, 'count' => int].
	 */
	public function get_referrers() {
		$referrers = get_option( self::OPTION_NAME, array() );
		if ( ! is_array( $referrers ) ) {
			return array();
		}
		
		// Normalize old format to new format for backward compatibility.
		$normalized = array();
		foreach ( $referrers as $domain => $data ) {
			if ( is_numeric( $data ) ) {
				// Old format: domain => timestamp
				$normalized[ $domain ] = array(
					'timestamp' => (int) $data,
					'count'     => 1, // Default count for old entries.
				);
			} elseif ( is_array( $data ) ) {
				// New format: domain => ['timestamp' => X, 'count' => Y]
				$normalized[ $domain ] = array(
					'timestamp' => isset( $data['timestamp'] ) ? (int) $data['timestamp'] : current_time( 'timestamp' ),
					'count'     => isset( $data['count'] ) ? (int) $data['count'] : 1,
				);
			}
		}
		
		return $normalized;
	}
	
	/**
	 * Get visit count for a specific domain.
	 *
	 * @param string $domain Domain name.
	 * @return int Visit count (0 if domain not found).
	 */
	public function get_domain_visit_count( $domain ) {
		$domain = $this->normalize_domain( $domain );
		$referrers = $this->get_referrers();
		
		if ( isset( $referrers[ $domain ] ) && is_array( $referrers[ $domain ] ) ) {
			return isset( $referrers[ $domain ]['count'] ) ? (int) $referrers[ $domain ]['count'] : 0;
		}
		
		return 0;
	}
	
	/**
	 * Get approved domains (domains that have sent referrers recently).
	 * Test domains are excluded from this list (they are never approved).
	 *
	 * @return array Array of approved domain names.
	 */
	public function get_approved_domains() {
		$referrers = $this->get_referrers();
		$settings = Loyalty_Links_Settings::get_instance();
		$retention_days = $settings->get_retention_days();
		$cutoff_time = current_time( 'timestamp' ) - ( $retention_days * DAY_IN_SECONDS );
		
		// Get test domains (these should never be approved).
		$test_domains = $this->get_test_domains();
		
		$approved = array();
		foreach ( $referrers as $domain => $data ) {
			// Skip test domains - they are never approved.
			if ( in_array( $domain, $test_domains, true ) ) {
				continue;
			}
			
			// Handle both old and new data formats.
			$timestamp = is_array( $data ) && isset( $data['timestamp'] ) 
				? (int) $data['timestamp'] 
				: (int) $data;
			
			if ( $timestamp >= $cutoff_time ) {
				$approved[] = $domain;
			}
		}
		
		return $approved;
	}
	
	/**
	 * Get list of test domains (domains that should never be approved).
	 *
	 * @return array Array of test domain names.
	 */
	public function get_test_domains() {
		$settings = Loyalty_Links_Settings::get_instance();
		return $settings->get_test_domains();
	}
	
	/**
	 * Add a domain to the test domains list.
	 *
	 * @param string $domain Domain name to mark as test domain.
	 * @return bool True if successful, false otherwise.
	 */
	private function add_test_domain( $domain ) {
		if ( empty( $domain ) ) {
			return false;
		}
		
		// Normalize domain.
		$domain = $this->normalize_domain( $domain );
		
		// Get existing test domains.
		$settings = Loyalty_Links_Settings::get_instance();
		$test_domains = $settings->get_test_domains();
		
		// Add domain if not already in list.
		if ( ! in_array( $domain, $test_domains, true ) ) {
			$test_domains[] = $domain;
			// Save updated list.
			return $settings->update_setting( 'test_domains', $test_domains );
		}
		
		return true;
	}
	
	/**
	 * Clean up old referrer records.
	 * Removes referrers older than the retention period.
	 */
	public function cleanup_referrers() {
		$referrers = $this->get_referrers();
		if ( empty( $referrers ) ) {
			return;
		}
		
		$settings = Loyalty_Links_Settings::get_instance();
		$retention_days = $settings->get_retention_days();
		$cutoff_time = current_time( 'timestamp' ) - ( $retention_days * DAY_IN_SECONDS );
		
		$cleaned = array();
		foreach ( $referrers as $domain => $data ) {
			// Handle both old and new data formats.
			$timestamp = is_array( $data ) && isset( $data['timestamp'] ) 
				? (int) $data['timestamp'] 
				: (int) $data;
			
			if ( $timestamp >= $cutoff_time ) {
				$cleaned[ $domain ] = $data; // Keep full data structure.
			}
		}
		
		// Only update if something was removed.
		if ( count( $cleaned ) !== count( $referrers ) ) {
			update_option( self::OPTION_NAME, $cleaned, false );
		}
	}
	
	/**
	 * Maybe cleanup referrers (runs periodically, not on every request).
	 */
	public function maybe_cleanup_referrers() {
		// Run cleanup on 1% of requests to avoid performance impact.
		if ( wp_rand( 1, 100 ) === 1 ) {
			$this->cleanup_referrers();
		}
	}
	
	/**
	 * Enqueue scripts and styles.
	 */
	public function enqueue_scripts() {
		// Only enqueue on frontend.
		if ( is_admin() ) {
			return;
		}
		// Enqueue JavaScript.
		// Load in footer but with async/defer for better performance.
		wp_enqueue_script(
			'loyalty-links-js',
			LOYALTY_LINKS_URL . '/assets/js/loyalty-links.js',
			array(),
			LOYALTY_LINKS_VERSION,
			true // Load in footer.
		);
		
		// Get monitored domains from settings (static, can be cached).
		$settings = Loyalty_Links_Settings::get_instance();
		$monitored_domains = $settings->get_monitored_domains();
		
		// Get test domains (static, can be cached).
		$test_domains = $this->get_test_domains();
		
		// Get current site domain for comparison (static).
		$site_url = home_url();
		$site_host = wp_parse_url( $site_url, PHP_URL_HOST );
		$site_domain = $this->normalize_domain( $site_host );
		
		// Localize script with static data only (monitored domains, test domains, site domain, REST API endpoints).
		// Approved domains are fetched dynamically via REST API to be cache-friendly.
		// Referrer tracking is also done via REST API for cache compatibility.
		wp_localize_script(
			'loyalty-links-js',
			'loyaltyLinks',
			array(
				'restUrlApproved' => rest_url( 'loyalty-links/v1/approved-domains' ),
				'restUrlTrack'    => rest_url( 'loyalty-links/v1/track-referrer' ),
				'restNonce'       => wp_create_nonce( 'wp_rest' ),
				'monitoredDomains' => $monitored_domains,
				'testDomains'      => $test_domains,
				'siteDomain'       => $site_domain,
			)
		);
	}
	
	/**
	 * Register REST API routes.
	 */
	public function register_rest_routes() {
		// Endpoint for getting approved domains (cache-friendly).
		register_rest_route(
			'loyalty-links/v1',
			'/approved-domains',
			array(
				'methods'             => 'GET',
				'callback'           => array( $this, 'rest_get_approved_domains' ),
				'permission_callback' => '__return_true', // Public endpoint, no authentication required.
			)
		);
		
		// Endpoint for tracking referrers (cache-friendly, requires nonce).
		register_rest_route(
			'loyalty-links/v1',
			'/track-referrer',
			array(
				'methods'             => 'POST',
				'callback'           => array( $this, 'rest_track_referrer' ),
				'permission_callback' => '__return_true', // Public endpoint, but nonce verified in callback.
				'args'               => array(
					'referrer' => array(
						'required'          => true,
						'type'              => 'string',
						'sanitize_callback' => 'esc_url_raw',
						'validate_callback' => function( $param ) {
							return ! empty( $param );
						},
					),
				),
			)
		);
	}
	
	/**
	 * REST API callback to get approved domains.
	 * This is called dynamically via AJAX, so it's cache-friendly.
	 *
	 * @param WP_REST_Request $request REST request object.
	 * @return WP_REST_Response|WP_Error REST response.
	 */
	public function rest_get_approved_domains( $request ) {
		// Get approved domains dynamically (based on current referrer history).
		$approved_domains = $this->get_approved_domains();
		
		// Get test domains (these should never be in approved domains).
		$test_domains = $this->get_test_domains();
		
		// Double-check: Remove any test domains from approved domains list (safety check).
		$approved_domains = array_diff( $approved_domains, $test_domains );
		$approved_domains = array_values( $approved_domains ); // Re-index array.
		
		// Return JSON response.
		return rest_ensure_response(
			array(
				'approvedDomains' => $approved_domains,
				'success'         => true,
			)
		);
	}
	
	/**
	 * REST API callback to track referrer.
	 * This endpoint is called from JavaScript and is cache-friendly.
	 *
	 * @param WP_REST_Request $request REST request object.
	 * @return WP_REST_Response|WP_Error REST response.
	 */
	public function rest_track_referrer( $request ) {
		// Verify nonce for security.
		$nonce = $request->get_header( 'X-WP-Nonce' );
		if ( ! $nonce || ! wp_verify_nonce( $nonce, 'wp_rest' ) ) {
			return new WP_Error(
				'rest_forbidden',
				__( 'Nonce verification failed.', 'loyalty-links' ),
				array( 'status' => 403 )
			);
		}
		
		// Get referrer URL from request.
		$referrer_url = $request->get_param( 'referrer' );
		
		if ( empty( $referrer_url ) ) {
			return new WP_Error(
				'rest_invalid_param',
				__( 'Referrer URL is required.', 'loyalty-links' ),
				array( 'status' => 400 )
			);
		}
		
		// Track the referrer.
		$tracked = $this->track_referrer_internal( $referrer_url );
		
		// Set cache prevention headers to avoid caching this endpoint.
		header( 'Cache-Control: no-cache, no-store, must-revalidate' );
		header( 'Pragma: no-cache' );
		header( 'Expires: 0' );
		
		// Return JSON response.
		return rest_ensure_response(
			array(
				'success' => $tracked,
				'message' => $tracked 
					? __( 'Referrer tracked successfully.', 'loyalty-links' )
					: __( 'Referrer not tracked (not in monitored list or internal).', 'loyalty-links' ),
			)
		);
	}
	
	/**
	 * Add async attribute to script tag for faster loading.
	 *
	 * @param string $tag    Script tag.
	 * @param string $handle Script handle.
	 * @return string Modified script tag.
	 */
	public function add_async_to_script( $tag, $handle ) {
		if ( 'loyalty-links-js' !== $handle ) {
			return $tag;
		}
		
		// Add async attribute.
		return str_replace( ' src', ' async src', $tag );
	}
}

// Initialize plugin.
Loyalty_Links::get_instance();
