<?php
/**
 * Settings page for Loyalty Links plugin.
 *
 * @package Loyalty_Links
 */

defined( 'ABSPATH' ) || exit;

/**
 * Settings page class.
 */
class Loyalty_Links_Settings {
	
	/**
	 * Option group name.
	 */
	const OPTION_GROUP = 'loyalty_links_settings';
	
	/**
	 * Single option name for all settings.
	 */
	const OPTION_NAME = 'loyalty_links_settings';
	
	/**
	 * Settings page slug.
	 */
	const PAGE_SLUG = 'give-take-links';
	
	/**
	 * Instance of this class.
	 *
	 * @var Loyalty_Links_Settings
	 */
	private static $instance = null;
	
	/**
	 * Get instance of this class.
	 *
	 * @return Loyalty_Links_Settings
	 */
	public static function get_instance() {
		if ( null === self::$instance ) {
			self::$instance = new self();
		}
		return self::$instance;
	}
	
	/**
	 * Constructor.
	 */
	private function __construct() {
		$this->init();
	}
	
	/**
	 * Initialize settings page.
	 */
	private function init() {
		add_action( 'admin_menu', array( $this, 'add_settings_page' ) );
		add_action( 'admin_init', array( $this, 'register_settings' ) );
	}
	
	/**
	 * Property to store test domains from sanitization callback.
	 *
	 * @var array
	 */
	private $pending_test_domains = array();
	
	/**
	 * Flag to prevent infinite loops during sanitization.
	 *
	 * @var bool
	 */
	private static $sanitizing = false;
	
	/**
	 * Add settings page to Settings menu.
	 */
	public function add_settings_page() {
		add_options_page(
			__( 'Loyalty Links', 'loyalty-links' ),
			__( 'Loyalty Links', 'loyalty-links' ),
			'manage_options',
			self::PAGE_SLUG,
			array( $this, 'render_settings_page' )
		);
	}
	
	/**
	 * Register settings using WordPress Settings API.
	 */
	public function register_settings() {
		// Register single settings option.
		register_setting(
			self::OPTION_GROUP,
			self::OPTION_NAME,
			array(
				'sanitize_callback' => array( $this, 'sanitize_settings' ),
				'default'           => $this->get_default_settings(),
			)
		);
		
		// Add settings section for domains.
		add_settings_section(
			'loyalty_links_domains_section',
			__( 'Monitored Domains', 'loyalty-links' ),
			array( $this, 'render_section_description' ),
			self::PAGE_SLUG
		);
		
		// Add settings field for domains.
		add_settings_field(
			'loyalty_links_domains_field',
			__( 'Domain List', 'loyalty-links' ),
			array( $this, 'render_domain_field' ),
			self::PAGE_SLUG,
			'loyalty_links_domains_section'
		);
		
		// Add settings section for retention.
		add_settings_section(
			'loyalty_links_retention_section',
			__( 'Retention Settings', 'loyalty-links' ),
			array( $this, 'render_retention_section_description' ),
			self::PAGE_SLUG
		);
		
		// Add settings field for retention days.
		add_settings_field(
			'loyalty_links_retention_field',
			__( 'Retention Days', 'loyalty-links' ),
			array( $this, 'render_retention_field' ),
			self::PAGE_SLUG,
			'loyalty_links_retention_section'
		);
		
		// Add settings section for test domains.
		add_settings_section(
			'loyalty_links_test_section',
			__( 'Test Referrer', 'loyalty-links' ),
			array( $this, 'render_test_section_description' ),
			self::PAGE_SLUG
		);
		
		// Add settings field for test domain.
		add_settings_field(
			'loyalty_links_test_field',
			__( 'Test Domain', 'loyalty-links' ),
			array( $this, 'render_test_domain_field' ),
			self::PAGE_SLUG,
			'loyalty_links_test_section'
		);
	}
	
	/**
	 * Get default settings array.
	 *
	 * @return array Default settings.
	 */
	private function get_default_settings() {
		return array(
			'monitored_domains' => '',
			'retention_days'    => defined( 'LOYALTY_LINKS_RETENTION_DAYS' ) ? LOYALTY_LINKS_RETENTION_DAYS : 30,
			'test_domains'      => array(),
		);
	}
	
	/**
	 * Get all settings.
	 *
	 * @return array Settings array.
	 */
	public function get_settings() {
		// Clear cache to ensure we get fresh data.
		wp_cache_delete( self::OPTION_NAME, 'options' );
		$settings = get_option( self::OPTION_NAME, $this->get_default_settings() );
		return wp_parse_args( $settings, $this->get_default_settings() );
	}
	
	/**
	 * Get a specific setting value.
	 *
	 * @param string $key Setting key.
	 * @param mixed  $default Default value if not set.
	 * @return mixed Setting value.
	 */
	public function get_setting( $key, $default = null ) {
		$settings = $this->get_settings();
		return isset( $settings[ $key ] ) ? $settings[ $key ] : $default;
	}
	
	/**
	 * Update a specific setting.
	 *
	 * @param string $key Setting key.
	 * @param mixed  $value Setting value.
	 * @return bool True if successful.
	 */
	public function update_setting( $key, $value ) {
		$settings = $this->get_settings();
		$settings[ $key ] = $value;
		return update_option( self::OPTION_NAME, $settings, false );
	}
	
	/**
	 * Sanitize all settings.
	 *
	 * @param array $input Raw input from form.
	 * @return array Sanitized settings.
	 */
	public function sanitize_settings( $input ) {
		// Verify nonce for security.
		if ( ! isset( $_POST['_wpnonce'] ) || ! wp_verify_nonce( sanitize_text_field( wp_unslash( $_POST[ '_wpnonce' ] ) ), self::OPTION_GROUP . '-options' ) ) {
			// Nonce verification failed, return current settings.
			return $this->get_settings();
		}
		
		// Prevent infinite loops.
		if ( self::$sanitizing ) {
			return $input;
		}
		
		self::$sanitizing = true;
		
		// Get current settings to preserve values not in form.
		$current_settings = $this->get_settings();
		$sanitized = array();
		
		// Sanitize monitored domains.
		if ( isset( $input['monitored_domains'] ) ) {
			// Unslash POST data before sanitization.
			$monitored_domains_raw = wp_unslash( $input['monitored_domains'] );
			$sanitized['monitored_domains'] = $this->sanitize_domain_list( $monitored_domains_raw );
		} else {
			$sanitized['monitored_domains'] = $current_settings['monitored_domains'];
		}
		
		// Sanitize retention days.
		if ( isset( $input['retention_days'] ) ) {
			// Unslash POST data before sanitization.
			$retention_days_raw = wp_unslash( $input['retention_days'] );
			$sanitized['retention_days'] = $this->sanitize_retention_days( $retention_days_raw );
		} else {
			$sanitized['retention_days'] = $current_settings['retention_days'];
		}
		
		// Process test domains directly (but only validate, don't call add_test_referrer to avoid loops).
		if ( isset( $input['test_domains'] ) ) {
			// Unslash POST data before processing.
			$test_domains_raw = wp_unslash( $input['test_domains'] );
			$test_domains_input = is_string( $test_domains_raw ) ? explode( "\n", $test_domains_raw ) : (array) $test_domains_raw;
			$sanitized['test_domains'] = $this->validate_test_domains( $test_domains_input, $sanitized['monitored_domains'] );
		} else {
			$sanitized['test_domains'] = $current_settings['test_domains'];
		}
		
		self::$sanitizing = false;
		
		return $sanitized;
	}
	
	/**
	 * Validate test domains without processing referrers (to avoid infinite loops).
	 *
	 * @param array  $test_domains_input Array of test domain strings.
	 * @param string $monitored_domains_string Newline-separated monitored domains string.
	 * @return array Array of normalized valid test domains.
	 */
	private function validate_test_domains( $test_domains_input, $monitored_domains_string ) {
		// Convert monitored domains string to array.
		$monitored_domains = array();
		if ( ! empty( $monitored_domains_string ) ) {
			$lines = explode( "\n", $monitored_domains_string );
			foreach ( $lines as $line ) {
				$line = trim( $line );
				if ( ! empty( $line ) ) {
					$monitored_domains[] = $this->normalize_domain( $line );
				}
			}
			$monitored_domains = array_unique( $monitored_domains );
		}
		
		$valid_test_domains = array();
		$invalid_domains = array();
		$not_monitored_domains = array();
		
		// Process each test domain.
		foreach ( $test_domains_input as $test_domain_raw ) {
			if ( is_string( $test_domain_raw ) ) {
				$test_domain_raw = trim( $test_domain_raw );
			}
			
			if ( empty( $test_domain_raw ) ) {
				continue;
			}
			
			// Normalize domain.
			$test_domain = $this->normalize_domain( $test_domain_raw );
			
			// Validate domain.
			if ( ! $this->is_valid_domain( $test_domain ) ) {
				$invalid_domains[] = $test_domain_raw;
				continue;
			}
			
			// Check if domain is in monitored list.
			if ( empty( $monitored_domains ) || ! in_array( $test_domain, $monitored_domains, true ) ) {
				$not_monitored_domains[] = $test_domain_raw;
				continue;
			}
			
			$valid_test_domains[] = $test_domain;
		}
		
		// Show error messages for invalid/not monitored domains.
		if ( ! empty( $invalid_domains ) ) {
			add_settings_error(
				'loyalty_links_messages',
				'loyalty_links_test_error',
				sprintf(
					/* translators: %s: Comma-separated list of domains */
					__( 'Invalid domain format: %s', 'loyalty-links' ),
					esc_html( implode( ', ', $invalid_domains ) )
				),
				'error'
			);
		}
		
		if ( ! empty( $not_monitored_domains ) ) {
			add_settings_error(
				'loyalty_links_messages',
				'loyalty_links_test_error',
				sprintf(
					/* translators: %s: Comma-separated list of domains */
					__( 'The following domains are not in your monitored domains list: %s', 'loyalty-links' ),
					esc_html( implode( ', ', $not_monitored_domains ) )
				),
				'error'
			);
		}
		
		// Return normalized valid test domains array (without processing referrers here).
		return array_unique( $valid_test_domains );
	}
	
	/**
	 * Render settings page.
	 */
	public function render_settings_page() {
		// Check user capabilities.
		if ( ! current_user_can( 'manage_options' ) ) {
			return;
		}
		
		// Show settings errors.
		settings_errors( 'loyalty_links_messages' );
		?>
		<div class="wrap">
			<h1><?php echo esc_html( get_admin_page_title() ); ?></h1>
			<form action="options.php" method="post">
				<?php
				settings_fields( self::OPTION_GROUP );
				do_settings_sections( self::PAGE_SLUG );
				submit_button( __( 'Save Settings', 'loyalty-links' ) );
				?>
			</form>
		</div>
		<?php
	}
	
	/**
	 * Render section description.
	 */
	public function render_section_description() {
		echo '<p>' . esc_html__( 'Enter the domains you want to monitor. The plugin will only apply link-removal logic to external links pointing to these domains. All other outbound links will remain untouched.', 'loyalty-links' ) . '</p>';
		echo '<p>' . esc_html__( 'Enter one domain per line. You can include or exclude the "www." prefix - both will work.', 'loyalty-links' ) . '</p>';
		echo '<p><strong>' . esc_html__( 'Example:', 'loyalty-links' ) . '</strong></p>';
		echo '<pre>example.com<br>another-domain.org<br>www.third-site.net</pre>';
	}
	
	/**
	 * Render domain list textarea field.
	 */
	public function render_domain_field() {
		$settings = $this->get_settings();
		$value = $settings['monitored_domains'];
		?>
		<textarea 
			name="<?php echo esc_attr( self::OPTION_NAME . '[monitored_domains]' ); ?>" 
			id="loyalty_links_monitored_domains" 
			rows="10" 
			cols="50" 
			class="large-text code"
			placeholder="example.com&#10;another-domain.org&#10;www.third-site.net"
		><?php echo esc_textarea( $value ); ?></textarea>
		<p class="description">
			<?php esc_html_e( 'Enter one domain per line. The plugin will only process external links pointing to these domains.', 'loyalty-links' ); ?>
		</p>
		<?php
		$domains = $this->get_monitored_domains();
		if ( ! empty( $domains ) ) {
			// Get visit counts for each domain.
			$loyalty_links = Loyalty_Links::get_instance();
			$domains_with_counts = array();
			foreach ( $domains as $domain ) {
				$count = $loyalty_links->get_domain_visit_count( $domain );
				$domains_with_counts[] = $domain . ' (' . $count . ')';
			}
			?>
			<p class="description" style="margin-top: 10px;">
				<strong><?php esc_html_e( 'Currently monitoring:', 'loyalty-links' ); ?></strong><br>
				<?php echo esc_html( implode( ', ', array_slice( $domains_with_counts, 0, 10 ) ) ); ?>
				<?php if ( count( $domains ) > 10 ) : ?>
					<?php 
					// translators: %d: Number of domains
					echo ' <em>(' . esc_html( sprintf( __( 'and %d more', 'loyalty-links' ), count( $domains ) - 10 ) ) . ')</em>'; ?>
				<?php endif; ?>
			</p>
			<?php
		}
	}
	
	/**
	 * Sanitize domain list input.
	 *
	 * @param string $input Raw input from textarea.
	 * @return string Sanitized domain list.
	 */
	public function sanitize_domain_list( $input ) {
		if ( empty( $input ) ) {
			return '';
		}
		
		// Split by newlines.
		$lines = explode( "\n", $input );
		$domains = array();
		
		foreach ( $lines as $line ) {
			$line = trim( $line );
			
			// Skip empty lines.
			if ( empty( $line ) ) {
				continue;
			}
			
			// Remove protocol if present.
			$line = preg_replace( '#^https?://#i', '', $line );
			
			// Remove trailing slashes.
			$line = rtrim( $line, '/' );
			
			// Remove path if present (keep only domain).
			$parts = explode( '/', $line );
			$domain = $parts[0];
			
			// Basic domain validation.
			if ( $this->is_valid_domain( $domain ) ) {
				// Normalize domain (remove www. prefix for consistency).
				$domain = $this->normalize_domain( $domain );
				$domains[] = $domain;
			}
		}
		
		// Remove duplicates and sort.
		$domains = array_unique( $domains );
		sort( $domains );
		
		// Return as newline-separated string.
		return implode( "\n", $domains );
	}
	
	/**
	 * Check if a string is a valid domain.
	 *
	 * @param string $domain Domain to validate.
	 * @return bool True if valid, false otherwise.
	 */
	private function is_valid_domain( $domain ) {
		// Basic domain validation regex.
		// Allows domains with subdomains, hyphens, etc.
		$pattern = '/^([a-z0-9]([a-z0-9\-]{0,61}[a-z0-9])?\.)+[a-z]{2,}$/i';
		return (bool) preg_match( $pattern, $domain );
	}
	
	/**
	 * Normalize domain name (remove www. prefix).
	 *
	 * @param string $domain Domain name.
	 * @return string Normalized domain.
	 */
	private function normalize_domain( $domain ) {
		return preg_replace( '/^www\./i', '', strtolower( $domain ) );
	}
	
	/**
	 * Get monitored domains as an array.
	 *
	 * @return array Array of normalized domain names.
	 */
	public function get_monitored_domains() {
		$settings = $this->get_settings();
		$value = $settings['monitored_domains'];
		
		if ( empty( $value ) ) {
			return array();
		}
		
		$lines = explode( "\n", $value );
		$domains = array();
		
		foreach ( $lines as $line ) {
			$line = trim( $line );
			if ( ! empty( $line ) ) {
				$domains[] = $this->normalize_domain( $line );
			}
		}
		
		return array_unique( $domains );
	}
	
	/**
	 * Get default retention days from constant or fallback to 30.
	 *
	 * @return int Default retention days.
	 */
	private function get_default_retention_days() {
		return defined( 'LOYALTY_LINKS_RETENTION_DAYS' ) ? LOYALTY_LINKS_RETENTION_DAYS : 30;
	}
	
	/**
	 * Render retention section description.
	 */
	public function render_retention_section_description() {
		echo '<p>' . esc_html__( 'Configure how long referrer records should be retained. Domains that have sent referrers within this period will be considered "approved" and their links will remain functional.', 'loyalty-links' ) . '</p>';
	}
	
	/**
	 * Render retention days input field.
	 */
	public function render_retention_field() {
		$settings = $this->get_settings();
		$value = $settings['retention_days'];
		?>
		<input 
			type="number" 
			name="<?php echo esc_attr( self::OPTION_NAME . '[retention_days]' ); ?>" 
			id="loyalty_links_retention_days" 
			value="<?php echo esc_attr( $value ); ?>" 
			min="1" 
			max="3650" 
			class="small-text" 
		/>
		<p class="description">
			<?php esc_html_e( 'Number of days to retain referrer records. Default: 30 days.', 'loyalty-links' ); ?>
		</p>
		<?php
	}
	
	/**
	 * Sanitize retention days input.
	 *
	 * @param mixed $input Raw input.
	 * @return int Sanitized retention days.
	 */
	public function sanitize_retention_days( $input ) {
		$value = absint( $input );
		
		// Ensure minimum value of 1.
		if ( $value < 1 ) {
			$value = $this->get_default_retention_days();
		}
		
		// Ensure maximum value of 3650 (10 years).
		if ( $value > 3650 ) {
			$value = 3650;
		}
		
		return $value;
	}
	
	/**
	 * Get retention days setting.
	 *
	 * @return int Retention days.
	 */
	public function get_retention_days() {
		$settings = $this->get_settings();
		$retention_days = isset( $settings['retention_days'] ) ? absint( $settings['retention_days'] ) : $this->get_default_retention_days();
		
		// Ensure valid range.
		if ( $retention_days < 1 ) {
			$retention_days = $this->get_default_retention_days();
		}
		if ( $retention_days > 3650 ) {
			$retention_days = 3650;
		}
		
		return $retention_days;
	}
	
	/**
	 * Render test section description.
	 */
	public function render_test_section_description() {
		echo '<p>' . esc_html__( 'Add test referrer entries to verify the plugin functionality without waiting for actual referrals. Enter domains and save settings to add test referrers. Test domains are never approved, so links will always be broken (non-functional) for testing purposes.', 'loyalty-links' ) . '</p>';
	}
	
	/**
	 * Render test domain field.
	 */
	public function render_test_domain_field() {
		// Get value from POST first (if form was just submitted), then from saved settings.
		$value = '';
		
		// Verify nonce before processing POST data.
		$option_name = self::OPTION_NAME;
		$post_data = array();
		if ( isset( $_POST[ $option_name ] ) && is_array( $_POST[ $option_name ] ) ) {
			// Verify nonce for security.
			$nonce_field = self::OPTION_GROUP . '-options';
			if ( isset( $_POST[ '_wpnonce' ] ) && wp_verify_nonce( sanitize_text_field( wp_unslash( $_POST[ '_wpnonce' ] ) ), $nonce_field ) ) {
				$post_data = sanitize_text_field( wp_unslash( $_POST[ $option_name ] ) );
			}
		}
		
		if ( isset( $post_data['test_domains'] ) && ! empty( $post_data['test_domains'] ) ) {
			// Show what was submitted (before processing).
			$test_domains_raw = $post_data['test_domains'];
			$value = is_array( $test_domains_raw ) 
				? implode( "\n", array_map( 'sanitize_text_field', $test_domains_raw ) )
				: sanitize_textarea_field( $test_domains_raw );
		} else {
			// Get saved test domains from settings.
			$settings = $this->get_settings();
			$test_domains = isset( $settings['test_domains'] ) && is_array( $settings['test_domains'] ) ? $settings['test_domains'] : array();
			$value = implode( "\n", $test_domains );
		}
		?>
		<textarea 
			name="<?php echo esc_attr( self::OPTION_NAME . '[test_domains]' ); ?>" 
			id="loyalty_links_test_domains" 
			rows="5" 
			cols="50" 
			class="large-text code"
			placeholder="example.com&#10;another-domain.org&#10;www.third-site.net"
		><?php echo esc_textarea( $value ); ?></textarea>
		<p class="description">
			<?php esc_html_e( 'Enter the domains you want to test, one per line. These domains must be in your monitored domains list above. Test domains are never approved, so links will always be broken (non-functional) regardless of retention settings.', 'loyalty-links' ); ?>
		</p>
		<?php
	}
	
	/**
	 * Process test domains.
	 *
	 * @param array  $test_domains_input Array of test domain strings.
	 * @param string $monitored_domains_string Newline-separated monitored domains string.
	 * @return array Array of normalized valid test domains.
	 */
	public function process_test_domains( $test_domains_input, $monitored_domains_string ) {
		// Convert monitored domains string to array.
		$monitored_domains = array();
		if ( ! empty( $monitored_domains_string ) ) {
			$lines = explode( "\n", $monitored_domains_string );
			foreach ( $lines as $line ) {
				$line = trim( $line );
				if ( ! empty( $line ) ) {
					$monitored_domains[] = $this->normalize_domain( $line );
				}
			}
			$monitored_domains = array_unique( $monitored_domains );
		}
		
		$valid_test_domains = array();
		$invalid_domains = array();
		$not_monitored_domains = array();
		
		// Process each test domain.
		foreach ( $test_domains_input as $test_domain_raw ) {
			if ( is_string( $test_domain_raw ) ) {
				$test_domain_raw = trim( $test_domain_raw );
			}
			
			if ( empty( $test_domain_raw ) ) {
				continue;
			}
			
			// Normalize domain.
			$test_domain = $this->normalize_domain( $test_domain_raw );
			
			// Validate domain.
			if ( ! $this->is_valid_domain( $test_domain ) ) {
				$invalid_domains[] = $test_domain_raw;
				continue;
			}
			
			// Check if domain is in monitored list.
			if ( empty( $monitored_domains ) || ! in_array( $test_domain, $monitored_domains, true ) ) {
				$not_monitored_domains[] = $test_domain_raw;
				continue;
			}
			
			$valid_test_domains[] = $test_domain;
		}
		
		// Show error messages for invalid/not monitored domains.
		if ( ! empty( $invalid_domains ) ) {
			add_settings_error(
				'loyalty_links_messages',
				'loyalty_links_test_error',
				sprintf(
					/* translators: %s: Comma-separated list of domains */
					__( 'Invalid domain format: %s', 'loyalty-links' ),
					esc_html( implode( ', ', $invalid_domains ) )
				),
				'error'
			);
		}
		
		if ( ! empty( $not_monitored_domains ) ) {
			add_settings_error(
				'loyalty_links_messages',
				'loyalty_links_test_error',
				sprintf(
					/* translators: %s: Comma-separated list of domains */
					__( 'The following domains are not in your monitored domains list: %s', 'loyalty-links' ),
					esc_html( implode( ', ', $not_monitored_domains ) )
				),
				'error'
			);
		}
		
		// Add test referrers for valid domains.
		if ( ! empty( $valid_test_domains ) ) {
			$loyalty_links = Loyalty_Links::get_instance();
			$test_days = 2; // Use 2 days ago (value doesn't matter since test domains are never approved).
			$processed_count = 0;
			
			foreach ( $valid_test_domains as $test_domain ) {
				$success = $loyalty_links->add_test_referrer( $test_domain, $test_days );
				if ( $success ) {
					$processed_count++;
				}
			}
			
			if ( $processed_count > 0 ) {
				add_settings_error(
					'loyalty_links_messages',
					'loyalty_links_test_success',
					sprintf(
						/* translators: %d: Number of domains */
						_n(
							'Test referrer added successfully for %d domain.',
							'Test referrers added successfully for %d domains.',
							$processed_count,
							'loyalty-links'
						),
						$processed_count
					),
					'success'
				);
			}
		}
		
		// Return normalized valid test domains array.
		return array_unique( $valid_test_domains );
	}
	
	/**
	 * Get test domains.
	 *
	 * @return array Array of test domain names.
	 */
	public function get_test_domains() {
		$settings = $this->get_settings();
		$test_domains = isset( $settings['test_domains'] ) && is_array( $settings['test_domains'] ) ? $settings['test_domains'] : array();
		return $test_domains;
	}
}
