/**
 * Loyalty Links - Frontend JavaScript
 * 
 * Breaks the connection of external links that are not from approved domains
 * (domains that have recently referred visitors to this site) by changing
 * their href to "#" while keeping them visible.
 */

(function() {
	'use strict';
	
	// Check if localization data exists.
	if (typeof loyaltyLinks === 'undefined') {
		return;
	}
	
	// Static data from PHP (can be cached).
	var monitoredDomains = loyaltyLinks.monitoredDomains || [];
	var testDomains = loyaltyLinks.testDomains || [];
	var siteDomain = loyaltyLinks.siteDomain || '';
	var restUrlApproved = loyaltyLinks.restUrlApproved || '';
	var restUrlTrack = loyaltyLinks.restUrlTrack || '';
	var restNonce = loyaltyLinks.restNonce || '';
	
	// Dynamic data (fetched via AJAX - cache-friendly).
	var approvedDomains = [];
	var approvedDomainsLoaded = false;
	
	/**
	 * Normalize domain name (remove www. prefix and convert to lowercase).
	 *
	 * @param {string} domain Domain name.
	 * @return {string} Normalized domain.
	 */
	function normalizeDomain(domain) {
		if (!domain) {
			return '';
		}
		return domain.toLowerCase().replace(/^www\./, '');
	}
	
	/**
	 * Extract domain from URL.
	 *
	 * @param {string} url URL string.
	 * @return {string} Domain name or empty string.
	 */
	function extractDomain(url) {
		if (!url || url.trim() === '') {
			return '';
		}
		
		try {
			// If URL doesn't have a protocol, add one temporarily for parsing.
			var urlToParse = url.trim();
			if (urlToParse.indexOf('://') === -1 && urlToParse.indexOf('//') !== 0) {
				// Check if it looks like a domain (contains a dot and doesn't start with /)
				if (urlToParse.indexOf('.') !== -1 && urlToParse.indexOf('/') !== 0) {
					// Add protocol temporarily for parsing
					urlToParse = 'http://' + urlToParse;
				} else {
					// Probably a relative URL
					return '';
				}
			}
			
			var parser = document.createElement('a');
			parser.href = urlToParse;
			var hostname = parser.hostname;
			
			// If hostname is empty or matches current location, it might be relative
			if (!hostname || hostname === window.location.hostname) {
				// Check if original URL looks like a domain
				var originalUrl = url.trim();
				if (originalUrl.indexOf('.') !== -1 && 
				    originalUrl.indexOf('/') !== 0 && 
				    originalUrl.indexOf('#') !== 0 &&
				    originalUrl.indexOf('mailto:') !== 0 &&
				    originalUrl.indexOf('tel:') !== 0) {
					// Extract domain manually (everything before first /)
					var domainPart = originalUrl.split('/')[0].split('?')[0].split('#')[0];
					return normalizeDomain(domainPart);
				}
				return '';
			}
			
			return normalizeDomain(hostname);
		} catch (e) {
			// Fallback: try to extract domain manually
			var cleanUrl = url.trim();
			if (cleanUrl.indexOf('.') !== -1 && 
			    cleanUrl.indexOf('/') !== 0 && 
			    cleanUrl.indexOf('#') !== 0) {
				var domainPart = cleanUrl.split('/')[0].split('?')[0].split('#')[0];
				return normalizeDomain(domainPart);
			}
			return '';
		}
	}
	
	/**
	 * Check if URL is external (different domain from site).
	 *
	 * @param {string} url URL to check.
	 * @return {boolean} True if external, false otherwise.
	 */
	function isExternal(url) {
		if (!url || url.trim() === '') {
			return false;
		}
		
		var trimmedUrl = url.trim();
		
		// Check for relative URLs, anchors, mailto, tel, etc.
		if (trimmedUrl.indexOf('#') === 0 || 
		    trimmedUrl.indexOf('mailto:') === 0 || 
		    trimmedUrl.indexOf('tel:') === 0 ||
		    trimmedUrl.indexOf('javascript:') === 0 ||
		    (trimmedUrl.indexOf('/') === 0 && trimmedUrl.indexOf('//') !== 0)) {
			return false;
		}
		
		var linkDomain = extractDomain(trimmedUrl);
		
		// If we couldn't extract a domain, it's probably not external
		if (!linkDomain || linkDomain === '') {
			return false;
		}
		
		// Compare with site domain
		return linkDomain !== siteDomain;
	}
	
	/**
	 * Check if domain is a test domain (should never be approved).
	 *
	 * @param {string} domain Domain to check.
	 * @return {boolean} True if test domain, false otherwise.
	 */
	function isTestDomain(domain) {
		if (!domain || testDomains.length === 0) {
			return false;
		}
		
		var normalizedDomain = normalizeDomain(domain);
		return testDomains.indexOf(normalizedDomain) !== -1;
	}
	
	/**
	 * Check if domain is in approved list.
	 *
	 * @param {string} domain Domain to check.
	 * @return {boolean} True if approved, false otherwise.
	 */
	function isApprovedDomain(domain) {
		if (!domain) {
			return false;
		}
		
		// Test domains are never approved.
		if (isTestDomain(domain)) {
			return false;
		}
		
		var normalizedDomain = normalizeDomain(domain);
		return approvedDomains.indexOf(normalizedDomain) !== -1;
	}
	
	/**
	 * Check if domain is in monitored list.
	 *
	 * @param {string} domain Domain to check.
	 * @return {boolean} True if monitored, false otherwise.
	 */
	function isMonitoredDomain(domain) {
		if (!domain) {
			return false;
		}
		
		// If no monitored domains are set, don't process any links.
		if (monitoredDomains.length === 0) {
			return false;
		}
		
		var normalizedDomain = normalizeDomain(domain);
		return monitoredDomains.indexOf(normalizedDomain) !== -1;
	}
	
	/**
	 * Process all links on the page.
	 */
	function processLinks() {
		var links = document.querySelectorAll('a[href]');
		var processedCount = 0;
		var brokenCount = 0;
		
		for (var i = 0; i < links.length; i++) {
			var link = links[i];
			var href = link.getAttribute('href');
			
			if (!href) {
				continue;
			}
			
			processedCount++;
			
			// Check if link is external.
			if (isExternal(href)) {
				var linkDomain = extractDomain(href);
				
				// Only process links if domain is in monitored list.
				if (isMonitoredDomain(linkDomain)) {
					// Check if it's a test domain first (test domains are never approved).
					var isTest = isTestDomain(linkDomain);
					var isApproved = isApprovedDomain(linkDomain);
					
					// If domain is not approved (or is a test domain), break the link connection.
					if (isTest || !isApproved) {
						// Store original href in data attribute for reference.
						link.setAttribute('data-original-href', href);
						
						// Change href to "#" to break the connection.
						link.setAttribute('href', '#');
						
						// Add class for styling (link remains visible but broken).
						link.classList.add('is-broken-link');
						link.classList.add('loyalty-links-broken');
						
						// Disable pointer events to prevent any interaction.
						link.style.pointerEvents = 'none';
						
						brokenCount++;
					} else {
						// Approved external links remain functional.
						link.classList.add('loyalty-links-approved');
						link.classList.remove('is-broken-link');
						link.classList.remove('loyalty-links-broken');
						
						// Restore pointer events for approved links.
						link.style.pointerEvents = '';
					}
				} else {
					// Domain is not monitored - leave link untouched.
					link.classList.add('loyalty-links-unmonitored');
				}
			} else {
				// Internal links are always functional.
				link.classList.add('loyalty-links-internal');
			}
		}
		
		// Add class to body to indicate processing is complete (for CSS).
		document.body.classList.add('loyalty-links-processed');
		
		// Debug log (remove in production if desired).
		if (console && console.log) {
			console.log('Loyalty Links: Processed ' + processedCount + ' links, broke ' + brokenCount + ' external links.');
			console.log('Loyalty Links: Approved domains:', approvedDomains);
			console.log('Loyalty Links: Test domains:', testDomains);
			console.log('Loyalty Links: Monitored domains:', monitoredDomains);
		}
	}
	
	/**
	 * Track referrer via REST API endpoint (cache-friendly).
	 * This runs once per page load to track the referrer.
	 */
	function trackReferrer() {
		// Get referrer from browser.
		var referrer = document.referrer;
		
		// Skip if no referrer.
		if (!referrer || referrer.trim() === '') {
			return;
		}
		
		// Extract referrer domain and check if it's external.
		var referrerDomain = extractDomain(referrer);
		var currentDomain = normalizeDomain(window.location.hostname);
		
		// Skip if referrer is from same domain (internal).
		if (!referrerDomain || referrerDomain === currentDomain || referrerDomain === siteDomain) {
			return;
		}
		
		// Skip if REST API URL not available.
		if (!restUrlTrack || !restNonce) {
			return;
		}
		
		// Use fetch API for modern browsers.
		if (typeof fetch !== 'undefined') {
			fetch(restUrlTrack, {
				method: 'POST',
				headers: {
					'Content-Type': 'application/json',
					'X-WP-Nonce': restNonce
				},
				credentials: 'same-origin',
				body: JSON.stringify({
					referrer: referrer
				})
			})
			.then(function(response) {
				if (!response.ok) {
					throw new Error('Network response was not ok');
				}
				return response.json();
			})
			.then(function(data) {
				// Debug log (optional).
				if (console && console.log && data && data.success) {
					console.log('Loyalty Links: Referrer tracked:', referrer);
				}
			})
			.catch(function(error) {
				// Silently fail - tracking is not critical for functionality.
				if (console && console.warn) {
					console.warn('Loyalty Links: Error tracking referrer:', error);
				}
			});
		} else {
			// Fallback for older browsers using XMLHttpRequest.
			var xhr = new XMLHttpRequest();
			xhr.open('POST', restUrlTrack, true);
			xhr.setRequestHeader('Content-Type', 'application/json');
			xhr.setRequestHeader('X-WP-Nonce', restNonce);
			xhr.onreadystatechange = function() {
				if (xhr.readyState === 4) {
					if (xhr.status === 200) {
						try {
							var data = JSON.parse(xhr.responseText);
							if (console && console.log && data && data.success) {
								console.log('Loyalty Links: Referrer tracked:', referrer);
							}
						} catch (e) {
							// Silently fail.
						}
					}
				}
			};
			xhr.send(JSON.stringify({ referrer: referrer }));
		}
	}
	
	/**
	 * Fetch approved domains from REST API endpoint.
	 * This is called dynamically to be cache-friendly.
	 *
	 * @param {Function} callback Function to call after domains are loaded.
	 */
	function fetchApprovedDomains(callback) {
		if (!restUrlApproved) {
			console.warn('Loyalty Links: REST API URL not available.');
			if (callback) {
				callback();
			}
			return;
		}
		
		// Use fetch API for modern browsers.
		if (typeof fetch !== 'undefined') {
			fetch(restUrlApproved, {
				method: 'GET',
				headers: {
					'X-WP-Nonce': restNonce
				},
				credentials: 'same-origin'
			})
			.then(function(response) {
				if (!response.ok) {
					throw new Error('Network response was not ok');
				}
				return response.json();
			})
			.then(function(data) {
				if (data && data.approvedDomains && Array.isArray(data.approvedDomains)) {
					approvedDomains = data.approvedDomains;
					approvedDomainsLoaded = true;
					
					// Debug log.
					if (console && console.log) {
						console.log('Loyalty Links: Approved domains loaded via REST API:', approvedDomains);
					}
					
					if (callback) {
						callback();
					}
				} else {
					console.warn('Loyalty Links: Invalid response from REST API.');
					if (callback) {
						callback();
					}
				}
			})
			.catch(function(error) {
				console.warn('Loyalty Links: Error fetching approved domains:', error);
				if (callback) {
					callback();
				}
			});
		} else {
			// Fallback for older browsers using XMLHttpRequest.
			var xhr = new XMLHttpRequest();
			xhr.open('GET', restUrlApproved, true);
			xhr.setRequestHeader('X-WP-Nonce', restNonce);
			xhr.onreadystatechange = function() {
				if (xhr.readyState === 4) {
					if (xhr.status === 200) {
						try {
							var data = JSON.parse(xhr.responseText);
							if (data && data.approvedDomains && Array.isArray(data.approvedDomains)) {
								approvedDomains = data.approvedDomains;
								approvedDomainsLoaded = true;
								
								// Debug log.
								if (console && console.log) {
									console.log('Loyalty Links: Approved domains loaded via REST API:', approvedDomains);
								}
								
								if (callback) {
									callback();
								}
							} else {
								console.warn('Loyalty Links: Invalid response from REST API.');
								if (callback) {
									callback();
								}
							}
						} catch (e) {
							console.warn('Loyalty Links: Error parsing REST API response:', e);
							if (callback) {
								callback();
							}
						}
					} else {
						console.warn('Loyalty Links: REST API request failed with status:', xhr.status);
						if (callback) {
							callback();
						}
					}
				}
			};
			xhr.send();
		}
	}
	
	/**
	 * Initialize when DOM is ready.
	 */
	function init() {
		// Track referrer immediately (doesn't need to wait for DOM).
		trackReferrer();
		
		// Wait for DOM to be ready, then fetch approved domains and process links.
		var domReady = function() {
			// Fetch approved domains from REST API (cache-friendly).
			fetchApprovedDomains(function() {
				// Process links after approved domains are loaded.
				processLinks();
			});
		};
		
		// Run immediately if DOM is already loaded.
		if (document.readyState === 'loading') {
			document.addEventListener('DOMContentLoaded', domReady);
		} else {
			domReady();
		}
		
		// Also process dynamically added content (for AJAX-loaded content).
		// Use MutationObserver for better performance than polling.
		if (window.MutationObserver) {
			var observer = new MutationObserver(function(mutations) {
				var shouldProcess = false;
				mutations.forEach(function(mutation) {
					if (mutation.addedNodes.length > 0) {
						// Check if any added nodes contain links.
						for (var i = 0; i < mutation.addedNodes.length; i++) {
							var node = mutation.addedNodes[i];
							if (node.nodeType === 1) { // Element node
								if (node.tagName === 'A' || node.querySelector('a')) {
									shouldProcess = true;
									break;
								}
							}
						}
					}
				});
				
				if (shouldProcess && approvedDomainsLoaded) {
					// Debounce: wait a bit before processing to avoid excessive calls.
					clearTimeout(window.loyaltyLinksTimeout);
					window.loyaltyLinksTimeout = setTimeout(processLinks, 100);
				}
			});
			
			observer.observe(document.body, {
				childList: true,
				subtree: true
			});
		}
	}
	
	// Start initialization.
	init();
	
})();
