// Function to parse stack trace and retrieve source, line number and column number
function logtasticParseStackTrace(stack) {
    
	const lines = stack.split('\n');

    for (let line of lines) {
        
		// Remove leading whitespace and 'at' if present
        line = line.trim().replace(/^at\s+/, '');

        // Match common stack formats:
        // - functionName@protocol://...:line:column
        // - at protocol://...:line:column
        // - protocol://...:line:column
        const match = line.match(/((?:[a-z]+):\/\/.+):(\d+):(\d+)/i);
        
		if (match) {
        
			const [, source, lineStr, columnStr] = match;
            const lineNum = Number(lineStr);
            const colNum = Number(columnStr);

            if (Number.isFinite(lineNum) && Number.isFinite(colNum)) {
                return {
                    source,
                    line: lineNum,
                    column: colNum
                };
            }
        }
		
    }

    return null; // No valid match found
}

// Catch runtime errors
window.onerror = function (message, source, lineno, colno, error) {
	const errorData = {
		type: 'runtime',
		message: message,
		source: source,
		line: lineno,
		column: colno,
		stack: error && error.stack ? error.stack : null,
		userAgent: navigator.userAgent,
		url: window.location.href,
	};

	logtasticSendJSError(errorData);
};

// Catch unhandled Promise rejections
window.addEventListener('unhandledrejection', function (event) {

	let errorData = {
		type: 'promiseRejection',
		message: event.reason ? event.reason.message || String(event.reason) : 'Unhandled rejection',
		stack: event.reason && event.reason.stack ? event.reason.stack : null,
		userAgent: navigator.userAgent,
		url: window.location.href,
	};

	// Parse stack trace for source, line number nad column number
	if ( event.reason && event.reason.stack ) {

		const parsed = logtasticParseStackTrace( event.reason.stack );

		if ( parsed ) {
			errorData.source = parsed.source;
			errorData.line = parsed.line;
			errorData.column = parsed.column;
		}

	}

	logtasticSendJSError(errorData);
	
});

// Error Sending Function
function logtasticSendJSError(errorData) {

	fetch(logtasticEnvData.ajax_url, {
		method: 'POST',
		headers: {
			'Content-Type': 'application/x-www-form-urlencoded'
		},
		body: new URLSearchParams({
			action: 'logtastic_log_js_error',
			_ajax_nonce: logtasticEnvData.log_js_error_nonce,
			data: JSON.stringify(errorData) // serialize if it's an object
		})
	});
}