<?php

/**
 * Functionality to create an mu loader file.
 *
 * @link              https://logtastic.net/
 * @since             1.0.0
 * @package           Logtastic
 * @author            Inspired Plugins
 * @copyright         2025 Morley Digital Limited
 * @license           GPL-2.0-or-later
 */

namespace Inspired_Plugins\Logtastic;

// If this file is called directly, abort.
if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly
}

class Logtastic_MU_Loader_Utilities {

	/**
	 * Check if MU Loader file exists and attempt to create
	 *
	 * @since      	  1.0.0
	 * @param string  $loader_name		The file name of the loader file
	 * @param string  $loader_content	The content for the loader file
	 */
	public function create_mu_loader($loader_name, $loader_content) {

		global $wp_filesystem;

		// Initialize WP_Filesystem if not already set
		if ( ! function_exists( 'request_filesystem_credentials' ) ) {
			require_once ABSPATH . 'wp-admin/includes/file.php';
		}

		if ( ! $wp_filesystem || ! is_object( $wp_filesystem ) ) {
			WP_Filesystem();
		}
		
		// Define file path for loader file
		$loader_file_path = WPMU_PLUGIN_DIR . '/' . $loader_name;
		
		// Check if file already already exists and content is the same as defined content
		if ( $wp_filesystem->exists( $loader_file_path ) ) {
			// Read contents
			$existing_content = $wp_filesystem->get_contents( $loader_file_path );
			
			if ( $existing_content !== false && md5( $loader_content ) === md5( $existing_content ) ) {
				return true;
			}
		}
		
		// If MU Plugin Dir doesn't exist, attempt to create it
		if ( !$wp_filesystem->is_dir( WPMU_PLUGIN_DIR ) ) {
			
			$created_mu_directory = $wp_filesystem->mkdir( WPMU_PLUGIN_DIR );
			
			if ( !$created_mu_directory ) {
				return false;
			}
			
		}
		
		// Check if the MU Plugin Dir is writable, return false if not
		if ( !$wp_filesystem->is_writable( WPMU_PLUGIN_DIR ) ) {
			return false;
		}
		
		// Attempt to create file and write contents
		$created_loader_file = $wp_filesystem->put_contents(
			$loader_file_path,
			$loader_content,
			FS_CHMOD_FILE // Use WordPress's recommended file permissions
		);
		
		// If file created, return true
		if ( $created_loader_file ) {
			return true;
		}

		return false;
		
	}


	/**
	 * Check if MU Loader file exists and attempt to delete
	 *
	 * @since      	  1.0.0
	 * @param string  $loader_name		The file name of the loader file
	 */
	public function delete_mu_loader($loader_name) {

		global $wp_filesystem;

		// Initialize WP_Filesystem if not already set
		if ( ! function_exists( 'request_filesystem_credentials' ) ) {
			require_once ABSPATH . 'wp-admin/includes/file.php';
		}

		if ( ! $wp_filesystem || ! is_object( $wp_filesystem ) ) {
			WP_Filesystem();
		}
		
		// Define file path for loader file
		$loader_file_path = WPMU_PLUGIN_DIR . '/' . $loader_name;
		
		// If file doesn't exist, return true
		if ( !$wp_filesystem->exists( $loader_file_path ) ) {
			return true;
		}
		
		// Attempt to delete file
		$deleted = wp_delete_file( $loader_file_path );	

		// If deleted, return true
		if ( $deleted ) {
			return true;
		}

		return false;
		
	}


	/**
	 * Define MU Loader file content
	 *
	 * @since      	  1.0.0
	 * @param string  $plugin_base_name		The base name (slug) of the plugin
	 */
	public function build_loader_content($plugin_base_name) {
		
		$loader = "<?php
/** 
 * Logtastic - MU Loader
 * 
 * This mu loader file ensures that the main Logtastic plugin is loaded ahead of all other plugins and themes, 
 * this ensures that all errors and actions generated by other plugins and themes are logged accurately.
 * This is automatically generated by the Logtastic plugin during installation. 
 * It is automatically disabled when disabling the main plugin. 
 * It is automatically removed when uninstalling the main plugin.
 *
 * @link              https://logtastic.net/
 * @since             1.0.0
 * @package           Logtastic
 * @author            Inspired Plugins
 * @copyright         2024 Morley Digital Limited
 * @license           GPL-2.0-or-later
 *
 * @wordpress-plugin
 * Plugin Name: 	Logtastic - MU Loader
 * Plugin URI:		https://logtastic.net/
 * Description:		This mu loader file ensures that the main Logtastic plugin is loaded ahead of all other plugins and themes, his ensures that all errors and actions generated by other plugins and themes are logged accurately.
 * Version:			1.0.0
 * Author:			Inspired Plugins
 * Author URI:		https://inspiredplugins.com/
 * License:			GPL v2 or later
 * License URI:		https://www.gnu.org/licenses/gpl-2.0.html
 * Text Domain: 	inspired-logtastic
 */

namespace Inspired_Plugins\Logtastic;

// If this file is called directly, or WP_PLUGIN_DIR constant is not defined, abort.
if ( ! defined( 'ABSPATH' ) || !defined( 'WP_PLUGIN_DIR' ) ) {
	exit; // Exit if accessed directly
}

// If the main plugin file exists, and the plugin is in the list of active plugins, load
if ( file_exists( WP_PLUGIN_DIR . '/{$plugin_base_name}' ) ) {
	if ( in_array( '{$plugin_base_name}', (array) get_option('active_plugins') ) ) {
		// Define constant to show that plugin was loaded via MU Loader	
		define( __NAMESPACE__ . '\PLUGIN_MU_LOADED', true );
		// Load the plugin
		include_once WP_PLUGIN_DIR . '/{$plugin_base_name}';
	}
}
";
		
		return $loader;
		
	}

}